/* $Id: CompoundType.java,v 1.5 2001/05/26 07:10:24 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.EOFException;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.ListIterator;

/**
 * The abstract implementation of an ASN.1 compound type.<p>
 *
 * This class is the superclass of SEQUENCE, SET, SEQUENCE OF and SET OF.<p>
 *
 * @version $Revision: 1.5 $
 * @author  Raif S. Naffah
 */
public abstract class CompoundType extends Type implements IIterativeType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(CompoundType.class.getName());

	/**
	 * The compound type components. Each element
	 */
	protected ArrayList components;

	// Constructor(s)
	// .......................................................................

	public CompoundType(String name, Tag tag) {
		super(name, tag == null ? null : new Tag(tag.getClazz(), tag.getValue(), tag.isExplicit(), true));
		this.initInternal();
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Encodes a compound type to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

      if (!this.isBlank())
         os.encodeStructure(this);
      else if (this.isOptional()) // do nothing. it's cool
         cat.warn("Skipping "+cn+" cause blank...");
      else
         throw new BlankElementException(cn);

		cat.debug("<== "+cn+".encode()");
	}

	/**
	 * Resets the elements of the current value of this instance to their
	 * initial state.
	 */
	public void reset() {
		((ArrayList) value).clear();
	}

	/**
	 * Returns true if the value of this instance is not set.
	 *
	 * @return true if this instance's value is not set.
	 */
	public boolean isBlank() {
		return ((ArrayList) value).isEmpty();
	}

	// IIterativeType interface implementation
	// .......................................................................

	/**
	 * Returns a ListIterator instance capable of iterating over this compund
	 * type element's instance(s).
	 *
	 * @return a java.util.ListIterator over the instance(s) of the element of
	 * this compound type.
	 */
	public ListIterator iterator() {
		return ((ArrayList) value).listIterator();
	}

	/**
	 * Returns the count of elements in the collection contained in a compound
	 * type.
	 *
	 * @return the count of elements in the underlying collection of a compound
	 * type.
	 */
	public int size() {
	   return (components == null) ? 0 : components.size();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Initialises the components of this instance.
	 */
	protected void initInternal() {
		components = new ArrayList();
		value = new ArrayList();
	}

	/**
	 * Returns a deep copy of this instance's components.
	 *
	 * @return a deep copy of this instance's components.
	 */
	protected ArrayList cloneComponents() {
		ArrayList result = null;
		try {
			ByteArrayOutputStream baos = new ByteArrayOutputStream(4096);
			ObjectOutputStream oos = new ObjectOutputStream(baos);

			oos.writeObject(components);

			oos.flush();
			baos.close();

			byte[] buffer = baos.toByteArray();

			ByteArrayInputStream bais = new ByteArrayInputStream(buffer);
			ObjectInputStream ois = new ObjectInputStream(bais);

			result = (ArrayList) ois.readObject();

			bais.close();
			buffer = null;
		} catch (IOException x) {
			cat.error("Got an IOException ("+x.getMessage()+") while cloning this instance's components. Returning null...");
		} catch (ClassNotFoundException x) {
			cat.error("Got a ClassNotFoundException ("+x.getMessage()+") while cloning this instance's components. Returning null...");
		}

		return result;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      if (obj == null)
         return false;

      if (!(obj instanceof ArrayList))
         return false;

      if (((ArrayList) obj).size() != this.size())
         return false;

		Iterator src = ((ArrayList) obj).iterator();
		Iterator dst = this.iterator();
		IType ts;
		IType td;
		while (src.hasNext() && dst.hasNext()) {
			ts = (IType) src.next();
			td = (IType) dst.next();
			if (!ts.equals(td))
			   return false;
		}

      return true;
   }
}