/* $Id: PackageProperties.java,v 1.1.1.1 2001/02/24 04:58:59 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.encoding;

import java.io.File;
import java.io.InputStream;
import java.io.PrintWriter;
import java.io.PrintStream;
import java.util.Enumeration;
import java.util.Properties;

/**
 * A class that acts as a central repository for a package specific properties.
 * It reads a Java properties file supposed to contain package-specific
 * properties. By convention, this properties file is assumed to be named:
 * <i>package</i>.properties, where <i>package</i> is the package name of
 * this class with the dots replaced by underscores; eg. if this class is
 * defined as a.b.c.PackageProperties, then its corresponding properties
 * file is assumed to be "a_b_c.properties" accessible from a root directory
 * visible from the CLASSPATH of the current thread running this class in the
 * file hierarchy: a/b/c/. Note that if there were more than one instance of
 * this file, under more that one such root, the order in which the JVM/Java
 * interpreter searches the nodes of the CLASSPATH will dictate which instance
 * will be loaded/used.<p>
 *
 * A practical way to allow modification of a package properties file after
 * it is jarred/zipped and deployed, is to define a special directory, say
 * "properties" for example, and include it in the CLASSPATH, before the jar/
 * zip file containing the distribution.
 *
 * @version $Revision: 1.1.1.1 $
 * @author  David Hopwood
 * @author  Jill Baker
 * @author  Raif S. Naffah
 */
class PackageProperties {

	// Constants and variables with relevant static code
	// .......................................................................

	static final boolean DEBUG_PACKAGE = true;

	private static final String PACKAGE_NAME; // Blank Final
	static {
		String pname = PackageProperties.class.getName();
		pname = pname.substring(0, pname.lastIndexOf("."));
		PACKAGE_NAME = pname;
   }

	/**
	 * Default properties in case .properties file was not found.
	 */
	private static final String[][] DEFAULT_PROPERTIES = {
		{ "Trace.*",       "false" },
		{ "Debug.Level.*", "0" }
	};

	private static final Properties properties; // Blank Final
	static {
		if (DEBUG_PACKAGE)
			System.err.println("*** Looking for "+PACKAGE_NAME+" properties");

		StringBuffer sb = new StringBuffer();

		sb.append(PACKAGE_NAME.replace('.', '_'));
		sb.append(".properties");

		String it = "/properties/" + sb.toString();
		if (DEBUG_PACKAGE)
			System.err.println("*** Loading "+it);

		boolean ok = false;
		Properties result = new Properties();
		try {
			InputStream is = PackageProperties.class.getResourceAsStream(it);
			result.load(is);
         ok = true;
			is.close();
			if (DEBUG_PACKAGE)
				System.out.println("*** Properties file loaded OK...");
		} catch (Exception ignored) {
		}

		if (!ok) {
			if (DEBUG_PACKAGE) {
				System.err.println("*** WARNING: Unable to load \""+it+"\" from CLASSPATH.");
				System.err.println("*** Will use default values instead...");
			}

			for (int i = 0, limit = DEFAULT_PROPERTIES.length; i < limit; i++)
				result.put(DEFAULT_PROPERTIES[i][0], DEFAULT_PROPERTIES[i][1]);

			if (DEBUG_PACKAGE)
				System.err.println("*** Default properties now set...");
		}

      properties = result;
	}

	// Constructor(s)
	// .......................................................................

	/**
	 * Trivial constructor to enforce Singleton pattern.
	 */
	private PackageProperties() {
		super();
	}


	// Properties methods (excluding load and save, which are deliberately not
	// supported).
	// .......................................................................

	/**
	 * Returns the value of a property for this algorithm.
	 */
	public static String getProperty(String key) {
		return properties.getProperty(key);
	}

	/**
	 * Returns the value of a property for this algorithm, or return
	 * <i>value</i> if the property was not set.
	 */
	public static String getProperty(String key, String value) {
		return properties.getProperty(key, value);
	}

	/**
	 * Lists algorithm properties to the PrintStream <i>out</i>.
	 */
	public static void list(PrintStream out) {
		list(new PrintWriter(out, true));
	}

	/**
	 * Lists algorithm properties to the PrintWriter <i>out</i>.
	 */
	public static void list(PrintWriter out) {
		out.println("#");
		out.println("# ----- Begin "+PACKAGE_NAME+" properties -----");
		out.println("#");
		String key, value;
		Enumeration enum = properties.propertyNames();
		while (enum.hasMoreElements()) {
			key = (String) enum.nextElement();
			value = getProperty(key);
			out.println(key + " = " + value);
		}

		out.println("#");
		out.println("# ----- End "+PACKAGE_NAME+" properties -----");
	}

	public static Enumeration propertyNames() {
		return properties.propertyNames();
	}


	// Developer support: Tracing and debugging enquiry methods (package-private)
	// .......................................................................

	/**
	 * Returns true if tracing is requested for a given class.<p>
	 *
	 * User indicates this by setting the tracing <code>boolean</code>
	 * property for <i>label</i> in the <tt>(algorithm).properties</tt>
	 * file. The property's key is "<code>Trace.<i>label</i></code>".<p>
	 *
	 * @param label  The name of a class.
	 * @return True iff a boolean true value is set for a property with
	 *      the key <tt>Trace.<i>label</i></tt>.
	 */
	static boolean isTraceable(String label) {
		String s = getProperty("Trace."+label);
		if (s == null)
			return false;

		return new Boolean(s.trim()).booleanValue();
	}

	/**
	 * Returns the debug level for a given class.<p>
	 *
	 * User indicates this by setting the numeric property with key
	 * "<tt>Debug.Level.<i>label</i></tt>".<p>
	 *
	 * If this property is not set, "<code>Debug.Level.*</code>" is looked up
	 * next. If neither property is set, or if the first property found is
	 * not a valid decimal integer, then this method returns 0.
	 *
	 * @param label  The name of a class.
	 * @return  The required debugging level for the designated class.
	 */
	static int getLevel(String label) {
		String s = getProperty("Debug.Level."+label);
		if (s == null) {
			s = getProperty("Debug.Level.*");
			if (s == null)
				return 0;
		}

		try {
			return Integer.parseInt(s.trim());
		} catch (NumberFormatException x) {
			return 0;
		}
	}

	/**
	* Returns the PrintWriter to which tracing and debugging output is to
	* be sent.<p>
	*
	* User indicates this by setting the property with key <code>Output</code>
	* to the literal <code>out</code> or <code>err</code>.<p>
	*
	* By default or if the set value is not allowed, <code>System.err</code>
	* will be used.
	*/
	static PrintWriter getOutput() {
		String name = getProperty("Output");
		return (name != null && name.trim().equals("out")) ?
			new PrintWriter(System.out, true) :
			new PrintWriter(System.err, true);
	}
}