/* $Id: ASNBoolean.java,v 1.4 2001/05/26 07:10:24 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;
import java.math.BigInteger;

/**
 * The basic implementation of an ASN.1 BOOLEAN type. The value of this type is
 * a <tt>java.lang.Boolean</tt>. A convenience method to obtain this value as
 * a <tt>java.lang.Boolean</tt> is supplied.<p>
 *
 * @version $Revision: 1.4 $
 * @author  Raif S. Naffah
 */
public class ASNBoolean extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(ASNBoolean.class.getName());

	/**
	 * The constant TRUE.
	 */
	public static final ASNBoolean TRUE = new ASNBoolean("TRUE", Boolean.TRUE);

	/**
	 * The constant FALSE.
	 */
	public static final ASNBoolean FALSE = new ASNBoolean("FALSE", Boolean.FALSE);

	// Constructor(s)
	//........................................................................

	public ASNBoolean() {
		super("", new Tag(Tag.BOOLEAN));
	}

	public ASNBoolean(String name) {
		super(name, new Tag(Tag.BOOLEAN));
	}

	public ASNBoolean(String name, Tag tag) {
		super(name, tag);
	}

	public ASNBoolean(String name, Object value) {
		this(name, new Tag(Tag.BOOLEAN), value);
	}

	/**
	 * Constructs a new instance of a BOOLEAN type, given a designated Tag and a
	 * designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value. Allowed types are:
	 * <tt>java.lang.String</tt>, <java.lang.Boolean</tt> and this type.
	 * @exception ClassCastException if the designated value is not appropriate.
	 */
	public ASNBoolean(String name, Tag tag, Object value) {
		super(name, tag);

		value(value);

		if (this.value != null)
			defaultValue(this.value);
	}

	// Class methods
	// .......................................................................

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNBoolean getInstance(String value) {
		return new ASNBoolean("", value);
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNBoolean getInstance(Boolean value) {
		return new ASNBoolean("", value);
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNBoolean getInstance(boolean value) {
		return new ASNBoolean("", new Boolean(value));
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes a BOOLEAN from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		is.mark(Integer.MAX_VALUE);
		try {
			Boolean result = is.decodeBoolean(this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Encodes an BOOLEAN to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

		Object val = this.value();
		if (val != null)
			os.encodeBoolean(this, (Boolean) val);
		else {
			val = this.defaultValue();
			if (val != null) {
				cat.warn("Encoding default value for "+cn);
				os.encodeBoolean(this, (Boolean) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
//			else {
//				cat.warn("Encoding a NULL for "+cn);
//				new Null(this.name()).encode(os);
//			}
		}

		cat.debug("<== "+cn+".encode()");
	}

	/**
	 * Sets the current value of this instance to the designated one.
	 *
	 * @param value the designated value. Allowed types are:
	 * <tt>java.lang.String</tt>, <java.lang.Boolean</tt> and this type.
	 * @exception ClassCastException if the designated value is not appropriate.
	 */
	public void setValue(Object value) {
		if (value == null)
			return;

		if (value instanceof String)
			this.value = new Boolean((String) value);
		else if (value instanceof Boolean)
			this.value = (Boolean) value;
		else if (value instanceof ASNBoolean)
			this.value = ((ASNBoolean) value).booleanValue();
		else
			throw new ClassCastException();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a java.lang.Boolean.
	 */
	public Boolean booleanValue() {
		return (Boolean) this.value;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      Boolean val = this.booleanValue();
      return (val == null) ? false : val.equals(obj);
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null)
			sb.append(String.valueOf(booleanValue()));
		else
			sb.append("N/A");

		return sb.toString();
	}
}