/* $Id: Tag.java,v 1.2 2001/05/24 19:25:52 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import java.io.Serializable;

/**
 * The representation of an ASN.1 Tag.<p>
 *
 * @version $Revision: 1.2 $
 * @author  Raif S. Naffah
 */
public class Tag implements Serializable {

	// Constants and variables
	// .......................................................................

	/**
	 * Constant values for Tag classes.
	 */
	public static final int UNIVERSAL   = 0x00;
	public static final int APPLICATION = 0x40;
	public static final int CONTEXT     = 0x80;
	public static final int PRIVATE     = 0xC0;

	/**
	 * Constant values for Tag values.
	 */
	public static final int BOOLEAN           = 0x01;
	public static final int INTEGER           = 0x02;
	public static final int BIT_STRING        = 0x03;
	public static final int OCTET_STRING      = 0x04;
	public static final int NULL              = 0x05;
	public static final int OBJECT_IDENTIFIER = 0x06;
	public static final int REAL              = 0x09;

	public static final int SEQUENCE          = 0x10; // 16
	public static final int SEQUENCE_OF       = 0x10;
	public static final int SET               = 0x11; // 17
	public static final int SET_OF            = 0x11;

	public static final int NUMERIC_STRING    = 0x12; // 18
	public static final int PRINTABLE_STRING  = 0x13; // 19
	public static final int T61_STRING        = 0x14; // 20
	public static final int VIDEOTEX_STRING   = 0x15; // 21
	public static final int IA5_STRING        = 0x16; // 22
	public static final int GRAPHIC_STRING    = 0x19; // 25
	public static final int ISO646_STRING     = 0x1A; // 26
	public static final int GENERAL_STRING    = 0x1B; // 27
	public static final int UNIVERSAL_STRING  = 0x1C; // 28
	public static final int BMP_STRING        = 0x1E; // 30

	public static final int UTC_TIME          = 0x17; // 23
	public static final int GENERALIZED_TIME  = 0x18; // 24

	int clazz; // the tag's class id
	int value; // the tag's actual value
	boolean explicit; // is it explicit or implicit?
	boolean constructed; // is it constructed?

	// Constructor(s)
	// .......................................................................

	/**
	 * Constructs an ASN.1 Tag instance.
	 *
	 * @param clazz the tag's class.
	 * @param value the tag's value.
	 * @param explicit whether this tag is explicit or implicit. Default is
	 *	explicit.
	 * @param constructed  Whether this tag is constructed or not. Default is not
	 *	constrcuted.
	 */
	public Tag(int clazz, int value, boolean explicit, boolean constructed) {
		super();

		this.clazz = clazz;
		this.value = value;
		this.explicit = explicit;
		this.constructed = constructed;
	}

	/**
	 * Convenience constructor. Constructs an ASN.1 Tag instance. The resulting
	 * tag is of a non-constructed type.
	 *
	 * @param clazz the tag's class.
	 * @param value the tag's value.
	 * @param explicit whether this tag is explicit or implicit. Default is
	 *	explicit.
	 */
	public Tag(int clazz, int value, boolean explicit) {
		this(clazz, value, explicit, false);
	}

	/**
	 * Convenience constructor. Constructs an ASN.1 Tag instance. The resulting
	 * tag is explicit, of a non-constructed type.
	 *
	 * @param clazz the tag's class.
	 * @param value the tag's value.
	 */
	public Tag(int clazz, int value) {
		this(clazz, value, true, false);
	}

	/**
	 * Convenience constructor. Constructs an ASN.1 Tag instance. The resulting
	 * tag is of class UNIVERSAL and is of a non-constructed type.
	 *
	 * @param value the tag's value.
	 * @param explicit whether this tag is explicit or implicit. Default is
	 *	explicit.
	 */
	public Tag(int value, boolean explicit) {
		this(Tag.UNIVERSAL, value, explicit, false);
	}

	/**
	 * Convenience constructor. Constructs an ASN.1 Tag instance. The resulting
	 * tag is of class UNIVERSAL, is explicit, and is of a non-constructed type.
	 *
	 * @param value the tag's value.
	 */
	public Tag(int value) {
		this(Tag.UNIVERSAL, value, true, false);
	}

	/**
	 * Private constructor for cloning purposes.
	 *
	 * @param that the tag to clone.
	 */
	private Tag(Tag that) {
		this(that.clazz, that.value, that.explicit, that.constructed);
	}

	// Cloneable interface methods
	// .......................................................................

	/**
	 * Returns a cloned instance of this.
	 *
	 * @return a cloned instance of this.
	 */
	public Object clone() {
		return new Tag(this);
	}

	// java.lang.Object overloaded methods
	// .......................................................................

	/**
	 * Tests for equality between this instance and a designated object.
	 *
	 * @param obj a designated object to test for equality with this instance.
	 * @return true if the designated object is an instance of Tag, and has the
	 * same class and value as this one.
	 */
	public boolean equals(Object obj) {
		if (!(obj instanceof Tag))
			return false;

		Tag that = (Tag) obj;
		if (this.getClazz() != that.getClazz())
			return false;

		return (this.getValue() == that.getValue());
	}

	// Accessor methods
	// .......................................................................

	/**
	 * Returns the tag's class.
	 *
	 * @return the tag's class.
	 */
	public int getClazz() {
		return clazz;
	}

	/**
	 * Returns the tag's class number.
	 *
	 * @return the tag's class number.
	 */
	public int getValue() {
		return value;
	}

	/**
	 * Returns true if the tag is explicit, false otherwise.
	 *
	 * @return true if the tag is explicit, false otherwise.
	 */
	public boolean isExplicit() {
		return explicit;
	}

	/**
	 * Returns true if the tag is constructed false otherwise.
	 *
	 * @return true if the tag is constructed false otherwise.
	 */
	public boolean isConstructed() {
		return constructed;
	}

	/**
	 * Convenience method. Returns true if this tag is UNIVERSAL.
	 *
	 * @return true if this tag's class is UNIVERSAL; false otherwise.
	 */
	public boolean isUniversal() {
		return (clazz == Tag.UNIVERSAL);
	}

	// Visualisation methods
	// .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer result = new StringBuffer("<Tag class=\"");
		switch (clazz) {
		case UNIVERSAL:   result.append("UNIVERSAL"); break;
		case APPLICATION: result.append("APPLICATION"); break;
		case CONTEXT:     result.append("CONTEXT"); break;
		case PRIVATE:     result.append("PRIVATE"); break;
		default:          result.append(clazz);
		}

		result.append("\" value=\"");
		if (clazz == CONTEXT)
			result.append(value);
		else
			switch (value) {
			case BOOLEAN:           result.append("BOOLEAN"); break;
			case INTEGER:           result.append("INTEGER"); break;
			case BIT_STRING:        result.append("BIT STRING"); break;
			case OCTET_STRING:      result.append("OCTET STRING"); break;
			case NULL:              result.append("NULL"); break;
			case OBJECT_IDENTIFIER: result.append("OBJECT IDENTIFIER"); break;

			case SEQUENCE:          result.append("SEQUENCE [OF]"); break;
			case SET:               result.append("SET [OF]"); break;

			case NUMERIC_STRING:    result.append("NumericString"); break;
			case PRINTABLE_STRING:  result.append("PrintableString"); break;
			case T61_STRING:        result.append("TeletexString"); break;
			case VIDEOTEX_STRING:   result.append("VideotexString"); break;
			case IA5_STRING:        result.append("IA5String"); break;
			case GRAPHIC_STRING:    result.append("GraphicString"); break;
			case ISO646_STRING:     result.append("VisibleString"); break;
			case GENERAL_STRING:    result.append("GeneralString"); break;
			case UNIVERSAL_STRING:  result.append("UniversalString"); break;
			case BMP_STRING:        result.append("BMPString"); break;

			case UTC_TIME:          result.append("UTCTime"); break;
			case GENERALIZED_TIME:  result.append("GeneralizedTime"); break;
			default:                result.append(value);
			}

		result.append("\" explicit=\"").append(String.valueOf(explicit));
		result.append("\" constructed=\"").append(String.valueOf(constructed)).append("\" />");
		return result.toString();
	}
}