/* $Id: SequenceOf.java,v 1.7 2001/06/30 01:56:29 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.ListIterator;

/**
 * The abstract implementation of the ASN.1 SEQUENCE OF type.<p>
 *
 * The value of such type is a <tt>java.util.ArrayList</tt>. This data
 * structure is supposed to hold the concrete value sets of 0 or more
 * instances of a SEQUENCE OF components. For example if this is an instance
 * of SEQUENCE OF TypeA, then the <tt>components</tt> of this SEQUENCE OF
 * contains <i>TypeA</i>, and its value is a <tt>java.util.ArrayList</tt> of 0
 * or more values of <i>TypeA</i>. The <tt>components</tt> field of this type
 * is itself a <tt>java.util.ArrayList</tt>.<p>
 *
 * A convenience method to return the value as a <tt>java.util.ArrayList</tt>
 * is supplied.<p>
 *
 * @version $Revision: 1.7 $
 * @author  Raif S. Naffah
 */
public abstract class SequenceOf extends CompoundType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(SequenceOf.class.getName());

	// Constructor(s)
	// .......................................................................

	public SequenceOf(Tag tag) {
		super("", tag);
	}

	public SequenceOf(String name, Tag tag) {
		super(name, tag);
	}

	/**
	 * Constructs a new instance of a SEQUENCE_OF type, given a designated Tag
	 * and a designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value.
	 * @exception ClassCastException if the designated value is not an ArrayList.
	 */
	public SequenceOf(String name, Tag tag, Object value) {
		super(name, tag);

		if (value != null)
			if (value instanceof ArrayList)
				value(value);
			else
				throw new ClassCastException();
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes a SEQUENCE OF from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		this.reset();

		is.mark(Integer.MAX_VALUE);
		ASNReader local = is.decodeStructure(this);
		if (local == null) {
			cat.warn(cn+" not found. Backtracking...");
			is.reset();
			throw new ElementNotFoundException(cn);
		}

		int found;
		while (true) {
			ArrayList e = cloneComponents();
			found = 0;
			cat.info("Looking for a valid "+cn+" (SEQUENCE OF) frame...");
			try {
				for (Iterator it = e.iterator(); it.hasNext(); ) {
					IType t = (Type) it.next();
					try {
						cat.info("Expecting a "+t.getClass().getName());
						t.decode(local);
						cat.info("Found a "+t.getClass().getName());
						found++;
					} catch (ElementNotFoundException x) {
						if (!t.isOptional()) {
							cat.warn(String.valueOf(x)+". Backtracking...");
							is.reset();
							throw x;
						} else
							cat.warn(String.valueOf(x)+". Ignoring...");
					}
				}

				cat.info("Eureka! Adding another frame to "+cn+"'s (SEQUENCE OF) collection...");
				this.iterator().add(e);

			} catch (EOFException x) {
				cat.warn("Exhausted the input stream...");

//				for (Iterator it = e.iterator(); it.hasNext(); )
//					if (!((Type) it.next()).isBlank()) {
            if (found > 0) {
						cat.info("Eureka! Adding last frame to "+cn+"'s (SEQUENCE OF) collection...");
						this.iterator().add(e);
						break;
					}

				break;
			}
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Sets the current value of this instance to the designated one. For a
	 * compound type, the value is expected to always be an ArrayList or an
	 * ANY with an ArrayList object as its value.
	 *
	 * @param value the new value to assign to this instance.
	 */
	public void value(Object value) {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".value()");
		cat.debug("value="+String.valueOf(value));
      cat.debug("value is a "+String.valueOf(value.getClass().getName()));
      if (value instanceof IType)
         cat.debug("Tag="+String.valueOf( ((IType) value).tag() ));

      this.reset();

      ArrayList e;
      Iterator frame;
      IType td;

      if (value instanceof IType) {
         value = ((IType) value).value();
   		cat.debug("value now="+String.valueOf(value));
         cat.debug("value is now a "+String.valueOf(value.getClass().getName()));
         if (value instanceof IType)
            cat.debug("Tag="+String.valueOf( ((IType) value).tag() ));
      }

      if (value instanceof ArrayList) {
         cat.warn("Assume a SEQUENCE OF one component...");
         frame = ((ArrayList) value).iterator();
         for (int i = 1; frame.hasNext(); i++) {
   		   cat.info("Processing frame #"+String.valueOf(i)+" of "+cn);
      		td = (IType) cloneComponents().iterator().next();
      		cat.info("Assigning value to "+td.getClass().getName());
            td.value(frame.next());
            this.iterator().add(td);
         }
      } else {
         RuntimeException x = new RuntimeException();
         x.printStackTrace(System.err);
         throw x;
      }

		cat.debug("<== "+cn+".value()");
	}

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		String cn = this.getClass().getName();
		String ls = System.getProperty("line.separator");
		StringBuffer sb = new StringBuffer(ls);
		sb.append("-- Begin ").append(cn).append(" (SEQUENCE OF)").append(ls);
		int i = 1;
		for (Iterator it = this.iterator(); it.hasNext(); ) {
			Object t = it.next();
			sb.append("-- frame #").append(i++)
				.append(" (").append(t.getClass().getName()).append("): ")
				.append(String.valueOf(t)).append(ls);
		}

		sb.append("-- End ").append(cn).append(" (SEQUENCE OF)");
		return sb.toString();
	}
}