/* $Id: ASNWriter.java,v 1.1.1.1 2001/02/24 04:59:02 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.io;

import cryptix.asn1.lang.IIterativeType;
import cryptix.asn1.lang.IType;

import java.io.IOException;
import java.io.OutputStream;
import java.math.BigInteger;
import java.util.Date;
import java.util.Properties;

/**
 * An abstract class that offers the minimal operations to encode values of
 * ASN.1 to an output stream according to some given encoding rules.<p>
 *
 * @version $Revision: 1.1.1.1 $
 * @author  Raif S. Naffah
 */
public abstract class ASNWriter extends OutputStream implements Cloneable {

	/**
	 * Creates and returns an object that encodes a stream according to the
	 * same rules as the concrete subclass.
	 *
	 * @return a new instance of an <tt>cryptix.asn1.io.ASNWriter</tt> that
	 * encodes according to the same rules as the concrete subclass on which
	 * this method is invoked.
	 */
	public abstract Object clone();

	/**
	 * Initialises this instance to encode to the designated output stream.
	 *
	 * @param os the designated output stream.
	 * @exception IllegalStateException if this instance is already initialised
	 * with an output stream. Caller should close the previous stream before
	 * invoking this method again.
	 */
	public abstract void open(OutputStream os);

	/**
	 * Encodes an ANY to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val the ANY value.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeAny(IType obj, Object val)
	throws IOException;

	/**
	 * Encodes an OBJECT IDENTIFIER to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void
	encodeObjectIdentifier(IType obj, String val)
	throws IOException;

	/**
	 * Encodes a NULL to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeNull(IType obj) throws IOException;

	/**
	 * Encodes a BOOLEAN to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeBoolean(IType obj, Boolean val)
	throws IOException;

	/**
	 * Encodes an INTEGER to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeInteger(IType obj, BigInteger val)
	throws IOException;

	/**
	 * Encodes a PrintableString to the output stream.
	 *
	 * @param tagValue the value of a Tag constant to differentiate between
	 * the different types of strings.
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeString(int tagValue, IType obj, String val)
	throws IOException;

	/**
	 * Encodes a BIT STRING to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeBitString(IType obj, byte[] val)
	throws IOException;

	/**
	 * Encodes an OCTET STRING to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 method.
	 */
	public abstract void encodeOctetString(IType obj, byte[] val)
	throws IOException;

	/**
	 * Encodes a UTCTime to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeUTCTime(IType obj, Date val)
	throws IOException;

	/**
	 * Encodes a GeneralizedTime to the output stream.
	 *
	 * @param obj the concrete ASN.1 type to encode
	 * @param val a compatible value with the designated ASN.1 type.
	 * @exception IOException if an exception occurs while executing this
	 * method.
	 */
	public abstract void encodeGeneralizedTime(IType obj, Date val)
	throws IOException;

	/**
	 * Encodes a constructed type (SEQUENCE/SET [OF]) to the underlying
	 * output stream.
	 *
	 * @param obj the compound type to encode.
	 * @exception IOException if an exception occurs while executing this
	 method.
	 */
	public abstract void encodeStructure(IIterativeType obj)
	throws IOException;
}