/* $Id: ObjectIdentifier.java,v 1.1.1.1 2001/02/24 04:59:06 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;

/**
 * The basic implementation of an ASN.1 OBJECT_IDENTIFIER type. The value of
 * such a type is a <tt>java.lang.String</tt>. A convenience method to return
 * the value as a <tt>java.lang.String</tt> is supplied.<p>
 *
 * @version $Revision: 1.1.1.1 $
 * @author  Raif S. Naffah
 */
public class ObjectIdentifier extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(ObjectIdentifier.class.getName());

	// Constructor(s)
	// .......................................................................

	public ObjectIdentifier() {
		super("", new Tag(Tag.OBJECT_IDENTIFIER));
	}

	public ObjectIdentifier(String name) {
		super(name, new Tag(Tag.OBJECT_IDENTIFIER));
	}

	public ObjectIdentifier(String name, Tag tag) {
		super(name, tag);
	}

	public ObjectIdentifier(String name, Object value) {
		this(name, new Tag(Tag.OBJECT_IDENTIFIER), value);
	}

	/**
	 * Constructs a new instance of an OBJECT_IDENTIFIER type, given a designated
	 * Tag and a designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value. Allowed types are:
	 * <tt>java.lang.String</tt>, and this type.
	 * @exception ClassCastException if the designated value is not a String.
	 */
	public ObjectIdentifier(String name, Tag tag, Object value) {
		super(name, tag);

		if (value != null)
			if (value instanceof String)
				setValue(value);
			else
				throw new ClassCastException();

		if (this.value != null)
			setDefaultValue(this.value);
	}

	// Class methods
	// .......................................................................

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ObjectIdentifier getInstance(String value) {
		return new ObjectIdentifier("", value);
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ObjectIdentifier getInstance(ObjectIdentifier value) {
		return new ObjectIdentifier("", value);
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes an OBJECT IDENTIFIER from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		is.mark(Integer.MAX_VALUE);
		try {
			String result = is.decodeObjectIdentifier(this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting input stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Encodes an OBJECT IDENTIFIER to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

		Object val = this.getValue();
		if (val != null)
			os.encodeObjectIdentifier(this, (String) val);
		else {
			val = this.getDefaultValue();
			if (val != null) {
				cat.warn("Encoding default value for "+cn);
				os.encodeObjectIdentifier(this, (String) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
			else {
				cat.warn("Encoding a NULL for "+cn);
				new Null(this.getName()).encode(os);
			}
		}

		cat.debug("<== "+cn+".encode()");
	}

	/**
	 * Sets the current value of this instance to the designated one.
	 *
	 * @param value the designated value. Allowed types are:
	 * <tt>java.lang.String</tt>, and this type.
	 * @exception ClassCastException if the designated value is not appropriate.
	 */
	public void setValue(Object value) {
		if (value == null)
			return;

		if (value instanceof String)
			this.value = (String) value;
		else if (value instanceof ObjectIdentifier)
			this.value = ((ObjectIdentifier) value).stringValue();
		else
			throw new ClassCastException();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a java.lang.String.
	 */
	public String stringValue() {
		return (String) this.value;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      String val = this.stringValue();
      return (val == null) ? false : val.equals(obj);
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null)
			sb.append(stringValue());
		else
			sb.append("N/A");

		return sb.toString();
	}
}