/* $Id: OctetString.java,v 1.1.1.1 2001/02/24 04:59:06 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;
import java.math.BigInteger;

/**
 * The basic implementation of the ASN.1 OCTET STRING type.<p>
 *
 * @version $Revision: 1.1.1.1 $
 * @author  Raif S. Naffah
 */
public class OctetString extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(OctetString.class.getName());

	// Constructor(s)
	//........................................................................

	public OctetString() {
		super("", new Tag(Tag.OCTET_STRING));
	}

	public OctetString(String name) {
		super(name, new Tag(Tag.OCTET_STRING));
	}

	public OctetString(String name, Tag tag) {
		super(name, tag);
	}

	public OctetString(String name, Object value) {
		this(name, new Tag(Tag.OCTET_STRING), value);
	}

	/**
	 * Constructs a new instance of an OCTET_STRING type, given a designated Tag
	 * and a designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value.
	 * @exception ClassCastException if the designated default value is not a byte
	 * array.
	 */
	public OctetString(String name, Tag tag, Object value) {
		super(name, tag);

		if (value != null)
			this.value = (byte[]) ((byte[]) value).clone();

		if (this.value != null)
			setDefaultValue(this.value);
	}

	// Class methods
	// .......................................................................

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 */
	public static OctetString getInstance(byte[] value) {
		return new OctetString("", value);
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes an OCTET_STRING from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		is.mark(Integer.MAX_VALUE);
		try {
			byte[] result = is.decodeOctetString(this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting input stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Encodes a OCTET_STRING to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

		Object val = this.getValue();
		if (val != null)
			os.encodeOctetString(this, (byte[]) val);
		else {
			val = this.getDefaultValue();
			if (val != null) {
				cat.info("Encoding default value for "+cn);
				os.encodeOctetString(this, (byte[]) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
			else {
				cat.warn("Encoding a NULL for "+cn);
				new Null(this.getName()).encode(os);
			}
		}

		cat.debug("<== "+cn+".encode()");
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a byte array.
	 */
	public byte[] byteArrayValue() {
		return (byte[]) this.value;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      if (obj == null)
         return false;

      byte[] other;
      try {
         other = (byte[]) obj;
      } catch (ClassCastException x) {
         return false;
      }

      byte[] val = this.byteArrayValue();
      if (val == null)
         return false;

      if (val.length != other.length)
         return false;

      for (int i = 0; i < val.length; i++)
         if (val[i] != other[i])
            return false;

      return true;
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance in hexadecimal.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null)
			sb.append("0x")
				.append(new BigInteger(1, byteArrayValue()).toString(16));
		else
			sb.append("N/A");

		return sb.toString();
	}
}