/* $Id: Factory.java,v 1.1.1.1 2001/02/24 04:58:58 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.encoding;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;

/**
 * A Factory class to locate/return registered/designated ASN.1 decoders and
 * encoders.<p>
 *
 * @version $Revision: 1.1.1.1 $
 * @author  Raif S. Naffah
 */
public class Factory {

	// Constructor(s)
	// .......................................................................

	/**
	 * Trivial private constructor to enforce Singleton pattern.
	 */
	private Factory() {
		super();
	}

   // Class methods
   // .......................................................................

	/**
	 * Returns an implementation that decodes an ASN.1 input stream according to
	 * a designated encoding rule set.
	 *
	 * @param encoding the designated name for the encoding. The concrete class
	 * is assumed to be visible with a name that is constructed from the given
	 * string by (a) converting it to lowercases, (b) changing the first
	 * character to its titlecase equivalent, and (c) appending to it the string
	 * "Decoder". For example the concrete class for the "DER" decoder would be
	 * "DerDecoder." The package where this class is assumed to be found is
	 * looked up in the properties file of this package under the key
	 * "${class}.package" where ${class} is the cosntructed name of the decoder
	 * as described before.
	 * @exception InstantiationException the designated class is either an
	 * interface or an abstract class.
	 * @exception IllegalAccessException if the designated class does not have
	 * a zero-argument public constructor.
	 * @exception ClassNotFoundException if the current class loader cannot find
	 * bytecode for the designated class.
	 * @exception ClassCastException if the designated class does not extend the
	 * ASNReader base class.
	 */
	public static ASNReader getDecoder(String encoding)
	throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		return ((ASNReader) getCodec(encoding, true));
	}

	/**
	 * Returns an implementation that encodes to an ASN.1 output stream according
	 * to a designated encoding rule set.
	 *
	 * @param encoding the designated name for the encoding. The concrete class
	 * is assumed to be visible with a name that is constructed from the given
	 * string by (a) converting it to lowercases, (b) changing the first
	 * character to its titlecase equivalent, and (c) appending to it the string
	 * "Encoder". For example the concrete class for the "DER" encoder would be
	 * "DerEncoder." The package where this class is assumed to be found is
	 * looked up in the properties file of this package under the key
	 * "${class}.package" where ${class} is the cosntructed name of the encoder
	 * as described before.
	 * @exception InstantiationException the designated class is either an
	 * interface or an abstract class.
	 * @exception IllegalAccessException if the designated class does not have
	 * a zero-argument public constructor.
	 * @exception ClassNotFoundException if the current class loader cannot find
	 * bytecode for the designated class.
	 * @exception ClassCastException if the designated class does not extend the
	 * the ASNWriter base class.
	 */
	public static ASNWriter getEncoder(String encoding)
	throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		return ((ASNWriter) getCodec(encoding, false));
	}

	/**
	 * Utility class method to instantitate a coder/decoder object from its type.
	 *
	 * @param type the type of the encoding.
	 * @param decoder true if the decoder object is required, false if the encoder
	 * object is required.
	 * @return an instance of designated Codec type.
	 */
	private static Object getCodec(String type, boolean decoder)
	throws InstantiationException, IllegalAccessException, ClassNotFoundException {
		type = type.toLowerCase();
		type = Character.toTitleCase(type.charAt(0))+type.substring(1);
		String clazz = type + (decoder ? "Decoder" : "Encoder");
		String pkg = PackageProperties.getProperty(clazz+".package");
		String name = pkg+"."+clazz;

		return (Class.forName(name).newInstance());
	}
}