/* $Id: ASNReader.java,v 1.1.1.1 2001/02/24 04:59:02 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.io;

import cryptix.asn1.lang.IIterativeType;
import cryptix.asn1.lang.IType;
import java.io.InputStream;
import java.io.IOException;
import java.math.BigInteger;
import java.util.Date;

/**
 * An abstract class that offers the minimal operations to decode a
 * concrete ASN.1 stream.<p>
 *
 * @version $Revision: 1.1.1.1 $
 * @author  Raif S. Naffah
 */
public abstract class ASNReader extends InputStream {

	/**
	 * Initialises this instance to decode from the designated input stream.
	 *
	 * @param is the designated input stream to decode.
	 * @exception IllegalStateException if this instance is already initialised
	 * with an input stream. Caller should close the previous stream before
	 * invoking this method again on a new input stream.
	 */
	public abstract void open(InputStream is);

	/**
	 * Decodes an ANY from the input stream.
	 *
	 * @param name the optional user-defined ASN.1 type name.
	 * @return the concrete object decoded from the underlying input stream.
	 * @exception IOException if an I/O related exception has occured.
	 */
	public abstract IType decodeAny(String name) throws IOException;

	/**
	 * Decodes an OBJECT IDENTIFIER from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return a String representation of the OID decoded from the stream.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract String decodeObjectIdentifier(IType obj) throws IOException;

	/**
	 * Decodes a NULL from the input stream.
	 *
	 * @param obj the element to decode.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract void decodeNull(IType obj) throws IOException;

	/**
	 * Decodes a BOOLEAN from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract Boolean decodeBoolean(IType obj) throws IOException;

	/**
	 * Decodes an INTEGER from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract BigInteger decodeInteger(IType obj) throws IOException;

	/**
	 * Decodes a PrintableString from the input stream.
	 *
	 * @param tagValue the value of a Tag constant to differentiate between
	 * the different types of strings.
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract String decodeString(int tagValue, IType obj) throws IOException;

	/**
	 * Decodes a BIT STRING from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract byte[] decodeBitString(IType obj) throws IOException;

	/**
	 * Decodes an OCTET STRING from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract byte[] decodeOctetString(IType obj) throws IOException;

	/**
	 * Decodes a UTCTime from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract Date decodeUTCTime(IType obj) throws IOException;

	/**
	 * Decodes a GeneralizedTime from the input stream.
	 *
	 * @param obj the element to decode.
	 * @return the concrete value of this ASN.1 type.
	 * @exception EOFException if the end-of-stream was encountered while
	 * decoding the element.
	 * @exception IOException if any other I/O related exception has occured.
	 */
	public abstract Date decodeGeneralizedTime(IType obj) throws IOException;

   /**
    * Decodes a compund type (SEQUENCE/SET [OF]) from the input stream.
    *
    * @param obj the compound element to decode.
    * @return a Decoder that parses the input stream according to the same
    * encoding rules as this one.
    * @exception EOFException if the end-of-stream was encountered while
    * decoding the element.
    * @exception IOException if any other I/O related exception has occured.
    */
   public abstract ASNReader decodeStructure(IType obj) throws IOException;
}