/* $Id: ASNInteger.java,v 1.2 2001/05/06 05:49:28 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;
import java.math.BigInteger;

/**
 * The basic implementation of an ASN.1 INTEGER type. The value of this type is
 * a <tt>java.math.BigInteger</tt>. A convenience method to obtain this value as
 * a <tt>java.math.BigInteger</tt> is supplied.<p>
 *
 * @version $Revision: 1.2 $
 * @author  Raif S. Naffah
 */
public class ASNInteger extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(ASNInteger.class.getName());

	// Constructor(s)
	//........................................................................

	public ASNInteger() {
		super("", new Tag(Tag.INTEGER));
	}

	public ASNInteger(String name) {
		super(name, new Tag(Tag.INTEGER));
	}

	public ASNInteger(String name, Tag tag) {
		super(name, tag);
	}

	public ASNInteger(String name, Object value) {
		this(name, new Tag(Tag.INTEGER), value);
	}

	/**
	 * Constructs a new instance of an INTEGER type, given a designated Tag and a
	 * designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value. Allowed types are:
	 * <tt>java.lang.String</tt>, <java.math.BigInteger</tt> and this type.
	 * @exception ClassCastException if the designated value is neither a String
	 * nor a BigInteger.
	 */
	public ASNInteger(String name, Tag tag, Object value) {
		super(name, tag);

		setValue(value);

		if (this.value != null)
			setDefaultValue(this.value);
	}

	// Class methods
	// .......................................................................

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNInteger getInstance(String value) {
		return new ASNInteger("", new BigInteger(value));
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNInteger getInstance(BigInteger value) {
		return new ASNInteger("", value);
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNInteger getInstance(long value) {
		return new ASNInteger("", BigInteger.valueOf(value));
	}

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 * @exception ClassCastException if the designated value is not appropriate
	 * (see constructor with 3 arguments for suitable types of value).
	 */
	public static ASNInteger getInstance(ASNInteger value) {
		return new ASNInteger("", value);
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes an INTEGER from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		is.mark(Integer.MAX_VALUE);
		try {
			BigInteger result = is.decodeInteger(this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Encodes an INTEGER to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

		Object val = this.getValue();
		if (val != null)
			os.encodeInteger(this, (BigInteger) val);
		else {
			val = this.getDefaultValue();
			if (val != null) {
				cat.warn("Encoding default value for "+cn);
				os.encodeInteger(this, (BigInteger) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
			else {
				cat.warn("Encoding a NULL for "+cn);
				new Null(this.getConstructName()).encode(os);
			}
		}

		cat.debug("<== "+cn+".encode()");
	}

	/**
	 * Sets the current value of this instance to the designated one.
	 *
	 * @param value the designated value. Allowed types are:
	 * <tt>java.lang.String</tt>, <java.math.BigInteger</tt> and this type.
	 * @exception ClassCastException if the designated value is not appropriate.
	 */
	public void setValue(Object value) {
		if (value == null)
			return;

		if (value instanceof String)
			this.value = new BigInteger((String) value);
		else if (value instanceof BigInteger)
			this.value = (BigInteger) value;
		else if (value instanceof ASNInteger)
			this.value = ((ASNInteger) value).intValue();
		else
			throw new ClassCastException();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a java.math.BigInteger.
	 */
	public BigInteger intValue() {
		return (BigInteger) this.value;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      if (obj == null || !(obj instanceof BigInteger))
         return false;

      BigInteger val = this.intValue();
      return (val == null) ? false : (val.compareTo(obj) == 0);
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance in both decimal and
	 * hexadecimal systems.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null) {
			BigInteger i = intValue();
			sb.append(i.toString());
			sb.append(" [0x").append(i.toString(16)).append("]");
		} else
			sb.append("N/A");

		return sb.toString();
	}
}