/* $Id: ASNString.java,v 1.2 2001/05/06 05:49:28 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;

/**
 * The abstract implementation of all ASN.1 string types.<p>
 *
 * @version $Revision: 1.2 $
 * @author  Raif S. Naffah
 */
public abstract class ASNString extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(ASNString.class.getName());

	// Constructor(s)
	// .......................................................................

	public ASNString(String name, Tag tag) {
		super(name, tag);
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes a String from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public abstract void decode(ASNReader is) throws IOException;

	/**
	 * Encodes a PrintableString to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public abstract void encode(ASNWriter os) throws IOException;

	/**
	 * Sets the current value of this instance to the designated one.
	 *
	 * @param value the designated value. Allowed types are:
	 * <tt>java.lang.String</tt>, and any subclass of <tt>cryptix.asn1.lang.ASNString</tt>.
	 * @exception ClassCastException if the designated value is not appropriate.
	 */
	public void setValue(Object value) {
		if (value == null)
			return;

		if (value instanceof String)
			this.value = (String) value;
		else if (value instanceof ASNString)
			this.value = ((ASNString) value).stringValue();
		else
			throw new ClassCastException();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a java.lang.String.
	 */
	public String stringValue() {
		return ((String) this.value);
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      String val = this.stringValue();
      return (val == null) ? false : val.equals(obj);
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null)
			sb.append("\"").append(stringValue()).append("\"");
		else
			sb.append("N/A");

		return sb.toString();
	}

	// Protected methods invoked by subclasses
	// .......................................................................

	void decodeInternal(ASNReader is, int tagValue) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decodeInternal("+tagValue+")");

		is.mark(Integer.MAX_VALUE);
		try {
			String result = is.decodeString(tagValue, this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decodeInternal()");
	}

	void encodeInternal(ASNWriter os, int tagValue) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encodeInternal("+tagValue+")");

		Object val = this.getValue();
		if (val != null)
			os.encodeString(tagValue, this, (String) val);
		else {
			val = this.getDefaultValue();
			if (val != null) {
				cat.warn("Encoding default value for "+cn);
				os.encodeString(tagValue, this, (String) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
			else {
				cat.warn("Encoding a NULL for "+cn);
				new Null(this.getConstructName()).encode(os);
			}
		}

		cat.debug("<== "+cn+".encodeInternal()");
	}
}