/* $Id: Sequence.java,v 1.3 2001/04/29 01:15:23 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.ListIterator;

/**
 * The basic implementation of the ASN.1 SEQUENCE type.<p>
 *
 * The value of such type is a (shallow) copy of the components of the
 * SEQUENCE. It is returned as a <tt>java.util.ArrayList</tt>, and is supposed
 * to hold the concrete values of the constituants of the SEQUENCE. Access to
 * those constituants is provided by an implementation of the
 * <tt>cryptix.asn1.lang.IIterativeType</tt> interface --which should return a
 * <tt>java.util.Iterator</tt> instance.<p>
 *
 * @version $Revision: 1.3 $
 * @author  Raif S. Naffah
 */
public abstract class Sequence extends CompoundType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(Sequence.class.getName());

	// Constructor(s)
	// .......................................................................

	public Sequence(Tag tag) {
		super("", tag);
	}

	public Sequence(String name, Tag tag) {
		super(name, tag);
	}

	/**
	 * Constructs a new instance of a SEQUENCE type, given a designated Tag and a
	 * designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value.
	 * @exception ClassCastException if the designated value is not an ArrayList.
	 */
	public Sequence(String name, Tag tag, Object value) {
		super(name, tag);

		if (value != null)
			if (value instanceof ArrayList)
				setValue(value);
			else
				throw new ClassCastException();
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes a SEQUENCE from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		this.reset();

		is.mark(Integer.MAX_VALUE);
		ASNReader local = is.decodeStructure(this);
		if (local == null) {
			cat.warn(cn+" not found. Backtracking...");
			is.reset();
			throw new ElementNotFoundException(cn);
		}

		cat.info("Iterating through "+cn+"'s components...");
		for (Iterator it = this.iterator(); it.hasNext(); ) {
			IType t = (Type) it.next();
			try {
				cat.info("Looking for "+t.getClass().getName());
				t.decode(local);
				cat.info("Found "+t.getClass().getName());
			} catch (ElementNotFoundException x) {
				if (!t.isOptional()) {
					cat.warn(String.valueOf(x)+". Backtracking...");
					is.reset();
					throw x;
				} else
					cat.warn(String.valueOf(x)+". Ignoring...");
			}
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Resets the current value of this instance to its initial state.
	 */
	public void reset() {
		for (Iterator it = this.iterator(); it.hasNext(); )
			((IType) it.next()).reset();
	}

	/**
	 * Returns the current value of this instance. For a SEQUENCE, this is
	 * effectively a deep copy of the components' elements.
	 *
	 * @return the current value of this instance.
	 */
	public Object getValue() {
		return cloneComponents();
	}

	/**
	 * Sets the current value of this instance to the designated one. For a
	 * SEQUENCE, this operation consists of iterating through the elements of
	 * the designated value, and assigning the similar element (with the same
	 * tag) in this instance's components, the value of the element from value.
	 *
	 * @param value the new value to assign to this instance.
	 */
	public void setValue(Object value) {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".setValue("+String.valueOf(value)+")");

		this.reset();

      if (value instanceof IType)
         value = ((IType) value).getValue();

		Iterator src;
		if (value instanceof ArrayList)
		   src = ((ArrayList) value).iterator();
		else
		   src = ((ArrayList) ((Any) value).getValue()).iterator();

		Iterator dst = this.iterator();
		IType ts;
		IType td;
		while (src.hasNext() && dst.hasNext()) {
			ts = (IType) src.next();
			td = (IType) dst.next();
			// what checks should we make? remember that any term may be an ANY
			// (with a null tag)?
			cat.info("Assigning value from "+ts.getClass().getName()+" to "+td.getClass().getName());
         if (ts instanceof IIterativeType)
   			td.setValue(ts);
   	   else
   			td.setValue(ts.getValue());
		}

		cat.debug("<== "+cn+".setValue()");
	}

	/**
	 * Returns true if the value of this instance is not set.
	 *
	 * @return true if this instance's value is not set.
	 */
	public boolean isBlank() {
		boolean result = true;
		IType t;
		for (Iterator it = this.iterator(); it.hasNext(); ) {
			t = (IType) it.next();
			if (!t.isBlank()) {
				result = false;
				break;
			}
		}

		return result;
	}

	// IIterativeType interface implementation
	// .......................................................................

	/**
	 * Returns a ListIterator instance capable of iterating over the types
	 * contained this SEQUENCE's components.
	 *
	 * @return a java.util.ListIterator over the types in the components of
	 * this SEQUENCE.
	 */
	public ListIterator iterator() {
		return components.listIterator();
	}

	// other instance methods
	// .......................................................................

	/**
	 * Initialises the components of this instance.
	 */
	protected void initInternal() {
		super.initInternal();
		value = null;
	}

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		String cn = this.getClass().getName();
		String ls = System.getProperty("line.separator");
		StringBuffer sb = new StringBuffer(ls);
		sb.append("-- Begin ").append(cn).append(" (SEQUENCE)").append(ls);
		int i = 1;
		for (Iterator it = this.iterator(); it.hasNext(); ) {
			IType t = (IType) it.next();
			sb.append("--   field #").append(i++)
				.append(" (").append(t.getClass().getName()).append("): ")
				.append(String.valueOf(t)).append(ls);
		}

		sb.append("-- End ").append(cn).append(" (SEQUENCE)");
		return sb.toString();
	}
}