/* $Id: Type.java,v 1.2 2001/05/06 05:49:28 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;

/**
 * The basic implementation of any ASN.1 type; the superclass of all concrete
 * implementation of ASN.1 types in this package.<p>
 *
 * @version $Revision: 1.2 $
 * @author  Raif S. Naffah
 */
public abstract class Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(Type.class.getName());

	/**
	 * The Name of this instance.
	 */
	protected String name;

	/**
	 * If/when known, the tag object of this instance.
	 */
	protected Tag tag;

	/**
	 * The current value of this object.
	 */
	protected Object value;

	/**
	 * The default value, if any, of this object.
	 */
	protected Object defaultValue;

	/**
	 * Flag indicating if this type (when used in a compound type) is optional or
	 * not.
	 */
	protected boolean optional; // default is false

	// Constructor(s)
	// .......................................................................

	/**
	 * A constructor for a subclass of ANY with a known Tag.
	 *
	 * @param name the name of this instance.
	 * @param tag the tag object for the concrete instance.
	 */
	protected Type(String name, Tag tag) {
		super();

		this.name = new String(name);
		if (tag != null)
			this.tag = (Tag) tag.clone();
	}

	// IType interface methods
	// .......................................................................

	public String getConstructName() {
		return this.name;
	}

	public void setConstructName(String name) {
		this.name = new String(name);
	}

	public Tag getTag() {
		return this.tag;
	}

	/**
	 * Decodes an instance from an input stream. This method always throws a
	 * <tt>java.lang.RuntimeException</tt>. Subclasses should override this
	 * method.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception RuntimeException always.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		RuntimeException x = new RuntimeException("Subclass should override decode() method");
		cat.error(cn+".decode()", x);
		throw x;
	}

	/**
	 * Encodes an instance of this object to an output stream. This method
	 * always throws a <tt>java.lang.RuntimeException</tt>. Subclasses should
	 * override this method.
	 *
	 * @param is the ASN.1 stream to write to.
	 * @exception RuntimeException always.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		RuntimeException x = new RuntimeException("Subclass should override encode() method");
		cat.error(cn+".encode()", x);
		throw x;
	}

	public void reset() {
		this.value = this.defaultValue;
	}

	public Object getValue() {
		return this.value;
	}

	public void setValue(Object value) {
		this.value = value;
	}

	public Object getDefaultValue() {
		return this.defaultValue;
	}

	public void setDefaultValue(Object defaultValue) {
		this.defaultValue = defaultValue;
	}

	public boolean isOptional() {
		return this.optional;
	}

	public void setOptional(boolean optional) {
		this.optional = optional;
	}

	public boolean isBlank() {
		return (value == null);
	}

	// Overloaded java.lang.Object methods
	// .......................................................................

   /**
    * Indicates whether some other object is "equal to" this one.<p>
    *
    * The equals method implements an equivalence relation:
    * <ul>
    *    <li>It is <i>reflexive</i>: for any reference value <tt>x</tt>,
    *    <tt>x.equals(x)</tt> should return <tt>true</tt>.</li>
    *    <li>It is <i>symmetric</i>: for any reference values <tt>x</tt> and
    *    <tt>y</tt>, <tt>x.equals(y)</tt> should return <tt>true</tt> if and
    *    only if <tt>y.equals(x)</tt> returns <tt>true</tt>.</li>
    *    <li>It is <i>transitive</i>: for any reference values <tt>x</tt>,
    *    <tt>y</tt>, and <tt>z</tt>, if <tt>x.equals(y)</tt> returns <tt>true</tt>
    *    and <tt>y.equals(z)</tt> returns <tt>true</tt>, then <tt>x.equals(z)</tt>
    *    should return <tt>true</tt>.</li>
    *    <li>It is <i>consistent</i>: for any reference values <tt>x</tt> and
    *    <tt>y</tt>, multiple invocations of <tt>x.equals(y)</tt> consistently
    *    return <tt>true</tt> or consistently return <tt>false</tt>, provided
    *    no information used in equals comparisons on the object is modified.</li>
    *    <li>For any non-null reference value <tt>x</tt>, <tt>x.equals(null)</tt>
    *    should return <tt>false</tt>.</li>
    * </ul>
    *
    * While the <tt>equals()</tt> method for class <tt>Object</tt> implements
    * the most discriminating possible equivalence relation on objects; that
    * is, for any reference values <tt>x</tt> and <tt>y</tt>, this method
    * returns <tt>true</tt> if and only if <tt>x</tt> and <tt>y</tt> refer to
    * the same object (<tt>x == y</tt> has the value <tt>true</tt>), this
    * method, on the other hand, returns <tt>true</tt> iff the designated
    * object (a) is a subclass of this class, and (b) has an equal <tt>Tag</tt>
    * and content values.

    * @param obj the reference object with which to compare.
    * @return <tt>true</tt> if this object is the same as the obj argument;
    * <tt>false</tt> otherwise.
    */
   public boolean equals(Object obj) {
      if (obj == null)
         return false;

      if (!(obj instanceof Type))
         return false;

      Type that = (Type) obj;
      if (that.tag == null || !that.tag.equals(this.tag))
         return false;

      return this.sameValue(that.value);
   }

	// other instance methods
	// .......................................................................

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * This default implementation invokes the {@link java.lang.Object.equals(java.lang.Object)}
    * method of class {@link java.lang.Object}. Concrete subclasses should
    * override this method to carry on a more meaningful test than an identity
    * test.
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      if (this.value == null || obj == null)
         return false;

      return this.value.equals(obj);
   }
}