/* $Id: UTCTime.java,v 1.2 2001/05/06 05:49:28 raif Exp $
 *
 * Copyright (C) 1997-2001 The Cryptix Foundation Limited. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject to
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General Licence along with this library; if
 * not, you can download a copy from http://www.cryptix.org/
 */
package cryptix.asn1.lang;

import cryptix.asn1.io.ASNReader;
import cryptix.asn1.io.ASNWriter;
import cryptix.asn1.io.BlankElementException;
import cryptix.asn1.io.ElementNotFoundException;

import org.apache.log4j.Category;

import java.io.EOFException;
import java.io.IOException;
import java.util.Date;

/**
 * The basic implementation of an ASN.1 UTCTime type. The value of such a type
 * is a <tt>java.util.Date</tt>. A convenience method to return the value as a
 * <tt>java.util.Date</tt> is supplied.<p>
 *
 * @version $Revision: 1.2 $
 * @author  Raif S. Naffah
 */
public class UTCTime extends Type implements IType {

	// Constants and vars
	// .......................................................................

	static Category cat = Category.getInstance(UTCTime.class.getName());

	// Constructor(s)
	// .......................................................................

	public UTCTime() {
		super("", new Tag(Tag.UTC_TIME));
	}

	public UTCTime(String name) {
		super(name, new Tag(Tag.UTC_TIME));
	}

	public UTCTime(String name, Tag tag) {
		super(name, tag);
	}

	public UTCTime(String name, Date value) {
		this(name, new Tag(Tag.UTC_TIME), value);
	}

	/**
	 * Constructs a new instance of a UTCTime type, given a designated Tag and a
	 * designated initial value.
	 *
	 * @param name the name of this instance.
	 * @param tag the designated Tag value.
	 * @param value the designated initial value.
	 * @exception ClassCastException if the designated value is not a Date.
	 */
	public UTCTime(String name, Tag tag, Object value) {
		super(name, tag);

		if (value != null)
			if (value instanceof Date)
	  			setValue(new Date(((Date) value).getTime()));
			else
				throw new ClassCastException();

		if (this.value != null)
			setDefaultValue(this.value);
	}

	// Class methods
	// .......................................................................

	/**
	 * Returns a new instance of this type with a trivial name and the
	 * designated value.
	 *
	 * @param value a designated initial value for the new instance.
	 * @return a new instance with the designated value.
	 */
	public static UTCTime getInstance(Date value) {
		return new UTCTime("", value);
	}

	// Redefinition of methods in superclass Type
	// .......................................................................

	/**
	 * Decodes a UTCTime from an input stream.
	 *
	 * @param is the ASN.1 stream to read from.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void decode(ASNReader is) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".decode()");

		is.mark(Integer.MAX_VALUE);
		try {
			Date result = is.decodeUTCTime(this);
			if (result == null)
				throw new ElementNotFoundException(cn);
			else
				this.setValue(result);
		} catch (IOException x) {
			cat.warn("Exception ("+String.valueOf(x)+") encountered while decoding a "+cn);
			if (x instanceof ASNException || x instanceof EOFException) {
				cat.warn("Resetting stream...");
				is.reset();
			}
			throw x;
		}

		cat.debug("<== "+cn+".decode()");
	}

	/**
	 * Encodes a UTCTime to an output stream.
	 *
	 * @param os the ASN.1 stream to write to.
	 * @exception IOException if an exception occurs during the operation.
	 */
	public void encode(ASNWriter os) throws IOException {
		String cn = this.getClass().getName();
		cat.debug("==> "+cn+".encode()");

		Object val = this.getValue();
		if (val != null)
			os.encodeUTCTime(this, (Date) val);
		else {
			val = this.getDefaultValue();
			if (val != null) {
				cat.info("Encoding default value for "+cn);
				os.encodeUTCTime(this, (Date) val);
			} else if (!this.isOptional())
				throw new BlankElementException(cn);
			else {
				cat.warn("Encoding a NULL for "+cn);
				new Null(this.getConstructName()).encode(os);
			}
		}

		cat.debug("<== "+cn+".encode()");
	}

	// other instance methods
	// .......................................................................

	/**
	 * Convenience method to facilitate type casting.
	 *
	 * @return the current value as a <tt>java.math.BigInteger</tt>.
	 */
	public Date dateValue() {
		return (Date) this.value;
	}

   /**
    * Returns <tt>true</tt> if the designated value is equal to the value of
    * this instance.<p>
    *
    * @param obj a value to compare with the value of this instance.
    * @return <tt>true</tt> if this instance has an equal, non-null value to
    * the designated one; <tt>false</tt> otherwise.
    */
   protected boolean sameValue(Object obj) {
      Date val = this.dateValue();
      return (val == null) ? false : val.equals(obj);
   }

   // java.lang.Object overloaded methods
   // .......................................................................

	/**
	 * Returns a string representation of this instance.
	 *
	 * @return a string representation of this instance.
	 */
	public String toString() {
		StringBuffer sb = new StringBuffer("-- ");
		if (value != null)
			sb.append(String.valueOf(dateValue()));
		else
			sb.append("N/A");

		return sb.toString();
	}
}