/* $Id: DevRandom.java,v 1.1 1999/09/03 09:53:06 gelderen Exp $
 *
 * Copyright (C) 1995-1999 Systemics Ltd.
 * on behalf of the Cryptix Development Team. All rights reserved.
 *
 * Use, modification, copying and distribution of this software is subject
 * the terms and conditions of the Cryptix General Licence. You should have
 * received a copy of the Cryptix General License along with this library;
 * if not, you can download a copy from http://www.cryptix.org/ .
 */
package cryptix.random;


import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.security.SecureRandomSpi;


/**
 * @author  Jeroen C. van Gelderen (gelderen@cryptix.org)
 * @version $Revision: 1.1 $
 */
public final class DevRandom extends SecureRandomSpi
{
    /** Name of randomness device */
    private static final File RANDOM_DEV = new File("/dev/urandom");


    /** Randomness device file */
    private static FileInputStream randomStream = null;


    /**
     * Mutex for randomStream access. All object instances share a single
     * randomStream object and we need to make sure this is thread safe.
     */
    private static final Object mutex = new Object();


    /**
     * Check whether the underlying randomness device is present on the
     * system.
     */
    public static boolean isAvailable()
    {
        File f = RANDOM_DEV;
        return f.exists() && f.canRead();
    }



    public DevRandom() {
        try {
            synchronized(mutex) {
                if( randomStream == null ) {
                    randomStream = new FileInputStream(RANDOM_DEV);
                }
            }
        } catch(FileNotFoundException e) {
            throw new RuntimeException(
                "Oops, somebody removed my randomness device?");
        }
    }


    protected void engineSetSeed(byte[] seed) {
        // user seeds are ignored
    }


    protected void engineNextBytes(byte[] bytes)
    {
        try {
            int count;
            int offset = 0;
            int todo   = bytes.length;
            while( todo > 0 ) {
                synchronized(mutex) {
                    if( (count = randomStream.read(bytes, offset, todo)) == -1 )
                        throw new IOException("EOF");
                }
                offset += count;
                todo   -= count;
            }
        } catch(IOException e) {
            throw new RuntimeException(
                "Cannot read from randomness device: " + e);
        }
    }


    protected byte[] engineGenerateSeed(int numBytes) {
        byte[] seed = new byte[numBytes];
        engineNextBytes(seed);
        return seed;
    }
}