
import cryptlib.*;
import java.io.*;
import java.nio.*;

class TestCryptlib
{
	public static int keySet;
	public static int caKeyPair;

	public static void openKeyStore() throws CryptException
	{
		keySet = crypt.KeysetOpen(crypt.UNUSED, crypt.KEYSET_FILE,
							  "keyset.p15", crypt.KEYOPT_CREATE);
	}

	public static void createKeyPairAndCert(String commonName, String label,
											boolean isCa, boolean isCodeSigning)
		throws CryptException, IOException, FileNotFoundException
	{
		System.out.println("Generating key pair and cert for " + label);

		//Create key pair
		int keyPair = crypt.CreateContext(crypt.UNUSED, crypt.ALGO_RSA);
		crypt.SetAttributeString(keyPair, crypt.CTXINFO_LABEL, label);
		crypt.SetAttribute(keyPair, crypt.CTXINFO_KEYSIZE, 128);
		crypt.GenerateKey(keyPair);
		if (isCa)
			caKeyPair = keyPair;

		//Create cert
		int cert = crypt.CreateCert(crypt.UNUSED, crypt.CERTTYPE_CERTIFICATE);
		if (isCa)
		{
			crypt.SetAttribute(cert, crypt.CERTINFO_SELFSIGNED, 1);
			crypt.SetAttribute(cert, crypt.CERTINFO_CA, 1);
		}
		if (isCodeSigning)
		{
			crypt.SetAttribute(cert, crypt.CERTINFO_EXTKEY_CODESIGNING, crypt.UNUSED);
		}
		crypt.SetAttribute(cert, crypt.CERTINFO_SUBJECTPUBLICKEYINFO, keyPair);
		crypt.SetAttributeString(cert, crypt.CERTINFO_COUNTRYNAME, "us");
		crypt.SetAttributeString(cert, crypt.CERTINFO_COMMONNAME, commonName);

		//Sign cert with keyPair
		crypt.SignCert(cert, caKeyPair);

		//Add keyPair and cert chain to pkcs15 keyset (.p15)
		crypt.AddPrivateKey(keySet, keyPair, "password");
		crypt.AddPublicKey(keySet, cert);

		//Write out cert chain to pkcs7 file (.p7b)
		byte[] buffer = new byte[1000];
		int certLength = crypt.ExportCert(buffer, crypt.CERTFORMAT_CERTCHAIN, cert);
		FileOutputStream fos = new FileOutputStream(label+".p7b");
		fos.write(buffer, 0, certLength);
		fos.close();

		//Write out private key and cert chain to pkcs12 file (.pfx)
		/*

		DISABLED CAUSED THE DEFAULT CRYPTLIB DOESN'T SUPPORT .PFX

		int pfxKeySet = crypt.KeysetOpen(crypt.UNUSED, crypt.KEYSET_FILE,
										 label+".pfx", crypt.KEYOPT_CREATE);
		crypt.AddPrivateKey(pfxKeySet, keyPair, "password");
		crypt.AddPublicKey(pfxKeySet, cert);
		crypt.KeysetClose(pfxKeySet);
		*/

		//Cleanup
		crypt.DestroyContext(cert);
		crypt.DestroyContext(keyPair);

		//Re-get the CA key pair so we get it attached to the certificate
		//This is a kludge, I wish I could just attach the cert directly
		//to the keypair, but we need to do this so we can later sign with
		//this keypair and have the cert automatically added to the chain
		if (isCa)
			caKeyPair = crypt.GetPrivateKey(keySet, crypt.KEYID_NAME, label, "password");
	}

	public static void closeKeyStore() throws CryptException
	{
		crypt.DestroyContext(caKeyPair);
		crypt.KeysetClose(keySet);
	}

	static int[] tempKeyPairs;
	static int[] tempCerts;
	static int tempKeySet;

	public static void makeTempKeyPairsAndCerts(boolean closeKeySet) throws CryptException
	{
		System.out.println("Generating temporary key pairs and certs");

		tempKeyPairs = new int[2];
		tempCerts = new int[2];

		tempKeySet = crypt.KeysetOpen(crypt.UNUSED, crypt.KEYSET_FILE, "keyset.p15", crypt.KEYOPT_CREATE);

		for (int count=0; count<2; count++)
		{
			//Generate temp key pair
			int k = crypt.CreateContext(crypt.UNUSED, crypt.ALGO_RSA);
			crypt.SetAttributeString(k, crypt.CTXINFO_LABEL, "Temp Key " + Integer.toString(count));
			crypt.SetAttribute(k, crypt.CTXINFO_KEYSIZE, 128);
			crypt.GenerateKey(k);

			//Generate good-for-anything self-signed cert for key pair
			int c = crypt.CreateCert(crypt.UNUSED, crypt.CERTTYPE_CERTIFICATE);
			crypt.SetAttribute(c, crypt.CERTINFO_XYZZY, 1); //Special good-for-anything cert
			crypt.SetAttribute(c, crypt.CERTINFO_SUBJECTPUBLICKEYINFO, k);
			crypt.SetAttributeString(c, crypt.CERTINFO_COMMONNAME, "Temp Cert" + Integer.toString(count));
			crypt.SignCert(c, k);

			//Kludge
			//Must write out key pair and cert to keyset and then read them back
			//to get cert attached to keypair.  Once the key pair  has a certificate
			//attached to it, we can use it as the server certificate for TLS, for example
			crypt.AddPrivateKey(tempKeySet, k, "password");
			crypt.AddPublicKey(tempKeySet, c);
			crypt.DestroyContext(k);
			k = crypt.GetPrivateKey(tempKeySet, crypt.KEYID_NAME, "Temp Key " + Integer.toString(count), "password");

			tempKeyPairs[count] = k;
			tempCerts[count] = c;
		}
		if (closeKeySet)
			crypt.KeysetClose(tempKeySet);
	}

	public static void destroyTempKeyPairsAndCerts(boolean closeKeySet) throws CryptException
	{
		for (int count=0; count<2; count++)
		{
			crypt.DestroyContext(tempKeyPairs[count]);
			crypt.DestroyCert(tempCerts[count]);
		}
		if (closeKeySet)
			crypt.KeysetClose(tempKeySet);
	}

	public static void testTLSClient(String serverName) throws CryptException
	{
		//Open an SSL session to the server
		int session = crypt.CreateSession(crypt.UNUSED, crypt.SESSION_SSL);
		crypt.SetAttributeString(session, crypt.SESSINFO_SERVER_NAME, serverName);
		crypt.SetAttribute(session, crypt.SESSINFO_ACTIVE, 1);

		//Set to blocking
		crypt.SetAttribute(crypt.UNUSED, crypt.OPTION_NET_TIMEOUT, 30);

		//Send 10MB of data in 10K packets
		ByteBuffer text = ByteBuffer.allocateDirect(10240);
		int bytesSent = 0;
		int bytesRead = 0;
		long startTime = System.currentTimeMillis();
		for (int count=0; count<512; count++)
		{
			bytesSent += crypt.PushData(session, text, 0, 10240);
			crypt.FlushData(session);
			bytesRead += crypt.PopData(session, text, 0, 10240);
		}
		long endTime = System.currentTimeMillis();
		System.out.println("Sent 10 MB over TLS 1.0 with 3DES in " +
						   new Float((endTime-startTime)/1000.0).toString() + " seconds");
		crypt.DestroySession(session);
	}

	public static void testTLSServer() throws CryptException
	{
		//Start up a single-threaded server, using the server keyPair/cert we
		//previously generated, and block waiting for a connection
		System.out.println("Waiting for client...");
		int session = crypt.CreateSession(crypt.UNUSED, crypt.SESSION_SSL_SERVER);
		crypt.SetAttribute(session, crypt.SESSINFO_PRIVATEKEY, tempKeyPairs[0]);
		crypt.SetAttribute(session, crypt.SESSINFO_ACTIVE, 1);
		System.out.println("Connection established...");

		//Set it to blocking
		crypt.SetAttribute(crypt.UNUSED, crypt.OPTION_NET_TIMEOUT, 30);

		//Send and Receive a total of 10MB in 10KB chunks
		ByteBuffer text = ByteBuffer.allocateDirect(10240);
		int bytesSent = 0;
		int bytesRead = 0;
		for (int count=0; count<512; count++)
		{
			bytesRead += crypt.PopData(session, text, 0, 10240);
			bytesSent += crypt.PushData(session, text, 0, 10240);
			crypt.FlushData(session);
		}

		crypt.DestroySession(session);
		System.out.println("Session closed...");
	}

	public static void testCMS() throws CryptException
	{
		String plaintext = "Hi, how are you";
		byte[] buffer = new byte[32 * 1024]; //equal to cryptlib's buffer size

		//First sign with temp key 0
		System.out.println("Signing message");
		int e = crypt.CreateEnvelope(crypt.UNUSED, crypt.FORMAT_CRYPTLIB);
		crypt.SetAttribute(e, crypt.ENVINFO_SIGNATURE, tempKeyPairs[0]);
		crypt.PushData(e, plaintext);
		crypt.FlushData(e);
		int bytesCopied = crypt.PopData(e, buffer, buffer.length);
		crypt.DestroyEnvelope(e);

		//Then encrypt to temp certs 0 and 1, using AES256
		System.out.println("Encrypting message");
		e = crypt.CreateEnvelope(crypt.UNUSED, crypt.FORMAT_CRYPTLIB);

		int c = crypt.CreateContext(crypt.UNUSED, crypt.ALGO_AES);
		crypt.SetAttribute(c, crypt.CTXINFO_KEYSIZE, 32);
		crypt.GenerateKey(c);
		crypt.SetAttribute(e, crypt.ENVINFO_SESSIONKEY, c);
		crypt.DestroyContext(c);

		crypt.SetAttribute(e, crypt.ENVINFO_PUBLICKEY, tempCerts[0]);
		crypt.SetAttribute(e, crypt.ENVINFO_PUBLICKEY, tempCerts[1]);

		crypt.PushData(e, buffer, 0, bytesCopied);
		crypt.FlushData(e);
		bytesCopied = crypt.PopData(e, buffer, buffer.length);
		crypt.DestroyEnvelope(e);


		//Prepare to decrypt encrypted data
		System.out.println("Decrypting message");
		e = crypt.CreateEnvelope(crypt.UNUSED, crypt.FORMAT_AUTO);

		//It tells us we need to add an encryption attribute, via Exceptions
		//This is an unfortunate result of translating status values as
		//Exceptions, but we can live with it
		try	{ crypt.PushData(e, buffer, 0, bytesCopied); } catch (CryptException ex){}
		try { crypt.FlushData(e); }	catch (CryptException ex){}

		//Iterate through encryption attributes until we find one that matches key 1
		crypt.SetAttribute(e, crypt.ENVINFO_CURRENT_COMPONENT, crypt.CURSOR_FIRST);
		while (true)
		{
			int attributeNeeded = crypt.GetAttribute(e, crypt.ENVINFO_CURRENT_COMPONENT);
			if (attributeNeeded ==crypt.ENVINFO_PRIVATEKEY)
			{
				try
				{
					//Test if key 1 works.  If so, exit the loop
					crypt.SetAttribute(e, crypt.ENVINFO_PRIVATEKEY, tempKeyPairs[1]);
					break;
				}
				catch (CryptException ex)
				{
					//This is not the correct key
					if (ex.getStatus() != crypt.ERROR_WRONGKEY)
						throw ex;
				}
			}
			crypt.SetAttribute(e, crypt.ENVINFO_CURRENT_COMPONENT, crypt.CURSOR_NEXT);
		}
		bytesCopied = crypt.PopData(e, buffer, buffer.length);
		crypt.DestroyEnvelope(e);

		//Prepare to verify decrypted data
		System.out.println("Verifying message");
		e = crypt.CreateEnvelope(crypt.UNUSED, crypt.FORMAT_AUTO);

		//Check that it was signed by one of the keys in tempKeySet
		crypt.SetAttribute(e, crypt.ENVINFO_KEYSET_SIGCHECK, tempKeySet);

		crypt.PushData(e, buffer, 0, bytesCopied);
		crypt.FlushData(e);
		bytesCopied = crypt.PopData(e, buffer, buffer.length);

		int result = crypt.GetAttribute(e, crypt.ENVINFO_SIGNATURE_RESULT);
		if (result != crypt.OK)
			throw new CryptException(result);
		System.out.println("Decrypted/Verified data is:\n" + new String(buffer, 0, bytesCopied));
	}

	public static void testCipher(int cryptAlgo, int keySize, String algoName) throws CryptException
	{
		//Create context with random IV (if it needs one) and a key derived
		//from a password using a salt and iteration count to hinder guessing attacks
		int encContext =
		crypt.CreateContext(crypt.UNUSED, cryptAlgo);
		crypt.SetAttribute(encContext, crypt.CTXINFO_KEYSIZE, keySize);
		crypt.SetAttribute(encContext, crypt.CTXINFO_KEYING_ITERATIONS, 10000);
		crypt.SetAttributeString(encContext, crypt.CTXINFO_KEYING_SALT, "salt1234".getBytes()); //can use byte arrays
		crypt.SetAttributeString(encContext, crypt.CTXINFO_KEYING_VALUE, "password123");		//or strings

		//Encrypt 10 MB of text
		//We do this in 10 KB chunks for the sake of demonstration
		//--------------------------------------------------------------

		//Create 10MB array
		//We use a direct ByteBuffer, otherwise the entire array needs to be
		//copied to/from JNI each time, which kills performance
		int tenMegs = 10 * 1024 * 1024;
		ByteBuffer text = ByteBuffer.allocateDirect(tenMegs);

		//Encrypt each chunk
		long startTime = System.currentTimeMillis();

		for (int count=0; count < 1024; count++)
			crypt.Encrypt(encContext, text, count*1024*10, 1024*10);    //encrypt chunk

		long endTime = System.currentTimeMillis();

		//Print results
		System.out.println(algoName +" 10 MB in " +
						   new Float((endTime-startTime)/1000.0).toString() + " seconds");

		//Destroy the context, zeroing the key
		crypt.DestroyContext(encContext);
	}

	public static void main(String[] args)
	{
		System.loadLibrary("cl32"); //or whatever the shared library is called

		try
		{

			crypt.Init();

			if (args.length == 0)
			{
				System.out.println(
									"\nCommands:\n" +
									"testcerts <caName> <webServerName> <codeSignerName>\n" +
									"testtls [server|client] <serverName (if client)>\n" +
									"testcms\n" +
									"testciphers");
				return;
			}

			if (args[0].equalsIgnoreCase("testcerts"))
			{
				openKeyStore();
				createKeyPairAndCert(args[1], "ca", true, false);
				createKeyPairAndCert(args[2], "webServer", false, false);
				createKeyPairAndCert(args[3], "codeSigner", false, true);
				closeKeyStore();
			}
			else if (args[0].equalsIgnoreCase("testtls"))
			{
				makeTempKeyPairsAndCerts(true);
				if (args[1].equals("client"))
					testTLSClient(args[2]);
				else
					testTLSServer();
				destroyTempKeyPairsAndCerts(false);
			}
			else if (args[0].equalsIgnoreCase("testcms"))
			{
				makeTempKeyPairsAndCerts(false);
				testCMS();
				destroyTempKeyPairsAndCerts(true);
			}
			else if (args[0].equalsIgnoreCase("testciphers"))
			{
				testCipher(crypt.ALGO_HMAC_SHA, 32, "HMAC-SHA1   Processed");
				testCipher(crypt.ALGO_RC4, 16, 		"RC4         Encrypted");
				testCipher(crypt.ALGO_AES, 16, 		"AES-128 CBC Encrypted");
				testCipher(crypt.ALGO_AES, 32, 		"AES-256 CBC Encrypted");
				testCipher(crypt.ALGO_3DES, 24,		"3DES CBC    Encrypted");
			}
			else
				System.out.println("unrecognized parameter");
			try
			{
				crypt.End();
			}
			catch (CryptException e2)
			{
				//This'll be thrown cause we don't clean up all objects
			}
		}
		catch (CryptException e)
		{
			//Cryptlib error
			e.printStackTrace();
		}
		catch (Exception e)
		{
			e.printStackTrace();
		}
	}
};
