
    
#include <jni.h>
#include <stdio.h>  //printf
#include <stdlib.h> //malloc, free
#include "cryptlibjni.h"
#include "../cryptlib.h"


/* Helper Functions */

int processStatus(JNIEnv *env, jint status)
{
	jclass exClass;
	jmethodID exConstructor;
	jthrowable obj;

	if (status >= cryptlib_crypt_OK)
		return 1;	

	exClass = (*env)->FindClass(env, "cryptlib/CryptException");
	if (exClass == 0)
	{
		printf("cryptlib.jni.c:processStatus - no class?!\n");
		return 0;
	}

	exConstructor = (*env)->GetMethodID(env, exClass, "<init>", "(I)V");
	if (exConstructor == 0)
	{
		printf("cryptlibjni.c:processStatus - no constructor?!\n");
		return 0;
	}

	obj = (*env)->NewObject(env, exClass, exConstructor, status);
	if (obj == 0)
	{
		printf("cryptlibjni.c:processStatus - no object?!\n");
		return 0;
	}

	if ((*env)->Throw(env, obj) < 0)
	{
		printf("cryptlibjni.c:processStatus - failed to throw?!\n");
		return 0;
	}
	return 0;
}

int checkIndicesArray(JNIEnv *env, jbyteArray array, int sequenceOffset, int sequenceLength)
{
	jsize arrayLength;
	jclass exClass;

	if (array == NULL)
	{
		if (sequenceOffset == 0)
			return 1;
		else
		{
			exClass = (*env)->FindClass(env, "java/lang/ArrayIndexOutOfBoundsException");
			if (exClass == 0)
				printf("cryptlibjni.c:checkIndicesArray - no class?!\n");
			else
			if ((*env)->ThrowNew(env, exClass, "") < 0)
				printf("cryptlibjni.c:checkIndicesArray - failed to throw?!\n");
			return 0;
		}
	}

	arrayLength = (*env)->GetArrayLength(env, array);

	if (sequenceOffset < 0 || 
		sequenceOffset >= arrayLength || 
		sequenceOffset + sequenceLength > arrayLength)
	{
	    exClass = (*env)->FindClass(env, "java/lang/ArrayIndexOutOfBoundsException");
		if (exClass == 0)
			printf("cryptlibjni.c:checkIndicesArray - no class?!\n");
		else
		if ((*env)->ThrowNew(env, exClass, "") < 0)
			printf("cryptlibjni.c:checkIndicesArray - failed to throw?!\n");
		return 0;
	}
	return 1;
}

int getPointerArray(JNIEnv* env, jbyteArray array, jbyte** bytesPtrPtr)
{
	jboolean isCopy;

	if (array == NULL)
	{
		(*bytesPtrPtr) = NULL;
		return 1;
	}

	(*bytesPtrPtr) = (*env)->GetByteArrayElements(env, array, &isCopy);

	if (*bytesPtrPtr == NULL)
	{
		printf("cryptlibjni.c:getPointer - failed to get elements of array?!\n");
		return 0;
	}
	return 1;
}

void releasePointerArray(JNIEnv* env,jbyteArray array, jbyte* bytesPtr)
{
	if (bytesPtr == NULL)
		return;
	(*env)->ReleaseByteArrayElements(env, array, bytesPtr, 0);
}

int checkIndicesNIO(JNIEnv *env, jobject byteBuffer, int sequenceOffset, int sequenceLength)
{
	jlong byteBufferLength;
	jclass exClass;

	if (byteBuffer == NULL)
	{
		if (sequenceOffset == 0)
			return 1;
		else
		{
			exClass = (*env)->FindClass(env, "java/lang/ArrayIndexOutOfBoundsException");
			if (exClass == 0)
				printf("cryptlibjni.c:checkIndicesNIO - no class?!\n");
			else
			if ((*env)->ThrowNew(env, exClass, "") < 0)
				printf("cryptlibjni.c:checkIndicesNIO - failed to throw?!\n");
			return 0;
		}
	}

	byteBufferLength = (*env)->GetDirectBufferCapacity(env, byteBuffer);
    if (byteBufferLength == -1)
    {
	    exClass = (*env)->FindClass(env, "java/lang/UnsupportedOperationException");
		if (exClass == 0)
			printf("cryptlibjni.c:checkIndicesNIO - no class?!\n");
		else
		if ((*env)->ThrowNew(env, exClass,
"Either a non-direct ByteBuffer was passed or your JVM doesn't support JNI access to direct ByteBuffers") < 0)
			printf("cryptlibjni.c:checkIndicesNIO - failed to throw?!\n");
		return 0;        
    }        

	if (sequenceOffset < 0 || 
		sequenceOffset >= byteBufferLength || 
		sequenceOffset + sequenceLength > byteBufferLength)
	{
	    exClass = (*env)->FindClass(env, "java/lang/ArrayIndexOutOfBoundsException");
		if (exClass == 0)
			printf("cryptlibjni.c:checkIndicesNIO - no class?!\n");
		else
		if ((*env)->ThrowNew(env, exClass, "") < 0)
			printf("cryptlibjni.c:checkIndicesNIO - failed to throw?!\n");
		return 0;
	}
	return 1;
}

int getPointerNIO(JNIEnv* env, jobject byteBuffer, jbyte** bytesPtrPtr)
{
	jclass exClass;
    
	if (byteBuffer == NULL)
	{
		(*bytesPtrPtr) = NULL;
		return 1;
	}

	(*bytesPtrPtr) = (*env)->GetDirectBufferAddress(env, byteBuffer);

	if (*bytesPtrPtr == NULL)
	{
	    exClass = (*env)->FindClass(env, "java/lang/UnsupportedOperationException");
		if (exClass == 0)
			printf("cryptlibjni.c:getPointerNIO - no class?!\n");
		else
		if ((*env)->ThrowNew(env, exClass, "Your JVM doesn't support JNI access to direct ByteBuffers") < 0)
			printf("cryptlibjni.c:getPointerNIO - failed to throw?!\n");
		return 0; 
	}
	return 1;
}

void releasePointerNIO(JNIEnv* env,jbyteArray array, jbyte* bytesPtr)
{
}

int getPointerString(JNIEnv* env, jstring str, jbyte** bytesPtrPtr)
{
	jboolean isCopy;
	jsize strLength;
	const jbyte* rawBytesPtr;
	jclass exClass;

	if (str == NULL)
	{
		(*bytesPtrPtr) = NULL;
		return 1;
	}

	rawBytesPtr = (*env)->GetStringUTFChars(env, str, &isCopy);

	if (rawBytesPtr == NULL)
	{
		printf("cryptlibjni.c:getPointerString - failed to get elements of String?!\n");
		return 0;
	}

	strLength = (*env)->GetStringUTFLength(env, str);

	(*bytesPtrPtr) = (jbyte*)malloc(strLength+1);
	if (*bytesPtrPtr == NULL)
	{
	    exClass = (*env)->FindClass(env, "java/lang/OutOfMemoryError");
		if (exClass == 0)
			printf("cryptlibjni.c:getPointerString - no class?!\n");
		else
		if ((*env)->ThrowNew(env, exClass, "") < 0)
			printf("cryptlibjni.c:getPointerString - failed to throw?!\n");
        (*env)->ReleaseStringUTFChars(env, str, rawBytesPtr); 
		return 0;
	}

	memcpy(*bytesPtrPtr, rawBytesPtr, strLength);

	(*bytesPtrPtr)[strLength] = 0;

    (*env)->ReleaseStringUTFChars(env, str, rawBytesPtr); 

	return 1;
}

void releasePointerString(JNIEnv* env, jstring str, jbyte* bytesPtr)
{
	if (bytesPtr == NULL)
		return;
	free(bytesPtr);
}


/*
 * Class:     cryptlib_crypt
 * Method:    Init
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Init
  (JNIEnv * env, jclass cryptClass)
{
	int status = 0;
	
	status = cryptInit();
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    End
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_End
  (JNIEnv * env, jclass cryptClass)
{
	int status = 0;
	
	status = cryptEnd();
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateContext
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateContext
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint cryptAlgo)
{
	int status = 0;
	jint cryptContext = 0;
	
	status = cryptCreateContext(&cryptContext, cryptUser, cryptAlgo);
	
	processStatus(env, status);
	return(cryptContext);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DestroyContext
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DestroyContext
  (JNIEnv * env, jclass cryptClass, jint cryptContext)
{
	int status = 0;
	
	status = cryptDestroyContext(cryptContext);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DestroyObject
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DestroyObject
  (JNIEnv * env, jclass cryptClass, jint cryptObject)
{
	int status = 0;
	
	status = cryptDestroyObject(cryptObject);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GenerateKey
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_GenerateKey
  (JNIEnv * env, jclass cryptClass, jint cryptContext)
{
	int status = 0;
	
	status = cryptGenerateKey(cryptContext);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GenerateKeyAsync
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_GenerateKeyAsync
  (JNIEnv * env, jclass cryptClass, jint cryptContext)
{
	int status = 0;
	
	status = cryptGenerateKeyAsync(cryptContext);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AsyncQuery
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AsyncQuery
  (JNIEnv * env, jclass cryptClass, jint cryptObject)
{
	int status = 0;
	
	status = cryptAsyncQuery(cryptObject);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AsyncCancel
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AsyncCancel
  (JNIEnv * env, jclass cryptClass, jint cryptObject)
{
	int status = 0;
	
	status = cryptAsyncCancel(cryptObject);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Encrypt
 * Signature: (ILjava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Encrypt__ILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint cryptContext, jobject buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesNIO(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerNIO(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptEncrypt(cryptContext, bufferPtr + bufferOffset, length);
	
	finish:
	releasePointerNIO(env, buffer, bufferPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Encrypt
 * Signature: (I[BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Encrypt__I_3BII
  (JNIEnv * env, jclass cryptClass, jint cryptContext, jbyteArray buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesArray(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerArray(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptEncrypt(cryptContext, bufferPtr + bufferOffset, length);
	
	finish:
	releasePointerArray(env, buffer, bufferPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Decrypt
 * Signature: (ILjava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Decrypt__ILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint cryptContext, jobject buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesNIO(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerNIO(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptDecrypt(cryptContext, bufferPtr + bufferOffset, length);
	
	finish:
	releasePointerNIO(env, buffer, bufferPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Decrypt
 * Signature: (I[BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Decrypt__I_3BII
  (JNIEnv * env, jclass cryptClass, jint cryptContext, jbyteArray buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesArray(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerArray(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptDecrypt(cryptContext, bufferPtr + bufferOffset, length);
	
	finish:
	releasePointerArray(env, buffer, bufferPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    SetAttribute
 * Signature: (III)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_SetAttribute
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType, jint value)
{
	int status = 0;
	
	status = cryptSetAttribute(cryptHandle, attributeType, value);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    SetAttributeString
 * Signature: (IILjava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_SetAttributeString__IILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType, jobject value, jint valueOffset, jint valueLength)
{
	int status = 0;
	jbyte* valuePtr = 0;
	
	if (!checkIndicesNIO(env, value, valueOffset, valueLength))
		goto finish;
	
	if (!getPointerNIO(env, value, &valuePtr))
		goto finish;
	
	status = cryptSetAttributeString(cryptHandle, attributeType, valuePtr + valueOffset, valueLength);
	
	finish:
	releasePointerNIO(env, value, valuePtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    SetAttributeString
 * Signature: (II[BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_SetAttributeString__II_3BII
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType, jbyteArray value, jint valueOffset, jint valueLength)
{
	int status = 0;
	jbyte* valuePtr = 0;
	
	if (!checkIndicesArray(env, value, valueOffset, valueLength))
		goto finish;
	
	if (!getPointerArray(env, value, &valuePtr))
		goto finish;
	
	status = cryptSetAttributeString(cryptHandle, attributeType, valuePtr + valueOffset, valueLength);
	
	finish:
	releasePointerArray(env, value, valuePtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetAttribute
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetAttribute
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType)
{
	int status = 0;
	jint value = 0;
	
	status = cryptGetAttribute(cryptHandle, attributeType, &value);
	
	processStatus(env, status);
	return(value);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetAttributeString
 * Signature: (IILjava/nio/ByteBuffer;I)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetAttributeString__IILjava_nio_ByteBuffer_2I
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType, jobject value, jint valueOffset)
{
	int status = 0;
	jint valueLength = 0;
	jbyte* valuePtr = 0;
	
	if (!processStatus(env, cryptGetAttributeString(cryptHandle, attributeType, NULL, &valueLength)))
		goto finish;
	
	if (!checkIndicesNIO(env, value, valueOffset, valueLength))
		goto finish;
	
	if (!getPointerNIO(env, value, &valuePtr))
		goto finish;
	
	status = cryptGetAttributeString(cryptHandle, attributeType, valuePtr + valueOffset, &valueLength);
	
	finish:
	releasePointerNIO(env, value, valuePtr);
	processStatus(env, status);
	return(valueLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetAttributeString
 * Signature: (II[BI)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetAttributeString__II_3BI
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType, jbyteArray value, jint valueOffset)
{
	int status = 0;
	jint valueLength = 0;
	jbyte* valuePtr = 0;
	
	if (!processStatus(env, cryptGetAttributeString(cryptHandle, attributeType, NULL, &valueLength)))
		goto finish;
	
	if (!checkIndicesArray(env, value, valueOffset, valueLength))
		goto finish;
	
	if (!getPointerArray(env, value, &valuePtr))
		goto finish;
	
	status = cryptGetAttributeString(cryptHandle, attributeType, valuePtr + valueOffset, &valueLength);
	
	finish:
	releasePointerArray(env, value, valuePtr);
	processStatus(env, status);
	return(valueLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeleteAttribute
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeleteAttribute
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jint attributeType)
{
	int status = 0;
	
	status = cryptDeleteAttribute(cryptHandle, attributeType);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddRandom
 * Signature: (Ljava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddRandom__Ljava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jobject randomData, jint randomDataOffset, jint randomDataLength)
{
	int status = 0;
	jbyte* randomDataPtr = 0;
	
	if (!checkIndicesNIO(env, randomData, randomDataOffset, randomDataLength))
		goto finish;
	
	if (!getPointerNIO(env, randomData, &randomDataPtr))
		goto finish;
	
	status = cryptAddRandom(randomDataPtr + randomDataOffset, randomDataLength);
	
	finish:
	releasePointerNIO(env, randomData, randomDataPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddRandom
 * Signature: ([BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddRandom___3BII
  (JNIEnv * env, jclass cryptClass, jbyteArray randomData, jint randomDataOffset, jint randomDataLength)
{
	int status = 0;
	jbyte* randomDataPtr = 0;
	
	if (!checkIndicesArray(env, randomData, randomDataOffset, randomDataLength))
		goto finish;
	
	if (!getPointerArray(env, randomData, &randomDataPtr))
		goto finish;
	
	status = cryptAddRandom(randomDataPtr + randomDataOffset, randomDataLength);
	
	finish:
	releasePointerArray(env, randomData, randomDataPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportKey
 * Signature: (Ljava/nio/ByteBuffer;III)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportKey__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject encryptedKey, jint encryptedKeyOffset, jint exportKey, jint sessionKeyContext)
{
	int status = 0;
	jint encryptedKeyLength = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!processStatus(env, cryptExportKey(NULL, &encryptedKeyLength, exportKey, sessionKeyContext)))
		goto finish;
	
	if (!checkIndicesNIO(env, encryptedKey, encryptedKeyOffset, encryptedKeyLength))
		goto finish;
	
	if (!getPointerNIO(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptExportKey(encryptedKeyPtr + encryptedKeyOffset, &encryptedKeyLength, exportKey, sessionKeyContext);
	
	finish:
	releasePointerNIO(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
	return(encryptedKeyLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportKey
 * Signature: ([BIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportKey___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray encryptedKey, jint encryptedKeyOffset, jint exportKey, jint sessionKeyContext)
{
	int status = 0;
	jint encryptedKeyLength = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!processStatus(env, cryptExportKey(NULL, &encryptedKeyLength, exportKey, sessionKeyContext)))
		goto finish;
	
	if (!checkIndicesArray(env, encryptedKey, encryptedKeyOffset, encryptedKeyLength))
		goto finish;
	
	if (!getPointerArray(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptExportKey(encryptedKeyPtr + encryptedKeyOffset, &encryptedKeyLength, exportKey, sessionKeyContext);
	
	finish:
	releasePointerArray(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
	return(encryptedKeyLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportKeyEx
 * Signature: (Ljava/nio/ByteBuffer;IIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportKeyEx__Ljava_nio_ByteBuffer_2IIII
  (JNIEnv * env, jclass cryptClass, jobject encryptedKey, jint encryptedKeyOffset, jint formatType, jint exportKey, jint sessionKeyContext)
{
	int status = 0;
	jint encryptedKeyLength = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!processStatus(env, cryptExportKeyEx(NULL, &encryptedKeyLength, formatType, exportKey, sessionKeyContext)))
		goto finish;
	
	if (!checkIndicesNIO(env, encryptedKey, encryptedKeyOffset, encryptedKeyLength))
		goto finish;
	
	if (!getPointerNIO(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptExportKeyEx(encryptedKeyPtr + encryptedKeyOffset, &encryptedKeyLength, formatType, exportKey, sessionKeyContext);
	
	finish:
	releasePointerNIO(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
	return(encryptedKeyLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportKeyEx
 * Signature: ([BIIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportKeyEx___3BIIII
  (JNIEnv * env, jclass cryptClass, jbyteArray encryptedKey, jint encryptedKeyOffset, jint formatType, jint exportKey, jint sessionKeyContext)
{
	int status = 0;
	jint encryptedKeyLength = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!processStatus(env, cryptExportKeyEx(NULL, &encryptedKeyLength, formatType, exportKey, sessionKeyContext)))
		goto finish;
	
	if (!checkIndicesArray(env, encryptedKey, encryptedKeyOffset, encryptedKeyLength))
		goto finish;
	
	if (!getPointerArray(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptExportKeyEx(encryptedKeyPtr + encryptedKeyOffset, &encryptedKeyLength, formatType, exportKey, sessionKeyContext);
	
	finish:
	releasePointerArray(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
	return(encryptedKeyLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportKey
 * Signature: (Ljava/nio/ByteBuffer;III)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_ImportKey__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject encryptedKey, jint encryptedKeyOffset, jint importKey, jint sessionKeyContext)
{
	int status = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!checkIndicesNIO(env, encryptedKey, encryptedKeyOffset, 1))
		goto finish;
	
	if (!getPointerNIO(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptImportKey(encryptedKeyPtr + encryptedKeyOffset, importKey, sessionKeyContext);
	
	finish:
	releasePointerNIO(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportKey
 * Signature: ([BIII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_ImportKey___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray encryptedKey, jint encryptedKeyOffset, jint importKey, jint sessionKeyContext)
{
	int status = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!checkIndicesArray(env, encryptedKey, encryptedKeyOffset, 1))
		goto finish;
	
	if (!getPointerArray(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptImportKey(encryptedKeyPtr + encryptedKeyOffset, importKey, sessionKeyContext);
	
	finish:
	releasePointerArray(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportKeyEx
 * Signature: (Ljava/nio/ByteBuffer;III)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_ImportKeyEx__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject encryptedKey, jint encryptedKeyOffset, jint importKey, jint sessionKeyContext)
{
	int status = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!checkIndicesNIO(env, encryptedKey, encryptedKeyOffset, 1))
		goto finish;
	
	if (!getPointerNIO(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptImportKeyEx(encryptedKeyPtr + encryptedKeyOffset, importKey, sessionKeyContext);
	
	finish:
	releasePointerNIO(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportKeyEx
 * Signature: ([BIII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_ImportKeyEx___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray encryptedKey, jint encryptedKeyOffset, jint importKey, jint sessionKeyContext)
{
	int status = 0;
	jbyte* encryptedKeyPtr = 0;
	
	if (!checkIndicesArray(env, encryptedKey, encryptedKeyOffset, 1))
		goto finish;
	
	if (!getPointerArray(env, encryptedKey, &encryptedKeyPtr))
		goto finish;
	
	status = cryptImportKeyEx(encryptedKeyPtr + encryptedKeyOffset, importKey, sessionKeyContext);
	
	finish:
	releasePointerArray(env, encryptedKey, encryptedKeyPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateSignature
 * Signature: (Ljava/nio/ByteBuffer;III)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateSignature__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject signature, jint signatureOffset, jint signContext, jint hashContext)
{
	int status = 0;
	jint signatureLength = 0;
	jbyte* signaturePtr = 0;
	
	if (!processStatus(env, cryptCreateSignature(NULL, &signatureLength, signContext, hashContext)))
		goto finish;
	
	if (!checkIndicesNIO(env, signature, signatureOffset, signatureLength))
		goto finish;
	
	if (!getPointerNIO(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCreateSignature(signaturePtr + signatureOffset, &signatureLength, signContext, hashContext);
	
	finish:
	releasePointerNIO(env, signature, signaturePtr);
	processStatus(env, status);
	return(signatureLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateSignature
 * Signature: ([BIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateSignature___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray signature, jint signatureOffset, jint signContext, jint hashContext)
{
	int status = 0;
	jint signatureLength = 0;
	jbyte* signaturePtr = 0;
	
	if (!processStatus(env, cryptCreateSignature(NULL, &signatureLength, signContext, hashContext)))
		goto finish;
	
	if (!checkIndicesArray(env, signature, signatureOffset, signatureLength))
		goto finish;
	
	if (!getPointerArray(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCreateSignature(signaturePtr + signatureOffset, &signatureLength, signContext, hashContext);
	
	finish:
	releasePointerArray(env, signature, signaturePtr);
	processStatus(env, status);
	return(signatureLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateSignatureEx
 * Signature: (Ljava/nio/ByteBuffer;IIIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateSignatureEx__Ljava_nio_ByteBuffer_2IIIII
  (JNIEnv * env, jclass cryptClass, jobject signature, jint signatureOffset, jint formatType, jint signContext, jint hashContext, jint extraData)
{
	int status = 0;
	jint signatureLength = 0;
	jbyte* signaturePtr = 0;
	
	if (!processStatus(env, cryptCreateSignatureEx(NULL, &signatureLength, formatType, signContext, hashContext, extraData)))
		goto finish;
	
	if (!checkIndicesNIO(env, signature, signatureOffset, signatureLength))
		goto finish;
	
	if (!getPointerNIO(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCreateSignatureEx(signaturePtr + signatureOffset, &signatureLength, formatType, signContext, hashContext, extraData);
	
	finish:
	releasePointerNIO(env, signature, signaturePtr);
	processStatus(env, status);
	return(signatureLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateSignatureEx
 * Signature: ([BIIIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateSignatureEx___3BIIIII
  (JNIEnv * env, jclass cryptClass, jbyteArray signature, jint signatureOffset, jint formatType, jint signContext, jint hashContext, jint extraData)
{
	int status = 0;
	jint signatureLength = 0;
	jbyte* signaturePtr = 0;
	
	if (!processStatus(env, cryptCreateSignatureEx(NULL, &signatureLength, formatType, signContext, hashContext, extraData)))
		goto finish;
	
	if (!checkIndicesArray(env, signature, signatureOffset, signatureLength))
		goto finish;
	
	if (!getPointerArray(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCreateSignatureEx(signaturePtr + signatureOffset, &signatureLength, formatType, signContext, hashContext, extraData);
	
	finish:
	releasePointerArray(env, signature, signaturePtr);
	processStatus(env, status);
	return(signatureLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CheckSignature
 * Signature: (Ljava/nio/ByteBuffer;III)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_CheckSignature__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject signature, jint signatureOffset, jint sigCheckKey, jint hashContext)
{
	int status = 0;
	jbyte* signaturePtr = 0;
	
	if (!checkIndicesNIO(env, signature, signatureOffset, 1))
		goto finish;
	
	if (!getPointerNIO(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCheckSignature(signaturePtr + signatureOffset, sigCheckKey, hashContext);
	
	finish:
	releasePointerNIO(env, signature, signaturePtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CheckSignature
 * Signature: ([BIII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_CheckSignature___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray signature, jint signatureOffset, jint sigCheckKey, jint hashContext)
{
	int status = 0;
	jbyte* signaturePtr = 0;
	
	if (!checkIndicesArray(env, signature, signatureOffset, 1))
		goto finish;
	
	if (!getPointerArray(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCheckSignature(signaturePtr + signatureOffset, sigCheckKey, hashContext);
	
	finish:
	releasePointerArray(env, signature, signaturePtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CheckSignatureEx
 * Signature: (Ljava/nio/ByteBuffer;III)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CheckSignatureEx__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject signature, jint signatureOffset, jint sigCheckKey, jint hashContext)
{
	int status = 0;
	jint extraData = 0;
	jbyte* signaturePtr = 0;
	
	if (!checkIndicesNIO(env, signature, signatureOffset, 1))
		goto finish;
	
	if (!getPointerNIO(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCheckSignatureEx(signaturePtr + signatureOffset, sigCheckKey, hashContext, &extraData);
	
	finish:
	releasePointerNIO(env, signature, signaturePtr);
	processStatus(env, status);
	return(extraData);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CheckSignatureEx
 * Signature: ([BIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CheckSignatureEx___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray signature, jint signatureOffset, jint sigCheckKey, jint hashContext)
{
	int status = 0;
	jint extraData = 0;
	jbyte* signaturePtr = 0;
	
	if (!checkIndicesArray(env, signature, signatureOffset, 1))
		goto finish;
	
	if (!getPointerArray(env, signature, &signaturePtr))
		goto finish;
	
	status = cryptCheckSignatureEx(signaturePtr + signatureOffset, sigCheckKey, hashContext, &extraData);
	
	finish:
	releasePointerArray(env, signature, signaturePtr);
	processStatus(env, status);
	return(extraData);
}

/*
 * Class:     cryptlib_crypt
 * Method:    KeysetOpen
 * Signature: (IILjava/lang/String;I)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_KeysetOpen
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint keysetType, jstring name, jint options)
{
	int status = 0;
	jint keyset = 0;
	jbyte* namePtr = 0;
	
	if (!getPointerString(env, name, &namePtr))
		goto finish;
	
	status = cryptKeysetOpen(&keyset, cryptUser, keysetType, namePtr, options);
	
	finish:
	releasePointerString(env, name, namePtr);
	processStatus(env, status);
	return(keyset);
}

/*
 * Class:     cryptlib_crypt
 * Method:    KeysetClose
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_KeysetClose
  (JNIEnv * env, jclass cryptClass, jint keyset)
{
	int status = 0;
	
	status = cryptKeysetClose(keyset);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetPublicKey
 * Signature: (IILjava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetPublicKey
  (JNIEnv * env, jclass cryptClass, jint keyset, jint keyIDtype, jstring keyID)
{
	int status = 0;
	jint cryptContext = 0;
	jbyte* keyIDPtr = 0;
	
	if (!getPointerString(env, keyID, &keyIDPtr))
		goto finish;
	
	status = cryptGetPublicKey(keyset, &cryptContext, keyIDtype, keyIDPtr);
	
	finish:
	releasePointerString(env, keyID, keyIDPtr);
	processStatus(env, status);
	return(cryptContext);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetPrivateKey
 * Signature: (IILjava/lang/String;Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetPrivateKey
  (JNIEnv * env, jclass cryptClass, jint keyset, jint keyIDtype, jstring keyID, jstring password)
{
	int status = 0;
	jint cryptContext = 0;
	jbyte* keyIDPtr = 0;
	jbyte* passwordPtr = 0;
	
	if (!getPointerString(env, keyID, &keyIDPtr))
		goto finish;
	if (!getPointerString(env, password, &passwordPtr))
		goto finish;
	
	status = cryptGetPrivateKey(keyset, &cryptContext, keyIDtype, keyIDPtr, passwordPtr);
	
	finish:
	releasePointerString(env, keyID, keyIDPtr);
	releasePointerString(env, password, passwordPtr);
	processStatus(env, status);
	return(cryptContext);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddPublicKey
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddPublicKey
  (JNIEnv * env, jclass cryptClass, jint keyset, jint certificate)
{
	int status = 0;
	
	status = cryptAddPublicKey(keyset, certificate);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddPrivateKey
 * Signature: (IILjava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddPrivateKey
  (JNIEnv * env, jclass cryptClass, jint keyset, jint cryptKey, jstring password)
{
	int status = 0;
	jbyte* passwordPtr = 0;
	
	if (!getPointerString(env, password, &passwordPtr))
		goto finish;
	
	status = cryptAddPrivateKey(keyset, cryptKey, passwordPtr);
	
	finish:
	releasePointerString(env, password, passwordPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeleteKey
 * Signature: (IILjava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeleteKey
  (JNIEnv * env, jclass cryptClass, jint keyset, jint keyIDtype, jstring keyID)
{
	int status = 0;
	jbyte* keyIDPtr = 0;
	
	if (!getPointerString(env, keyID, &keyIDPtr))
		goto finish;
	
	status = cryptDeleteKey(keyset, keyIDtype, keyIDPtr);
	
	finish:
	releasePointerString(env, keyID, keyIDPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateCert
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateCert
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint certType)
{
	int status = 0;
	jint certificate = 0;
	
	status = cryptCreateCert(&certificate, cryptUser, certType);
	
	processStatus(env, status);
	return(certificate);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DestroyCert
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DestroyCert
  (JNIEnv * env, jclass cryptClass, jint certificate)
{
	int status = 0;
	
	status = cryptDestroyCert(certificate);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetCertExtension
 * Signature: (ILjava/lang/String;Ljava/nio/ByteBuffer;I)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetCertExtension__ILjava_lang_String_2Ljava_nio_ByteBuffer_2I
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jstring oid, jobject extension, jint extensionOffset)
{
	int status = 0;
	jint extensionLength = 0;
	jint criticalFlag = 0;
	jbyte* oidPtr = 0;
	jbyte* extensionPtr = 0;
	
	if (!getPointerString(env, oid, &oidPtr))
		goto finish;
	
	if (!processStatus(env, cryptGetCertExtension(cryptHandle, oidPtr, &criticalFlag, NULL, &extensionLength)))
		goto finish;
	
	if (!checkIndicesNIO(env, extension, extensionOffset, extensionLength))
		goto finish;
	
	if (!getPointerNIO(env, extension, &extensionPtr))
		goto finish;
	
	status = cryptGetCertExtension(cryptHandle, oidPtr, &criticalFlag, extensionPtr + extensionOffset, &extensionLength);
	
	finish:
	releasePointerNIO(env, extension, extensionPtr);
	releasePointerString(env, oid, oidPtr);
	processStatus(env, status);
	return(extensionLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    GetCertExtension
 * Signature: (ILjava/lang/String;[BI)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_GetCertExtension__ILjava_lang_String_2_3BI
  (JNIEnv * env, jclass cryptClass, jint cryptHandle, jstring oid, jbyteArray extension, jint extensionOffset)
{
	int status = 0;
	jint extensionLength = 0;
	jint criticalFlag = 0;
	jbyte* oidPtr = 0;
	jbyte* extensionPtr = 0;
	
	if (!getPointerString(env, oid, &oidPtr))
		goto finish;
	
	if (!processStatus(env, cryptGetCertExtension(cryptHandle, oidPtr, &criticalFlag, NULL, &extensionLength)))
		goto finish;
	
	if (!checkIndicesArray(env, extension, extensionOffset, extensionLength))
		goto finish;
	
	if (!getPointerArray(env, extension, &extensionPtr))
		goto finish;
	
	status = cryptGetCertExtension(cryptHandle, oidPtr, &criticalFlag, extensionPtr + extensionOffset, &extensionLength);
	
	finish:
	releasePointerArray(env, extension, extensionPtr);
	releasePointerString(env, oid, oidPtr);
	processStatus(env, status);
	return(extensionLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddCertExtension
 * Signature: (ILjava/lang/String;ILjava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddCertExtension__ILjava_lang_String_2ILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint certificate, jstring oid, jint criticalFlag, jobject extension, jint extensionOffset, jint extensionLength)
{
	int status = 0;
	jbyte* oidPtr = 0;
	jbyte* extensionPtr = 0;
	
	if (!getPointerString(env, oid, &oidPtr))
		goto finish;
	
	if (!checkIndicesNIO(env, extension, extensionOffset, extensionLength))
		goto finish;
	
	if (!getPointerNIO(env, extension, &extensionPtr))
		goto finish;
	
	status = cryptAddCertExtension(certificate, oidPtr, criticalFlag, extensionPtr + extensionOffset, extensionLength);
	
	finish:
	releasePointerNIO(env, extension, extensionPtr);
	releasePointerString(env, oid, oidPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    AddCertExtension
 * Signature: (ILjava/lang/String;I[BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_AddCertExtension__ILjava_lang_String_2I_3BII
  (JNIEnv * env, jclass cryptClass, jint certificate, jstring oid, jint criticalFlag, jbyteArray extension, jint extensionOffset, jint extensionLength)
{
	int status = 0;
	jbyte* oidPtr = 0;
	jbyte* extensionPtr = 0;
	
	if (!getPointerString(env, oid, &oidPtr))
		goto finish;
	
	if (!checkIndicesArray(env, extension, extensionOffset, extensionLength))
		goto finish;
	
	if (!getPointerArray(env, extension, &extensionPtr))
		goto finish;
	
	status = cryptAddCertExtension(certificate, oidPtr, criticalFlag, extensionPtr + extensionOffset, extensionLength);
	
	finish:
	releasePointerArray(env, extension, extensionPtr);
	releasePointerString(env, oid, oidPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeleteCertExtension
 * Signature: (ILjava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeleteCertExtension
  (JNIEnv * env, jclass cryptClass, jint certificate, jstring oid)
{
	int status = 0;
	jbyte* oidPtr = 0;
	
	if (!getPointerString(env, oid, &oidPtr))
		goto finish;
	
	status = cryptDeleteCertExtension(certificate, oidPtr);
	
	finish:
	releasePointerString(env, oid, oidPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    SignCert
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_SignCert
  (JNIEnv * env, jclass cryptClass, jint certificate, jint signContext)
{
	int status = 0;
	
	status = cryptSignCert(certificate, signContext);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CheckCert
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_CheckCert
  (JNIEnv * env, jclass cryptClass, jint certificate, jint sigCheckKey)
{
	int status = 0;
	
	status = cryptCheckCert(certificate, sigCheckKey);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportCert
 * Signature: (Ljava/nio/ByteBuffer;III)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ImportCert__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject certObject, jint certObjectOffset, jint certObjectLength, jint cryptUser)
{
	int status = 0;
	jint certificate = 0;
	jbyte* certObjectPtr = 0;
	
	if (!checkIndicesNIO(env, certObject, certObjectOffset, certObjectLength))
		goto finish;
	
	if (!getPointerNIO(env, certObject, &certObjectPtr))
		goto finish;
	
	status = cryptImportCert(certObjectPtr + certObjectOffset, certObjectLength, cryptUser, &certificate);
	
	finish:
	releasePointerNIO(env, certObject, certObjectPtr);
	processStatus(env, status);
	return(certificate);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ImportCert
 * Signature: ([BIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ImportCert___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray certObject, jint certObjectOffset, jint certObjectLength, jint cryptUser)
{
	int status = 0;
	jint certificate = 0;
	jbyte* certObjectPtr = 0;
	
	if (!checkIndicesArray(env, certObject, certObjectOffset, certObjectLength))
		goto finish;
	
	if (!getPointerArray(env, certObject, &certObjectPtr))
		goto finish;
	
	status = cryptImportCert(certObjectPtr + certObjectOffset, certObjectLength, cryptUser, &certificate);
	
	finish:
	releasePointerArray(env, certObject, certObjectPtr);
	processStatus(env, status);
	return(certificate);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportCert
 * Signature: (Ljava/nio/ByteBuffer;III)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportCert__Ljava_nio_ByteBuffer_2III
  (JNIEnv * env, jclass cryptClass, jobject certObject, jint certObjectOffset, jint certFormatType, jint certificate)
{
	int status = 0;
	jint certObjectLength = 0;
	jbyte* certObjectPtr = 0;
	
	if (!processStatus(env, cryptExportCert(NULL, &certObjectLength, certFormatType, certificate)))
		goto finish;
	
	if (!checkIndicesNIO(env, certObject, certObjectOffset, certObjectLength))
		goto finish;
	
	if (!getPointerNIO(env, certObject, &certObjectPtr))
		goto finish;
	
	status = cryptExportCert(certObjectPtr + certObjectOffset, &certObjectLength, certFormatType, certificate);
	
	finish:
	releasePointerNIO(env, certObject, certObjectPtr);
	processStatus(env, status);
	return(certObjectLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    ExportCert
 * Signature: ([BIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_ExportCert___3BIII
  (JNIEnv * env, jclass cryptClass, jbyteArray certObject, jint certObjectOffset, jint certFormatType, jint certificate)
{
	int status = 0;
	jint certObjectLength = 0;
	jbyte* certObjectPtr = 0;
	
	if (!processStatus(env, cryptExportCert(NULL, &certObjectLength, certFormatType, certificate)))
		goto finish;
	
	if (!checkIndicesArray(env, certObject, certObjectOffset, certObjectLength))
		goto finish;
	
	if (!getPointerArray(env, certObject, &certObjectPtr))
		goto finish;
	
	status = cryptExportCert(certObjectPtr + certObjectOffset, &certObjectLength, certFormatType, certificate);
	
	finish:
	releasePointerArray(env, certObject, certObjectPtr);
	processStatus(env, status);
	return(certObjectLength);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CAAddItem
 * Signature: (II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_CAAddItem
  (JNIEnv * env, jclass cryptClass, jint keyset, jint certificate)
{
	int status = 0;
	
	status = cryptCAAddItem(keyset, certificate);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CAGetItem
 * Signature: (IIILjava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CAGetItem
  (JNIEnv * env, jclass cryptClass, jint keyset, jint certType, jint keyIDtype, jstring keyID)
{
	int status = 0;
	jint certificate = 0;
	jbyte* keyIDPtr = 0;
	
	if (!getPointerString(env, keyID, &keyIDPtr))
		goto finish;
	
	status = cryptCAGetItem(keyset, &certificate, certType, keyIDtype, keyIDPtr);
	
	finish:
	releasePointerString(env, keyID, keyIDPtr);
	processStatus(env, status);
	return(certificate);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CADeleteItem
 * Signature: (IILjava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_CADeleteItem
  (JNIEnv * env, jclass cryptClass, jint keyset, jint keyIDtype, jstring keyID)
{
	int status = 0;
	jbyte* keyIDPtr = 0;
	
	if (!getPointerString(env, keyID, &keyIDPtr))
		goto finish;
	
	status = cryptCADeleteItem(keyset, keyIDtype, keyIDPtr);
	
	finish:
	releasePointerString(env, keyID, keyIDPtr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CACertManagement
 * Signature: (IIII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CACertManagement
  (JNIEnv * env, jclass cryptClass, jint action, jint keyset, jint caKey, jint certRequest)
{
	int status = 0;
	jint certificate = 0;
	
	status = cryptCACertManagement(&certificate, action, keyset, caKey, certRequest);
	
	processStatus(env, status);
	return(certificate);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateEnvelope
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateEnvelope
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint formatType)
{
	int status = 0;
	jint envelope = 0;
	
	status = cryptCreateEnvelope(&envelope, cryptUser, formatType);
	
	processStatus(env, status);
	return(envelope);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DestroyEnvelope
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DestroyEnvelope
  (JNIEnv * env, jclass cryptClass, jint envelope)
{
	int status = 0;
	
	status = cryptDestroyEnvelope(envelope);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    CreateSession
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_CreateSession
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint formatType)
{
	int status = 0;
	jint session = 0;
	
	status = cryptCreateSession(&session, cryptUser, formatType);
	
	processStatus(env, status);
	return(session);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DestroySession
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DestroySession
  (JNIEnv * env, jclass cryptClass, jint session)
{
	int status = 0;
	
	status = cryptDestroySession(session);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    PushData
 * Signature: (ILjava/nio/ByteBuffer;II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_PushData__ILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint envelope, jobject buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jint bytesCopied = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesNIO(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerNIO(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptPushData(envelope, bufferPtr + bufferOffset, length, &bytesCopied);
	
	finish:
	releasePointerNIO(env, buffer, bufferPtr);
	processStatus(env, status);
	return(bytesCopied);
}

/*
 * Class:     cryptlib_crypt
 * Method:    PushData
 * Signature: (I[BII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_PushData__I_3BII
  (JNIEnv * env, jclass cryptClass, jint envelope, jbyteArray buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jint bytesCopied = 0;
	jbyte* bufferPtr = 0;
	
	if (!checkIndicesArray(env, buffer, bufferOffset, length))
		goto finish;
	
	if (!getPointerArray(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptPushData(envelope, bufferPtr + bufferOffset, length, &bytesCopied);
	
	finish:
	releasePointerArray(env, buffer, bufferPtr);
	processStatus(env, status);
	return(bytesCopied);
}

/*
 * Class:     cryptlib_crypt
 * Method:    FlushData
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_FlushData
  (JNIEnv * env, jclass cryptClass, jint envelope)
{
	int status = 0;
	
	status = cryptFlushData(envelope);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    PopData
 * Signature: (ILjava/nio/ByteBuffer;II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_PopData__ILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint envelope, jobject buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jint bytesCopied = 0;
	jbyte* bufferPtr = 0;
	
	//CryptPopData is a special case that doesn't have the length querying call
	
	if (!checkIndicesNIO(env, buffer, bufferOffset, bytesCopied))
		goto finish;
	
	if (!getPointerNIO(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptPopData(envelope, bufferPtr + bufferOffset, length, &bytesCopied);
	
	finish:
	releasePointerNIO(env, buffer, bufferPtr);
	processStatus(env, status);
	return(bytesCopied);
}

/*
 * Class:     cryptlib_crypt
 * Method:    PopData
 * Signature: (I[BII)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_PopData__I_3BII
  (JNIEnv * env, jclass cryptClass, jint envelope, jbyteArray buffer, jint bufferOffset, jint length)
{
	int status = 0;
	jint bytesCopied = 0;
	jbyte* bufferPtr = 0;
	
	//CryptPopData is a special case that doesn't have the length querying call
	
	if (!checkIndicesArray(env, buffer, bufferOffset, bytesCopied))
		goto finish;
	
	if (!getPointerArray(env, buffer, &bufferPtr))
		goto finish;
	
	status = cryptPopData(envelope, bufferPtr + bufferOffset, length, &bytesCopied);
	
	finish:
	releasePointerArray(env, buffer, bufferPtr);
	processStatus(env, status);
	return(bytesCopied);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeviceOpen
 * Signature: (IILjava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_DeviceOpen
  (JNIEnv * env, jclass cryptClass, jint cryptUser, jint deviceType, jstring name)
{
	int status = 0;
	jint device = 0;
	jbyte* namePtr = 0;
	
	if (!getPointerString(env, name, &namePtr))
		goto finish;
	
	status = cryptDeviceOpen(&device, cryptUser, deviceType, namePtr);
	
	finish:
	releasePointerString(env, name, namePtr);
	processStatus(env, status);
	return(device);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeviceClose
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeviceClose
  (JNIEnv * env, jclass cryptClass, jint device)
{
	int status = 0;
	
	status = cryptDeviceClose(device);
	
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeviceCreateContext
 * Signature: (II)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_DeviceCreateContext
  (JNIEnv * env, jclass cryptClass, jint device, jint cryptAlgo)
{
	int status = 0;
	jint cryptContext = 0;
	
	status = cryptDeviceCreateContext(device, &cryptContext, cryptAlgo);
	
	processStatus(env, status);
	return(cryptContext);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeviceControlEx
 * Signature: (IILjava/nio/ByteBuffer;IILjava/nio/ByteBuffer;II)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeviceControlEx__IILjava_nio_ByteBuffer_2IILjava_nio_ByteBuffer_2II
  (JNIEnv * env, jclass cryptClass, jint device, jint controlType, jobject data1, jint data1Offset, jint data1Length, jobject data2, jint data2Offset, jint data2Length)
{
	int status = 0;
	jbyte* data1Ptr = 0;
	jbyte* data2Ptr = 0;
	
	if (!checkIndicesNIO(env, data1, data1Offset, data1Length))
		goto finish;
	if (!checkIndicesNIO(env, data2, data2Offset, data2Length))
		goto finish;
	
	if (!getPointerNIO(env, data1, &data1Ptr))
		goto finish;
	if (!getPointerNIO(env, data2, &data2Ptr))
		goto finish;
	
	status = cryptDeviceControlEx(device, controlType, data1Ptr + data1Offset, data1Length, data2Ptr + data2Offset, data2Length);
	
	finish:
	releasePointerNIO(env, data1, data1Ptr);
	releasePointerNIO(env, data2, data2Ptr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    DeviceControlEx
 * Signature: (II[BII[BII)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_DeviceControlEx__II_3BII_3BII
  (JNIEnv * env, jclass cryptClass, jint device, jint controlType, jbyteArray data1, jint data1Offset, jint data1Length, jbyteArray data2, jint data2Offset, jint data2Length)
{
	int status = 0;
	jbyte* data1Ptr = 0;
	jbyte* data2Ptr = 0;
	
	if (!checkIndicesArray(env, data1, data1Offset, data1Length))
		goto finish;
	if (!checkIndicesArray(env, data2, data2Offset, data2Length))
		goto finish;
	
	if (!getPointerArray(env, data1, &data1Ptr))
		goto finish;
	if (!getPointerArray(env, data2, &data2Ptr))
		goto finish;
	
	status = cryptDeviceControlEx(device, controlType, data1Ptr + data1Offset, data1Length, data2Ptr + data2Offset, data2Length);
	
	finish:
	releasePointerArray(env, data1, data1Ptr);
	releasePointerArray(env, data2, data2Ptr);
	processStatus(env, status);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Login
 * Signature: (Ljava/lang/String;Ljava/lang/String;)I
 */
JNIEXPORT jint JNICALL Java_cryptlib_crypt_Login
  (JNIEnv * env, jclass cryptClass, jstring name, jstring password)
{
	int status = 0;
	jint user = 0;
	jbyte* namePtr = 0;
	jbyte* passwordPtr = 0;
	
	if (!getPointerString(env, name, &namePtr))
		goto finish;
	if (!getPointerString(env, password, &passwordPtr))
		goto finish;
	
	status = cryptLogin(&user, namePtr, passwordPtr);
	
	finish:
	releasePointerString(env, name, namePtr);
	releasePointerString(env, password, passwordPtr);
	processStatus(env, status);
	return(user);
}

/*
 * Class:     cryptlib_crypt
 * Method:    Logout
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_cryptlib_crypt_Logout
  (JNIEnv * env, jclass cryptClass, jint user)
{
	int status = 0;
	
	status = cryptLogout(user);
	
	processStatus(env, status);
}

