// Digit sequence arithmetic

#ifndef _CL_DS_H
#define _CL_DS_H

#include "cl_types.h"
#include "cl_D.h"

// Digit sequence (DS) - nur intern verwendet -
// Zusammenhngender Speicherbereich mit n (ein uintC) Digits,
// zwischen zwei Pointer MSDptr und LSDptr.
//  MSDptr                  LSDptr
// | MSD ............. LSW |
// [abgekrzt: MSDptr/n/LSDptr ]
// In 68000-Manier (vgl. ADDX, SUBX) ist das Most significant Digit an der
// untersten Adresse, nmlich MSDptr. LSDptr = MSDptr + n zeigt hinter die DS.
// Falls n = 0, wird die Zahl 0 dargestellt.
// Falls n > 0, ist das hchstwertige Bit (nmlich  Bit (intDsize-1) von
//              *MSDptr) das Vorzeichenbit. Schreibt man es noch unendlich
//              oft an, so erhlt man die "unendliche Bitfolge".
// Normalisierte Digit sequence (NDS) ist eine solche, bei der das MSD ntig
// ist, also n = 0 oder (n > 0 und nicht alle hchstwertigen intDsize+1 Bits
// sind gleich).
// In C:
//   uintD* MSWptr und uintC len.
//   MSWptr[0] ... MSWptr[len-1] sind die Digits.

// Unsigned Digit sequence (UDS) - nur intern verwendet -
// wie DS (MSD unten, LSD oben), nur ohne Vorzeichen.
// Normalized Unsigned Digit sequence (NUDS):
// wie UDS, nur ist entweder n=0 (Zahl 0) oder bei n>0 : *MSDptr >0.
// (d.h. die Zahl >=0 kann nicht mit weniger Digits als UDS dargestellt werden).
// In C:
//   uintD* MSWptr und uintC len.
//   MSWptr[0] ... MSWptr[(uintL)len-1] sind die Digits.

// Zur Konstruktion konstanter DS: D(byte0,byte1,byte2,byte3,) liefert
// die 32 Bits von {byte0,byte1,byte2,byte3} als 32/intDsize Digits.
  #if (intDsize==8)
    #define D(byte0,byte1,byte2,byte3,dummy)  byte0,byte1,byte2,byte3,
  #endif
  #if (intDsize==16)
    #define D(byte0,byte1,byte2,byte3,dummy)  ((byte0<<8)|byte1),((byte2<<8)|byte3),
  #endif
  #if (intDsize==32)
    #define D(byte0,byte1,byte2,byte3,dummy)  \
      (((uintD)(byte0)<<24)|((uintD)(byte1)<<16)|((uintD)(byte2)<<8)|((uintD)(byte3))),
  #endif

struct DS {
	uintD* MSDptr;
	unsigned int len;
	uintD* LSDptr;
};


// These functions on digit sequences are either inline C++ functions
// or external assembler functions (see files cl_asm_*).


// See which functions are defined as external functions.
#define INCLUDED_FROM_C
#include "cl_asm.cc"


// Declare the external functions.

extern "C" {

#ifdef COPY_LOOPS

extern uintD* copy_loop_up (const uintD* sourceptr, uintD* destptr, uintC count);

extern uintD* copy_loop_down (const uintD* sourceptr, uintD* destptr, uintC count);

#endif

#ifdef FILL_LOOPS

extern uintD* fill_loop_up (uintD* destptr, uintC count, uintD filler);

extern uintD* fill_loop_down (uintD* destptr, uintC count, uintD filler);

#endif

#ifdef CLEAR_LOOPS

extern uintD* clear_loop_up (uintD* destptr, uintC count);

extern uintD* clear_loop_down (uintD* destptr, uintC count);

#endif

#ifdef LOG_LOOPS

extern void or_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void xor_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void and_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void eqv_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void nand_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void nor_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void andc2_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void orc2_loop_up (uintD* xptr, const uintD* yptr, uintC count);

extern void not_loop_up (uintD* xptr, uintC count);

#endif

#ifdef TEST_LOOPS

extern cl_boolean and_test_loop_up (const uintD* xptr, const uintD* yptr, uintC count);

extern cl_boolean test_loop_up (const uintD* ptr, uintC count);

extern cl_signean compare_loop_up (const uintD* xptr, const uintD* yptr, uintC count);

#endif

#ifdef ADDSUB_LOOPS

extern uintD add_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count);

extern uintD addto_loop_down (const uintD* sourceptr, uintD* destptr, uintC count);

extern uintD inc_loop_down (uintD* ptr, uintC count);

extern uintD sub_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count);

extern uintD subx_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count, uintD carry);

extern uintD subfrom_loop_down (const uintD* sourceptr, uintD* destptr, uintC count);

extern uintD dec_loop_down (uintD* ptr, uintC count);

extern uintD neg_loop_down (uintD* ptr, uintC count);

#endif

#ifdef SHIFT_LOOPS

extern uintD shift1left_loop_down (uintD* ptr, uintC count);

extern uintD shiftleft_loop_down (uintD* ptr, uintC count, uintC i, uintD carry);

extern uintD shiftleftcopy_loop_down (const uintD* sourceptr, uintD* destptr, uintC count, uintC i);

extern uintD shift1right_loop_up (uintD* ptr, uintC count, uintD carry);

extern uintD shiftright_loop_up (uintD* ptr, uintC count, uintC i);

extern uintD shiftrightsigned_loop_up (uintD* ptr, uintC count, uintC i);

extern uintD shiftrightcopy_loop_up (const uintD* sourceptr, uintD* destptr, uintC count, uintC i, uintD carry);

#endif

#ifdef MUL_LOOPS

extern uintD mulusmall_loop_down (uintD digit, uintD* ptr, uintC len, uintD newdigit);

extern void mulu_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len);

extern uintD muluadd_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len);

extern uintD mulusub_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len);

#endif

#ifdef DIV_LOOPS

extern uintD divu_loop_up (uintD digit, uintD* ptr, uintC len);

extern uintD divucopy_loop_up (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len);

#endif

} // "C" extern.


// Define the missing functions as inline functions.

#ifndef COPY_LOOPS

// Kopierschleife:
// destptr = copy_loop_up(sourceptr,destptr,count);
// kopiert count (uintC>=0) Digits aufwrts von sourceptr nach destptr
// und liefert das neue destptr.
  inline uintD* copy_loop_up (const uintD* sourceptr, uintD* destptr, uintC count)
    { dotimesC(count,count, { *destptr++ = *sourceptr++; } );
      return destptr;
    }

// Kopierschleife:
// destptr = copy_loop_down(sourceptr,destptr,count);
// kopiert count (uintC>=0) Digits abwrts von sourceptr nach destptr
// und liefert das neue destptr.
  inline uintD* copy_loop_down (const uintD* sourceptr, uintD* destptr, uintC count)
    { dotimesC(count,count, { *--destptr = *--sourceptr; } );
      return destptr;
    }

#endif

#ifndef FILL_LOOPS

// Fllschleife:
// destptr = fill_loop_up(destptr,count,filler);
// kopiert count (uintC>=0) mal das Digit filler aufwrts nach destptr
// und liefert das neue destptr.
  inline uintD* fill_loop_up (uintD* destptr, uintC count, uintD filler)
    { dotimesC(count,count, { *destptr++ = filler; } );
      return destptr;
    }

// Fllschleife:
// destptr = fill_loop_down(destptr,count,filler);
// kopiert count (uintC>=0) mal das Digit filler abwrts nach destptr
// und liefert das neue destptr.
  inline uintD* fill_loop_down (uintD* destptr, uintC count, uintD filler)
    { dotimesC(count,count, { *--destptr = filler; } );
      return destptr;
    }

#endif

#ifndef CLEAR_LOOPS

// Lsch-Schleife:
// destptr = clear_loop_up(destptr,count);
// lscht count (uintC>=0) Digits aufwrts ab destptr
// und liefert das neue destptr.
  inline uintD* clear_loop_up (uintD* destptr, uintC count)
    { dotimesC(count,count, { *destptr++ = 0; } );
      return destptr;
    }

// Lsch-Schleife:
// destptr = clear_loop_down(destptr,count);
// lscht count (uintC>=0) Digits abwrts ab destptr
// und liefert das neue destptr.
  inline uintD* clear_loop_down (uintD* destptr, uintC count)
    { dotimesC(count,count, { *--destptr = 0; } );
      return destptr;
    }

#endif

#ifndef LOG_LOOPS

// OR-Schleife:
// or_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch OR.
  inline void or_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { *xptr++ |= *yptr++; } ); }

// XOR-Schleife:
// xor_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch XOR.
  inline void xor_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { *xptr++ ^= *yptr++; } ); }

// AND-Schleife:
// and_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch AND.
  inline void and_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { *xptr++ &= *yptr++; } ); }

// EQV-Schleife:
// eqv_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch EQV (NOT XOR).
  inline void eqv_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count,
        {var uintD temp = ~ (*xptr ^ *yptr++); *xptr++ = temp; }
        );
    }

// NAND-Schleife:
// nand_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch NAND (NOT AND).
  inline void nand_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count,
        {var uintD temp = ~ (*xptr & *yptr++); *xptr++ = temp; }
        );
    }

// NOR-Schleife:
// nor_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch NOR (NOT OR).
  inline void nor_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count,
        {var uintD temp = ~ (*xptr | *yptr++); *xptr++ = temp; }
        );
    }

// ANDC2-Schleife:
// andc2_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch ANDC2 (AND NOT).
  inline void andc2_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { *xptr++ &= ~(*yptr++); } ); }

// ORC2-Schleife:
// orc2_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr
// mit Ziel ab xptr durch ORC2 (OR NOT).
  inline void orc2_loop_up (uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { *xptr++ |= ~(*yptr++); } ); }

// NOT-Schleife:
// not_loop_up(xptr,count);
// verknpft count (uintC>0) Digits aufwrts ab xptr mit Ziel ab xptr
// durch NOT.
  inline void not_loop_up (uintD* xptr, uintC count)
    { dotimespC(count,count,
        {var uintD temp = ~ (*xptr); *xptr++ = temp; }
        );
    }

#endif

#ifndef TEST_LOOPS

// AND-Test-Schleife:
// and_test_loop_up(xptr,yptr,count);
// verknpft count (uintC>=0) Digits aufwrts ab xptr und ab yptr durch AND
// und testet, ob sich dabei ein Digit /=0 ergibt. Ergebnis /=0, falls ja.
  inline cl_boolean and_test_loop_up (const uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count, { if (*xptr++ & *yptr++) return cl_true; } );
      return cl_false;
    }

// Test-Schleife:
// test_loop_up(ptr,count)
// bzw.  if_test_loop_up(ptr,count, statement1, statement2)
// testet count (uintC>=0) Digits aufwrts ab ptr, ob darunter eines /=0 ist.
// Ergebnis /=0, falls ja.
  inline cl_boolean test_loop_up (const uintD* ptr, uintC count)
    { dotimesC(count,count, { if (*ptr++) return cl_true; } );
      return cl_false;
    }

// Vergleichsschleife:
// result = compare_loop_up(xptr,yptr,count);
// vergleicht nacheinander xptr[0] mit yptr[0], xptr[1] mit yptr[1], usw.,
// insgesamt count Digits, und liefert 0 falls alle gleich sind,
// +1 falls zuerst ein xptr[i]>yptr[i] ist,
// -1 falls zuerst ein xptr[i]<yptr[i] ist.
  inline cl_signean compare_loop_up (const uintD* xptr, const uintD* yptr, uintC count)
    { dotimesC(count,count,
        { if (!(*xptr++ == *yptr++))
            // verschiedene Digits gefunden
            return (*--xptr > *--yptr ? signean_plus : signean_minus);
        });
      return signean_null; // alle Digits gleich
    }

#endif

#ifndef ADDSUB_LOOPS

// Additionsschleife:
// bertrag = add_loop_down(sourceptr1,sourceptr2,destptr,count);
// addiert count (uintC>=0) Digits abwrts von sourceptr1, von sourceptr2
// abwrts nach destptr und liefert den bertrag (0 oder /=0, was 1 bedeutet).
  inline uintD add_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count)
    { var uintD source1;
      var uintD source2;
      if (!(count==0))
      do { source1 = *--sourceptr1;
           source2 = *--sourceptr2;
           *--destptr = source1 + source2;
           if (source1 > (uintD)(~source2)) goto carry_1;
           carry_0:
           count--;
         }
         until (count==0);
      return 0;
      do { source1 = *--sourceptr1;
           source2 = *--sourceptr2;
           *--destptr = source1 + source2 + 1;
           if (source1 < (uintD)(~source2)) goto carry_0;
           carry_1:
           count--;
         }
         until (count==0);
      return 1;
    }

// Additionsschleife:
// bertrag = addto_loop_down(sourceptr,destptr,count);
// addiert count (uintC>=0) Digits abwrts von sourceptr, von destptr
// abwrts nach destptr und liefert den bertrag (0 oder /=0, was 1 bedeutet).
  inline uintD addto_loop_down (const uintD* sourceptr, uintD* destptr, uintC count)
    { var uintD source1;
      var uintD source2;
      if (!(count==0))
      do { source1 = *--sourceptr;
           source2 = *--destptr;
           *destptr = source1 + source2;
           if (source1 > (uintD)(~source2)) goto carry_1;
           carry_0:
           count--;
         }
         until (count==0);
      return 0;
      do { source1 = *--sourceptr;
           source2 = *--destptr;
           *destptr = source1 + source2 + 1;
           if (source1 < (uintD)(~source2)) goto carry_0;
           carry_1:
           count--;
         }
         until (count==0);
      return 1;
    }

// Incrementierschleife:
// bertrag = inc_loop_down(ptr,count);
// incrementiert count (uintC>=0) Digits abwrts von ptr, so lange bis kein
// bertrag mehr auftritt und liefert den bertrag (0 oder /=0, was 1 bedeutet).
  inline uintD inc_loop_down (uintD* ptr, uintC count)
    { dotimesC(count,count,
        { if (!( ++(*--ptr) == 0 )) return 0; } // kein weiterer bertrag
        );
      return 1; // weiterer bertrag
    }

// Subtraktionsschleife:
// bertrag = sub_loop_down(sourceptr1,sourceptr2,destptr,count);
// subtrahiert count (uintC>=0) Digits abwrts von sourceptr1, von sourceptr2
// abwrts nach destptr und liefert den bertrag (0 oder /=0, was -1 bedeutet).
  inline uintD sub_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count)
    { var uintD source1;
      var uintD source2;
      if (!(count==0))
      do { source1 = *--sourceptr1;
           source2 = *--sourceptr2;
           *--destptr = source1 - source2;
           if (source1 < source2) goto carry_1;
           carry_0:
           count--;
         }
         until (count==0);
      return 0;
      do { source1 = *--sourceptr1;
           source2 = *--sourceptr2;
           *--destptr = source1 - source2 - 1;
           if (source1 > source2) goto carry_0;
           carry_1:
           count--;
         }
         until (count==0);
      return -1;
    }

// Subtraktionsschleife:
// bertrag = subx_loop_down(sourceptr1,sourceptr2,destptr,count,carry);
// subtrahiert count (uintC>=0) Digits abwrts von sourceptr1 und addiert
// einen Carry (0 oder -1), von sourceptr2 abwrts nach destptr und
// liefert den bertrag (0 oder /=0, was -1 bedeutet).
  inline uintD subx_loop_down (const uintD* sourceptr1, const uintD* sourceptr2, uintD* destptr, uintC count, uintD carry)
    { var uintD source1;
      var uintD source2;
      if (carry==0)
        { if (!(count==0))
            do { source1 = *--sourceptr1;
                 source2 = *--sourceptr2;
                 *--destptr = source1 - source2;
                 if (source1 < source2) goto carry_1;
                 carry_0:
                 count--;
               }
               until (count==0);
          return 0;
        }
        else
        { if (!(count==0))
            do { source1 = *--sourceptr1;
                 source2 = *--sourceptr2;
                 *--destptr = source1 - source2 - 1;
                 if (source1 > source2) goto carry_0;
                 carry_1:
                 count--;
               }
               until (count==0);
          return -1;
    }   }

// Subtraktionsschleife:
// bertrag = subfrom_loop_down(sourceptr,destptr,count);
// subtrahiert count (uintC>=0) Digits abwrts von sourceptr, von destptr
// abwrts nach destptr (dest := dest - source)
// und liefert den bertrag (0 oder /=0, was -1 bedeutet).
  inline uintD subfrom_loop_down (const uintD* sourceptr, uintD* destptr, uintC count)
    { var uintD source1;
      var uintD source2;
      if (!(count==0))
      do { source1 = *--destptr;
           source2 = *--sourceptr;
           *destptr = source1 - source2;
           if (source1 < source2) goto carry_1;
           carry_0:
           count--;
         }
         until (count==0);
      return 0;
      do { source1 = *--destptr;
           source2 = *--sourceptr;
           *destptr = source1 - source2 - 1;
           if (source1 > source2) goto carry_0;
           carry_1:
           count--;
         }
         until (count==0);
      return -1;
    }

// Decrementierschleife:
// bertrag = dec_loop_down(ptr,count);
// decrementiert count (uintC>=0) Digits abwrts von ptr, so lange bis kein
// bertrag mehr auftritt und liefert den bertrag (0 oder -1).
  inline uintD dec_loop_down (uintD* ptr, uintC count)
    { dotimesC(count,count,
        { if (!( (*--ptr)-- == 0 )) return 0; } // kein weiterer bertrag
        );
      return -1; // weiterer bertrag
    }

// Negierschleife:
// bertrag = neg_loop_down(ptr,count);
// negiert count (uintC>=0) Digits abwrts von ptr,
// und liefert den bertrag (0 oder -1).
  inline uintD neg_loop_down (uintD* ptr, uintC count)
    { // erstes Digit /=0 suchen:
      until (count==0) { if (!(*--ptr == 0)) goto L1; count--; }
      return 0;
      L1: // erstes Digit /=0 gefunden, ab jetzt gibt's Carrys
      *ptr = - *ptr; count--; // 1 Digit negieren
      dotimesC(count,count, { --ptr; *ptr = ~ *ptr; } ); // alle anderen Digits invertieren
      return -1;
    }

#endif

#ifndef SHIFT_LOOPS

// Schiebeschleife um 1 Bit nach links:
// bertrag = shift1left_loop_down(ptr,count);
// schiebt count (uintC>=0) Digits abwrts von ptr um 1 Bit nach links,
// und liefert den bertrag (0 oder /=0, was 1 bedeutet).
  #if HAVE_DD
  inline uintD shift1left_loop_down (uintD* ptr, uintC count)
    { var uintDD accu = 0;
      dotimesC(count,count,
        { accu = ((uintDD)(*--ptr)<<1)+accu; *ptr = lowD(accu);
          accu = (uintDD)(highD(accu));
        });
      return (uintD)accu;
    }
  #else
  inline uintD shift1left_loop_down (uintD* ptr, uintC count)
    { var uintD carry = 0;
      dotimesC(count,count,
        { var uintD accu = *--ptr;
          *ptr = (accu<<1) | carry;
          carry = accu>>(intDsize-1);
        });
      return carry;
    }
  #endif

// Schiebeschleife um i Bits nach links:
// bertrag = shiftleft_loop_down(ptr,count,i,bertrag_init);
// schiebt count (uintC>=0) Digits abwrts von ptr um i Bits (0<i<intDsize)
// nach links, schiebt dabei die i Bits aus bertrag_init rechts rein,
// und liefert den bertrag (was links rauskommt, >=0, <2^i).
  #if HAVE_DD
  inline uintD shiftleft_loop_down (uintD* ptr, uintC count, uintC i, uintD carry)
    { var uintDD accu = (uintDD)carry;
      dotimesC(count,count,
        { accu = ((uintDD)(*--ptr)<<i)+accu; *ptr = lowD(accu);
          accu = (uintDD)(highD(accu));
        });
      return (uintD)accu;
    }
  #else
  inline uintD shiftleft_loop_down (uintD* ptr, uintC count, uintC i, uintD carry)
    { var uintC j = intDsize-i;
      dotimesC(count,count,
        { var uintD accu = *--ptr;
          *ptr = (accu<<i) | carry;
          carry = accu>>j;
        });
      return carry;
    }
  #endif

// Schiebe- und Kopierschleife um i Bits nach links:
// bertrag = shiftleftcopy_loop_down(sourceptr,destptr,count,i);
// kopiert count (uintC>=0) Digits abwrts von sourceptr nach destptr
// und schiebt sie dabei um i Bits (0<i<intDsize) nach links,
// wobei ganz rechts mit i Nullbits aufgefllt wird,
// und liefert den bertrag (was links rauskommt, >=0, <2^i).
  #if HAVE_DD
  inline uintD shiftleftcopy_loop_down (const uintD* sourceptr, uintD* destptr, uintC count, uintC i)
    { var uintDD accu = 0;
      dotimesC(count,count,
        { accu = ((uintDD)(*--sourceptr)<<i)+accu; *--destptr = lowD(accu);
          accu = (uintDD)(highD(accu));
        });
      return (uintD)accu;
    }
  #else
  inline uintD shiftleftcopy_loop_down (const uintD* sourceptr, uintD* destptr, uintC count, uintC i)
    { var uintC j = intDsize-i;
      var uintD carry = 0;
      dotimesC(count,count,
        { var uintD accu = *--sourceptr;
          *--destptr = (accu<<i) | carry;
          carry = accu>>j;
        });
      return carry;
    }
  #endif

// Schiebeschleife um 1 Bit nach rechts:
// bertrag = shift1right_loop_up(ptr,count,bertrag_init);
// schiebt count (uintC>=0) Digits aufwrts von ptr um 1 Bit nach rechts,
// wobei links das Bit bertrag_init (sollte =0 oder =-1 sein) hineingeschoben
// wird, und liefert den bertrag (0 oder /=0, was 1 bedeutet).
  #if HAVE_DD
  inline uintD shift1right_loop_up (uintD* ptr, uintC count, uintD carry)
    { var uintDD accu = (sintDD)(sintD)carry & ((uintDD)1 << (2*intDsize-1)); // 0 oder bit(2*intDsize-1)
      dotimesC(count,count,
        { accu = (highlowDD_0(*ptr)>>1)+accu; *ptr++ = highD(accu);
          accu = highlowDD_0(lowD(accu));
        });
      return highD(accu);
    }
  #else
  inline uintD shift1right_loop_up (uintD* ptr, uintC count, uintD carry)
    { carry = carry << (intDsize-1); // carry zu einem einzigen Bit machen
      dotimesC(count,count,
        { var uintD accu = *ptr;
          *ptr++ = (accu >> 1) | carry;
          carry = accu << (intDsize-1);
        });
      return carry;
    }
  #endif

// Schiebeschleife um i Bits nach rechts:
// bertrag = shiftright_loop_up(ptr,count,i);
// schiebt count (uintC>=0) Digits aufwrts von ptr um i Bits (0<i<intDsize)
// nach rechts, wobei links Nullen eingeschoben werden,
// und liefert den bertrag (was rechts rauskommt, als Bits intDsize-1..intDsize-i).
  #if HAVE_DD
  inline uintD shiftright_loop_up (uintD* ptr, uintC count, uintC i)
    { var uintDD accu = 0;
      dotimesC(count,count,
        { // Die oberen i Bits von (uintD)accu bilden hier den bertrag.
          accu = highlowDD_0(lowD(accu));
          // Die oberen i Bits von (uintDD)accu bilden hier den bertrag.
          accu = (highlowDD_0(*ptr)>>i)+accu; *ptr++ = highD(accu);
        });
      return lowD(accu);
    }
  #else
  inline uintD shiftright_loop_up (uintD* ptr, uintC count, uintC i);
    { var uintC j = intDsize-i;
      var uintD carry = 0;
      dotimesC(count,count,
        { var uintD accu = *ptr;
          *ptr++ = (accu >> i) | carry;
          carry = accu << j;
        });
      return carry;
    }
  #endif

// Schiebeschleife um i Bits nach rechts:
// bertrag = shiftrightsigned_loop_up(ptr,count,i);
// schiebt count (uintC>0) Digits aufwrts von ptr um i Bits (0<i<intDsize)
// nach rechts, wobei links das MSBit ver-i-facht wird,
// und liefert den bertrag (was rechts rauskommt, als Bits intDsize-1..intDsize-i).
  #if HAVE_DD
  inline uintD shiftrightsigned_loop_up (uintD* ptr, uintC count, uintC i)
    { var uintDD accu = // bertrag mit i Vorzeichenbits initialisieren
                           highlowDD_0(sign_of_sintD((sintD)(*ptr)))>>i;
      dotimespC(count,count,
        { // Die oberen i Bits von (uintD)accu bilden hier den bertrag.
          accu = highlowDD_0(lowD(accu));
          // Die oberen i Bits von (uintDD)accu bilden hier den bertrag.
          accu = (highlowDD_0(*ptr)>>i)+accu; *ptr++ = highD(accu);
        });
      return lowD(accu);
    }
  #else
  inline uintD shiftrightsigned_loop_up (uintD* ptr, uintC count, uintC i)
    { var uintC j = intDsize-i;
      var uintD carry;
      { var uintD accu = *ptr;
        *ptr++ = (sintD)accu >> i;
        carry = accu << j;
        count--;
      }
      dotimesC(count,count,
        { var uintD accu = *ptr;
          *ptr++ = (accu >> i) | carry;
          carry = accu << j;
        });
      return carry;
    }
  #endif

// Schiebe- und Kopier-Schleife um i Bits nach rechts:
// bertrag = shiftrightcopy_loop_up(sourceptr,destptr,count,i,carry);
// kopiert count (uintC>=0) Digits aufwrts von sourceptr nach destptr
// und schiebt sie dabei um i Bits (0<i<intDsize) nach rechts, wobei carry
// (sozusagen als sourceptr[-1]) die i Bits ganz links bestimmt,
// und liefert den bertrag (was rechts rauskommt, als Bits intDsize-1..intDsize-i).
  #if HAVE_DD
  inline uintD shiftrightcopy_loop_up (const uintD* sourceptr, uintD* destptr, uintC count, uintC i, uintD carry)
    { var uintDD accu = // bertrag mit carry initialisieren
                           highlowDD_0(carry)>>i;
      dotimesC(count,count,
        { // Die oberen i Bits von (uintD)accu bilden hier den bertrag.
          accu = highlowDD_0(lowD(accu));
          // Die oberen i Bits von (uintDD)accu bilden hier den bertrag.
          accu = (highlowDD_0(*sourceptr++)>>i)+accu; *destptr++ = highD(accu);
        });
      return lowD(accu);
    }
  #else
  inline uintD shiftrightcopy_loop_up (const uintD* sourceptr, uintD* destptr, uintC count, uintC i, uintD carry)
    { var uintC j = intDsize-i;
      carry = carry << j;
      dotimesC(count,count,
        { var uintD accu = *sourceptr++;
          *destptr++ = (accu >> i) | carry;
          carry = accu << j;
        });
      return carry;
    }
  #endif

#endif

#ifndef MUL_LOOPS

// Multiplikations-Einfachschleife:
// Multipliziert eine UDS mit einem kleinen Digit und addiert ein kleines Digit.
// mulusmall_loop_down(digit,ptr,len,newdigit)
// multipliziert die UDS  ptr[-len..-1]  mit digit (>=2, <=36),
// addiert dabei newdigit (>=0, <digit) zur letzten Ziffer,
// und liefert den Carry (>=0, <digit).
  #if HAVE_DD
  inline uintD mulusmall_loop_down (uintD digit, uintD* ptr, uintC len, uintD newdigit)
    { var uintDD carry = newdigit;
      dotimesC(len,len,
        { // Hier ist 0 <= carry < digit.
          carry = carry + muluD(digit,*--ptr);
          // Hier ist 0 <= carry < 2^intDsize*digit.
          *ptr = lowD(carry);
          carry = (uintDD)highD(carry); // carry := floor(carry/2^intDsize) < digit
        });
      return lowD(carry);
    }
  #else
  inline uintD mulusmall_loop_down (uintD digit, uintD* ptr, uintC len, uintD newdigit)
    { var uintD carry = newdigit;
      dotimesC(len,len,
        { // Hier ist 0 <= carry < digit.
          var uintD hi;
          var uintD lo;
          muluD(digit,*--ptr,hi=,lo=);
          // Hier ist 0 <= 2^intDsize*hi + lo + carry < 2^intDsize*digit.
          lo += carry; if (lo < carry) { hi += 1; }
          *ptr = lo;
          carry = hi;
        });
      return carry;
    }
  #endif

// Multiplikations-Einfachschleife:
// Multipliziert eine UDS mit einem Digit und legt das Ergebnis in einer
// zweiten UDS ab.
// mulu_loop_down(digit,sourceptr,destptr,len);
// multipliziert die UDS  sourceptr[-len..-1]  (len>0)
// mit dem einzelnen  digit
// und legt das Ergebnis in der UDS  destptr[-len-1..-1]  ab.
  #if HAVE_DD
  inline void mulu_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintDD carry = 0;
      dotimespC(len,len,
        { // Hier ist carry=digit=0 oder 0 <= carry < digit.
          carry = carry + muluD(digit,*--sourceptr);
          // Hier ist carry=digit=0 oder 0 <= carry < 2^intDsize*digit.
          *--destptr = lowD(carry);
          carry = (uintDD)highD(carry); // carry := floor(carry/2^intDsize) < digit
        });
      *--destptr = lowD(carry);
    }
  #else
  inline void mulu_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintD carry = 0;
      dotimespC(len,len,
        { // Hier ist carry=digit=0 oder 0 <= carry < digit.
          var uintD hi;
          var uintD lo;
          muluD(digit,*--sourceptr,hi=,lo=);
          // Hier ist 0 <= 2^intDsize*hi + lo + carry < 2^intDsize*digit oder hi=lo=carry=digit=0.
          lo += carry; if (lo < carry) { hi += 1; }
          *--destptr = lo;
          carry = hi;
        });
      *--destptr = carry;
    }
  #endif

// Multiplikations-Einfachschleife mit Akkumulation:
// Multipliziert eine UDS mit einem Digit und addiert das Ergebnis zu einer
// zweiten UDS auf.
// muluadd_loop_down(digit,sourceptr,destptr,len);
// multipliziert die UDS  sourceptr[-len..-1]  (len>0)
// mit dem einzelnen digit, legt das Ergebnis in der UDS  destptr[-len..-1]
// ab und liefert den weiteren bertrag.
  #if HAVE_DD
  inline uintD muluadd_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintDD carry = 0;
      if (!(digit==0))
        { dotimespC(len,len,
            { // Hier ist 0 <= carry <= digit.
              carry = carry + muluD(digit,*--sourceptr) + (uintDD)*--destptr;
              // Hier ist 0 <= carry <= 2^intDsize*digit + 2^intDsize-1.
              *destptr = lowD(carry);
              carry = (uintDD)highD(carry); // carry := floor(carry/2^intDsize) <= digit
            });
        }
      return lowD(carry);
    }
  #else
  inline uintD muluadd_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintD carry = 0;
      if (!(digit==0))
        { dotimespC(len,len,
            { // Hier ist 0 <= carry <= digit.
              var uintD hi;
              var uintD lo;
              muluD(digit,*--sourceptr,hi=,lo=);
              // Hier ist 0 <= 2^intDsize*hi + lo + carry + *--destptr <= 2^intDsize*digit+2^intDsize-1.
              lo += carry; if (lo < carry) { hi += 1; }
              carry = *--destptr;
              lo += carry; if (lo < carry) { hi += 1; }
              *destptr = lo;
              carry = hi;
            });
        }
      return carry;
    }
  #endif

// Multiplikations-Einfachschleife mit Diminution:
// Multipliziert eine UDS mit einem Digit und subtrahiert das Ergebnis von
// einer zweiten UDS.
// mulusub_loop_down(digit,sourceptr,destptr,len);
// multipliziert die UDS  sourceptr[-len..-1]  (len>0)  mit dem einzelnen
// digit, subtrahiert das Ergebnis von der UDS  destptr[-len..-1]  und liefert
// den weiteren bertrag (>=0, evtl. von destptr[-len-1] zu subtrahieren).
  #if HAVE_DD
  inline uintD mulusub_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintDD carry = 0;
      if (!(digit==0))
        { dotimespC(len,len,
            { // Hier ist 0 <= carry <= digit.
              carry = carry + muluD(digit,*--sourceptr) + (uintD)(~(*--destptr));
              // Hier ist 0 <= carry <= 2^intDsize*digit + 2^intDsize-1.
              *destptr = ~lowD(carry);
              carry = (uintDD)highD(carry); // carry := floor(carry/2^intDsize) <= digit
              // Hier ist 0 <= carry <= digit.
            });
          return lowD(carry);
        }
        else
        return 0; // nichts zu subtrahieren -> kein bertrag
    }
  #else
  inline uintD mulusub_loop_down (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintD carry = 0;
      if (!(digit==0))
        { dotimespC(len,len,
            { // Hier ist 0 <= carry <= digit.
              var uintD hi;
              var uintD lo;
              muluD(digit,*--sourceptr,hi=,lo=);
              // Hier ist 0 <= 2^intDsize*hi + lo + carry + ~(*--destptr) <= 2^intDsize*digit+2^intDsize-1.
              lo += carry; if (lo < carry) { hi += 1; }
              carry = *--destptr;
              *destptr = carry - lo; if (carry < lo) { hi += 1; }
              carry = hi;
            });
          return carry;
        }
        else
        return 0; // nichts zu subtrahieren -> kein bertrag
    }
  #endif

#endif

#ifndef DIV_LOOPS

// Divisions-Einfachschleife:
// Dividiert eine UDS durch ein Digit.
// divu_loop_up(digit,ptr,len)
// dividiert die UDS  ptr[0..len-1] durch digit,
// legt das Ergebnis in derselben UDS ab, und liefert den Rest (>=0, <digit).
  #if HAVE_DD
  inline uintD divu_loop_up (uintD digit, uintD* ptr, uintC len)
    { var uintD rest = 0;
      dotimesC(len,len,
        { divuD(highlowDD(rest,*ptr),digit,*ptr =, rest =); ptr++; }
        );
      return rest;
    }
  #else
  inline uintD divu_loop_up (uintD digit, uintD* ptr, uintC len)
    { var uintD rest = 0;
      dotimesC(len,len,
        { divuD(rest,*ptr,digit,*ptr =, rest =); ptr++; }
        );
      return rest;
    }
  #endif

// Divisions-Einfachschleife:
// Dividiert eine UDS durch ein Digit und legt das Ergebnis in einer
// zweiten UDS ab.
// divucopy_loop_up(digit,sourceptr,destptr,len)
// dividiert die UDS  sourceptr[0..len-1]  durch digit,
// legt das Ergebnis in der UDS  destptr[0..len-1]  ab,
// und liefert den Rest (>=0, <digit).
  #if HAVE_DD
  inline uintD divucopy_loop_up (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintD rest = 0;
      dotimesC(len,len,
        { divuD(highlowDD(rest,*sourceptr++),digit,*destptr++ =, rest =); }
        );
      return rest;
    }
  #else
  inline uintD divucopy_loop_up (uintD digit, const uintD* sourceptr, uintD* destptr, uintC len)
    { var uintD rest = 0;
      dotimesC(len,len,
        { divuD(rest,*sourceptr++,digit,*destptr++ =, rest =); }
        );
      return rest;
    }
  #endif

#endif


// Umwandlungsroutinen Digit-Sequence-Teil <--> Longword:

// get_32_Dptr(ptr)
//   holt die nchsten 32 Bits aus den 32/intDsize Digits ab ptr.
// set_32_Dptr(ptr,wert);
//   speichert den Wert wert (32 Bits) in die 32/intDsize Digits ab ptr.
// get_max32_Dptr(count,ptr)
//   holt die nchsten count Bits aus den ceiling(count/intDsize) Digits ab ptr.
// set_max32_Dptr(count,ptr,wert)
//   speichert wert (count Bits) in die ceiling(count/intDsize) Digits ab ptr.
// Jeweils ptr eine Variable vom Typ uintD*,
//         wert eine Variable vom Typ uint32,
//         count eine Variable oder constant-expression mit Wert >=0, <=32.
  #if (intDsize==32)
    inline uint32 get_32_Dptr (const uintD* ptr)
    {
	return ptr[0];
    }
    inline void set_32_Dptr (uintD* ptr, uint32 wert)
    {
	ptr[0] = wert;
    }
    inline uint32 get_max32_Dptr (uintC count, const uintD* ptr)
    {
	return count==0 ? 0 :
                          ptr[0];
    }
    inline void set_max32_Dptr (uintC count, uintD* ptr, uint32 wert)
    {
	if (count==0) return;
	ptr[0] = wert; return;
    }
  #endif
  #if (intDsize==16)
    inline uint32 get_32_Dptr (const uintD* ptr)
    {
	return ((uint32)(ptr[0])<<16) | (uint32)(ptr[1]);
    }
    inline void set_32_Dptr (uintD* ptr, uint32 wert)
    {
	ptr[0] = (uintD)(wert>>16); ptr[1] = (uintD)wert;
    }
    inline uint32 get_max32_Dptr (uintC count, const uintD* ptr)
    {
	return count==0 ? 0 :
	       count<=16 ? ptr[0] :
	                   ((uint32)(ptr[0])<<16) | (uint32)(ptr[1]);
    }
    inline void set_max32_Dptr (uintC count, uintD* ptr, uint32 wert)
    {
	if (count==0) return;
	if (count<=16) { ptr[0] = (uintD)wert; return; }
	ptr[0] = (uintD)(wert>>16); ptr[1] = (uintD)wert; return;
    }
  #endif
  #if (intDsize==8)
    inline uint32 get_32_Dptr (const uintD* ptr)
    {
	return ((((((uint32)(ptr[0]) <<8) | (uint32)(ptr[1])) <<8) | (uint32)(ptr[2])) <<8) | (uint32)(ptr[3]);
    }
    inline void set_32_Dptr (uintD* ptr, uint32 wert)
    {
	ptr[0] = (uintD)(wert>>24); ptr[1] = (uintD)(wert>>16); ptr[2] = (uintD)(wert>>8); ptr[3] = (uintD)wert;
    }
    inline uint32 get_max32_Dptr (uintC count, const uintD* ptr)
    {
	return count==0 ? 0 :
	       count<=8 ? ptr[0] :
	       count<=16 ? ((uint32)(ptr[0])<<8) | (uint32)(ptr[1]) :
	       count<=24 ? ((((uint32)(ptr[0])<<8) | (uint32)(ptr[1]))<<8) | (uint32)(ptr[2]) :
	                   ((((((uint32)(ptr[0])<<8) | (uint32)(ptr[1]))<<8) | (uint32)(ptr[2]))<<8) | (uint32)(ptr[3]);
    }
    inline void set_max32_Dptr (uintC count, uintD* ptr, uint32 wert)
    {
	if (count==0) return;
	if (count<=8) { ptr[0] = (uintD)wert; return; }
	if (count<=16) { ptr[0] = (uintD)(wert>>8); ptr[1] = (uintD)wert; return; }
	if (count<=24) { ptr[0] = (uintD)(wert>>16); ptr[1] = (uintD)(wert>>8); ptr[2] = (uintD)wert; return; }
	ptr[0] = (uintD)(wert>>24); ptr[1] = (uintD)(wert>>16); ptr[2] = (uintD)(wert>>8); ptr[3] = (uintD)wert; return;
    }
  #endif

// get_uint1D_Dptr(ptr)  holt 1 Digit (unsigned) ab ptr
// get_uint2D_Dptr(ptr)  holt 2 Digits (unsigned) ab ptr
// get_uint3D_Dptr(ptr)  holt 3 Digits (unsigned) ab ptr
// get_uint4D_Dptr(ptr)  holt 4 Digits (unsigned) ab ptr
// get_sint1D_Dptr(ptr)  holt 1 Digit (signed) ab ptr
// get_sint2D_Dptr(ptr)  holt 2 Digits (signed) ab ptr
// get_sint3D_Dptr(ptr)  holt 3 Digits (signed) ab ptr
// get_sint4D_Dptr(ptr)  holt 4 Digits (signed) ab ptr
// Jeweils ptr eine Variable vom Typ uintD*.
  inline uint32 get_uint1D_Dptr (const uintD* ptr)
  {
	return ptr[0];
  }
  inline sint32 get_sint1D_Dptr (const uintD* ptr)
  {
	return (sint32)(sintD)(ptr[0]);
  }
  #if (intDsize < 32)
  inline uint32 get_uint2D_Dptr (const uintD* ptr)
  {
	return ((uint32)(ptr[0]) << intDsize) | (uint32)(ptr[1]);
  }
  inline sint32 get_sint2D_Dptr (const uintD* ptr)
  {
	return ((uint32)(sint32)(sintD)(ptr[0]) << intDsize) | (uint32)(ptr[1]);
  }
  #else
  #define get_uint2D_Dptr(ptr)  get_uint1D_Dptr(&(ptr)[1])
  #define get_sint2D_Dptr(ptr)  (sint32)get_uint2D_Dptr(ptr)
  #endif
  #if (intDsize < 16)
  inline uint32 get_uint3D_Dptr (const uintD* ptr)
  {
	return ((((uint32)(ptr[0]) << intDsize) | (uint32)(ptr[1])) << intDsize) | (uint32)(ptr[2]);
  }
  inline sint32 get_sint3D_Dptr (const uintD* ptr)
  {
	return ((((uint32)(sint32)(sintD)(ptr[0]) << intDsize) | (uint32)(ptr[1])) << intDsize) | (uint32)(ptr[2]);
  }
  inline uint32 get_uint4D_Dptr (const uintD* ptr)
  {
	return ((((((uint32)(ptr[0]) << intDsize) | (uint32)(ptr[1])) << intDsize) | (uint32)(ptr[2])) << intDsize) | (uint32)(ptr[3]);
  }
  inline sint32 get_sint4D_Dptr (const uintD* ptr)
  {
	return ((((((uint32)(sint32)(sintD)(ptr[0]) << intDsize) | (uint32)(ptr[1])) << intDsize) | (uint32)(ptr[2])) << intDsize) | (uint32)(ptr[3]);
  }
  #else
  #define get_uint3D_Dptr(ptr)  get_uint2D_Dptr(&(ptr)[1])
  #define get_sint3D_Dptr(ptr)  (sint32)get_uint3D_Dptr(ptr)
  #define get_uint4D_Dptr(ptr)  get_uint2D_Dptr(&(ptr)[2])
  #define get_sint4D_Dptr(ptr)  (sint32)get_uint4D_Dptr(ptr)
  #endif


// NUM_STACK ist eine Art Zahlen-Stack-Pointer.
// Verwendung:
//   {SAVE_NUM_STACK
//    ...
//    num_stack_need(...);
//    ...
//    num_stack_need(...);
//    ...
//   }
// SAVE_NUM_STACK rettet den aktuellen Wert von NUM_STACK.
// Dann darf beliebig oft mit num_stack_need Platz auf dem Zahlen-Stack
// belegt werden.
// Beim Ende des Blocks wird NUM_STACK wieder auf den vorigen Wert gesetzt,
// und der Platz gilt als wieder freigegeben.
// In jeder C-Funktion sollte SAVE_NUM_STACK nur einmal aufgerufen werden.

// num_stack_need(need, low_addr = , high_addr = );
// belegt need Digits auf dem Zahlen-Stack und legt die untere Grenze des
// allozierten Bereichs (den MSDptr) in low_addr und die obere Grenze (den
// LSDptr) in high_addr ab. Jedes von beiden ist optional.

// num_stack_need_1(need, low_addr = , high_addr = );
// wie num_stack_need, nur da unterhalb von low_addr noch ein Digit Platz
// zustzlich belegt wird.

#if defined(__GNUC__) && !defined(__riscos) && !defined(__convex__)
  #define SAVE_NUM_STACK
  #define num_stack_need(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need);					  \
     var uintD* __array = (uintD*)__builtin_alloca(__need*sizeof(uintD)); \
     unused (low_zuweisung &__array[0]); unused (high_zuweisung &__array[__need]); \
    }
  #define num_stack_need_1(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need)+1;				  \
     var uintD* __array = (uintD*)__builtin_alloca(__need*sizeof(uintD)); \
     unused (low_zuweisung &__array[1]); unused (high_zuweisung &__array[__need]); \
    }
#elif !defined(NO_ALLOCA) && !defined(SPARC)
  #define SAVE_NUM_STACK
  #define num_stack_need(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need);					\
     var uintD* __array = (uintD*)alloca(__need*sizeof(uintD));		\
     low_zuweisung &__array[0]; high_zuweisung &__array[__need];	\
    }
  #define num_stack_need_1(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need)+1;				\
     var uintD* __array = (uintD*)alloca(__need*sizeof(uintD));		\
     low_zuweisung &__array[1]; high_zuweisung &__array[__need];	\
    }
#else
  #include "cl_malloca.h"
  // Save a malloca()/freea() "stack" pointer,
  // restore it automatically when the block or the function is exited.
  class num_stack {
	void* pointer;
  public:
	num_stack () { pointer = cl_malloca_getstack(); }
	~num_stack () { cl_malloca_setstack(pointer); }
  };
  #define SAVE_NUM_STACK  num_stack CONCAT(num_stack_,__LINE__);
  #define num_stack_need(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need);					\
     var uintD* __array = (uintD*)cl_malloca(__need*sizeof(uintD));	\
     low_zuweisung &__array[0]; high_zuweisung &__array[__need];	\
    }
  #define num_stack_need_1(need,low_zuweisung,high_zuweisung)  \
    {var uintL __need = (uintL)(need)+1;				\
     var uintD* __array = (uintD*)cl_malloca(__need*sizeof(uintD));	\
     low_zuweisung &__array[1]; high_zuweisung &__array[__need];	\
    }
#endif


// Macro: In der DS MSDptr/len/LSDptr wird eine 1 unterhalb des Pointers ptr
// addiert. Unterhalb von MSDptr mu 1 Digit Platz sein.
// Dabei ist  ptr - MSDptr = count  und  0 < count <= len .
// Eventuell wird MSDptr erniedrigt und len erhht.
  #define DS_1_plus(ptr,count)  \
    {var uintD* ptr_from_DS_1_plus = (ptr);				\
     var uintC count_from_DS_1_plus = (count);				\
     loop { if (--count_from_DS_1_plus==0) /* Zhler erniedrigen      */\
              { /* Beim Most Significant Digit angelangt              */\
                *(--ptr_from_DS_1_plus) += 1;				\
                /* jetzt ist ptr_from_DS_1_plus = MSDptr              */\
                if (*ptr_from_DS_1_plus == (uintD)bit(intDsize-1))	\
                  { /* 7FFF + 1 mu zu 00008000 werden:               */\
                    *--MSDptr = 0;					\
                    len++;						\
                  }							\
                break;							\
              }								\
            if (!((*(--ptr_from_DS_1_plus) += 1) == 0)) /* weiterincrementieren */\
              break; /* kein weiterer bertrag -> Schleife abbrechen  */\
    }     }

// Macro: In der DS MSDptr/len/LSDptr wird eine 1 unterhalb des Pointers ptr
// subtrahiert. Unterhalb von MSDptr mu 1 Digit Platz sein.
// Dabei ist  ptr - MSDptr = count  und  0 < count <= len .
// Eventuell wird MSDptr erniedrigt und len erhht.
  #define DS_minus1_plus(ptr,count)  \
    {var uintD* ptr_from_DS_minus1_plus = (ptr);			\
     var uintC count_from_DS_minus1_plus = (count);			\
     loop { if (--count_from_DS_minus1_plus==0) /* Zhler erniedrigen */\
              { /* Beim Most Significant Digit angelangt              */\
                *(--ptr_from_DS_minus1_plus) -= 1;			\
                /* jetzt ist ptr_from_DS_minus1_plus = MSDptr         */\
                if (*ptr_from_DS_minus1_plus == (uintD)bit(intDsize-1)-1) \
                  { /* 8000 - 1 mu zu FFFF7FFF werden:               */\
                    *--MSDptr = (uintD)(-1);				\
                    len++;						\
                  }							\
                break;							\
              }								\
            if (!((sintD)(*(--ptr_from_DS_minus1_plus) -= 1) == -1)) /* weiterdecrementieren */\
              break; /* kein weiterer bertrag -> Schleife abbrechen  */\
    }     }


// Multiplikations-Doppelschleife:
// Multipliziert zwei UDS und legt das Ergebnis in einer dritten UDS ab.
// cl_mulu_2loop_down(sourceptr1,len1,sourceptr2,len2,destptr);
// multipliziert die UDS  sourceptr1[-len1..-1]  (len1>0)
//           mit der UDS  sourceptr2[-len1..-1]  (len2>0)
// und legt das Ergebnis in der UDS  destptr[-len..-1]  (len=len1+len2) ab.
// Unterhalb von destptr werden len Digits Platz bentigt.
extern void cl_mulu_2loop_down (const uintD* sourceptr1, uintC len1,
                                const uintD* sourceptr2, uintC len2,
                                uintD* destptr);

// Multipliziert zwei Unsigned-Digit-sequences.
// UDS_UDS_mal_UDS(len1,LSDptr1, len2,LSDptr2, MSDptr=,len=,LSDptr=);
// multipliziert die UDS ../len1/LSDptr1 und ../len2/LSDptr2.
// Dabei sollte len1>0 und len2>0 sein.
// Ergebnis ist die UDS MSDptr/len/LSDptr, mit len=len1+len2, im Stack.
// Dabei wird num_stack erniedrigt.
  #define UDS_UDS_mal_UDS(len1,LSDptr1,len2,LSDptr2, MSDptr_zuweisung,len_zuweisung,LSDptr_zuweisung)  \
    var uintL len_from_UDSmal = (uintL)(len1) + (uintL)(len2);		\
    var uintD* LSDptr_from_UDSmal;					\
    unused (len_zuweisung len_from_UDSmal);				\
    num_stack_need(len_from_UDSmal,MSDptr_zuweisung,LSDptr_zuweisung LSDptr_from_UDSmal =); \
    cl_mulu_2loop_down((LSDptr1),(len1),(LSDptr2),(len2),LSDptr_from_UDSmal);

// Multipliziert zwei Digit-sequences.
// DS_DS_mal_DS(MSDptr1,len1,LSDptr1, MSDptr2,len2,LSDptr2, MSDptr=,len=,LSDptr=);
// multipliziert die DS MSDptr1/len1/LSDptr1 und MSDptr2/len2/LSDptr2.
// Dabei sollte len1>0 und len2>0 sein. Alles sollten Variablen sein!
// Ergebnis ist die DS MSDptr/len/LSDptr, mit len=len1+len2, im Stack.
// Dabei wird num_stack erniedrigt.
  // Methode:
  // Erst unsigned multiplizieren. Dann bis zu zwei Subtraktionen.
  // Sei b=2^intDsize, k=len1, l=len2, n=DS1, m=DS2.
  // Gesucht ist n * m.
  // Wir errechnen erst das unsigned-product p (mod b^(k+l)).
  // n>0, m>0: p = n*m,             n*m = p
  // n<0, m>0: p = (n+b^k)*m,       n*m + b^(k+l) = p - b^k * m (mod b^(k+l)).
  // n>0, m<0: p = n*(m+b^l),       n*m + b^(k+l) = p - b^l * n (mod b^(k+l)).
  // n<0, m<0: p = (n+b^k)*(m+b^l),
  //           n*m = p - b^k * (m+b^l) - b^l * (n+b^k) (mod b^(k+l)).
  #define DS_DS_mal_DS(MSDptr1,len1,LSDptr1,MSDptr2,len2,LSDptr2, MSDptr_zuweisung,len_zuweisung,LSDptr_zuweisung)  \
    var uintD* LSDptr0;							\
    UDS_UDS_mal_UDS(len1,LSDptr1,len2,LSDptr2, MSDptr_zuweisung,len_zuweisung,LSDptr_zuweisung LSDptr0 = ); \
    if ((sintD)(MSDptr1[0]) < 0) /* n<0 ?                             */\
      /* mu m bzw. m+b^l subtrahieren, um k Digits verschoben:       */\
      { subfrom_loop_down(LSDptr2,&LSDptr0[-(uintP)len1],len2); }	\
    if ((sintD)(MSDptr2[0]) < 0) /* m<0 ?                             */\
      /* mu n bzw. n+b^k subtrahieren, um l Digits verschoben:       */\
      { subfrom_loop_down(LSDptr1,&LSDptr0[-(uintP)len2],len1); }


// Dividiert zwei Unsigned Digit sequences durcheinander.
// UDS_divide(a_MSDptr,a_len,a_LSDptr, b_MSDptr,b_len,b_LSDptr, &q,&r);
// Die UDS a = a_MSDptr/a_len/a_LSDptr (a>=0) wird durch
// die UDS b = b_MSDptr/b_len/b_LSDptr (b>=0) dividiert:
// a = q * b + r mit 0 <= r < b. Bei b=0 Error.
// q der Quotient, r der Rest.
// q = q_MSDptr/q_len/q_LSDptr, r = r_MSDptr/r_len/r_LSDptr beides
// Normalized Unsigned Digit sequences.
// Vorsicht: q_LSDptr <= r_MSDptr,
//           Vorzeichenerweiterung von r kann q zerstren!
//           Vorzeichenerweiterung von q ist erlaubt.
// a und b werden nicht modifiziert.
// num_stack wird erniedrigt.
  #define UDS_divide(a_MSDptr,a_len,a_LSDptr,b_MSDptr,b_len,b_LSDptr,q_,r_)  \
    /* Platz frs Ergebnis machen. Brauche maximal a_len+1 Digits.    */\
    var uintC _a_len = (a_len);						\
    var uintD* roomptr; num_stack_need_1(_a_len+1,roomptr=,);		\
    cl_UDS_divide(a_MSDptr,_a_len,a_LSDptr,b_MSDptr,b_len,b_LSDptr,roomptr,q_,r_);
  extern void cl_UDS_divide (const uintD* a_MSDptr, uintC a_len, const uintD* a_LSDptr,
                             const uintD* b_MSDptr, uintC b_len, const uintD* b_LSDptr,
                             uintD* roomptr, DS* q_, DS* r_);


// Bildet zu einer Unsigned Digit sequence a die Wurzel
// (genauer: Gauklammer aus Wurzel aus a).
// UDS_sqrt(a_MSDptr,a_len,a_LSDptr, &b, squarep=)
// > a_MSDptr/a_len/a_LSDptr: eine UDS
// < NUDS b: Gauklammer der Wurzel aus a
// < squarep: cl_true falls a = b^2, cl_false falls b^2 < a < (b+1)^2.
// a wird nicht modifiziert.
// Vorzeichenerweiterung von b ist erlaubt.
// num_stack wird erniedrigt.
  #define UDS_sqrt(a_MSDptr,a_len,a_LSDptr,b_,squarep_zuweisung)  \
    { /* ceiling(a_len,2) Digits Platz frs Ergebnis machen:          */\
      var uintC _a_len = (a_len);					\
      num_stack_need_1(ceiling(_a_len,2),(b_)->MSDptr=,);		\
      squarep_zuweisung cl_UDS_sqrt(a_MSDptr,_a_len,a_LSDptr,b_);	\
    }
  extern cl_boolean cl_UDS_sqrt (const uintD* a_MSDptr, uintC a_len, const uintD* a_LSDptr, DS* b_);


// Hilfsfunktion zur Ausgabe von Integers

// cl_digits_need(len,base) liefert eine obere Abschtzung fr die Anzahl der
// Ziffern im Stellenwertsystem der Basis base, die eine UDS der Lnge len
// braucht.
  extern uintL cl_digits_need (uintC len, uintL base);

// Wandelt eine UDS in ein Stellensystem um.
// UDS_to_digits(MSDptr,len,base, &ergebnis);
// > MSDptr/len/..: eine UDS
// > base: Stellensystem-Basis, 2 <= base <= 36.
// > ergebnis.LSBptr: darunter ist mindestens digits_need(len) Bytes Platz
// < ergebnis: fertige Folge MSBptr/len/LSBptr von Ziffern
// Die UDS MSDptr/len/.. wird zerstrt.
  typedef struct { uintB* MSBptr; uintL len; uintB* LSBptr; } cl_digits;
  extern void UDS_to_digits (uintD* MSDptr, uintC len, uintD base, cl_digits* erg);


#endif /* _CL_DS_H */
