// Basic definitions of numbers

#ifndef _CL_NUMBER_H
#define _CL_NUMBER_H

#include "cl_config.h"
#include "cl_machine.h"
#include "cl_types.h"

// Type hierachy:
// Number (N) =
//    Real (R) =
//       Float (F) =
//          Short float (SF)
//          Single float (FF)
//          Double float (DF)
//          Long float (LF)
//       Rational (RA) =
//          Integer (I) =
//             Fixnum (FN)
//             Bignum (BN)
//          Ratio (RT)
//    Complex (C)


// Forward declarations.
class cl_number;
class cl_N;
class cl_R;
class cl_F;
class cl_SF;
class cl_FF;
class cl_DF;
class cl_LF;
class cl_RA;
class cl_I;


// A number is either a pointer to heap allocated data
//             or immediate data.


// It is possible to distinguish these because pointers are aligned.
// cl_uint_alignment is the guaranteed alignment of a `void*' or `long'
// in memory. Must be > 1.
#if defined(__m68k__)
  #define cl_word_alignment  2
#endif
#if defined(__i386__) || defined(__mips__) || defined(__sparc__) || defined(__hppa__) || defined(__arm__) || defined(__rs6000__) || defined(__m88k__) || defined(__convex__)
  #define cl_word_alignment  4
#endif
#if defined(__alpha__)
  #define cl_word_alignment  8
#endif
#if !defined(cl_word_alignment)
  #error "Define cl_word_alignment for your CPU!"
#endif


// Immediate data is a word, as wide as a pointer.
typedef sintP  cl_sint;
typedef uintP  cl_uint;  // This ought to be called `cl_word'.
#define cl_word_size intPsize
#if (cl_word_size==64) // defined(__alpha__)
  #define CL_WIDE_POINTERS
#endif

// Distinguish immediate data from pointers.
inline cl_boolean cl_pointer_p (cl_uint word)
{
	return (cl_boolean)((word & (cl_word_alignment-1)) == 0);
}
inline cl_boolean cl_immediate_p (cl_uint word)
{
	return (cl_boolean)((word & (cl_word_alignment-1)) != 0);
}

// Immediate data: Fixnum, Short Float, maybe Single Float.
// They have type tags.
//   |...............................|......|
//               cl_value             cl_tag

// Number of bits reserverd for tagging information:
#if (cl_word_alignment <= 4)
  #define cl_tag_len	2
#else
  #define cl_tag_len	3
#endif
#define cl_tag_shift	0
#if (cl_word_size == 64)
  #define cl_value_shift  32
#else
  #define cl_value_shift  (cl_tag_len+cl_tag_shift)
#endif
#define cl_value_len	(cl_word_size - cl_value_shift)
#define cl_tag_mask	(((1UL << cl_tag_len) - 1) << cl_tag_shift)
#define cl_value_mask	(((1UL << cl_value_len) - 1) << cl_value_shift)

// Return the tag of a word.
inline cl_uint cl_tag (cl_uint word)
{
	return (word & cl_tag_mask) >> cl_tag_shift;
}

// Return the value (unsigned) of a word.
inline cl_uint cl_value (cl_uint word)
{
	// This assumes cl_value_shift + cl_value_len == cl_word_size.
	return word >> cl_value_shift;
}

// Return the value (sign extended) of a word.
inline cl_sint cl_svalue (cl_uint word)
{
	// This assumes cl_value_shift + cl_value_len == cl_word_size.
	return (cl_sint)word >> cl_value_shift;
}

// Return a word, combining a value and a tag.
inline cl_uint cl_combine (cl_uint tag, cl_uint value)
{
	return (value << cl_value_shift) + (tag << cl_tag_shift);
}
inline cl_uint cl_combine (cl_uint tag, cl_sint value)
{
	// This assumes cl_value_shift + cl_value_len == cl_word_size.
	return (value << cl_value_shift) + (tag << cl_tag_shift);
}
// Keep the compiler happy.
inline cl_uint cl_combine (cl_uint tag, unsigned int value)
{ return cl_combine(tag,(cl_uint)value); }
inline cl_uint cl_combine (cl_uint tag, int value)
{ return cl_combine(tag,(cl_sint)value); }

// Definition of the tags.
#if !defined(CL_WIDE_POINTERS)
  #if (cl_word_alignment == 2)
    #define cl_FN_tag	1
    #define cl_SF_tag	3	// must satisfy the cl_immediate_p predicate!
  #endif
  #if (cl_word_alignment == 4)
    #define cl_FN_tag	1
    #define cl_SF_tag	2
  #endif
#else // CL_WIDE_POINTERS
  // Single Floats are immediate as well.
  #define cl_FN_tag	1
  #define cl_SF_tag	2
  #define cl_FF_tag	3
#endif


// Heap allocated data contains a header, for two purposes:
// - dynamic typing,
// - reference count (a portable alternative to garbage collection).
// typedef
struct cl_heap {
	int refcount;	// reference count, at least 24 bits
	int typetag;	// type tag, at least 8 bits
};

typedef enum {
	cl_typetag_none = 0,
	cl_typetag_bignum,
	cl_typetag_ratio,
#if !defined(CL_WIDE_POINTERS)
	cl_typetag_ffloat,
#endif
	cl_typetag_dfloat,
	cl_typetag_lfloat,
	cl_typetag_complex
};

// Free a number on heap.
extern void cl_free_heap_number (cl_heap* pointer);


// cl_private_thing: An immediate value or a pointer into the heap.
// This must be as wide as a `cl_uint'.
// (Actually, this ought to be a  union { void*; cl_uint; }, but using
// a pointer type generates better code.)
// Never throw away a cl_private_thing, or reference counts will be wrong!
typedef struct cl_anything * cl_private_thing;

// Increment the reference count.
inline void inc_pointer_refcount (cl_heap* pointer)
{
	pointer->refcount++;
}

// Decrement the reference count.
inline void dec_pointer_refcount (cl_heap* pointer)
{
	if (--pointer->refcount == 0)
		cl_free_heap_number(pointer);
}

// Increment the reference count.
// This must be a macro, not an inline function, because pointer_p() is a
// non-virtual member function, so that the compiler can optimize it. 
#define inc_refcount(x)  \
	if ((x).pointer_p())					\
		inc_pointer_refcount((x).heappointer);		\

// Decrement the reference count.
// This must be a macro, not an inline function, because pointer_p() is a
// non-virtual member function, so that the compiler can optimize it. 
#define dec_refcount(x)  \
	if ((x).pointer_p())					\
		dec_pointer_refcount((x).heappointer);		\

// The declaration of a copy constructor.
#define CL_DEFINE_COPY_CONSTRUCTOR(_class_)			\
inline _class_::_class_ (const _class_& x)			\
{								\
	inc_refcount(x);					\
	word = x.word;						\
}

// The declaration of an assignment operator.
#define CL_DEFINE_ASSIGNMENT_OPERATOR(dest_class,src_class)	\
inline dest_class& dest_class::operator= (const src_class& x)	\
{								\
	/* Be careful, we might be assigning x to itself. */	\
	inc_refcount(x);					\
	dec_refcount(*this);					\
	word = x.word;						\
	return *this;						\
}

// We have a small problem with destructors: The specialized destructor
// of a leaf class such as `cl_SF' should be more efficient than the
// general destructor for `cl_N'. Since (by C++ specs) destructing a cl_SF
// would run the destructors for cl_SF, cl_F, cl_R, cl_N (in that order),
// and in the last step the compiler does not know any more that the object
// actually is a cl_SF, there is no way to optimize the destructor!
// ("progn-reversed" method combination is evil.)
// And if we define "mirror"/"shadow" classes with no destructors (such
// that `cl_F' inherits from `cl_F_no_destructor' buts adds a destructor)
// then we need to add explicit conversion operators cl_SF -> cl_F -> cl_R ...,
// with the effect that calling an overloaded function like `as_cl_F'
// (which has two signatures `as_cl_F(cl_number)' and `as_cl_F(cl_F)')
// with a cl_SF argument gives an "call of overloaded function is ambiguous"
// error.
// There is no help: If we want overloaded functions to be callable in a way
// that makes sense, `cl_SF' has to be a subclass of `cl_F', and then the
// destructor of `cl_SF' will do at least as much computation as the `cl_F'
// destructor. Praise C++ ! :-((
// (Even making `pointer_p()' a virtual function would not help.)


// Concrete class of all numbers.
// typedef
class cl_number {
protected:
	union {
		void*   pointer;
		cl_heap* heappointer;
		cl_uint word;
	};
public:
// Default constructor. (Used for objects with no initializer.)
	cl_number ();
// Copy constructor. (Used for function argument passing and function
// return value, and of course for objects with initializers of the same type.)
	cl_number (const cl_number& x);
// Converters. (Used for function argument passing and function return values.)
// Assignment operators. (Used for assignments.)
	cl_number& operator= (const cl_number&);
	cl_number& operator= (const cl_N&);
	cl_number& operator= (const cl_R&);
	cl_number& operator= (const cl_F&);
	cl_number& operator= (const cl_SF&);
	cl_number& operator= (const cl_FF&);
	cl_number& operator= (const cl_DF&);
	cl_number& operator= (const cl_LF&);
	cl_number& operator= (const cl_RA&);
	cl_number& operator= (const cl_I&);
// Destructor. (Used when a variable goes out of scope.)
	~cl_number ();
// Distinguish immediate data from pointer.
	cl_boolean pointer_p() const
		{ return cl_pointer_p(word); }
// Return the type tag of an immediate number.
	cl_uint nonpointer_tag () const
		{ return cl_tag(word); }
// Return the type tag of a heap-allocated number.
	int pointer_typetag () const
		{ return ((cl_heap *)pointer)->typetag; }
// Constructors and assignment operators from C numeric types.
	cl_number (const int);		// |argument| must be < 2^29
	cl_number (const unsigned int);	// argument must be < 2^29
	cl_number (const long);
	cl_number (const unsigned long);
	cl_number (const float);
	cl_number (const double);
	cl_number& operator= (const int);	// |argument| must be < 2^29
	cl_number& operator= (const unsigned int); // argument must be < 2^29
	cl_number& operator= (const long);
	cl_number& operator= (const unsigned long);
	cl_number& operator= (const float);
	cl_number& operator= (const double);
// Other constructors.
//	cl_number (const char *);
// Private pointer manipulations.
	operator cl_private_thing () const;
private:
// Friend declarations. They are for the compiler. Just ignore them.
	friend cl_boolean cl_N_p (const cl_number& x);
	friend cl_boolean cl_R_p (const cl_number& x);
	friend cl_boolean cl_F_p (const cl_number& x);
	friend cl_boolean cl_SF_p (const cl_number& x);
	friend cl_boolean cl_FF_p (const cl_number& x);
	friend cl_boolean cl_DF_p (const cl_number& x);
	friend cl_boolean cl_LF_p (const cl_number& x);
	friend cl_boolean cl_RA_p (const cl_number& x);
	friend cl_boolean cl_I_p (const cl_number& x);
	friend struct cl_complex* TheComplex (const cl_number& obj);
	friend struct cl_complex* allocate_complex (const cl_R& real, const cl_R& imag);
	friend cl_boolean realp (const cl_N& x);
	friend cl_boolean complexp (const cl_N& x);
	friend cl_boolean rationalp (const cl_R& x);
	friend cl_boolean integerp (const cl_R& x);
	friend cl_boolean floatp (const cl_R& x);
	friend cl_boolean eq (const cl_R& x, sint32 y);
#if !defined(CL_WIDE_POINTERS)
	friend struct cl_ffloat* TheFfloat (const cl_number& obj);
#endif
	friend struct cl_dfloat* TheDfloat (const cl_number& obj);
	friend struct cl_lfloat* TheLfloat (const cl_number& obj);
	friend struct cl_ratio* TheRatio (const cl_number& obj);
	friend struct cl_ratio* allocate_ratio (const cl_I& num, const cl_I& den);
	friend cl_boolean rationalp (const cl_RA& x);
	friend cl_boolean integerp (const cl_RA& x);
	friend cl_boolean zerop (const cl_RA& x);
	friend cl_boolean eq (const cl_RA& x, sint32 y);
	friend struct cl_bignum* TheBignum (const cl_number& obj);
};

inline cl_number::~cl_number ()
	{ dec_refcount(*this); }

// Abstract classes for N, R, F, RA, I.

// typedef
class cl_N : public cl_number {
public:
// Default constructor.
	cl_N ();
// Converters.
// Assignment operators.
	cl_N& operator= (const cl_N&);
	cl_N& operator= (const cl_R&);
	cl_N& operator= (const cl_F&);
	cl_N& operator= (const cl_SF&);
	cl_N& operator= (const cl_FF&);
	cl_N& operator= (const cl_DF&);
	cl_N& operator= (const cl_LF&);
	cl_N& operator= (const cl_RA&);
	cl_N& operator= (const cl_I&);
// Constructors and assignment operators from C numeric types.
	cl_N (const int);		// |argument| must be < 2^29
	cl_N (const unsigned int);	// argument must be < 2^29
	cl_N (const long);
	cl_N (const unsigned long);
	cl_N (const float);
	cl_N (const double);
	cl_N& operator= (const int);		// |argument| must be < 2^29
	cl_N& operator= (const unsigned int);	// argument must be < 2^29
	cl_N& operator= (const long);
	cl_N& operator= (const unsigned long);
	cl_N& operator= (const float);
	cl_N& operator= (const double);
// Other constructors.
	cl_N (const char *);
// Private constructor.
	cl_N (cl_private_thing);
	cl_N (struct cl_complex *);
private:
// Friend declarations. They are for the compiler. Just ignore them.
};

// typedef
class cl_R : public cl_N {
public:
// Default constructor.
	cl_R ();
// Converters.
// Assignment operators.
	cl_R& operator= (const cl_R&);
	cl_R& operator= (const cl_F&);
	cl_R& operator= (const cl_SF&);
	cl_R& operator= (const cl_FF&);
	cl_R& operator= (const cl_DF&);
	cl_R& operator= (const cl_LF&);
// Constructors and assignment operators from C numeric types.
	cl_R (const int);		// |argument| must be < 2^29
	cl_R (const unsigned int);	// argument must be < 2^29
	cl_R (const long);
	cl_R (const unsigned long);
	cl_R (const float);
	cl_R (const double);
	cl_R& operator= (const int);		// |argument| must be < 2^29
	cl_R& operator= (const unsigned int);	// argument must be < 2^29
	cl_R& operator= (const long);
	cl_R& operator= (const unsigned long);
	cl_R& operator= (const float);
	cl_R& operator= (const double);
// Other constructors.
	cl_R (const char *);
// Private constructor.
	cl_R (cl_private_thing);
private:
// Friend declarations. They are for the compiler. Just ignore them.
};

// typedef
class cl_F : public cl_R {
public:
// Default constructor.
	cl_F ();
// Converters.
// Assignment operators.
	cl_F& operator= (const cl_F&);
	cl_F& operator= (const cl_SF&);
	cl_F& operator= (const cl_FF&);
	cl_F& operator= (const cl_DF&);
	cl_F& operator= (const cl_LF&);
// Constructors and assignment operators from C numeric types.
	cl_F (const float);
	cl_F (const double);
	cl_F& operator= (const float);
	cl_F& operator= (const double);
// Other constructors.
	cl_F (const char *);
// Private constructor.
	cl_F (cl_private_thing);
private:
// Friend declarations. They are for the compiler. Just ignore them.
};

// typedef
class cl_SF : public cl_F {
public:
// Default constructor.
	cl_SF ();
// Assignment operators.
	cl_SF& operator= (const cl_SF&);
// Optimization of method pointer_p().
	cl_boolean pointer_p() const
		{ return cl_false; }
// Faster pointer_p() gives a faster copy constructor (but not destructor!!!).
	cl_SF (const cl_SF& x);
// Other constructors.
	cl_SF (const char *);
// Private constructor.
	cl_SF (cl_private_thing);
	cl_SF (struct cl_sfloat * /* NULL! */, cl_uint);
private:
// Friend declarations. They are for the compiler. Just ignore them.
	friend uintL SF_uexp (const cl_SF& x);
	friend cl_signean SF_sign (const cl_SF& x);
	friend uintL SF_mant (const cl_SF& x);
	friend cl_boolean zerop (const cl_SF& x);
	friend cl_boolean minusp (const cl_SF& x);
	friend cl_signean cl_compare (const cl_SF& x, const cl_SF& y);
	friend cl_SF operator+ (const cl_SF& x1, const cl_SF& x2);
	friend cl_SF operator- (const cl_SF& x);
	friend cl_SF operator- (const cl_SF& x1, const cl_SF& x2);
	friend cl_SF fround (const cl_SF& x);
	friend cl_SF ftruncate (const cl_SF& x);
	friend cl_SF futruncate (const cl_SF& x);
};

// typedef
class cl_FF : public cl_F {
public:
// Default constructor.
	cl_FF ();
// Assignment operators.
	cl_FF& operator= (const cl_FF&);
// Optimization of method pointer_p().
	cl_boolean pointer_p() const
#if defined(CL_WIDE_POINTERS)
		{ return cl_false; }
#else
		{ return cl_true; }
#endif
// Faster pointer_p() gives a faster copy constructor (but not destructor!!!).
	cl_FF (const cl_FF& x);
// Constructors and assignment operators from C numeric types.
	cl_FF (const float);
	cl_FF& operator= (const float);
// Other constructors.
	cl_FF (const char *);
// Private constructor.
	cl_FF (cl_private_thing);
#if defined(CL_WIDE_POINTERS)
	cl_FF (struct cl_ffloat * /* NULL! */, cl_uint);
#else
	cl_FF (struct cl_ffloat *);
// Private pointer manipulations.
	operator struct cl_ffloat * () const;
#endif
private:
// Friend declarations. They are for the compiler. Just ignore them.
#if defined(CL_WIDE_POINTERS)
	friend uint32 cl_ffloat_value (const cl_FF& x);
	friend cl_private_thing cl_FF_pointer_value (cl_FF_pointer p);
#endif
};

// typedef
class cl_DF : public cl_F {
public:
// Default constructor.
	cl_DF ();
// Assignment operators.
	cl_DF& operator= (const cl_DF&);
// Optimization of method pointer_p().
	cl_boolean pointer_p() const
		{ return cl_true; }
// Faster pointer_p() gives a faster copy constructor (but not destructor!!!).
	cl_DF (const cl_DF& x);
// Constructors and assignment operators from C numeric types.
	cl_DF (const double);
	cl_DF& operator= (const double);
// Other constructors.
	cl_DF (const char *);
// Private constructor.
	cl_DF (cl_private_thing);
	cl_DF (struct cl_dfloat *);
// Private pointer manipulations.
	operator struct cl_dfloat * () const;
private:
// Friend declarations. They are for the compiler. Just ignore them.
};

// typedef
class cl_LF : public cl_F {
public:
// Default constructor.
	cl_LF ();
// Assignment operators.
	cl_LF& operator= (const cl_LF&);
// Optimization of method pointer_p().
	cl_boolean pointer_p() const
		{ return cl_true; }
// Faster pointer_p() gives a faster copy constructor (but not destructor!!!).
	cl_LF (const cl_LF& x);
// Other constructors.
	cl_LF (const char *);
// Private constructor.
	cl_LF (cl_private_thing);
	cl_LF (struct cl_lfloat *);
// Private pointer manipulations.
	operator struct cl_lfloat * () const;
private:
// Friend declarations. They are for the compiler. Just ignore them.
	friend cl_LF LF_LF_plus_LF (const cl_LF& arg1, const cl_LF& arg2);
};

// typedef
class cl_RA : public cl_R {
public:
// Default constructor.
	cl_RA ();
// Converters.
// Assignment operators.
	cl_RA& operator= (const cl_RA&);
	cl_RA& operator= (const cl_I&);
// Constructors and assignment operators from C numeric types.
	cl_RA (const int);		// |argument| must be < 2^29
	cl_RA (const unsigned int);	// argument must be < 2^29
	cl_RA (const long);
	cl_RA (const unsigned long);
	cl_RA& operator= (const int);		// |argument| must be < 2^29
	cl_RA& operator= (const unsigned int);	// argument must be < 2^29
	cl_RA& operator= (const long);
	cl_RA& operator= (const unsigned long);
// Other constructors.
	cl_RA (const char *);
// Private constructor.
	cl_RA (cl_private_thing);
	cl_RA (struct cl_ratio *);
private:
// Friend declarations. They are for the compiler. Just ignore them.
};

// typedef
class cl_I : public cl_RA {
public:
// Default constructor.
	cl_I ();
// Assignment operators.
	cl_I& operator= (const cl_I&);
// Constructors and assignment operators from C numeric types.
	cl_I (const int);		// |argument| must be < 2^29
	cl_I (const unsigned int);	// argument must be < 2^29
	cl_I (const long);
	cl_I (const unsigned long);
	cl_I& operator= (const int);		// |argument| must be < 2^29
	cl_I& operator= (const unsigned int);	// argument must be < 2^29
	cl_I& operator= (const long);
	cl_I& operator= (const unsigned long);
// Other constructors.
	cl_I (const char *);
// Private constructor.
	cl_I (cl_private_thing);
	cl_I (struct cl_fixnum * /* NULL! */, cl_uint);
	cl_I (struct cl_bignum *);
private:
// Friend declarations. They are for the compiler. Just ignore them.
	friend cl_boolean minusp (const cl_I& x);
	friend cl_boolean zerop (const cl_I& x);
	friend cl_boolean eq (const cl_I& x, sint32 y);
	friend cl_uint cl_FN_word (const cl_I& x);
	friend sint32 FN_to_L (const cl_I& x);
	friend cl_boolean cl_equal (const cl_I& x, const cl_I& y);
	friend cl_signean cl_compare (const cl_I& x, const cl_I& y);
	friend cl_I logior (const cl_I& x, const cl_I& y);
	friend cl_I logxor (const cl_I& x, const cl_I& y);
	friend cl_I logand (const cl_I& x, const cl_I& y);
	friend cl_I logeqv (const cl_I& x, const cl_I& y);
	friend cl_I lognand (const cl_I& x, const cl_I& y);
	friend cl_I lognor (const cl_I& x, const cl_I& y);
	friend cl_I logandc2 (const cl_I& x, const cl_I& y);
	friend cl_I logorc2 (const cl_I& x, const cl_I& y);
	friend cl_I lognot (const cl_I& x);
	friend cl_boolean logtest (const cl_I& x, const cl_I& y);
	friend cl_boolean oddp (const cl_I& x);
	friend cl_I plus1 (const cl_I& x);
	friend cl_I minus1 (const cl_I& x);
};

// The assignment operators:
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_number)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_N)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_R)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_F)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_SF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_FF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_DF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_LF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_RA)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_number, cl_I)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_N)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_R)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_F)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_SF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_FF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_DF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_LF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_RA)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_N, cl_I)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_R)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_F)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_SF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_FF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_DF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_R, cl_LF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_F, cl_F)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_F, cl_SF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_F, cl_FF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_F, cl_DF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_F, cl_LF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_SF, cl_SF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_FF, cl_FF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_DF, cl_DF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_LF, cl_LF)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_RA, cl_RA)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_RA, cl_I)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_I, cl_I)

// The default constructors.
inline cl_number::cl_number ()
	{ word = cl_combine(cl_FN_tag,0); }
inline cl_N::cl_N ()
	{ word = cl_combine(cl_FN_tag,0); }
inline cl_R::cl_R ()
	{ word = cl_combine(cl_FN_tag,0); }
inline cl_F::cl_F ()
	{ word = cl_combine(cl_SF_tag,0); }
inline cl_SF::cl_SF ()
	{ word = cl_combine(cl_SF_tag,0); }
#if defined(CL_WIDE_POINTERS)
inline cl_FF::cl_FF ()
	{ word = cl_combine(cl_FF_tag,0); }
// `cl_FF_pointer' is just `cl_FF' without destructor.
typedef cl_FF cl_FF_pointer;
inline cl_private_thing cl_FF_pointer_value (const cl_FF_pointer p)
	{ return p.pointer; }
#else
// Private pointer manipulations. Never throw away a `struct cl_ffloat *'!
inline cl_FF::operator struct cl_ffloat * () const
{
	inc_refcount(*this);
	return (struct cl_ffloat *) pointer;
}
inline cl_FF::cl_FF ()
	{ extern cl_FF cl_FF_0;
	  pointer = (struct cl_ffloat *) cl_FF_0;
	}
#if 0 // see cl_FF.h
inline cl_FF::cl_FF (struct cl_ffloat * ptr)
	{ pointer = ptr; }
#endif
// `cl_FF_pointer' is just `cl_FF' as a pointer, without destructor.
typedef struct cl_ffloat * cl_FF_pointer;
inline cl_private_thing cl_FF_pointer_value (const cl_FF_pointer p)
	{ return (cl_private_thing)p; }
#endif
// Private pointer manipulations. Never throw away a `struct cl_dfloat *'!
inline cl_DF::operator struct cl_dfloat * () const
{
	inc_refcount(*this);
	return (struct cl_dfloat *) pointer;
}
inline cl_DF::cl_DF ()
	{ extern cl_DF cl_DF_0;
	  pointer = (struct cl_dfloat *) cl_DF_0;
	}
#if 0 // see cl_DF.h
inline cl_DF::cl_DF (struct cl_dfloat * ptr)
	{ pointer = ptr; }
#endif
// Private pointer manipulations. Never throw away a `struct cl_lfloat *'!
inline cl_LF::operator struct cl_lfloat * () const
{
	inc_refcount(*this);
	return (struct cl_lfloat *) pointer;
}
inline cl_LF::cl_LF ()
	{ extern cl_LF cl_LF_0;
	  pointer = (struct cl_lfloat *) cl_LF_0;
	}
inline cl_RA::cl_RA ()
	{ word = cl_combine(cl_FN_tag,0); }
inline cl_I::cl_I ()
	{ word = cl_combine(cl_FN_tag,0); }

// The copy constructors.
CL_DEFINE_COPY_CONSTRUCTOR(cl_number)
CL_DEFINE_COPY_CONSTRUCTOR(cl_SF)
CL_DEFINE_COPY_CONSTRUCTOR(cl_FF)
CL_DEFINE_COPY_CONSTRUCTOR(cl_DF)
CL_DEFINE_COPY_CONSTRUCTOR(cl_LF)

// Constructors and assignment operators from C numeric types.

#define CL_DEFINE_INT_CONSTRUCTOR(_class_,_type_)  \
inline _class_::_class_ (_type_ wert)					\
{									\
	word = cl_combine(cl_FN_tag,wert);				\
}
#define CL_DEFINE_INT_CONSTRUCTORS(_class_)  \
CL_DEFINE_INT_CONSTRUCTOR(_class_, int)					\
CL_DEFINE_INT_CONSTRUCTOR(_class_, unsigned int)

CL_DEFINE_INT_CONSTRUCTORS(cl_number)
CL_DEFINE_INT_CONSTRUCTORS(cl_N)
CL_DEFINE_INT_CONSTRUCTORS(cl_R)
CL_DEFINE_INT_CONSTRUCTORS(cl_RA)
CL_DEFINE_INT_CONSTRUCTORS(cl_I)

#define CL_DEFINE_INT_ASSIGNMENT_OPERATOR(_class_,_type_)  \
inline _class_& _class_::operator= (_type_ wert)			\
{									\
	dec_refcount(*this);						\
	word = cl_combine(cl_FN_tag,wert);				\
	return *this;							\
}
#define CL_DEFINE_INT_ASSIGNMENT_OPERATORS(_class_)  \
CL_DEFINE_INT_ASSIGNMENT_OPERATOR(_class_, int)				\
CL_DEFINE_INT_ASSIGNMENT_OPERATOR(_class_, unsigned int)

CL_DEFINE_INT_ASSIGNMENT_OPERATORS(cl_number)
CL_DEFINE_INT_ASSIGNMENT_OPERATORS(cl_N)
CL_DEFINE_INT_ASSIGNMENT_OPERATORS(cl_R)
CL_DEFINE_INT_ASSIGNMENT_OPERATORS(cl_RA)
CL_DEFINE_INT_ASSIGNMENT_OPERATORS(cl_I)

#if (long_bitsize==32)
// `long' == `sintL', `unsigned long' == `uintL'.
#define CL_DEFINE_LONG_CONSTRUCTORS(_class_)  \
inline _class_::_class_ (long wert)					\
{									\
	extern cl_I L_to_I (sint32 wert);				\
	*this = L_to_I(wert);						\
}									\
inline _class_::_class_ (unsigned long wert)				\
{									\
	extern cl_I UL_to_I (uint32 wert);				\
	*this = UL_to_I(wert);						\
}
#elif (long_bitsize==64)
// `long' == `sintQ', `unsigned long' == `uintQ'.
#define CL_DEFINE_LONG_CONSTRUCTORS(_class_)  \
inline _class_::_class_ (long wert)					\
{									\
	extern cl_I Q_to_I (sint64 wert);				\
	*this = Q_to_I(wert);						\
}									\
inline _class_::_class_ (unsigned long wert)				\
{									\
	extern cl_I UQ_to_I (uint64 wert);				\
	*this = UQ_to_I(wert);						\
}
#endif
// CL_DEFINE_LONG_CONSTRUCTORS shouldn't generate a temporary cl_I. ??

CL_DEFINE_LONG_CONSTRUCTORS(cl_number)
CL_DEFINE_LONG_CONSTRUCTORS(cl_N)
CL_DEFINE_LONG_CONSTRUCTORS(cl_R)
CL_DEFINE_LONG_CONSTRUCTORS(cl_RA)
CL_DEFINE_LONG_CONSTRUCTORS(cl_I)

#if (long_bitsize==32)
// `long' == `sintL', `unsigned long' == `uintL'.
#define CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(_class_)  \
inline _class_& _class_::operator= (long wert)				\
{									\
	extern cl_I L_to_I (sint32 wert);				\
	*this = L_to_I(wert);						\
	return *this;							\
}									\
inline _class_& _class_::operator= (unsigned long wert)			\
{									\
	extern cl_I UL_to_I (uint32 wert);				\
	*this = UL_to_I(wert);						\
	return *this;							\
}
#elif (long_bitsize==64)
// `long' == `sintQ', `unsigned long' == `uintQ'.
#define CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(_class_)  \
inline _class_& _class_::operator= (long wert)				\
{									\
	extern cl_I Q_to_I (sint64 wert);				\
	*this = Q_to_I(wert);						\
	return *this;							\
}									\
inline _class_& _class_::operator= (unsigned long wert)			\
{									\
	extern cl_I UQ_to_I (uint64 wert);				\
	*this = UQ_to_I(wert);						\
	return *this;							\
}
#endif

CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(cl_number)
CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(cl_N)
CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(cl_R)
CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(cl_RA)
CL_DEFINE_LONG_ASSIGNMENT_OPERATORS(cl_I)

// Conversions to subtypes:
// As(cl_I)(x) returns x as a cl_I. It first checks that x is a cl_I
// and then returns it without change of representation.
#if 0 // no debug information
  #define As(type)  as_##type
extern const cl_N& as_cl_N (const cl_number& x);
extern const cl_R& as_cl_R (const cl_number& x);
extern const cl_F& as_cl_F (const cl_number& x);
extern const cl_SF& as_cl_SF (const cl_number& x);
extern const cl_FF& as_cl_FF (const cl_number& x);
extern const cl_DF& as_cl_DF (const cl_number& x);
extern const cl_LF& as_cl_LF (const cl_number& x);
extern const cl_RA& as_cl_RA (const cl_number& x);
extern const cl_I& as_cl_I (const cl_number& x);
// These special cases need no checking.
inline const cl_N& as_cl_N (const cl_N& x) { return x; }
inline const cl_R& as_cl_R (const cl_R& x) { return x; }
inline const cl_F& as_cl_F (const cl_F& x) { return x; }
inline const cl_SF& as_cl_SF (const cl_SF& x) { return x; }
inline const cl_FF& as_cl_FF (const cl_FF& x) { return x; }
inline const cl_DF& as_cl_DF (const cl_DF& x) { return x; }
inline const cl_LF& as_cl_LF (const cl_LF& x) { return x; }
inline const cl_RA& as_cl_RA (const cl_RA& x) { return x; }
inline const cl_I& as_cl_I (const cl_I& x) { return x; }
#else // Line number information for ease of debugging.
  #define As(type)  as_##type cl_as_aux
  #define cl_as_aux(expr)  (expr,__FILE__,__LINE__)
extern const cl_N& as_cl_N (const cl_number& x, const char * filename, int line);
extern const cl_R& as_cl_R (const cl_number& x, const char * filename, int line);
extern const cl_F& as_cl_F (const cl_number& x, const char * filename, int line);
extern const cl_SF& as_cl_SF (const cl_number& x, const char * filename, int line);
extern const cl_FF& as_cl_FF (const cl_number& x, const char * filename, int line);
extern const cl_DF& as_cl_DF (const cl_number& x, const char * filename, int line);
extern const cl_LF& as_cl_LF (const cl_number& x, const char * filename, int line);
extern const cl_RA& as_cl_RA (const cl_number& x, const char * filename, int line);
extern const cl_I& as_cl_I (const cl_number& x, const char * filename, int line);
// These special cases need no checking.
inline const cl_N& as_cl_N (const cl_N& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_R& as_cl_R (const cl_R& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_F& as_cl_F (const cl_F& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_SF& as_cl_SF (const cl_SF& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_FF& as_cl_FF (const cl_FF& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_DF& as_cl_DF (const cl_DF& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_LF& as_cl_LF (const cl_LF& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_RA& as_cl_RA (const cl_RA& x, const char * filename, int line) { (void)filename; (void)line; return x; }
inline const cl_I& as_cl_I (const cl_I& x, const char * filename, int line) { (void)filename; (void)line; return x; }
#endif


// Constructors and assignment operators from C numeric types.

// from `float':
extern cl_FF_pointer cl_float_to_FF_pointer (const union ffloatjanus& val);

#define CL_DEFINE_FLOAT_CONSTRUCTOR(_class_)				\
inline _class_ :: _class_ (const float x)				\
{									\
	pointer = cl_FF_pointer_value(cl_float_to_FF_pointer(*(const union ffloatjanus *)&x)); \
}									\
inline _class_& _class_::operator= (const float x)			\
{									\
	cl_private_thing tmp = cl_FF_pointer_value(cl_float_to_FF_pointer(*(const union ffloatjanus *)&x)); \
	dec_refcount(*this);						\
	pointer = tmp;							\
	return *this;							\
}

CL_DEFINE_FLOAT_CONSTRUCTOR(cl_FF)
CL_DEFINE_FLOAT_CONSTRUCTOR(cl_F)
CL_DEFINE_FLOAT_CONSTRUCTOR(cl_R)
CL_DEFINE_FLOAT_CONSTRUCTOR(cl_N)

// from `double':
extern struct cl_dfloat * cl_double_to_DF_pointer (const union dfloatjanus& val);

#define CL_DEFINE_DOUBLE_CONSTRUCTOR(_class_)				\
inline _class_::_class_ (const double x)				\
{									\
	pointer = cl_double_to_DF_pointer(*(const union dfloatjanus *)&x); \
}									\
inline _class_& _class_::operator= (const double x)			\
{									\
	struct cl_dfloat * tmp = cl_double_to_DF_pointer(*(const union dfloatjanus *)&x); \
	dec_refcount(*this);						\
	pointer = tmp;							\
	return *this;							\
}

CL_DEFINE_DOUBLE_CONSTRUCTOR(cl_DF)
CL_DEFINE_DOUBLE_CONSTRUCTOR(cl_F)
CL_DEFINE_DOUBLE_CONSTRUCTOR(cl_R)
CL_DEFINE_DOUBLE_CONSTRUCTOR(cl_N)


// Private constructors.
inline cl_number::operator cl_private_thing () const
{
	inc_refcount(*this);
	return (cl_private_thing) pointer;
}
#if 0 // See cl_FF.h, cl_DF.h, cl_LF_impl.h etc.
inline cl_N::cl_N (cl_private_thing ptr) { pointer = ptr; }
inline cl_R::cl_R (cl_private_thing ptr) { pointer = ptr; }
inline cl_F::cl_F (cl_private_thing ptr) { pointer = ptr; }
inline cl_SF::cl_SF (cl_private_thing ptr) { pointer = ptr; }
inline cl_FF::cl_FF (cl_private_thing ptr) { pointer = ptr; }
inline cl_DF::cl_DF (cl_private_thing ptr) { pointer = ptr; }
inline cl_LF::cl_LF (cl_private_thing ptr) { pointer = ptr; }
inline cl_RA::cl_RA (cl_private_thing ptr) { pointer = ptr; }
inline cl_I::cl_I (cl_private_thing ptr) { pointer = ptr; }
#endif


// Return type for integer_decode_float:
struct cl_idecoded_float {
	cl_I mantissa;
	cl_I exponent;
	cl_I sign;
// Constructor.
	cl_idecoded_float () {}
	cl_idecoded_float (const cl_I& m, const cl_I& e, const cl_I& s) : mantissa(m), exponent(e), sign(s) {}
};


// Conversion:
//inline cl_N& cl_as_N (const cl_number& x)
//{ return *(cl_N*)(&x); }


// Hack section.

// Conversions to subtypes without checking:
// The(cl_I)(x) converts x to a cl_I, without change of representation!
  #define The(type)  *(const type *) & cl_number_identity
// This inline function is for type checking purposes only.
  inline const cl_number& cl_number_identity (const cl_number& x) { return x; }

// Mutable(type,x);
// x should be a variable `const type x' or `const type& x'.
// This macro introduces a new variable `type& x' whose value can be
// modified. Useful for modifying the argument of a function which takes
// a `const type &x'.
// Warning: To apply this to a function's formal parameter, a block { ... }
// must be inserted.
  #define Mutable(type,x)  \
    type __copied_##x = x;						\
    type& x = __copied_##x;

// DeclareType(type,x);
// x should be a variable of some subtype of `cl_number'. type should be
// a subtype of `cl_number'. A new variable of the given type is declared,
// with name x and which refers to x (by reference, with const attribute).
  #define DeclareType(type,x)  \
    const type& __tmp_##x = *(const type*) &x;				\
    const type& x = __tmp_##x;


#endif /* _CL_NUMBER_H */
