// Public rational number operations.

#ifndef _CL_RATIONAL_H
#define _CL_RATIONAL_H

#include "cl_number.h"


// Liefert (- r), wo r eine rationale Zahl ist.
extern cl_RA operator- (const cl_RA& r);

// (+ r s), wo r und s rationale Zahlen sind.
extern cl_RA operator+ (const cl_RA& r, const cl_RA& s);
// Dem C++-Compiler mu man auch das Folgende sagen:
inline cl_RA operator+ (const int x, const cl_RA& y)
	{ return cl_I(x) + y; }
inline cl_RA operator+ (const unsigned int x, const cl_RA& y)
	{ return cl_I(x) + y; }
inline cl_RA operator+ (const long x, const cl_RA& y)
	{ return cl_I(x) + y; }
inline cl_RA operator+ (const unsigned long x, const cl_RA& y)
	{ return cl_I(x) + y; }
inline cl_RA operator+ (const cl_RA& x, const int y)
	{ return x + cl_I(y); }
inline cl_RA operator+ (const cl_RA& x, const unsigned int y)
	{ return x + cl_I(y); }
inline cl_RA operator+ (const cl_RA& x, const long y)
	{ return x + cl_I(y); }
inline cl_RA operator+ (const cl_RA& x, const unsigned long y)
	{ return x + cl_I(y); }

// (- r s), wo r und s rationale Zahlen sind.
extern cl_RA operator- (const cl_RA& r, const cl_RA& s);
// Dem C++-Compiler mu man auch das Folgende sagen:
inline cl_RA operator- (const int x, const cl_RA& y)
	{ return cl_I(x) - y; }
inline cl_RA operator- (const unsigned int x, const cl_RA& y)
	{ return cl_I(x) - y; }
inline cl_RA operator- (const long x, const cl_RA& y)
	{ return cl_I(x) - y; }
inline cl_RA operator- (const unsigned long x, const cl_RA& y)
	{ return cl_I(x) - y; }
inline cl_RA operator- (const cl_RA& x, const int y)
	{ return x - cl_I(y); }
inline cl_RA operator- (const cl_RA& x, const unsigned int y)
	{ return x - cl_I(y); }
inline cl_RA operator- (const cl_RA& x, const long y)
	{ return x - cl_I(y); }
inline cl_RA operator- (const cl_RA& x, const unsigned long y)
	{ return x - cl_I(y); }

// (1+ r), wo r eine rationale Zahl ist.
extern cl_RA plus1 (const cl_RA& r);

// (1- r), wo r eine rationale Zahl ist.
extern cl_RA minus1 (const cl_RA& r);

// cl_equal(r,s) vergleicht zwei rationale Zahlen r und s auf Gleichheit.
extern cl_boolean cl_equal (const cl_RA& r, const cl_RA& s);

// cl_compare(r,s) vergleicht zwei rationale Zahlen r und s.
// Ergebnis: 0 falls r=s, +1 falls r>s, -1 falls r<s.
extern cl_signean cl_compare (const cl_RA& r, const cl_RA& s);

inline bool operator== (const cl_RA& x, const cl_RA& y)
	{ return cl_equal(x,y); }
inline bool operator!= (const cl_RA& x, const cl_RA& y)
	{ return !cl_equal(x,y); }
inline bool operator<= (const cl_RA& x, const cl_RA& y)
	{ return cl_compare(x,y)<=0; }
inline bool operator< (const cl_RA& x, const cl_RA& y)
	{ return cl_compare(x,y)<0; }
inline bool operator>= (const cl_RA& x, const cl_RA& y)
	{ return cl_compare(x,y)>=0; }
inline bool operator> (const cl_RA& x, const cl_RA& y)
	{ return cl_compare(x,y)>0; }

// Kehrwert (/ r), wo r eine rationale Zahl ist.
extern cl_RA recip (const cl_RA& r);

// Liefert (* r s), wo r und s rationale Zahlen sind.
extern cl_RA operator* (const cl_RA& r, const cl_RA& s);

// Liefert (/ r s), wo r und s rationale Zahlen sind.
extern cl_RA operator/ (const cl_RA& r, const cl_RA& s);

// Return type for rounding operators.
// x / y  --> (q,r) with x = y*q+r.
struct cl_RA_div_t {
	cl_I quotient;
	cl_RA remainder;
// Constructor.
	cl_RA_div_t () {}
	cl_RA_div_t (const cl_I& q, const cl_RA& r) : quotient(q), remainder(r) {}
};

// Liefert ganzzahligen und gebrochenen Anteil einer rationalen Zahl.
// (q,r) := (floor x)
// floor2(x)
// > x: rationale Zahl
// < q,r: Quotient q, ein Integer, Rest r, eine rationale Zahl
  extern cl_RA_div_t floor2 (const cl_RA& x);
  extern cl_I floor1 (const cl_RA& x);

// Liefert ganzzahligen und gebrochenen Anteil einer rationalen Zahl.
// (q,r) := (ceiling x)
// ceiling2(x)
// > x: rationale Zahl
// < q,r: Quotient q, ein Integer, Rest r, eine rationale Zahl
  extern cl_RA_div_t ceiling2 (const cl_RA& x);
  extern cl_I ceiling1 (const cl_RA& x);

// Liefert ganzzahligen und gebrochenen Anteil einer rationalen Zahl.
// (q,r) := (truncate x)
// truncate2(x)
// > x: rationale Zahl
// < q,r: Quotient q, ein Integer, Rest r, eine rationale Zahl
  extern cl_RA_div_t truncate2 (const cl_RA& x);
  extern cl_I truncate1 (const cl_RA& x);

// Liefert ganzzahligen und gebrochenen Anteil einer rationalen Zahl.
// (q,r) := (round x)
// round2(x)
// > x: rationale Zahl
// < q,r: Quotient q, ein Integer, Rest r, eine rationale Zahl
  extern cl_RA_div_t round2 (const cl_RA& x);
  extern cl_I round1 (const cl_RA& x);

// floor2(x,y) liefert (floor x y).
extern cl_RA_div_t floor2 (const cl_RA& x, const cl_RA& y);
extern cl_I floor1 (const cl_RA& x, const cl_RA& y);

// ceiling2(x,y) liefert (ceiling x y).
extern cl_RA_div_t ceiling2 (const cl_RA& x, const cl_RA& y);
extern cl_I ceiling1 (const cl_RA& x, const cl_RA& y);

// truncate2(x,y) liefert (truncate x y).
extern cl_RA_div_t truncate2 (const cl_RA& x, const cl_RA& y);
extern cl_I truncate1 (const cl_RA& x, const cl_RA& y);

// round2(x,y) liefert (round x y).
extern cl_RA_div_t round2 (const cl_RA& x, const cl_RA& y);
extern cl_I round1 (const cl_RA& x, const cl_RA& y);

// signum(x) liefert (signum x), wo x eine rationale Zahl ist.
extern cl_RA signum (const cl_RA& x);

// (expt x y), wo x eine rationale Zahl und y ein Integer >0 ist.
extern cl_RA expt_pos (const cl_RA& x, uintL y);
extern cl_RA expt_pos (const cl_RA& x, const cl_I& y);

// (expt x y), wo x eine rationale Zahl und y ein Integer ist.
extern cl_RA expt (const cl_RA& x, sintL y);
extern cl_RA expt (const cl_RA& x, const cl_I& y);

// Stellt fest, ob eine rationale Zahl >=0 das Quadrat einer rationalen Zahl
// ist.
// sqrtp(x,&w)
// > x: eine rationale Zahl >=0
// < w: rationale Zahl (sqrt x) falls x Quadratzahl
// < ergebnis: cl_true   ..................., cl_false sonst
  extern cl_boolean sqrtp (const cl_RA& x, cl_RA* w);

// sqrt(x) = (sqrt x) zieht die Wurzel aus einer rationalen Zahl x >=0.
extern cl_R sqrt (const cl_RA& x);

// Stellt fest, ob eine rationale Zahl >=0 die n-te Potenz einer rationalen Zahl
// ist.
// rootp(x,n,&w)
// > x: eine rationale Zahl >=0
// > n: ein Integer >0
// < w: exakte n-te Wurzel (expt x (/ n)) falls x eine n-te Potenz
// < ergebnis: cl_true                    ........................, cl_false sonst
  extern cl_boolean rootp (const cl_RA& x, uintL n, cl_RA* w);
  extern cl_boolean rootp (const cl_RA& x, const cl_I& n, cl_RA* w);

// Liefert zu Integers a>0, b>1 den Logarithmus log(a,b),
// falls er eine rationale Zahl ist.
// logp(a,b,&l)
// > a: ein Integer >0
// > b: ein Integer >1
// < l: log(a,b)       falls er eine exakte rationale Zahl ist
// < ergebnis: cl_true ......................................., cl_false sonst
  extern cl_boolean logp (const cl_I& a, const cl_I& b, cl_RA* l);

// Liefert zu rationalen Zahlen a>0, b>0 den Logarithmus log(a,b),
// falls er eine rationale Zahl ist.
// logp(a,b,&l)
// > a: eine rationale Zahl >0
// > b: eine rationale Zahl >0, /=1
// < l: log(a,b)       falls er eine exakte rationale Zahl ist
// < ergebnis: cl_true ......................................., cl_false sonst
  extern cl_boolean logp (const cl_RA& a, const cl_RA& b, cl_RA* l);


#ifdef WANT_OBFUSCATING_OPERATORS
// This could be optimized to use in-place operations.
inline cl_RA& operator+= (cl_RA& x, const cl_RA& y) { return x = x + y; }
inline cl_RA& operator+= (cl_RA& x, const int y) { return x = x + y; }
inline cl_RA& operator+= (cl_RA& x, const unsigned int y) { return x = x + y; }
inline cl_RA& operator+= (cl_RA& x, const long y) { return x = x + y; }
inline cl_RA& operator+= (cl_RA& x, const unsigned long y) { return x = x + y; }
inline cl_RA& operator++ /* prefix */ (cl_RA& x) { return x = plus1(x); }
inline void operator++ /* postfix */ (cl_RA& x, int dummy) { x = plus1(x); }
inline cl_RA& operator-= (cl_RA& x, const cl_RA& y) { return x = x - y; }
inline cl_RA& operator-= (cl_RA& x, const int y) { return x = x - y; }
inline cl_RA& operator-= (cl_RA& x, const unsigned int y) { return x = x - y; }
inline cl_RA& operator-= (cl_RA& x, const long y) { return x = x - y; }
inline cl_RA& operator-= (cl_RA& x, const unsigned long y) { return x = x - y; }
inline cl_RA& operator-- /* prefix */ (cl_RA& x) { return x = minus1(x); }
inline void operator-- /* postfix */ (cl_RA& x, int dummy) { x = minus1(x); }
inline cl_RA& operator*= (cl_RA& x, const cl_RA& y) { return x = x * y; }
inline cl_RA& operator/= (cl_RA& x, const cl_RA& y) { return x = x / y; }
#endif


#endif /* _CL_RATIONAL_H */
