// Low-level arithmetic: operations on 16-bit and 32-bit words

#ifndef _CL_LOW_H
#define _CL_LOW_H


// Determines the sign of a 32-bit number.
// sign_of(wert)
// > wert: eine 32-Bit-Zahl
// < sint32 ergebnis: 0 falls wert>=0, -1 falls wert<0.
inline sint32 sign_of (sint32 wert)
{
#if defined(__sparc__) || defined(__arm__)
	return wert >> 31;
#else
	return (wert >= 0 ? 0 : -1);
#endif
}

// Determines the sign of a 16-bit number.
// sign_of(wert)
// > wert: eine 16-Bit-Zahl
// < sint16 ergebnis: 0 falls wert>=0, -1 falls wert<0.
inline sint16 sign_of (sint16 wert)
{
#if defined(__sparc__)
	return (sint32)wert >> 31;
#else
	return (wert >= 0 ? 0 : -1);
#endif
}


// High-Word einer 32-Bit-Zahl bestimmen
// high16(wert)
inline uint16 high16 (uint32 wert)
{
	return wert >> 16;
}

// Low-Word einer 32-Bit-Zahl bestimmen
// low16(wert)
inline uint16 low16 (uint32 wert)
{
	return (uint16)wert;
}

// Eine 32-Bit-Zahl aus ihrem High-Word und ihrem Low-Word bestimmen:
// highlow32(uint16 high, uint16 low)
inline uint32 highlow32 (uint16 high, uint16 low)
{
	return ((uint32)high << 16) | (uint32)low;
}

// Eine 32-Bit-Zahl aus ihrem High-Word und ihrem Low-Word 0 bestimmen:
// highlow32_0(uint16 high)
inline uint32 highlow32_0 (uint16 high)
{
	return (uint32)high << 16;
}

// Eine 32-Bit-Zahl aus zwei aufeinanderfolgenden 16-Bit-Digits einer UDS
// zusammensetzen: highlow32_at(ptr)
#define highlow32_at(ptr)  highlow32(((uint16*)(ptr))[0],((uint16*)(ptr))[1])

// Eine 32-Bit-Zahl in zwei aufeinanderfolgende 16-Bit-Digits einer UDS abspeichern:
// set_highlow32_at(ptr,value32); wobei ptr und value32 Variablen.
#define set_highlow32_at(ptr,value32)  (((uint16*)(ptr))[0]=high16(value32),((uint16*)(ptr))[1]=low16(value32))


// Multipliziert zwei 16-Bit-Zahlen miteinander und liefert eine 32-Bit-Zahl:
// mulu16(arg1,arg2)
// > arg1, arg2 : zwei 16-Bit-Zahlen
// < ergebnis: eine 32-Bit-Zahl
#if defined(__GNUC__) && defined(__sparc__) && defined(FAST_DOUBLE)
// Ist das schneller als mulu16_ ??
inline uint32 mulu16 (uint16 arg1, uint16 arg2)
{
	union { double f; uint32 i[2]; } __fi;
	__fi.f = (double)(sint32)arg1 * (double)(sint32)arg2
		 + (double)(4503599627370496.0L); // + 2^52, zum Normalisieren
	return __fi.i[1]; // untere 32 Bit herausholen (benutzt CL_BIG_ENDIAN_P !)
}
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
inline uint32 mulu16 (uint16 arg1, uint16 arg2)
{
	register uint16 _hi;
	register uint16 _lo;
	__asm__("mulw %2"
		: "=d" /* %dx */ (_hi), "=a" /* %ax */ (_lo)
		: "rm" (arg1), "1" /* %eax */ (arg2)
	       );
	return highlow32(_hi,_lo);
}
#elif defined(__sparc__)
  extern "C" uint32 mulu16_ (uint16 arg1, uint16 arg2);
  #define mulu16  mulu16_  // extern in Assembler
#else
inline uint32 mulu16 (uint16 arg1, uint16 arg2)
{
	return arg1 * arg2;
}
#endif

// Multipliziert zwei 24-Bit-Zahlen zusammen und liefert eine 48-Bit-Zahl.
// mulu24(arg1,arg2,hi=,lo=);
// > arg1, arg2 : zwei 24-Bit-Zahlen
// < 2^32*hi+lo : eine 48-Bit-Zahl
#if defined(__sparc__) && defined(FAST_DOUBLE)
  #define mulu24(x,y,hi_zuweisung,lo_zuweisung)  \
    { var uint32 _x = (x);					\
      var uint32 _y = (y);					\
      var union { double f; uint32 i[2]; uint16 s[4]; } __fi;	\
      __fi.f = (double)(sint32)(_x)*(double)(sint32)(_y)	\
               + (double)(4503599627370496.0L); /* + 2^52, zum Normalisieren */\
      hi_zuweisung __fi.s[1]; /* mittlere 16 Bit herausholen, (benutzt CL_BIG_ENDIAN_P !) */\
      lo_zuweisung __fi.i[1]; /* untere 32 Bit herausholen (benutzt CL_BIG_ENDIAN_P !)    */\
    }
#else
  #define mulu24  mulu32
#endif

// Multipliziert zwei 32-Bit-Zahlen miteinander und liefert eine 64-Bit-Zahl:
// mulu32(arg1,arg2,hi=,lo=);
// > arg1, arg2 : zwei 32-Bit-Zahlen
// < 2^32*hi+lo : eine 64-Bit-Zahl
  extern "C" uint32 mulu32_ (uint32 arg1, uint32 arg2); // -> Low-Teil
  extern "C" uint32 mulu32_high;                        // -> High-Teil
#if defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ var uint32 _x = (x); \
       var uint32 _y = (y); \
       var uint32 _hi;      \
       var uint32 _lo;      \
       __asm__("mulul %3,%0:%1" : "=d" (_hi), "=d"(_lo) : "1" (_x), "dm" (_y) ); \
       hi_zuweisung _hi;         \
       lo_zuweisung _lo;         \
     })
#elif defined(__GNUC__) && defined(__m68k__)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ var uint32 _x = (x);						\
       var uint32 _y = (y);						\
       var uint16 _x1 = high16(_x);					\
       var uint16 _x0 = low16(_x);					\
       var uint16 _y1 = high16(_y);					\
       var uint16 _y0 = low16(_y);					\
       var uint32 _hi = mulu16(_x1,_y1); /* obere Portion */		\
       var uint32 _lo = mulu16(_x0,_y0); /* untere Portion */		\
       {var uint32 _mid = mulu16(_x0,_y1); /* 1. mittlere Portion */	\
        _hi += high16(_mid); _mid = highlow32_0(low16(_mid));		\
        _lo += _mid; if (_lo < _mid) { _hi += 1; } /* 64-Bit-Addition */\
       }								\
       {var uint32 _mid = mulu16(_x1,_y0); /* 2. mittlere Portion */	\
        _hi += high16(_mid); _mid = highlow32_0(low16(_mid));		\
        _lo += _mid; if (_lo < _mid) { _hi += 1; } /* 64-Bit-Addition */\
       }								\
       hi_zuweisung _hi;						\
       lo_zuweisung _lo;						\
     })
#elif defined(__GNUC__) && defined(__sparc__)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ lo_zuweisung mulu32_(x,y); /* extern in Assembler */	\
      {var register uint32 _hi __asm__("%g1");			\
       hi_zuweisung _hi;					\
     }})
#elif defined(__GNUC__) && defined(__arm__)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ lo_zuweisung mulu32_(x,y); /* extern in Assembler */	\
      {var register uint32 _hi __asm__("%r1"/*"%a2"*/);		\
       hi_zuweisung _hi;					\
     }})
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ var register uint32 _hi;                                  \
       var register uint32 _lo;                                  \
       __asm__("mull %2"                                         \
               : "=d" /* %edx */ (_hi), "=a" /* %eax */ (_lo)    \
               : "g" ((uint32)(x)), "1" /* %eax */ ((uint32)(y)) \
              );                                                 \
       hi_zuweisung _hi; lo_zuweisung _lo;                       \
     })
#elif defined(__GNUC__) && defined(__mips__) && !defined(NO_ASM)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ var register uint32 _hi;                       \
       var register uint32 _lo;                       \
       __asm__("multu %3,%2 ; mfhi %0 ; mflo %1"      \
               : "=r" (_hi), "=r" (_lo)               \
               : "r" ((uint32)(x)), "r" ((uint32)(y)) \
              );                                      \
       hi_zuweisung _hi; lo_zuweisung _lo;            \
     })
#elif defined(__GNUC__) && defined(HAVE_LONGLONG)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    ({ var register uint64 _prod = (uint64)(x) * (uint64)(y); \
       hi_zuweisung (uint32)(_prod>>32);                      \
       lo_zuweisung (uint32)(_prod);                          \
     })
#elif defined(WATCOM) && defined(__i386__) && !defined(NO_ASM)
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    { var register uint32 _hi;                  \
      var register uint32 _lo;                  \
      _lo = mulu32_(x,y), _hi = mulu32_high_(); \
      hi_zuweisung _hi; lo_zuweisung _lo;       \
    }
  extern "C" uint32 mulu32_high_ (void);
  #pragma aux mulu32_ = 0xF7 0xE2 /* mull %edx */ parm [eax] [edx] value [eax] modify [eax edx];
  #pragma aux mulu32_high_ = /* */ value [edx] modify [];
#else
  #define mulu32(x,y,hi_zuweisung,lo_zuweisung)  \
    { lo_zuweisung mulu32_(x,y); hi_zuweisung mulu32_high; }
  #if defined(__m68k__) || defined(__sparc__) || defined(__arm__) || (defined(__i386__) && !defined(WATCOM)) || defined(__mips__) || defined(__hppa__)
    // mulu32_ extern in Assembler
    #if defined(__sparc__)
      extern "C" uint32 _get_g1 (void);
      #define mulu32_high  (_get_g1()) // Rckgabe im Register %g1
    #elif !defined(__hppa__)
      #define NEED_VAR_mulu32_high
    #endif
  #else
    #define NEED_FUNCTION_mulu32_
  #endif
#endif

// Multipliziert zwei 32-Bit-Zahlen miteinander und liefert eine 32-Bit-Zahl:
// mulu32_unchecked(arg1,arg2)
// > arg1, arg2 : zwei 32-Bit-Zahlen
// < ergebnis : eine 32-Bit-Zahl
// Es wird vorausgesetzt, da arg1*arg2 < 2^32.
#if defined(__sparc__)
  extern "C" uint32 mulu32_unchecked (uint32 x, uint32 y); // extern in Assembler
#else
  // Wir knnen dafr auch die Bibliotheksroutine des C-Compilers nehmen:
  inline uint32 mulu32_unchecked (uint32 arg1, uint32 arg2)
  {
	return arg1 * arg2;
  }
#endif


// Dividiert eine 16-Bit-Zahl durch eine 16-Bit-Zahl und
// liefert einen 16-Bit-Quotienten und einen 16-Bit-Rest.
// divu_1616_1616(x,y,q=,r=);
// > uint16 x: Zhler
// > uint16 y: Nenner
// < uint16 q: floor(x/y)
// < uint16 r: x mod y
// < x = q*y+r
  #define divu_1616_1616(x,y,q_zuweisung,r_zuweisung)  \
    { var uint16 __x = (x);					\
      var uint16 __y = (y);					\
      q_zuweisung floor(__x,__y);				\
      r_zuweisung (__x % __y);					\
    }

// Dividiert eine 32-Bit-Zahl durch eine 16-Bit-Zahl und
// liefert einen 16-Bit-Quotienten und einen 16-Bit-Rest.
// divu_3216_1616(x,y,q=,r=);
// > uint32 x: Zhler
// > uint16 y: Nenner
// > Es sei bekannt, da 0 <= x < 2^16*y .
// < uint16 q: floor(x/y)
// < uint16 r: x mod y
// < x = q*y+r
  extern "C" uint16 divu_3216_1616_ (uint32 x, uint16 y); // -> Quotient q
  extern "C" uint16 divu_16_rest;                         // -> Rest r
#if defined(__GNUC__) && defined(__sparc__)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    ({ var uint32 __qr = divu_3216_1616_(x,y); /* extern in Assembler */\
       q_zuweisung low16(__qr);						\
       r_zuweisung high16(__qr);					\
     })
#elif defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    ({var uint32 __x = (x);						\
      var uint16 __y = (y);						\
      var uint32 __qr;							\
      __asm__ __volatile__ ("						\
        divu %2,%0							\
        " : "=d" (__qr) : "0" (__x), "dm" (__y));			\
      q_zuweisung low16(__qr);						\
      r_zuweisung high16(__qr);						\
     })
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    ({var uint32 __x = (x);						\
      var uint16 __y = (y);						\
      var uint16 __q;							\
      var uint16 __r;							\
      __asm__("divw %4"							\
              : "=a" /* %ax */ (__q), "=d" /* %dx */ (__r)		\
              : "1" /* %dx */ ((uint16)(high16(__x))), "0" /* %ax */ ((uint16)(low16(__x))), "rm" (__y) \
             );								\
      q_zuweisung __q;							\
      r_zuweisung __r;							\
     })
#elif defined(__GNUC__)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    ({var uint32 __x = (x);						\
      var uint16 __y = (y);						\
      var uint16 __q = floor(__x,__y);					\
      q_zuweisung __q;							\
      r_zuweisung (__x - __q * __y);					\
     })
#elif defined(__sparc__)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    { var uint32 __qr = divu_3216_1616_(x,y); /* extern in Assembler */	\
      q_zuweisung low16(__qr);						\
      r_zuweisung high16(__qr);						\
    }
#elif defined(__arm__)
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    { q_zuweisung divu_3216_1616_(x,y); /* extern in Assembler */	\
      r_zuweisung divu_16_rest;						\
    }
  #define NEED_VAR_divu_16_rest
#else
  #define divu_3216_1616(x,y,q_zuweisung,r_zuweisung)  \
    { q_zuweisung divu_3216_1616_(x,y); r_zuweisung divu_16_rest; }
  #define NEED_FUNCTION_divu_3216_1616_
#endif

// Dividiert eine 32-Bit-Zahl durch eine 16-Bit-Zahl und
// liefert einen 32-Bit-Quotienten und einen 16-Bit-Rest.
// divu_3216_3216(x,y,q=,r=);
// > uint32 x: Zhler
// > uint16 y: Nenner
// Es sei bekannt, da y>0.
// < uint32 q: floor(x/y)
// < uint16 r: x mod y
// < x = q*y+r
  extern "C" uint32 divu_3216_3216_ (uint32 x, uint16 y); // -> Quotient q
  extern "C" uint16 divu_16_rest;                         // -> Rest r
#if defined(__sparc__) || defined(__i386__)
  #define divu_3216_3216  divu_3232_3232
#else
  // Methode: (beta = 2^16)
  // x = x1*beta+x0 schreiben.
  // Division mit Rest: x1 = q1*y + r1, wobei 0 <= x1 < beta <= beta*y.
  // Also 0 <= q1 < beta, 0 <= r1 < y.
  // Division mit Rest: (r1*beta+x0) = q0*y + r0, wobei 0 <= r1*beta+x0 < beta*y.
  // Also 0 <= q0 < beta, 0 <= r0 < y
  // und x = x1*beta+x0 = (q1*beta+q0)*y + r0.
  // Setze q := q1*beta+q0 und r := r0.
  #define divu_3216_3216(x,y,q_zuweisung,r_zuweisung)  \
    { var uint32 _x = (x);						\
      var uint16 _y = (y);						\
      var uint16 _q1;							\
      var uint16 _q0;							\
      var uint16 _r1;							\
      divu_3216_1616(high16(_x),_y, _q1 = , _r1 = );			\
      divu_3216_1616(highlow32(_r1,low16(_x)),_y, _q0 = , r_zuweisung);	\
      q_zuweisung highlow32(_q1,_q0);					\
    }
#endif

// Dividiert eine 32-Bit-Zahl durch eine 32-Bit-Zahl und
// liefert einen 32-Bit-Quotienten und einen 32-Bit-Rest.
// divu_3232_3232(x,y,q=,r=);
// > uint32 x: Zhler
// > uint32 y: Nenner
// Es sei bekannt, da y>0.
// < uint32 q: floor(x/y)
// < uint32 r: x mod y
// < x = q*y+r
  extern "C" uint32 divu_3232_3232_ (uint32 x, uint32 y); // -> Quotient q
  extern "C" uint32 divu_32_rest;                         // -> Rest r
#if defined(__sparc__) || defined(__i386__)
  #define divu_3232_3232(x,y,q_zuweisung,r_zuweisung)  \
    divu_6432_3232(0,x,y,q_zuweisung,r_zuweisung)
  #define divu_3232_3232_(x,y) divu_6432_3232_(0,x,y)
#else
  // Methode: (beta = 2^n = 2^16, n = 16)
  // Falls y < beta, handelt es sich um eine 32-durch-16-Bit-Division.
  // Falls y >= beta:
  // Quotient  q = floor(x/y) < beta  (da 0 <= x < beta^2, y >= beta).
  // y habe genau n+k Bits (1 <= k <= n), d.h. 2^(n+k-1) <= y < 2^(n+k).
  // Schreibe  x = 2^k*x1 + x0  mit  x1 := floor(x/2^k)
  // und       y = 2^k*y1 + y0  mit  y1 := floor(y/2^k)
  // und bilde den Nherungs-Quotienten floor(x1/y1)
  // oder (noch besser) floor(x1/(y1+1)).
  // Wegen 0 <= x1 < 2^(2n) und 0 < 2^(n-1) <= y1 < 2^n
  // und  x1/(y1+1) <= x/y < x1/(y1+1) + 2
  // (denn x1/(y1+1) = (x1*2^k)/((y1+1)*2^k) <= (x1*2^k)/y <= x/y
  // und x/y - x1/(y1+1) = (x+x*y1-x1*y)/(y*(y1+1))
  // = (x+x0*y1-x1*y0)/(y*(y1+1)) <= (x+x0*y1)/(y*(y1+1))
  // <= x/(y*(y1+1)) + x0/y
  // <= 2^(2n)/(2^(n+k-1)*(2^(n-1)+1)) + 2^k/2^(n+k-1)
  // = 2^(n-k+1)/(2^(n-1)+1) + 2^(1-n) <= 2^n/(2^(n-1)+1) + 2^(1-n) < 2 )
  // gilt  floor(x1/(y1+1)) <= floor(x/y) <= floor(x1/(y1+1)) + 2  .
  // Man bildet also  q:=floor(x1/(y1+1))  (ein Shift um n Bit oder
  // eine (2n)-durch-n-Bit-Division, mit Ergebnis q <= floor(x/y) < beta)
  // und x-q*y und mu hiervon noch hchstens 2 mal y abziehen und q
  // incrementieren, um den Quotienten  q = floor(x/y)  und den Rest
  // x-floor(x/y)*y  der Division zu bekommen.
  #define divu_3232_3232(x,y,q_zuweisung,r_zuweisung)  \
    { var uint32 _x = (x);						\
      var uint32 _y = (y);						\
      if (_y <= (uint32)(bit(16)-1))					\
        { var uint16 _q1;						\
          var uint16 _q0;						\
          var uint16 _r1;						\
          divu_3216_1616(high16(_x),_y, _q1 = , _r1 = );		\
          divu_3216_1616(highlow32(_r1,low16(_x)),_y, _q0 = , r_zuweisung); \
          q_zuweisung highlow32(_q1,_q0);				\
        }								\
        else								\
        { var uint32 _x1 = _x; /* x1 := x */				\
          var uint32 _y1 = _y; /* y1 := y */				\
          var uint16 _q;						\
          do { _x1 = floor(_x1,2); _y1 = floor(_y1,2); } /* k erhhen */\
             until (_y1 <= (uint32)(bit(16)-1)); /* bis y1 < beta */	\
          { var uint16 _y2 = low16(_y1)+1; /* y1+1 bilden */		\
            if (_y2==0)							\
              { _q = high16(_x1); } /* y1+1=beta -> ein Shift */	\
              else							\
              { divu_3216_1616(_x1,_y2,_q=,); } /* Division von x1 durch y1+1 */\
          }								\
          /* _q = q = floor(x1/(y1+1)) */				\
          /* x-q*y bilden (eine 16-mal-32-Bit-Multiplikation ohne berlauf): */\
          _x -= highlow32_0(mulu16(_q,high16(_y))); /* q * high16(y) * beta */\
          /* gefahrlos, da q*high16(y) <= q*y/beta <= x/beta < beta */	\
          _x -= mulu16(_q,low16(_y)); /* q * low16(y) */		\
          /* gefahrlos, da q*high16(y)*beta + q*low16(y) = q*y <= x */	\
          /* Noch hchstens 2 mal y abziehen: */			\
          if (_x >= _y)							\
            { _q += 1; _x -= _y;					\
              if (_x >= _y)						\
                { _q += 1; _x -= _y; }					\
            }								\
          r_zuweisung _x;						\
          q_zuweisung (uint32)(_q);					\
    }   }
  #define NEED_FUNCTION_divu_3232_3232_
#endif

// Dividiert eine 64-Bit-Zahl durch eine 32-Bit-Zahl und
// liefert einen 32-Bit-Quotienten und einen 32-Bit-Rest.
// divu_6432_3232(xhi,xlo,y,q=,r=);
// > uint32 xhi,xlo: x = 2^32*xhi+xlo = Zhler
// > uint32 y: Nenner
// > Es sei bekannt, da 0 <= x < 2^32*y .
// < uint32 q: floor(x/y)
// < uint32 r: x mod y
// < x = q*y+r
  extern "C" uint32 divu_6432_3232_ (uint32 xhi, uint32 xlo, uint32 y); // -> Quotient q
  extern "C" uint32 divu_32_rest;                                       // -> Rest r
#if defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    ({var uint32 __xhi = (xhi);						\
      var uint32 __xlo = (xlo);						\
      var uint32 __y = (y);						\
      var uint32 __q;							\
      var uint32 __r;							\
      __asm__ __volatile__ ("						\
        divul %4,%1:%0							\
        " : "=d" (__q), "=d" (__r) : "1" (__xhi), "0" (__xlo), "dm" (__y)); \
      q_zuweisung __q;							\
      r_zuweisung __r;							\
     })
  #define divu_6432_3232_(xhi,xlo,y) \
    ({var uint32 ___q; divu_6432_3232(xhi,xlo,y,___q=,); ___q; })
#elif defined(__GNUC__) && defined(__sparc__)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    ({ var uint32 _q = divu_6432_3232_(xhi,xlo,y); /* extern in Assembler */\
       var register uint32 _r __asm__("%g1");				    \
       q_zuweisung _q; r_zuweisung _r;					    \
     })
#elif defined(__GNUC__) && defined(__arm__)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    ({ var uint32 _q = divu_6432_3232_(xhi,xlo,y); /* extern in Assembler */\
       var register uint32 _r __asm__("%r1"/*"%a2"*/);			    \
       q_zuweisung _q; r_zuweisung _r;					    \
     })
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    ({var uint32 __xhi = (xhi);						\
      var uint32 __xlo = (xlo);						\
      var uint32 __y = (y);						\
      var uint32 __q;							\
      var uint32 __r;							\
      __asm__ __volatile__ (						\
         "divl %4"							\
         : "=a" /* %eax */ (__q), "=d" /* %edx */ (__r)			\
         : "1" /* %edx */ (__xhi), "0" /* %eax */ (__xlo), "rm" (__y)	\
         );								\
      q_zuweisung __q;							\
      r_zuweisung __r;							\
     })
  #define divu_6432_3232_(xhi,xlo,y) \
    ({var uint32 ___q; divu_6432_3232(xhi,xlo,y,___q=,); ___q; })
#elif defined(__GNUC__) && defined(HAVE_LONGLONG)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung) \
    ({var uint32 __xhi = (xhi);						\
      var uint32 __xlo = (xlo);						\
      var uint64 __x = (uint64)__xhi<<32 | (uint64)__xlo;		\
      var uint32 __y = (y);						\
      var uint32 __q = floor(__x,(uint64)__y);				\
      q_zuweisung __q; r_zuweisung __xlo - __q * __y;			\
     })
  #define divu_6432_3232_(xhi,xlo,y) \
    ({var uint32 ___q; divu_6432_3232(xhi,xlo,y,___q=,); ___q; })
#elif defined(WATCOM) && defined(__i386__) && !defined(NO_ASM)
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    { var uint32 __xhi = (xhi);						\
      var uint32 __xlo = (xlo);						\
      var uint32 __y = (y);						\
      var uint32 __q;							\
      var uint32 __r;							\
      __q = divu_6432_3232_(__xhi,__xlo,__y); __r = divu_6432_3232_rest(); \
      q_zuweisung __q;							\
      r_zuweisung __r;							\
    }
  extern "C" uint32 divu_6432_3232_rest (void);
  #pragma aux divu_6432_3232_ = 0xF7 0xF1 /* divl %ecx */ parm [edx] [eax] [ecx] value [eax] modify [eax edx];
  #pragma aux divu_6432_3232_rest = /* */ value [edx] modify [];
#else
  #define divu_6432_3232(xhi,xlo,y,q_zuweisung,r_zuweisung)  \
    { q_zuweisung divu_6432_3232_(xhi,xlo,y); r_zuweisung divu_32_rest; }
  #if defined(__m68k__) || defined(__sparc__) || defined(__arm__) || (defined(__i386__) && !defined(WATCOM)) || defined(__hppa__)
    // divu_6432_3232_ extern in Assembler
    #if defined(__sparc__)
      extern "C" uint32 _get_g1 (void);
      #define divu_32_rest  (_get_g1()) // Rckgabe im Register %g1
    #else
      #define NEED_VAR_divu_32_rest
    #endif
  #else
    #define NEED_FUNCTION_divu_6432_3232_
  #endif
#endif


// Zieht die Ganzzahl-Wurzel aus einer 32-Bit-Zahl und
// liefert eine 16-Bit-Wurzel und einen Rest.
// isqrt_32_16(x,y=,sqrtp=);
// > uint32 x: Radikand, >= 2^30, < 2^32
// < uint16 y: floor(sqrt(x)), >= 2^15, < 2^16
// < boolean sqrtp: /=0, falls x=y^2
  // Methode:
  // y := 2^16 als Anfangswert,
  // y := floor((y + floor(x/y))/2) als nchster Wert,
  // solange z := floor(x/y) < y, setze y := floor((y+z)/2).
  // y ist fertig; x=y^2 genau dann, wenn z=y und die letzte Division aufging.
  // (Beweis:
  //  1. Die Folge der y ist streng monoton fallend.
  //  2. Stets gilt y >= floor(sqrt(x)) (denn fr alle y>0 ist
  //     y + x/y >= 2*sqrt(x) und daher  floor((y + floor(x/y))/2) =
  //     floor(y/2 + x/(2*y)) >= floor(sqrt(x)) ).
  //  3. Am Schlu gilt x >= y^2.
  // )
  #define isqrt_32_16(x,y_zuweisung,sqrtp_zuweisung)  \
    { var uint32 _x = (x);						\
      var uint16 _x1 = high16(_x);					\
      var uint16 _y = floor(_x1,2) | bit(16-1);				\
      loop								\
        { var uint16 _z;						\
          var uint16 _r;						\
          if (_x1 >= _y) /* Division _x/_y ergbe berlauf -> _z > _y */\
            { unused (sqrtp_zuweisung FALSE); break; } 			\
          divu_3216_1616(_x,_y, _z=,_r=); /* Dividiere _x/_y */		\
          if (_z >= _y)							\
            { unused (sqrtp_zuweisung (_z == _y) && (_r == 0)); break; } \
          _y = floor((uint16)(_z+_y),2) | bit(16-1); /* _y mu >= 2^15 bleiben */\
        }								\
      y_zuweisung _y;							\
    }

// Zieht die Ganzzahl-Wurzel aus einer 64-Bit-Zahl und
// liefert eine 32-Bit-Wurzel und einen Rest.
// isqrt_64_32(xhi,xlo,y=,sqrtp=);
// > uint32 xhi,xlo: Radikand x = 2^32*xhi+xlo, >= 2^62, < 2^64
// < uint32 y: floor(sqrt(x)), >= 2^31, < 2^32
// < boolean sqrtp: /=0, falls x=y^2
#if defined(__sparc__) || defined(__m68k__) || defined(__hppa__)
  // Methode:
  // y := 2^32 als Anfangswert,
  // y := floor((y + floor(x/y))/2) als nchster Wert,
  // solange z := floor(x/y) < y, setze y := floor((y+z)/2).
  // y ist fertig; x=y^2 genau dann, wenn z=y und die letzte Division aufging.
  // (Beweis:
  //  1. Die Folge der y ist streng monoton fallend.
  //  2. Stets gilt y >= floor(sqrt(x)) (denn fr alle y>0 ist
  //     y + x/y >= 2*sqrt(x) und daher  floor((y + floor(x/y))/2) =
  //     floor(y/2 + x/(2*y)) >= floor(sqrt(x)) ).
  //  3. Am Schlu gilt x >= y^2.
  // )
  #define isqrt_64_32(xhi,xlo,y_zuweisung,sqrtp_zuweisung)  \
    { var uint32 _xhi = (xhi);						\
      var uint32 _xlo = (xlo);						\
      var uint32 _y = floor(_xhi,2) | bit(32-1);			\
      loop								\
        { var uint32 _z;						\
          var uint32 _rest;						\
          if (_xhi >= _y) /* Division _x/_y ergbe berlauf -> _z > _y */\
            { sqrtp_zuweisung FALSE; break; }				\
          divu_6432_3232(_xhi,_xlo,_y, _z=,_rest=); /* Dividiere _x/_y */\
          if (_z >= _y)							\
            { sqrtp_zuweisung (_z == _y) && (_rest == 0); break; }	\
          _y = floor(_z+_y,2) | bit(32-1); /* _y mu >= 2^31 bleiben */	\
        }								\
      y_zuweisung _y;							\
    }
#else
  // Methode:
  // Wie bei UDS_sqrt mit n=2.
  // y = 2^16*yhi + ylo ansetzen.
  // Dann mu
  //   yhi = floor(y/2^16) = floor(floor(sqrt(x))/2^16)
  //       = floor(sqrt(x)/2^16) = floor(sqrt(x/2^32)) = isqrt(xhi)
  // sein. Es folgt yhi >= 2^15.
  // Danach sucht man das grte ylo >=0 mit
  // x - 2^32*yhi^2 >= 2*2^16*yhi*ylo + ylo^2.
  // Dazu setzen wir  xhi*2^32+xlo := x - 2^32*yhi^2
  // (also xhi := xhi - yhi^2, das ist >=0, <=2*yhi).
  // Die Schtzung fr die zweite Ziffer
  //     ylo' := min(2^16-1,floor((xhi*2^32+xlo)/(2*2^16*yhi)))
  // erfllt ylo'-1 <= ylo <= ylo', ist also um hchstens 1 zu gro.
  // (Beweis: Rechte Ungleichung klar, da  ylo < 2^16  und
  //   xhi*2^32+xlo >= 2*2^16*yhi*ylo + ylo^2 >= 2*2^16*yhi*ylo
  //   ==> (xhi*2^32+xlo)/(2*2^16*yhi) >= ylo  gelten mu.
  //   Linke Ungleichung: Falls floor(...)>=2^16, ist
  //   xhi*2^32+xlo >= 2*2^16*2^16*yhi >= 2*2^16*yhi*(2^16-1) + 2^32
  //                >= 2*2^16*yhi*(2^16-1) + (2^16-1)^2
  //   und xhi*2^32+xlo < 2*2^16*2^16*yhi + (2^16)^2, also
  //   ylo = 2^16-1 = ylo'.
  //   Sonst ist ylo' = floor((xhi*2^32+xlo)/(2*2^16*yhi)), also
  //   xhi*2^32+xlo >= 2*2^16*yhi*ylo' >= 2*2^16*yhi*(ylo'-1) + 2^32
  //                >= 2*2^16*yhi*(ylo'-1) + (ylo'-1)^2,
  //   also ylo >= ylo'-1 nach Definition von ylo.)
  #define isqrt_64_32(xhi,xlo,y_zuweisung,sqrtp_zuweisung)  \
    { var uint32 _xhi = (xhi);						\
      var uint32 _xlo = (xlo);						\
      var uint16 _yhi;							\
      var uint16 _ylo;							\
      /* erste Ziffer berechnen: */					\
      isqrt_32_16(_xhi,_yhi=,); /* yhi := isqrt(xhi) */			\
      _xhi -= mulu16(_yhi,_yhi); /* jetzt 0 <= xhi <= 2*yhi */		\
      /* x = 2^32*yhi^2 + 2^32*xhi + xlo */				\
      /* Schtzung fr die zweite Ziffer berechnen: */			\
      /* ylo := min(2^16-1,floor((xhi*2^32+xlo)/(2*2^16*yhi))) bilden: */\
     {var uint32 _z = (_xhi << 15) | (_xlo >> 17); /* < 2^15*(2*yhi+1) */\
      var uint32 _r = highlow32_0(_yhi);				\
      if (_z >= _r)							\
        { _ylo = bit(16)-1; _r = _z - _r + (uint32)_yhi; }		\
        else								\
        { divu_3216_1616(_z,_yhi, _ylo=,_r=); }				\
      /* x = 2^32*yhi^2 + 2*2^16*yhi*ylo + 2^17*r + (xlo mod 2^17), */	\
      /* 0 <= r < yhi + 2^15 */						\
      _xlo = (_r << 17) | (_xlo & (bit(17)-1));				\
      /* x = 2^32*yhi^2 + 2*2^16*yhi*ylo + 2^32*floor(r/2^15) + xlo */	\
      _z = mulu16(_ylo,_ylo); /* z = ylo^2 */				\
      /* Versuche vom Rest 2^32*floor(r/2^15) + xlo  z zu subtrahieren. */\
      /* Falls Rest >= z (d.h. r>=2^15 oder xlo>=z), ist ylo fertig, */	\
      /* und es gilt x=y^2 genau dann, wenn r<2^15 und xlo=z. */	\
      /* Sonst (d.h. r<2^15 und xlo<z), mu man ylo erniedrigen. Dazu */\
      /* setzt man  ylo := ylo-1, z := z-(2*ylo+1), */			\
      /* Rest := Rest + 2^17*yhi = xlo + 2^17*yhi >= 2^32 > z, also x>y^2. */\
      if (_r < bit(15))							\
        { if (_xlo < _z)						\
            { _ylo -= 1; sqrtp_zuweisung FALSE; }			\
            else							\
            { sqrtp_zuweisung (_xlo == _z); }				\
        }								\
        else								\
        { sqrtp_zuweisung FALSE; }					\
      y_zuweisung highlow32(_yhi,_ylo);					\
    }}
#endif

// Zieht die Ganzzahl-Wurzel aus einer 32-Bit-Zahl und
// liefert eine 16-Bit-Wurzel.
// isqrt(x)
// > uintL x : Radikand, >=0, <2^32
// < uintL ergebnis : Wurzel, >=0, <2^16
  extern uintL isqrt (uintL x);

// Zieht die Ganzzahl-Wurzel aus einer 64-Bit-Zahl und
// liefert eine 32-Bit-Wurzel.
// isqrt(x1,x0)
// > uintL2 x = x1*2^32+x0 : Radikand, >=0, <2^64
// < uintL ergebnis : Wurzel, >=0, <2^32
  extern uintL isqrt (uintL x1, uintL x0);


// Bits einer 8-Bit-Zahl zhlen:
// integerlength8(digit,size=);
// setzt size auf die hchste in digit vorkommende Bitnummer.
// > digit: ein uint8 >0
// < size: >0, <=8, mit 2^(size-1) <= digit < 2^size
#if defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define integerlength8(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit            */\
      __asm__("bfffo %1{#0:#8},%0" : "=d" (zero_counter) : "dm" ((uint8)(digit)) ); \
      size_zuweisung (8-zero_counter);                                              \
    }
#elif defined(__sparc__)
  #define integerlength8(digit,size_zuweisung)  \
    integerlength32((uint32)(digit),size_zuweisung) // siehe unten
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define integerlength8(digit,size_zuweisung)  \
    integerlength16((uint16)(digit),size_zuweisung)
#else
  #define integerlength8(digit,size_zuweisung)  \
    { var uintC bitsize = 1;					\
      var uintL x8 = (uint8)(digit);				\
      /* x8 hat hchstens 8 Bits.                             */\
      if (x8 >= bit(4)) { x8 = x8>>4; bitsize += 4; }		\
      /* x8 hat hchstens 4 Bits.                             */\
      if (x8 >= bit(2)) { x8 = x8>>2; bitsize += 2; }		\
      /* x8 hat hchstens 2 Bits.                             */\
      if (x8 >= bit(1)) { /* x8 = x8>>1; */ bitsize += 1; }	\
      /* x8 hat hchstens 1 Bit. Dieses Bit mu gesetzt sein. */\
      size_zuweisung bitsize;					\
    }
#endif

// Bits einer 16-Bit-Zahl zhlen:
// integerlength16(digit,size=);
// setzt size auf die hchste in digit vorkommende Bitnummer.
// > digit: ein uint16 >0
// < size: >0, <=16, mit 2^(size-1) <= digit < 2^size
#if defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define integerlength16(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit              */\
      __asm__("bfffo %1{#0:#16},%0" : "=d" (zero_counter) : "dm" ((uint16)(digit)) ); \
      size_zuweisung (16-zero_counter);                                               \
    }
#elif defined(__sparc__)
  #define integerlength16(digit,size_zuweisung)  \
    integerlength32((uint32)(digit),size_zuweisung) // siehe unten
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define integerlength16(digit,size_zuweisung)  \
    { var uintW one_position; /* Position der fhrenden 1                 */\
      __asm__("bsrw %1,%0" : "=r" (one_position) : "r" ((uint16)(digit)) ); \
      size_zuweisung (1+one_position);                                      \
    }
// Die weiteren kommen von gcc/longlong.h :
#elif defined(__GNUC__) && defined(__ibm032__) && !defined(NO_ASM) // RT/ROMP
  #define integerlength16(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit   */\
      __asm__("clz %0,%1" : "=r" (zero_counter) : "r" ((uint32)(digit)) ); \
      size_zuweisung (16-zero_counter);                                    \
    }
#else
  #define integerlength16(digit,size_zuweisung)  \
    { var uintC bitsize = 1;						\
      var uintWL x16 = (uint16)(digit);					\
      /* x16 hat hchstens 16 Bits.                                   */\
      if (x16 >= bit(8)) { x16 = x16>>8; bitsize += 8; }		\
      /* x16 hat hchstens 8 Bits.                                    */\
      if (x16 >= bit(4)) { x16 = x16>>4; bitsize += 4; }		\
      /* x16 hat hchstens 4 Bits.                                    */\
      if (x16 >= bit(2)) { x16 = x16>>2; bitsize += 2; }		\
      /* x16 hat hchstens 2 Bits.                                    */\
      if (x16 >= bit(1)) { /* x16 = x16>>1; */ bitsize += 1; }		\
      /* x16 hat hchstens 1 Bit. Dieses Bit mu gesetzt sein.        */\
      size_zuweisung bitsize;						\
    }
#endif

// Bits einer 32-Bit-Zahl zhlen:
// integerlength32(digit,size=);
// setzt size auf die hchste in digit vorkommende Bitnummer.
// > digit: ein uint32 >0
// < size: >0, <=32, mit 2^(size-1) <= digit < 2^size
#if defined(__GNUC__) && defined(__m68k__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit              */\
      __asm__("bfffo %1{#0:#32},%0" : "=d" (zero_counter) : "dm" ((uint32)(digit)) ); \
      size_zuweisung (32-zero_counter);                                               \
    }
#elif defined(__sparc__) && defined(FAST_DOUBLE)
  #define integerlength32(digit,size_zuweisung)  \
    {var union { double f; uint32 i[2]; } __fi;				\
     const int df_mant_len = 52;  /* mantissa bits (excl. hidden bit) */\
     const int df_exp_mid = 1022; /* exponent bias */                   \
     /* Bilde 2^52 + digit:                                           */\
     __fi.i[0] = (uint32)(df_mant_len+1+df_exp_mid) << (df_mant_len-32); /* Vorzeichen 0, Exponent 53 */\
     __fi.i[1] = (digit); /* untere 32 Bits setzen (benutzt BIG_ENDIAN_P !) */\
     /* subtrahiere 2^52:                                             */\
     __fi.f = __fi.f - (double)(4503599627370496.0L);			\
     /* Hole davon den Exponenten:                                    */\
     size_zuweisung ((__fi.i[0] >> (df_mant_len-32)) - df_exp_mid);	\
    }
#elif defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL one_position; /* Position der fhrenden 1                  */\
      __asm__("bsrl %1,%0" : "=r" (one_position) : "rm" ((uint32)(digit)) ); \
      size_zuweisung (1+one_position);                                       \
    }
#elif defined(__hppa__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    size_zuweisung length32(digit);
  extern "C" uintL length32 (uintL digit); // extern in Assembler
// Die weiteren kommen von gcc/longlong.h :
#elif defined(__GNUC__) && (defined(__a29k__) || defined(___AM29K__)) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit   */\
      __asm__("clz %0,%1" : "=r" (zero_counter) : "r" ((uint32)(digit)) ); \
      size_zuweisung (32-zero_counter);                                    \
    }
#elif defined(__GNUC__) && defined(__gmicro__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit      */\
      __asm__("bsch/1 %1,%0" : "=g" (zero_counter) : "g" ((uint32)(digit)) ); \
      size_zuweisung (32-zero_counter);                                       \
    }
#elif defined(__GNUC__) && defined(__rs6000__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL zero_counter; /* zhlt die fhrenden Nullbits in digit     */\
      __asm__("cntlz %0,%1" : "=r" (zero_counter) : "r" ((uint32)(digit)) ); \
      size_zuweisung (32-zero_counter);                                      \
    }
#elif defined(__GNUC__) && defined(__m88k__) && !defined(NO_ASM)
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL one_position; /* Position der fhrenden 1                */\
      __asm__("ff1 %0,%1" : "=r" (one_position) : "r" ((uint32)(digit)) ); \
      size_zuweisung (1+one_position);                                     \
    }
#elif defined(__GNUC__) && defined(__ibm032__) && !defined(NO_ASM) // RT/ROMP
  #define integerlength32(digit,size_zuweisung)  \
    { var uintL x32 = (uint32)(digit);				\
      if (x32 >= bit(16))					\
        { integerlength16(x32>>16,size_zuweisung 16 + ); }	\
        else							\
        { integerlength16(x32,size_zuweisung); }		\
    }
#else
  #define integerlength32(digit,size_zuweisung)  \
    { var uintC bitsize = 1;						\
      var uintL x32 = (uint32)(digit);					\
      /* x32 hat hchstens 32 Bits.                                   */\
      if (x32 >= bit(16)) { x32 = x32>>16; bitsize += 16; }		\
      /* x32 hat hchstens 16 Bits.                                   */\
      if (x32 >= bit(8)) { x32 = x32>>8; bitsize += 8; }		\
      /* x32 hat hchstens 8 Bits.                                    */\
      if (x32 >= bit(4)) { x32 = x32>>4; bitsize += 4; }		\
      /* x32 hat hchstens 4 Bits.                                    */\
      if (x32 >= bit(2)) { x32 = x32>>2; bitsize += 2; }		\
      /* x32 hat hchstens 2 Bits.                                    */\
      if (x32 >= bit(1)) { /* x32 = x32>>1; */ bitsize += 1; }		\
      /* x32 hat hchstens 1 Bit. Dieses Bit mu gesetzt sein.        */\
      size_zuweisung bitsize;						\
    }
#endif


// Hintere Nullbits eines 32-Bit-Wortes zhlen:
// ord2_32(digit,count=);
// setzt size auf die kleinste in digit vorkommende Bitnummer.
// > digit: ein uint32 >0
// < count: >=0, <32, mit 2^count | digit, digit/2^count ungerade
  #if defined(__GNUC__) && defined(__i386__) && !defined(NO_ASM)
    #define ord2_32(digit,count_zuweisung)  \
      { var uintL one_position; /* Position der letzten 1                    */\
        __asm__("bsfl %1,%0" : "=r" (one_position) : "rm" ((uint32)(digit)) ); \
        count_zuweisung one_position;                                          \
      }
    #define FAST_ORD2
  #else
    // Sei n = ord2(x). Dann ist logxor(x,x-1) = 2^n + (2^n-1) = 2^(n+1)-1.
    // Also  (ord2 x) = (1- (integer-length (logxor x (1- x)))) .
    #define ord2_32(digit,count_zuweisung)  \
      { var uint32 _digit = digit ^ (digit - 1);	\
        integerlength32(_digit,count_zuweisung -1 + )	\
      }
  #endif

#endif /* _CL_LOW_H */
