// Public float operations.

#ifndef _CL_FLOAT_H
#define _CL_FLOAT_H

#include "cl_number.h"
#include "cl_floatformat.h"

// Liefert (- x), wo x ein Float ist.
extern cl_F operator- (const cl_F& x);

// Liefert (+ x y), wo x und y Floats sind.
extern cl_F operator+ (const cl_F& x, const cl_F& y);

// Liefert (- x y), wo x und y Floats sind.
extern cl_F operator- (const cl_F& x, const cl_F& y);

// Liefert (* x y), wo x und y Floats sind.
extern cl_F operator* (const cl_F& x, const cl_F& y);

// Liefert (/ x y), wo x und y Floats sind.
extern cl_F operator/ (const cl_F& x, const cl_F& y);

// Liefert (abs x), wo x ein Float ist.
extern cl_F abs (const cl_F& x);

// Liefert zu einem Float x>=0 : (sqrt x), ein Float.
extern cl_F sqrt (const cl_F& x);

// recip(x) liefert (/ x), wo x ein Float ist.
extern cl_F recip (const cl_F& x);

// (1+ x), wo x ein Float ist.
inline cl_F plus1 (const cl_F& x) // { return x + cl_I(1); }
{
	extern cl_F cl_float (const cl_R&, const cl_F&);
	return x + cl_float(cl_I(1),x);
}

// (1- x), wo x ein Float ist.
inline cl_F minus1 (const cl_F& x) // { return x + cl_I(-1); }
{
	extern cl_F cl_float (const cl_R&, const cl_F&);
	return x + cl_float(cl_I(-1),x);
}

// zerop(x) testet, ob (= x 0).
extern cl_boolean zerop (const cl_F& x);

// minusp(x) testet, ob (< x 0).
extern cl_boolean minusp (const cl_F& x);

// cl_compare(x,y) vergleicht zwei Floats x und y.
// Ergebnis: 0 falls x=y, +1 falls x>y, -1 falls x<y.
extern cl_signean cl_compare (const cl_F& x, const cl_F& y);

inline bool operator== (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)==0; }
inline bool operator!= (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)!=0; }
inline bool operator<= (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)<=0; }
inline bool operator< (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)<0; }
inline bool operator>= (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)>=0; }
inline bool operator> (const cl_F& x, const cl_F& y)
	{ return cl_compare(x,y)>0; }


// ffloor(x) liefert (ffloor x), wo x ein Float ist.
extern cl_F ffloor (const cl_F& x);

// fceiling(x) liefert (fceiling x), wo x ein Float ist.
extern cl_F fceiling (const cl_F& x);

// ftruncate(x) liefert (ftruncate x), wo x ein Float ist.
extern cl_F ftruncate (const cl_F& x);

// fround(x) liefert (fround x), wo x ein Float ist.
extern cl_F fround (const cl_F& x);


// Return type for frounding operators.
// x / y  --> (q,r) with x = y*q+r.
struct cl_F_fdiv_t {
	cl_F quotient;
	cl_F remainder;
// Constructor.
	cl_F_fdiv_t () {}
	cl_F_fdiv_t (const cl_F& q, const cl_F& r) : quotient(q), remainder(r) {}
};

// ffloor2(x) liefert (ffloor x), wo x ein F ist.
extern cl_F_fdiv_t ffloor2 (const cl_F& x);

// fceiling2(x) liefert (fceiling x), wo x ein F ist.
extern cl_F_fdiv_t fceiling2 (const cl_F& x);

// ftruncate2(x) liefert (ftruncate x), wo x ein F ist.
extern cl_F_fdiv_t ftruncate2 (const cl_F& x);

// fround2(x) liefert (fround x), wo x ein F ist.
extern cl_F_fdiv_t fround2 (const cl_F& x);


// Return type for rounding operators.
// x / y  --> (q,r) with x = y*q+r.
struct cl_F_div_t {
	cl_I quotient;
	cl_F remainder;
// Constructor.
	cl_F_div_t () {}
	cl_F_div_t (const cl_I& q, const cl_F& r) : quotient(q), remainder(r) {}
};

// floor2(x) liefert (floor x), wo x ein F ist.
extern cl_F_div_t floor2 (const cl_F& x);
extern cl_I floor1 (const cl_F& x);

// ceiling2(x) liefert (ceiling x), wo x ein F ist.
extern cl_F_div_t ceiling2 (const cl_F& x);
extern cl_I ceiling1 (const cl_F& x);

// truncate2(x) liefert (truncate x), wo x ein F ist.
extern cl_F_div_t truncate2 (const cl_F& x);
extern cl_I truncate1 (const cl_F& x);

// round2(x) liefert (round x), wo x ein F ist.
extern cl_F_div_t round2 (const cl_F& x);
extern cl_I round1 (const cl_F& x);

// floor2(x,y) liefert (floor x y), wo x und y Floats sind.
extern cl_F_div_t floor2 (const cl_F& x, const cl_F& y);
inline cl_I floor1 (const cl_F& x, const cl_F& y) { return floor1(x/y); }

// ceiling2(x,y) liefert (ceiling x y), wo x und y Floats sind.
extern cl_F_div_t ceiling2 (const cl_F& x, const cl_F& y);
inline cl_I ceiling1 (const cl_F& x, const cl_F& y) { return ceiling1(x/y); }

// truncate2(x,y) liefert (truncate x y), wo x und y Floats sind.
extern cl_F_div_t truncate2 (const cl_F& x, const cl_F& y);
inline cl_I truncate1 (const cl_F& x, const cl_F& y) { return truncate1(x/y); }

// round2(x,y) liefert (round x y), wo x und y Floats sind.
extern cl_F_div_t round2 (const cl_F& x, const cl_F& y);
inline cl_I round1 (const cl_F& x, const cl_F& y) { return round1(x/y); }


// The default float format used when converting rational numbers to floats.
extern cl_float_format_t cl_default_float_format;

// Returns the smallest float format which guarantees at least n decimal digits
// in the mantissa (after the decimal point).
extern cl_float_format_t cl_float_format (uintL n);

// cl_float(x,y) wandelt ein Float x in das Float-Format des Floats y um
// und rundet dabei ntigenfalls.
// > x,y: Floats
// < ergebnis: (float x y)
extern cl_F cl_float (const cl_F& x, const cl_F& y);

// cl_float(x,f) wandelt ein Float x in das Float-Format f um
// und rundet dabei ntigenfalls.
// > x: ein Float
// > f: eine Float-Format-Spezifikation
// < ergebnis: (float x f)
extern cl_F cl_float (const cl_F& x, cl_float_format_t f);

// cl_float(x,y) wandelt ein Integer x in das Float-Format des Floats y um
// und rundet dabei ntigenfalls.
// > x: ein Integer
// > y: ein Float
// < ergebnis: (float x y)
extern cl_F cl_float (const cl_I& x, const cl_F& y);

// cl_float(x,y) wandelt ein Integer x in das Float-Format f um
// und rundet dabei ntigenfalls.
// > x: ein Integer
// > f: eine Float-Format-Spezifikation
// < ergebnis: (float x f)
extern cl_F cl_float (const cl_I& x, cl_float_format_t f);

// cl_float(x,y) wandelt eine rationale Zahl x in das Float-Format des
// Floats y um und rundet dabei ntigenfalls.
// > x: eine rationale Zahl
// > y: ein Float
// < ergebnis: (float x y)
extern cl_F cl_float (const cl_RA& x, const cl_F& y);

// cl_float(x,y) wandelt eine rationale Zahl x in das Float-Format f um
// und rundet dabei ntigenfalls.
// > x: eine rationale Zahl
// > f: eine Float-Format-Spezifikation
// < ergebnis: (float x f)
extern cl_F cl_float (const cl_RA& x, cl_float_format_t f);

// The C++ compilers are not clever enough to guess this:
inline cl_F cl_float (int x, const cl_F& y)
	{ return cl_float(cl_I(x),y); }
inline cl_F cl_float (unsigned int x, const cl_F& y)
	{ return cl_float(cl_I(x),y); }
inline cl_F cl_float (int x, cl_float_format_t y)
	{ return cl_float(cl_I(x),y); }
inline cl_F cl_float (unsigned int x, cl_float_format_t y)
	{ return cl_float(cl_I(x),y); }


// Return type for decode_float:
struct cl_decoded_float {
	cl_F mantissa;
	cl_I exponent;
	cl_F sign;
// Constructor.
	cl_decoded_float () {}
	cl_decoded_float (const cl_F& m, const cl_I& e, const cl_F& s) : mantissa(m), exponent(e), sign(s) {}
};

// decode_float(x) liefert zu einem Float x: (decode-float x).
// x = 0.0 liefert (0.0, 0, 1.0).
// x = (-1)^s * 2^e * m liefert ((-1)^0 * 2^0 * m, e als Integer, (-1)^s).
extern cl_decoded_float decode_float (const cl_F& x);

// float_exponent(x) liefert zu einem Float x:
// den Exponenten von (decode-float x).
// x = 0.0 liefert 0.
// x = (-1)^s * 2^e * m liefert e.
extern sintL float_exponent (const cl_F& x);

// float_radix(x) liefert (float-radix x), wo x ein Float ist.
inline sintL float_radix (const cl_F& x)
{
	(void)x; // unused x
	return 2;
}

// float_sign(x) liefert (float-sign x), wo x ein Float ist.
extern cl_F float_sign (const cl_F& x);

// float_sign(x,y) liefert (float-sign x y), wo x und y Floats sind.
extern cl_F float_sign (const cl_F& x, const cl_F& y);

// float_digits(x) liefert (float-digits x), wo x ein Float ist.
// < ergebnis: ein uintL >0
extern uintL float_digits (const cl_F& x);

// float_precision(x) liefert (float-precision x), wo x ein Float ist.
// < ergebnis: ein uintL >=0
extern uintL float_precision (const cl_F& x);

// Returns the floating point format of a float.
inline cl_float_format_t cl_float_format (const cl_F& x)
	{ return (cl_float_format_t) float_digits(x); }


// integer_decode_float(x) liefert zu einem Float x: (integer-decode-float x).
// x = 0.0 liefert (0, 0, 1).
// x = (-1)^s * 2^e * m bei Float-Precision p liefert
//   (Mantisse 2^p * m als Integer, e-p als Integer, (-1)^s als Fixnum).
extern cl_idecoded_float integer_decode_float (const cl_F& x);


// scale_float(x,delta) liefert x*2^delta, wo x ein Float ist.
extern cl_F scale_float (const cl_F& x, sintL delta);
extern cl_F scale_float (const cl_F& x, const cl_I& delta);


// signum(x) liefert (signum x), wo x ein Float ist.
extern cl_F signum (const cl_F& x);


// Returns the largest (most positive) floating point number in float format f.
extern cl_F most_positive_float (cl_float_format_t f);

// Returns the smallest (most negative) floating point number in float format f.
extern cl_F most_negative_float (cl_float_format_t f);

// Returns the least positive floating point number (i.e. > 0 but closest to 0)
// in float format f.
extern cl_F least_positive_float (cl_float_format_t f);

// Returns the least negative floating point number (i.e. < 0 but closest to 0)
// in float format f.
extern cl_F least_negative_float (cl_float_format_t f);

// Returns the smallest floating point number e > 0 such that 1+e != 1.
extern cl_F float_epsilon (cl_float_format_t f);

// Returns the smallest floating point number e > 0 such that 1-e != 1.
extern cl_F float_negative_epsilon (cl_float_format_t f);


#ifdef WANT_OBFUSCATING_OPERATORS
// This could be optimized to use in-place operations.
inline cl_F& operator+= (cl_F& x, const cl_F& y) { return x = x + y; }
inline cl_F& operator++ /* prefix */ (cl_F& x) { return x = plus1(x); }
inline void operator++ /* postfix */ (cl_F& x, int dummy) { x = plus1(x); }
inline cl_F& operator-= (cl_F& x, const cl_F& y) { return x = x - y; }
inline cl_F& operator-- /* prefix */ (cl_F& x) { return x = minus1(x); }
inline void operator-- /* postfix */ (cl_F& x, int dummy) { x = minus1(x); }
inline cl_F& operator*= (cl_F& x, const cl_F& y) { return x = x * y; }
inline cl_F& operator/= (cl_F& x, const cl_F& y) { return x = x / y; }
#endif


// If this is true, floating point underflow returns zero instead of an error.
extern cl_boolean cl_inhibit_floating_point_underflow;


#endif /* _CL_FLOAT_H */
