// Strings.

#ifndef _CL_STRING_H
#define _CL_STRING_H

#include "cl_object.h"
#include "cl_abort.h"
#include <string.h>

// General, reference counted and garbage collected strings.
struct cl_heap_string : public cl_heap {
private:
	unsigned long length;	// length (in characters)
	char data[1];		// the characters, plus a '\0' at the end
	// Standard allocation disabled.
	void* operator new (size_t size) { (void)size; cl_abort (); return NULL; }
	// Standard deallocation disabled.
	void operator delete (void* ptr) { unused ptr; cl_abort(); }
	// No default constructor.
	cl_heap_string ();
private:
// Friend declarations. They are for the compiler. Just ignore them.
	friend class cl_string;
	friend cl_heap_string* cl_make_heap_string (unsigned long len);
	friend cl_heap_string* cl_make_heap_string (const char * s);
	friend cl_heap_string* cl_make_heap_string (const char * ptr, unsigned long len);
	friend cl_string operator+ (const cl_string& str1, const cl_string& str2);
	friend cl_string operator+ (const char* str1, const cl_string& str2);
	friend cl_string operator+ (const cl_string& str1, const char* str2);
};

struct cl_string : public cl_gcpointer {
public:
	// Conversion to simple string.
	// NOTE! The resulting pointer is valid only as long as the string
	// is live, i.e. you must keep the string in a variable until you
	// are done with the pointer to the characters.
	const char * asciz () const
	{
		return &((cl_heap_string*)pointer)->data[0];
	}
	// Return the length (number of characters).
	unsigned long length () const
	{
		return ((cl_heap_string*)pointer)->length;
	}
	// Return a specific character.
	char operator[] (unsigned long i) const
	{
		if (!(i < length())) cl_abort(); // Range check.
		return ((cl_heap_string*)pointer)->data[i];
	}
	// Constructors.
	cl_string ();
	cl_string (const cl_string&);
	cl_string (const char * s);
	cl_string (const char * ptr, unsigned long len);
	// Assignment operators.
	cl_string& operator= (const cl_string&);
	cl_string& operator= (const char *);
	// Optimization of method pointer_p().
	cl_boolean pointer_p() const { return cl_true; }
	// Private pointer manipulations.
	operator cl_heap_string* () const;
	cl_string (cl_heap_string* str) { pointer = str; }
	cl_string (cl_private_thing p) : cl_gcpointer (p) {}
};
CL_DEFINE_COPY_CONSTRUCTOR(cl_string)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_string,cl_string)
inline cl_string::cl_string (const char * s)
{
	pointer = cl_make_heap_string(s);
}
inline cl_string& cl_string::operator= (const char * s)
{
	cl_heap_string* tmp = cl_make_heap_string(s);
	cl_dec_refcount(*this);
	pointer = tmp;
	return *this;
}

// Length.
inline unsigned long strlen (const cl_string& str)
{
	return str.length();
}
// Conversion to `const char *'.
inline const char * asciz (const char * s) { return s; }
inline const char * asciz (const cl_string& s) { return s.asciz(); }

// Comparison.
inline bool equal (const cl_string& str1, const cl_string& str2)
{
    return str1.length() == str2.length()
           && !strcmp(str1.asciz(), str2.asciz());
}
inline bool equal (const char * str1, const cl_string& str2)
{
    return !strcmp(str1, str2.asciz());
}
inline bool equal (const cl_string& str1, const char * str2)
{
    return !strcmp(str1.asciz(), str2);
}

// Private pointer manipulations. Never throw away a `struct cl_heap_string *'!
inline cl_string::operator cl_heap_string* () const
{
	cl_inc_refcount(*this);
	return (cl_heap_string*)pointer;
}
inline cl_string::cl_string ()
{
	extern cl_string cl_null_string;
	pointer = (cl_heap_string*) cl_null_string;
}

// Hash code.
extern unsigned long hashcode (const cl_string& str);

#endif /* _CL_STRING_H */
