// Univariate Polynomials over the rational numbers.

#ifndef _CL_UNIVPOLY_RATIONAL_H
#define _CL_UNIVPOLY_RATIONAL_H

#include "cl_ring.h"
#include "cl_univpoly.h"
#include "cl_number.h"

// Normal univariate polynomials with stricter static typing:
// `cl_RA' instead of `cl_ring_element'.

#ifdef notyet

typedef cl_UP_specialized<cl_RA> cl_UP_RA;
typedef cl_univpoly_specialized_ring<cl_RA> cl_univpoly_rational_ring;
//typedef cl_heap_univpoly_specialized_ring<cl_RA> cl_heap_univpoly_rational_ring;

#else

class cl_heap_univpoly_rational_ring;

class cl_univpoly_rational_ring : public cl_univpoly_ring {
public:
	// Default constructor.
	cl_univpoly_rational_ring () : cl_univpoly_ring () {}
	// Copy constructor.
	cl_univpoly_rational_ring (const cl_univpoly_rational_ring&);
	// Assignment operator.
	cl_univpoly_rational_ring& operator= (const cl_univpoly_rational_ring&);
	// Automatic dereferencing.
	cl_heap_univpoly_rational_ring* operator-> () const
	{ return (cl_heap_univpoly_rational_ring*)heappointer; }
};
// Copy constructor and assignment operator.
CL_DEFINE_COPY_CONSTRUCTOR2(cl_univpoly_rational_ring,cl_univpoly_ring)
CL_DEFINE_ASSIGNMENT_OPERATOR(cl_univpoly_rational_ring,cl_univpoly_rational_ring)

class cl_UP_RA : public cl_UP {
public:
	const cl_univpoly_rational_ring& ring () const { return The(cl_univpoly_rational_ring)(_ring); }
	// Conversion.
	CL_DEFINE_CONVERTER(cl_ring_element)
	// Destructive modification.
	void set_coeff (uintL index, const cl_RA& y);
	void finalize();
	// Evaluation.
	cl_RA operator() (const cl_RA& y) const;
public:	// Ability to place an object at a given address.
	void* operator new (size_t size) { return cl_malloc_hook(size); }
	void* operator new (size_t size, cl_UP_RA* ptr) { (void)size; return ptr; }
	void operator delete (void* ptr) { cl_free_hook(ptr); }
};

class cl_heap_univpoly_rational_ring : public cl_heap_univpoly_ring {
	SUBCLASS_cl_heap_univpoly_ring()
	// High-level operations.
	void fprint (cl_ostream stream, const cl_UP_RA& x)
	{
		cl_heap_univpoly_ring::fprint(stream,x);
	}
	cl_boolean equal (const cl_UP_RA& x, const cl_UP_RA& y)
	{
		return cl_heap_univpoly_ring::equal(x,y);
	}
	cl_UP_RA zero ()
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::zero());
	}
	cl_boolean zerop (const cl_UP_RA& x)
	{
		return cl_heap_univpoly_ring::zerop(x);
	}
	cl_UP_RA plus (const cl_UP_RA& x, const cl_UP_RA& y)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::plus(x,y));
	}
	cl_UP_RA minus (const cl_UP_RA& x, const cl_UP_RA& y)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::minus(x,y));
	}
	cl_UP_RA uminus (const cl_UP_RA& x)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::uminus(x));
	}
	cl_UP_RA one ()
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::one());
	}
	cl_UP_RA canonhom (const cl_I& x)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::canonhom(x));
	}
	cl_UP_RA mul (const cl_UP_RA& x, const cl_UP_RA& y)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::mul(x,y));
	}
	cl_UP_RA square (const cl_UP_RA& x)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::square(x));
	}
	cl_UP_RA expt_pos (const cl_UP_RA& x, const cl_I& y)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::expt_pos(x,y));
	}
	sintL degree (const cl_UP_RA& x)
	{
		return cl_heap_univpoly_ring::degree(x);
	}
	cl_UP_RA monomial (const cl_RA& x, uintL e)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::monomial(cl_ring_element(cl_RA_ring,x),e));
	}
	cl_RA coeff (const cl_UP_RA& x, uintL index)
	{
		return The(cl_RA)(cl_heap_univpoly_ring::coeff(x,index));
	}
	cl_UP_RA create (sintL deg)
	{
		return The2(cl_UP_RA)(cl_heap_univpoly_ring::create(deg));
	}
	void set_coeff (cl_UP_RA& x, uintL index, const cl_RA& y)
	{
		cl_heap_univpoly_ring::set_coeff(x,index,cl_ring_element(cl_RA_ring,y));
	}
	void finalize (cl_UP_RA& x)
	{
		cl_heap_univpoly_ring::finalize(x);
	}
	cl_RA eval (const cl_UP_RA& x, const cl_RA& y)
	{
		return The(cl_RA)(cl_heap_univpoly_ring::eval(x,cl_ring_element(cl_RA_ring,y)));
	}
private:
	// No need for any constructors.
	cl_heap_univpoly_rational_ring ();
};

// Lookup of polynomial rings.
inline cl_univpoly_rational_ring cl_find_univpoly_ring (const cl_rational_ring& r)
{ return The(cl_univpoly_rational_ring) (cl_find_univpoly_ring((const cl_ring&)r)); }
inline cl_univpoly_rational_ring cl_find_univpoly_ring (const cl_rational_ring& r, const cl_symbol& varname)
{ return The(cl_univpoly_rational_ring) (cl_find_univpoly_ring((const cl_ring&)r,varname)); }

// Operations on polynomials.

// Add.
inline cl_UP_RA operator+ (const cl_UP_RA& x, const cl_UP_RA& y)
	{ return x.ring()->plus(x,y); }

// Negate.
inline cl_UP_RA operator- (const cl_UP_RA& x)
	{ return x.ring()->uminus(x); }

// Subtract.
inline cl_UP_RA operator- (const cl_UP_RA& x, const cl_UP_RA& y)
	{ return x.ring()->minus(x,y); }

// Multiply.
inline cl_UP_RA operator* (const cl_UP_RA& x, const cl_UP_RA& y)
	{ return x.ring()->mul(x,y); }

// Squaring.
inline cl_UP_RA square (const cl_UP_RA& x)
	{ return x.ring()->square(x); }

// Exponentiation x^y, where y > 0.
inline cl_UP_RA expt_pos (const cl_UP_RA& x, const cl_I& y)
	{ return x.ring()->expt_pos(x,y); }

// Scalar multiplication.
inline cl_UP_RA operator* (const cl_I& x, const cl_UP_RA& y)
	{ return y.ring()->mul(y.ring()->canonhom(x),y); }
inline cl_UP_RA operator* (const cl_UP_RA& x, const cl_I& y)
	{ return x.ring()->mul(x.ring()->canonhom(y),x); }

// Coefficient.
inline cl_RA coeff (const cl_UP_RA& x, uintL index)
	{ return x.ring()->coeff(x,index); }

// Destructive modification.
inline void set_coeff (cl_UP_RA& x, uintL index, const cl_RA& y)
	{ x.ring()->set_coeff(x,index,y); }
inline void finalize (cl_UP_RA& x)
	{ x.ring()->finalize(x); }
inline void cl_UP_RA::set_coeff (uintL index, const cl_RA& y)
	{ ring()->set_coeff(*this,index,y); }
inline void cl_UP_RA::finalize ()
	{ ring()->finalize(*this); }

// Evaluation. (No extension of the base ring allowed here for now.)
inline cl_RA cl_UP_RA::operator() (const cl_RA& y) const
{
	return ring()->eval(*this,y);
}

// Derivative.
inline cl_UP_RA deriv (const cl_UP_RA& x)
	{ return The2(cl_UP_RA)(deriv((const cl_UP&)x)); }

#endif


// Returns the n-th Legendre polynomial (n >= 0).
extern cl_UP_RA cl_legendre (sintL n);

#endif /* _CL_UNIVPOLY_RATIONAL_H */
