// atanhx().

// General includes.
#include "cl_sysdep.h"

// Specification.
#include "cl_R_tran.h"
#include "cl_F.h"
#include "cl_lfloat.h"
#include "cl_LF.h"


// Implementation.

#include "cl_real.h"
#include "cl_float.h"
#include "cl_low.h"

cl_F atanhx (const cl_F& x)
{
// Methode:
// e := Exponent aus (decode-float x), d := (float-digits x)
// Bei x=0.0 oder e<=-d/2 liefere x
//   (denn bei e<=-d/2 ist x^2 < 2^(-d), also
//   1 <= atanh(x)/x = 1+x^2/3+x^4/5+... < 1+x^2/2 < 1+2^(-d-1) < 1+2^(-d),
//   also ist atanh(x)/x, auf d Bits gerundet, gleich 1.0).
// Bei groem d verwende die Formel ln((1+x)/(1-x))/2 (asymptotisch schneller),
//   aber erhhe die Genauigkeit, so da beim Bilden von 1+x keine Bits verloren
//   gehen.
// Bei e<=-sqrt(d) verwende die Potenzreihe
//   atanh(x)/x = sum(j=0..inf,(x^2)^j/(2j+1)):
//   a:=x^2, b:=1, i:=1, sum:=0,
//   while (/= sum (setq sum (+ sum (/ b i)))) do i:=i+2, b:=b*a.
//   Ergebnis x*sum.
// Sonst setze y := x/(1+sqrt(1-x^2)), berechne rekursiv z:=atanh(y)
//   und liefere 2*z = (scale-float z 1).
// Diese Rekursion wird entrekursiviert. Statt k mal hintereinander
//   x := x/(1+sqrt(1-x^2)) zu bilden, arbeitet man lieber mit den Kehrwerten,
//   setzt also x := 1/|x|, dann k mal x := x+sqrt(x^2-1), dann x := +- 1/x.
// Aufwand: asymptotisch d^2.5 .

	if (zerop(x))
		return x;
	var uintL d = float_digits(x);
	var sintL e = float_exponent(x);
	if (e <= (sintL)(-d)>>1) // e <= -d/2 <==> e <= -ceiling(d/2)
		return x; // ja -> x als Ergebnis
	if (longfloatp(x) && (TheLfloat(x)->len >= 31)) {
		DeclareType(cl_LF,x);
		var cl_LF xx = extend(x,TheLfloat(x)->len+ceiling((uintL)(-e),intDsize));
		return cl_float(scale_float(ln((1+xx)/(1-xx)),-1),x);
	}
	var uintL k = 0; // Rekursionszhler k:=0
	var uintL sqrt_d = floor(isqrt(d)*5,8); // limit_slope*floor(sqrt(d))
	// Bei e <= -1-limit_slope*floor(sqrt(d)) kann die Potenzreihe
	// angewandt werden. limit_slope = 1.0 ist schlecht (ca. 15% zu
	// schlecht). Ein guter Wert ist limit_scope = 0.625 = 5/8.
	var cl_F xx = x;
	if (e >= (sintL)(-sqrt_d)) {
		// e > -1-limit_slope*floor(sqrt(d)) -> mu |x| verkleinern.
		var sintL e_limit = 1+sqrt_d; // 1+limit_slope*floor(sqrt(d))
		xx = recip(abs(xx)); // 1/|x|
		do {
		  // nchstes x nach der Formel x := x+sqrt(x^2 - 1) berechnen:
		  xx = sqrt(square(xx) + cl_float(-1,xx)) + xx;
		  k = k+1;
		} until (float_exponent(xx) > e_limit);
		// Schleifenende mit Exponent(x) > 1+limit_slope*floor(sqrt(d)),
		// also x >= 2^(1+limit_slope*floor(sqrt(d))),
		// also 1/x <= 2^(-1-limit_slope*floor(sqrt(d))).
		// Nun kann die Potenzreihe auf 1/x angewandt werden.
		xx = recip(xx);
		if (minusp(x))
			xx = - xx; // Vorzeichen wieder rein
	}
	// Potenzreihe anwenden:
	var int i = 1;
	var cl_F a = square(xx); // a = x^2
	var cl_F b = cl_float(1,xx); // b := (float 1 x)
	var cl_F sum = cl_float(0,xx); // sum := (float 0 x)
	loop {
		var cl_F new_sum = sum + b / (cl_I)i; // (+ sum (/ b i))
		if (new_sum == sum) // = sum ?
			break; // ja -> Potenzreihe abbrechen
		sum = new_sum;
		b = b*a;
		i = i+2;
	}
	var cl_F erg = sum*xx; // sum*x als Ergebnis
	return scale_float(erg,k); // wegen Rekursion noch mal 2^k
}
// Bit complexity (N = length(x)): O(log(N)^2*M(N)).

// Timings of the above algorithms, on an i486 33 MHz, running Linux,
// applied to x = sqrt(2)-1 = 0.414...
//   N      naive  use ln
//   10     0.013   0.020
//   25     0.059   0.080
//   50     0.24    0.17
//  100     1.03    0.53
//  250     7.6     2.5
//  500    36.1     9.3
// 1000   176      28.2
// ==> using ln faster for N >= 31.
