/*
Copyright (C) 2000  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include "header.h"
FILE *yyin;
int yyparse (void);
void  yyrestart(FILE *new_file);
void patchfunclist(void);
void inittype(void);
void initoperators(void);
int linecount,errs;
int yydebug;
static int optgen=0,opttree=0,opttype=0,optinfo=0;
int optcleanvar=0;/*used in genbrace to suppress {} optimization*/
int indentlevel=2;
void printlistfunc(FILE *fout);
void outputtype(FILE *fout);
void cleancode(int n, int p);
void cleanvar(int n);
void initdesc(const char *s);
void dump(FILE *fout)
{
  int i,j;
  for(i=0;i<nfunc;i++)
  {
    gpfunc *gp=&lfunc[i].gp;
    context *fc=block+lfunc[i].bl;
    fprintf(fout,"Function:\n %s",gp->cname);
    for(j=0;j<fc->n;j++)
    {
      if (fc->c[j].flag&(1<<Carg))
	fprintf(fout,"%c%s %s",j?',':'(',
		Gname[fc->c[j].t],fc->c[j].var);
    }
    fprintf(fout,")\n");
    if (strcmp(gp->gpname,gp->cname))
      fprintf(fout,"GP name: %s\n",gp->gpname);
    fprintf(fout,"code: %s\n",gp->code);
    fprintf(fout,"return type: %s\n",Gname[gp->type]);
    fprintf(fout,"mode=%d \t  spec=%d\n",gp->mode,gp->spec);
    for(   ;j<fc->n;j++)
    {
      ctxvar *c=fc->c+j;
      if (!((c->flag&(1<<Carg)) || ((c->flag&(1<<Cconst)) && c->val==-1)))
	fprintf(fout,"%s %s\n",Gname[c->t],c->var);
    }
    fprintf(fout,"\n");
  }
}

void
compile(FILE *fin, char *nom)
{
  char *descfile;
  int startnode;
  linecount=1;
  nameparse=nom;
  yyin=fin;
  nnode=0;tree=NULL;node_alloc=0;
  nvalue=0;value=NULL;value_alloc=0;
  nfunc=0;lfunc=NULL;func_alloc=0;
  nctx=0;ctxstack=NULL;ctx_alloc=0;
  nbloc=0;block=NULL;bloc_alloc=0;
  naff=0;affstack=NULL;aff_alloc=0;
  currfunc=-1;
  initoperators();
  patchfunclist();
  inittype();
  if (!(descfile=getenv("GP2C_FUNC_DSC")))
    descfile=FUNCDSC_PATH;
  initdesc(descfile);
  if(opttype) {outputtype(stdout);exit(0);}

  /*Node 0 should be a (Gvoid)Fgnil*/
  tree[newnode(Fgnil,-1,-1)].t=Gvoid;
  ungetc('@',fin);/*mark start of file*/
  errs=0;
  if(yyparse())
  {
    fprintf(stderr,"Errors found: aborting...\n");
    exit(1);
  }
  if (errs)
  {
    fprintf(stderr,"%d error%s found: aborting...\n",errs,errs==1?"":"s");
    exit(1);
  }
  if (!nnode)
  {
    fprintf(stdout,"/*No code was generated*/\n");
    exit(0);
  }
  startnode=nnode-1;
  if (optgen)
  {
    printnode(stdout,startnode);
    printf("\n");
    return;
  }
  if (yydebug) fprintf(stderr,"End of parsing\n");

  genblock(startnode,-1);
  startnode=addinitfunc(startnode);
  if (opttree)
  {
    maketree(stderr,startnode);
    fprintf(stderr,";\n");
  }
  if (debug)
    printnode(stderr,startnode);
  do
  {
    lastpass=0;
    gentype(startnode);
  }while(lastpass);
  if (debug>=2)
  {
    fprintf(stderr,"\n--------------END GENTYPE------------\n");
    printnode(stderr,startnode);
  }
  moveblock(startnode,-1,0,NULL,NULL);
  if (debug)
  {
    fprintf(stderr,"\n--------------END MOVEBLOCK------------\n");
    printnode(stderr,startnode);
  }
  do
  {
    lastpass=0;
    varlist(startnode);
  }while(lastpass);
  if (!optcleanvar)
    cleanvar(startnode);
  if (autogc)
  {
    pilelist(startnode);
    pileclean(startnode);
  }
  if (optinfo) dump(stderr);
  cleancode(startnode,-1);
  if (opttree)
  {
    maketree(stderr,startnode);
    fprintf(stderr,";\n");
  }
  if (debug)
  {
    fprintf(stderr,"\n--------------END CLEANCODE------------\n");
    printnode(stderr,startnode);
  }
  genheader(stdout);
  gencode(stdout,startnode);
}
void version(void)
{
  printf("GP to C compiler version %s \n   targeted at PARI/GP %s\n",VERSION,PARI_VERSION);
  printf("Copyright 2000 The PARI Group\n");
  printf("GP2C is free software, covered by the GNU General Public License, and
you are welcome to change it and/or distribute copies of it under
certain conditions.  There is absolutely no warranty for GP2C.\n");
}
void usage(FILE *fout, char *s)
{
  fprintf(fout,"%s [-gihfltvydTGV] [file.gp]
GP to C translator.

user option:
 -g : Generate automatic garbage collection code.
 -iN: Set indentation level to N spaces (default 2).
 -W : Output information about global variable.
query options:
 -h : this help.
 -f : Dump information about functions to stderr.
 -l : Output the list of functions known to the compiler.
 -t : Output the table of types known to the compiler.
 -v : Output version information and exit.
debugging options:
 -d : Increase debugging level.
 -y : Switch parser to debug mode.
 -T : Output syntactic tree in treetool format.
 -G : Generate GP code in place of C code. Don't smile.
 -V : do not clean up variables.

file.gp: file to be processed, default to stdin.
The generated C code is output to stdout.
",s);
}
int main(int argc, char **argv)
{
  char c;
  FILE *fin;
  autogc=0;
  warn=0;
  while((c=getopt(argc,argv,"gi:hfltvydTGVW"))!=-1)
  {
    switch(c)
    {
    case 'g':
      autogc=1-autogc;
      break;
    case 'i':
      indentlevel=atoi(optarg);
      break;
    case 'h':
      usage(stdout,argv[0]);
      exit(0);
      break;
    case 'f':
      optinfo=1-optinfo;
      break;
    case 'l':
      patchfunclist();
      printlistfunc(stdout);
      exit(0);
    case 't':
      opttype=1-opttype;
      break;
    case 'v':
      version();
      exit(0);
    case 'd':
      debug++;
      break;
    case 'y':
      yydebug++;
      break;
    case 'T':
      opttree=1-opttree;
      break;
    case 'G':
      optgen=1-optgen;
      break;
    case 'V':
      optcleanvar=1-optcleanvar;
      break;
    case 'W':
      warn=1-warn;
      break;
    case '?':
      usage(stderr,argv[0]);
      exit(1);
      break;
    }
  }
  if (argc-optind>1)
  {
    usage(stderr,argv[0]);
    exit(1);
  }
  if (argc==optind)
    compile(stdin,"stdin");
  else
  {
    char *nom=argv[optind];
    char *nomf=nom;
    if ((fin=fopen(nomf,"r"))==NULL)
    {
      perror(argv[0]);
      exit(errno);
    }
    compile(fin,nom);
    fclose(fin);
  }
  return 0;
}
