/*
Copyright (C) 2000  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "header.h"
char *Ffuncname[]={"Fseq",
		   "Fmatrix","FmatrixL","FmatrixR",
		   "Faffect",
		   "Ffacteurmat",
		   "Fmatrixelts","Fmatrixlines",
		   "Femptyvec","Femptymat","Fmat",
		   "Flistarg",

		   "Fconst","Fstring","Fsmall",
		   "Frefarg","FtrucQ","Ftag",
		   "Fentry","Fentryfunc","Fdeffunc",

		   /*These nodes are generated by genblock, not by parser*/
		   "Fblock","Fgnil"
};

int isanint(const char *s)
{
  do
    if (*s<'0' || *s>'9')
      return 0;
  while(*++s);
  return 1;
}
int isasmall(const char *s, long *res)
{
  char *endptr;
  if (!isanint(s))
    return 0;
  errno=0;
  *res=strtol(s,&endptr,10);
  if (!*endptr && !errno)
    return 1;
  else 
    return 0;
}
int linecount;
int newnode(Ffunc f, int x, int y)
{
  if (nnode>=node_alloc)
  {
    node_alloc=nnode+100;
    if (debug) fprintf(stderr,"tree=%x alloc node: %d/%d\n",(unsigned int)tree,nnode,node_alloc);
    tree=realloc(tree,node_alloc*sizeof(*tree));
    if (tree==NULL)
      fprintf(stderr,"Oups... alloc node: %d/%d\n",nnode,node_alloc);
  }
  tree[nnode].f=f;
  tree[nnode].x=x;
  tree[nnode].y=y;
  tree[nnode].t=Gnotype;
  tree[nnode].m=0;
  tree[nnode].lineno=linecount;
  return nnode++;
}
int newopcall(OPerator op, int x, int y)
{
  if (y==-1)
    return newnode(Fentryfunc,op,x);
  else
    return newnode(Fentryfunc,op,newnode(Flistarg,x,y));
}
int newvalue(Ffunc func, const char *s)
{
  long small;
  if (nvalue>=value_alloc)
  {
    value_alloc=nvalue+100;
    if (debug) fprintf(stderr,"alloc value: %d/%d\n",nvalue,value_alloc);
    value=realloc(value,value_alloc*sizeof(*value));
    if (value==NULL)
      fprintf(stderr,"Oups... alloc value: %d/%d\n",nvalue,value_alloc);
  }
  switch(func)
  {
  case Fconst:
    if (isasmall(s,&small))
    {
      value[nvalue].type=Vsmall;
      value[nvalue].val.small=small;
    }
    else
    {
      value[nvalue].type=Vstr;
      value[nvalue].val.str=strdup(s);
    }
    break;
  case FtrucQ:/*should be special*/
  case Fstring:
    value[nvalue].type=Vstr;
    value[nvalue].val.str=strdup(s);
    break;
  default:
    die(-1,"Internal error: unhandled func %d in newvalue", func);
  }
  return nvalue++;
}
int newconstnode(const char *s)
{
  long small;
  if (isasmall(s,&small))
    return newnode(Fsmall,small,-1);
  else
    return newnode(Fconst,newvalue(Fconst,s),-1);
}
int newentry(const char *s)
{
  return newvalue(Fstring,s);
}
int isfunc(int n, const char *s)
{
  return tree[n].f==Fentryfunc && !strcmp(s,value[tree[n].x].val.str);
}
int newuserentry(const char *s)
{
  const char *p;
  int n;
  if (s[0]=='p' || s[0]=='l')
  {
    for(p=s+1; *p=='_'; p++);
    if (isdigit(*p))
    {
      for (   ; isdigit(*p); p++);
      if (!*p)
      {
	char *q=calloc(sizeof(*s),2+strlen(s));
	sprintf(q,"%c_%s",s[0],s+1);
	n=newvalue(Fstring,q);
	free(q);
	return n;
      }
    }
  }
  return newvalue(Fstring,s);
}
int newmember(const char *s)
{
  char *p;
  int n;
  p=calloc(sizeof(*s),3+strlen(s));
  sprintf(p,"_.%s",s);
  n=newvalue(Fstring,p);
  free(p);
  return n;
}
void initoperators(void)
{
  int i;
  for (i=0;i<OPnboperator;i++)
    newentry(opname[i]);
}
int newleaf(int n)
{
  int r;
  if (n==-1)
    return newnode(Fgnil,-1,-1);
  r=newnode(tree[n].f,tree[n].x,tree[n].y);
  tree[r]=tree[n];
  return r;
}
int getlvalue(int n)
{
  if (n==-1)
    return -1;
  switch(tree[n].f)
  {
  case Fentry:
    return n;
  case Ftag:
    return getlvalue(tree[n].x);
  case Ffacteurmat:
    return getlvalue(tree[n].x);
  default:
    return -1;
  }
}
int getlvaluerr(int n)
{
  int ret=getlvalue(n);
  if(ret==-1)
    die(n,"not an lvalue");
  return ret;
}
Gtype strtotype(char *s)
{
  int i;
  for(i=0;i<Gnbtype;i++)
  {
    if (!strcmp(s,Gname[i]))
      return i;
  }
  return -1;
}
Gtype nodetype(char *s)
{
  int t=strtotype(s);
  if (t==-1)
    die(-1,"unknow type %s",s);
  return t;
}
Mmode strtomode(char *s)
{
  int i;
  for(i=0;i<Mnbmode;i++)
  {
    if (!strcmp(s,Mname[i]))
      return i;
  }
  return -1;
}
Mmode nodemode(char *s)
{
  int t=strtomode(s);
  if (t==-1)
    die(-1,"unknow mode %s",t);
  return t;
}

int listtoseq(int *stack, int nb)
{
  int n,i;
  if (nb==-1) return -1;
  for(n=stack[0],i=1;i<nb;n=newnode(Flistarg,n,stack[i++]));
  return n;
}
static int hasproto=0;
void printnodeparens(FILE *fout, int n, int parens)
{
  int i,v;
  context *fc;
  int x,y;
  if (n<0)
    return;
  x=tree[n].x;
  y=tree[n].y;
  if (debug>=3) fprintf(fout,"/*%s:%d:%s*/",Gname[tree[n].t],tree[n].m,Ffuncname[tree[n].f]);
  switch(tree[n].f)
  {
  case Fseq:
    printnode(fout,x);
    if (x>=0 && tree[x].f!=Fblock && 
	(tree[x].f!=Fseq || tree[tree[x].y].f!=Fblock))
      fprintf(fout,";\n");
    printnode(fout,y);
    break;
  case Fmatrix:
    fprintf(fout,"[");
    printnode(fout,x);
    fprintf(fout,"]");
    break;
  case FmatrixL:
    fprintf(fout,"[");
    printnode(fout,x);
    fprintf(fout,",]");
    break;
  case FmatrixR:
    fprintf(fout,"[,");
    printnode(fout,x);
    fprintf(fout,"]");
    break;
  case Ffacteurmat:
    printnode(fout,x);
    printnode(fout,y);
    break;
  case Faffect:
    printnode(fout,x);
    fprintf(fout,"=");
    printnode(fout,y);
    break;
  case Fconst:
  case Fstring:
    if (value[x].type==Vsmall)
      fprintf(fout,"%ld",value[x].val.small);
    else
      fprintf(fout,"\"%s\"",value[x].val.str);
    break;
  case Fsmall:
    fprintf(fout,"%d",x);
    break;
  case FtrucQ:
    fprintf(fout,"'%s",value[x].val.str);
    break;
  case Fmatrixelts:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fmatrixlines:
    printnode(fout,x);
    fprintf(fout,";");
    printnode(fout,y);
    break;
  case Femptyvec:
    fprintf(fout,"[]");
    break;
  case Femptymat:
    fprintf(fout,"[;]");
    break;
  case Fmat:
    fprintf(fout,"[");
    printnode(fout,x);
    fprintf(fout,"]");
    break;
  case Flistarg:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fentry:
    fprintf(fout,"%s",value[x].val.str);
    break;  
  case Fentryfunc:
    if (x<OPnboperator || (value[x].val.str[0]=='_' && value[x].val.str[1]=='.'))
    {
      int arg[1024];
      int nb=genlistargs(n,arg,0,1024);
      int i=0;
      char *p,*name=value[x].val.str;
      if (parens)
	fputc('(',fout);
      for (p=name;*p;p++)
      {
	if (*p=='_')
	{
	  if (i==nb)
	    die(n,"too many arguments for operator %s",name);
	  printnodeparens(fout,arg[i++],1);

	}
	else
	  fputc(*p,fout);
      }
      if (i!=nb)
	die(n,"too few arguments for operator %s",name);
      if (parens)
	fputc(')',fout);
    }
    else if (!hasproto || strcmp(value[x].val.str,"local"))
    {
      fprintf(fout,"%s",value[x].val.str);
      fprintf(fout,"(");
      printnode(fout,y);
      fprintf(fout,")");
    }
    break;
  case Frefarg:
    fprintf(fout,"&%s",value[x].val.str);
    break;
  case Fdeffunc:
    printnode(fout,x);
    fprintf(fout,"=\n");
    if (tree[y].f!=Fblock)
      fprintf(fout,"{\n");
    printnode(fout,y);
    fprintf(fout,"\n");
    if (tree[y].f!=Fblock)
      fprintf(fout,"}\n");
    break;
  case Fblock:
    fprintf(fout,"{");
    fc=block+x;
    v=fc->ret;
    if (v>=0)
      fprintf(fout,"/*=%s*/",value[tree[v].x].val.str);
    fprintf(fout,"\n");
    if (fc->n)
    {
      fprintf(fout,"local(");
      for(i=0;i<fc->n;i++)
	fprintf(fout,"%s%c",fc->c[i].var,i==fc->n-1?')':',');
      fprintf(fout,";\n");
    }    
    printnode(fout,y);
    fprintf(fout,"\n}\n");
    break;
  case Fgnil:
    fprintf(fout,"gnil");
    break;
  case Ftag:
    printnode(fout,x);    
    fprintf(fout,":%s",Gname[y]);
    break;
  }
}
void printnode(FILE *fout, int n)
{
  printnodeparens(fout,n,0);
}

void maketree(FILE *fout,int n)
{
  switch(tree[n].f)
  {
  case Ftag:
    fprintf(fout,"(");
    maketree(fout,tree[n].x);
    fprintf(fout,",");
    fprintf(fout,"%s",Gname[tree[n].y]);
    fprintf(fout,")");
    break;
  default:
    if (tree[n].f<FneedENTRY)
    {
      if (tree[n].x!=-1)
      {
	fprintf(fout,"(");
	maketree(fout,tree[n].x);
	if (tree[n].y!=-1 )
	{
	  fprintf(fout,",");
	  maketree(fout,tree[n].y);
	}
	fprintf(fout,")");
      }
    }
    else
    {
      if (tree[n].y!=-1)
      {
	fprintf(fout,"(");
	maketree(fout,tree[n].y);
	fprintf(fout,")");
      }
    }
  }
  fprintf(fout,"%s_%d",Ffuncname[tree[n].f],n);
}
char *funcname(int f)
{
  if (f<0 || f>Fgnil)
    return "Funknown";
  else
    return Ffuncname[f];
}
