/*
Copyright (C) 2002  The PARI group.

This file is part of the GP2C package.

PARI/GP is free software; you can redistribute it and/or modify it under the
terms of the GNU General Public License as published by the Free Software
Foundation. It is distributed in the hope that it will be useful, but WITHOUT
ANY WARRANTY WHATSOEVER.

Check the License for details. You should have received a copy of it, along
with the package; see the file 'COPYING'. If not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. */

#include "config.h"
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include "header.h"
const char *Ffuncname[]={"Fseq",
		   "Fmatrix","FmatrixL","FmatrixR",
		   "Faffect",
		   "Ffacteurmat",
		   "Fmatrixelts","Fmatrixlines",
		   "Femptyvec","Femptymat","Fmat",
		   "Flistarg",

		   "Fconst","Fsmall","Fgnil",
		   "Frefarg","Ftag",
		   "Fentry","Fentryfunc","Fdeffunc",

		   /*These nodes are generated by genblock, not by parser*/
		   "Fblock"
};

int isanint(const char *s)
{
  do
    if (*s<'0' || *s>'9')
      return 0;
  while(*++s);
  return 1;
}
/* 0 not an integer
 * 1 a true integer
 * 2 a real which is an integer (1.,17.00)
 */
int isarealint(const char *s)
{
  while (*s>='0' && *s<='9')
    s++;
  if (!*s) return 1;
  if (*s != '.') return 0;
  while(*++s)
    if (*s != '0') return 0;
  return 2;
}

int linecount;
int newcomment(void)
{
  int n=stack_new(&s_comment);
  comment *c=com+n;
  stack_init(&c->s,sizeof(*c->txt),(void **)&c->txt);
  return n;
}

void pushcomment(int n, char x)
{
  comment *c=com+n;
  int m=stack_new(&c->s);
  c->txt[m]=x;
}

int newnodecom(Ffunc f, int x, int y, int com)
{
  int n=stack_new(&s_node);
  tree[n].f=f;
  tree[n].x=x;
  tree[n].y=y;
  tree[n].t=Gnotype;
  tree[n].m=0;
  tree[n].lineno=linecount;
  tree[n].comment=com;
  return n;
}
int newnode(Ffunc f, int x, int y)
{
  return newnodecom(f,x,y,-1);
}
int newopcall(OPerator op, int x, int y)
{
  if (y==-1)
    return newnode(Fentryfunc,op,x);
  else
    return newnode(Fentryfunc,op,newnode(Flistarg,x,y));
}

int newstringvalue(const char *s)
{
  int n=stack_new(&s_value);
  value[n].type=CSTstr;
  value[n].val.str=s;
  return n++;
}

int newquotevalue(const char *s)
{
  int n=stack_new(&s_value);
  value[n].type=CSTquote;
  value[n].val.str=s;
  return n++;
}

int newsmallrealvalue(long small)
{
  int n=stack_new(&s_value);
  value[n].type=CSTsmallreal;
  value[n].val.small=small;
  return n++;
}

int newnumvalue(const char *s)
{
  int n=stack_new(&s_value);
  value[n].type=CSTexpr;
  value[n].val.str=s;
  return n++;
}

int newdotvalue(const char *s)
{
  int n=stack_new(&s_value);
  char *t=calloc(strlen(s)+2,sizeof(s));
  sprintf(t,"%s.",s);
  value[n].type=CSTexpr;
  value[n].val.str=t;
  return n++;
}

/* newxxxnode functions
 * These unctions take a token value as a string and a comment index.
 * They must return a new Fconst or Fsmall node.
 * They must free the string if they do not reference it.
 * the cast free((char *)s) is to remove the const qualifier.
 */
int newintnode(const char *s, int c)
{
  long small;
  char *endptr;
  errno=0;
  small=strtol(s,&endptr,10);
  if (!*endptr && !errno)
  {
    free((char *)s);
    return newnodecom(Fsmall,small,-1,c);
  }
  else
    return newnodecom(Fconst,newnumvalue(s),-1,c);
}

int newrealnode(const char *s, int c)
{
  char *endptr;
  int val;
  if (!isarealint(s))
    val=newnumvalue(s);
  else
  {
    long small;
    errno=0;/* for catching strtol overflows*/
    small=strtol(s,&endptr,10);
    if (*endptr=='.' && !errno)
    {
      val=newsmallrealvalue(small);
      free((char *)s);
    }
    else 
      val=newnumvalue(s);
  }
  return newnodecom(Fconst,val,-1,c);
}

int newdotnode(const char *s, int c)
{
  char *endptr;
  int val;
  long small;
  errno=0;
  small=strtol(s,&endptr,10);
  if (!*endptr && !errno)
    val=newsmallrealvalue(small);
  else 
    val=newdotvalue(s);
  free((char *)s);
  return newnodecom(Fconst,val,-1,c);
}

int is_const(int n, CSTtype t)
{
  int f=tree[n].f;
  int x=tree[n].x;
  return f==Fconst && value[x].type==t;
}

int newentry(const char *s)
{
  return newstringvalue(s);
}

int isfunc(int n, const char *s)
{
  return tree[n].f==Fentryfunc && !strcmp(s,value[tree[n].x].val.str);
}

extern char *optprefix;
const char *usercname(const char *s)
{
  const char *p;
  if (!optprefix)
  {
    if (s[0]=='p' || s[0]=='l')
    {
      for(p=s+1; *p=='_'; p++);
      if (isdigit(*p))
      {
	for (   ; isdigit(*p); p++);
	if (!*p)
	{
	  char *q=calloc(sizeof(*s),2+strlen(s));
	  sprintf(q,"%c_%s",s[0],s+1);
          return q;
	}
      }
    }
  }
  else 
  {
    char *q=calloc(sizeof(*s),strlen(optprefix)+1+strlen(s));
    sprintf(q,"%s%s",optprefix,s);
    return q;
  }
  return s;
}

int newmember(const char *s)
{
  char *p;
  int n;
  p=calloc(sizeof(*s),3+strlen(s));
  sprintf(p,"_.%s",s);
  n=newentry(p);
  free((char *)s);
  return n;
}
void initoperators(void)
{
  int i;
  for (i=0;i<OPnboperator;i++)
    newentry(opname[i]);
}

const char *entryname(int n)
{
  return value[tree[n].x].val.str;
}

int newleaf(int n)
{
  int r;
  if (n==-1)
    return GNIL;
  r=newnode(tree[n].f,tree[n].x,tree[n].y);
  tree[r]=tree[n];
  return r;
}
int getlvalue(int n)
{
  if (n==-1)
    return -1;
  switch(tree[n].f)
  {
    case Fentry:
      return n;
    case Ftag:
      return getlvalue(tree[n].x);
    case Ffacteurmat:
      return getlvalue(tree[n].x);
    default:
      return -1;
  }
}
int detag(int n)
{
  while(tree[n].f==Ftag) n=tree[n].x;
  return n;
}
int getlvaluerr(int n)
{
  int ret=getlvalue(n);
  if(ret==-1)
    die(n,"not an lvalue");
  return ret;
}
Gtype strtotype(char *s)
{
  int i;
  for(i=0;i<Gnbtype;i++)
  {
    if (!strcmp(s,Gname[i]))
      return i;
  }
  return -1;
}
Gtype nodetype(err_name e, char *s)
{
  int t;
  if (!*s)
    return Gnotype;
  t=strtotype(s);
  if (t==-1)
    die(e,"unknow type %s or colon ':' used as semi-colon';' ",s);
  return t;
}
Mmode strtomode(char *s)
{
  int i;
  for(i=0;i<Mnbmode;i++)
  {
    if (!strcmp(s,Mname[i]))
      return i;
  }
  return -1;
}
Mmode nodemode(char *s)
{
  int t=strtomode(s);
  if (t==-1)
    die(-1,"unknow mode %s",s);
  return t;
}

int listtoseq(int *stack, int nb)
{
  int n,i;
  if (nb==-1) return -1;
  for(n=stack[0],i=1;i<nb;n=newnode(Flistarg,n,stack[i++]));
  return n;
}

void
printentryname(FILE *fout, const char *s)
{
  if (isdigit(*s))
    fprintf(fout,"p%s",s);
  else
    fprintf(fout,"%s",s);
}

void
printentry(FILE *fout, int x)
{
  const char *s=value[x].val.str;
  printentryname(fout, s);
}

extern int indent;
void printnodeparens(FILE *fout, int n, int parens)
{
  int i,v;
  context *fc;
  int x,y;
  if (n<0)
    return;
  x=tree[n].x;
  y=tree[n].y;
  gencomment(fout,n,!indent);
  if (debug>=3) fprintf(fout,"/*%s:%d:%s*/",Gname[tree[n].t],tree[n].m,funcname(tree[n].f));
  switch(tree[n].f)
  {
  case Fseq:
    printnode(fout,x);
    if (x>=0 && tree[x].f==Fseq) x=tree[x].y;
    if (x>=0 && tree[x].f!=Fblock && tree[x].f!=Fdeffunc)
    {
      if (indent)
        fprintf(fout,";\n");
      else
        fprintf(fout,"; \\\n");
      genindent(fout);
    }
    printnode(fout,y);
    break;
  case Fmatrix:
    fprintf(fout,"[");
    printnode(fout,x);
    if (y>0)
    {
      fprintf(fout,",");
      printnode(fout,y);
    }
    fprintf(fout,"]");
    break;
  case FmatrixL:
    fprintf(fout,"[");
    printnode(fout,x);
    fprintf(fout,",]");
    break;
  case FmatrixR:
    fprintf(fout,"[,");
    printnode(fout,x);
    fprintf(fout,"]");
    break;
  case Ffacteurmat:
    printnodeparens(fout,x,1);
    printnode(fout,y);
    break;
  case Faffect:
    if (parens)
      fputc('(',fout);
    printnode(fout,x);
    fprintf(fout,"=");
    printnode(fout,y);
    if (parens)
      fputc(')',fout);
    break;
  case Fconst:
    switch(value[x].type)
    {
    case CSTsmall:
      fprintf(fout,"%ld",value[x].val.small);
      break;
    case CSTsmallreal:
      fprintf(fout,"%ld.",value[x].val.small);
      break;
    case CSTexpr: 
      fprintf(fout,"%s",value[x].val.str);
      break;
    case CSTstr:
      fprintf(fout,"\"%s\"",value[x].val.str);
      break;
    case CSTquote:
      fprintf(fout,"'%s",value[x].val.str);
      break;
    }
    break;
  case Fsmall:
    fprintf(fout,"%d",x);
    break;
  case Fmatrixelts:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fmatrixlines:
    printnode(fout,x);
    fprintf(fout,";");
    /*Would be nice to indent matrix, alas I have no way to know where is the
     * bottomrow*/
    printnode(fout,y);
    break;
  case Femptyvec:
    fprintf(fout,"[]");
    break;
  case Femptymat:
    fprintf(fout,"[;]");
    break;
  case Fmat:
    fprintf(fout,"[");
    printnode(fout,x);
    fprintf(fout,"]");
    break;
  case Flistarg:
    printnode(fout,x);
    fprintf(fout,",");
    printnode(fout,y);
    break;
  case Fentry:
    printentry(fout,x);
    break;  
  case Fentryfunc:
    if (x<OPnboperator || (value[x].val.str[0]=='_' && value[x].val.str[1]=='.'))
    {
      int arg[1024];
      int nb=genlistargs(n,arg,0,1024);
      int i=0;
      const char *p, *name=value[x].val.str;
      if (parens && x!=OPcat)
	fputc('(',fout);
      for (p=name;*p;p++)
      {
	if (*p=='_')
	{
	  if (i==nb)
	    die(n,"too few arguments for operator %s",name);
	  printnodeparens(fout,arg[i++],1);

	}
	else
	  fputc(*p,fout);
      }
      if (i!=nb)
	die(n,"too many arguments for operator %s",name);
      if (parens && x!=OPcat)
	fputc(')',fout);
    }
    else 
    {
      fprintf(fout,"%s",value[x].val.str);
      fprintf(fout,"(");
      if (y!=GNOARG)
        printnode(fout,y);
      fprintf(fout,")");
    }
    break;
  case Frefarg:
    fprintf(fout,"&%s",value[x].val.str);
    break;
  case Fdeffunc:
    fprintf(fout,"\n");
    printnode(fout,x);
    fprintf(fout,"=\n");
    genindent(fout);
    if (tree[y].f!=Fblock)
    {
      fprintf(fout,"{\n");
      indent++;
      genindent(fout);
    }
    printnode(fout,y);
    fprintf(fout,"\n");
    indent--;
    genindent(fout);
    if (tree[y].f!=Fblock)
    {
      fprintf(fout,"}\n");
      genindent(fout);
    }
    break;
  case Fblock:
    fprintf(fout,"{");
    fc=block+x;
    v=fc->ret;
    if (v>=0)
    {
      fprintf(fout,"/*=");
      printentry(fout,tree[v].x);
      fprintf(fout,"*/");
    }
    indent++;
    fprintf(fout,"\n");
    genindent(fout);
    if (fc->s.n)
    {
      int k=fc->s.n;
      fprintf(fout,"local(");
      for(i=0;i<k;i++)
      {
        if (i>0) fprintf(fout,", ");
        printentryname(fout,fc->c[i].var);
      }
      fprintf(fout,");\n");
      genindent(fout);
    }    
    printnode(fout,y);
    fprintf(fout,"\n");
    indent--;
    genindent(fout);
    fprintf(fout,"}\n");
    genindent(fout);
    break;
  case Fgnil:
    if ( n!= GNOARG )
      fprintf(fout,"gnil");
    break;
  case Ftag:
    printnodeparens(fout,x,1);    
    fprintf(fout,":%s",Gname[y]);
    break;
  }
}
void printnode(FILE *fout, int n)
{
  printnodeparens(fout,n,0);
}

void maketreeGRL_node(FILE *fout,int n)
{
  int x,y,f;
  if (n<2) return;
  x=tree[n].x;
  y=tree[n].y;
  f=tree[n].f;
  switch(tree[n].f)
  {
  case Ftag:
    fprintf(fout,"node: { title: \"%d\" label: \"(%s)\" }\n",n,Gname[y]);
    fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color: red}\n",n,x);
    maketreeGRL_node(fout,x);
    break;
  default:
    if (f<FneedENTRY)
    {
      fprintf(fout,"node: { title: \"%d\" label: \"%s\" }\n",n,Ffuncname[f]);
      if (x!=-1)
      {
        fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color:red}\n",n,x);
	maketreeGRL_node(fout,x);
	if (y!=-1 )
	{
          fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color:blue}\n",n,y);
	  maketreeGRL_node(fout,y);
	}
      }
    }
    else
    {
      fprintf(fout,"node: { title: \"%d\" ",n);
      switch(f)
      {
      case Fsmall:
        fprintf(fout,"label: \"%d\" }\n",x);
        break;
      case Fdeffunc:
        fprintf(fout,"label: \"def(%s)\" }\n",entryname(x));
        break;
      case Fblock:
        fprintf(fout,"label: \"bloc(%d)\" }\n",x);
        break;
      default:
        fprintf(fout,"label: \"%s\" }\n",entryname(n));
      }
      if (tree[n].y!=-1)
      {
        fprintf(fout,"edge: { sourcename: \"%d\" targetname: \"%d\" class:2 color: blue}\n",n,y);
	maketreeGRL_node(fout,y);
      }
    }
  }
}

void maketreeGRL(FILE *fout,int n)
{
  fprintf(fout,"graph: { title:\"test\"\n"); 
  fprintf(fout,"xmax: 700 ymax: 700 x: 30 y: 30\n"); 
  fprintf(fout,"layout_downfactor: 8\n"); 
  fprintf(fout,"node: { title: \"%d\" label: \"GNIL\" }\n",GNIL);
  fprintf(fout,"node: { title: \"%d\" label: \"GNOARG\" }\n",GNOARG);
  maketreeGRL_node(fout,n);
  fprintf(fout,"}\n"); 
  
}

void maketree(FILE *fout,int n)
{
  switch(tree[n].f)
  {
  case Ftag:
    fprintf(fout,"(");
    maketree(fout,tree[n].x);
    fprintf(fout,",");
    fprintf(fout,"%s",Gname[tree[n].y]);
    fprintf(fout,")");
    break;
  default:
    if (tree[n].f<FneedENTRY)
    {
      if (tree[n].x!=-1)
      {
	fprintf(fout,"(");
	maketree(fout,tree[n].x);
	if (tree[n].y!=-1 )
	{
	  fprintf(fout,",");
	  maketree(fout,tree[n].y);
	}
	fprintf(fout,")");
      }
    }
    else
    {
      if (tree[n].y!=-1)
      {
	fprintf(fout,"(");
	maketree(fout,tree[n].y);
	fprintf(fout,")");
      }
    }
  }
  fprintf(fout,"%s_%d",Ffuncname[tree[n].f],n);
}
const char *funcname(int f)
{
  if (f<0 || f>Flastfunc)
    return "Funknown";
  else
    return Ffuncname[f];
}
