/*
 * STAN - Stream Analyser
 * Copyright (c) 2001 Konrad Rieck <kr@roqe.org>
 * The Roqefellaz, http://www.roqe.org/stan
 *
 * $Id: stan.c,v 1.18 2001/04/10 14:58:46 kr Exp $
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <fcntl.h>
#include <unistd.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>

#include <stan.h>
#include <data.h>
#include <stats.h>
#include <bits.h>
#include <config.h>

int errno, verbose = 0, progress = 0, bits = 0;

void print_usage()
{
    printf
	("usage: stan [-bhvV] [-f filter] [-n number] [-p patlen] [file ...]\n"
	 " -b             analyse bits, usefull for measuring prng's.\n"
	 " -h             give this help\n"
	 " -v             be verbose and print progress information.\n"
	 " -V             print version.\n"
	 " -p patlen      pattern length to analyse. (default 2)\n"
	 " -f filter      filter charakters, 0=none, 1=printables, 2=letters.\n"
	 " -n number      number of patterns to print.\n"
	 " file...        files to analyse. If none given, use standard input.\n");
}

void print_version()
{
    printf("STAN - Stream Analyser v%s - Copyright (c) 2001 Konrad Rieck "
	   "<kr@roqe.org>\n", VERSION);
}

void print_error(const char *txt)
{
    perror(txt);
    exit(EXIT_FAILURE);
}

void print_progress()
{
    switch (progress % 4) {
    case 0:
	printf("\b|");
	break;
    case 1:
	printf("\b/");
	break;
    case 2:
	printf("\b-");
	break;
    case 3:
	printf("\b\\");
	break;
    }
    fflush(stdout);
}

int main(int argc, char **argv)
{
    extern char *optarg;
    extern int optind;
    int c, patlen = 0, filter = 0, number = 0;

    while ((c = getopt(argc, argv, "bf:hn:p:vV")) != EOF) {
	switch (c) {
	case 'b':
	    bits = 1;
	    break;
	case 'p':
	    patlen = atoi(optarg);
	    break;
	case 'f':
	    filter = atoi(optarg);
	    break;
	case '?':
	case 'h':
	    print_usage();
	    exit(EXIT_SUCCESS);
	    break;
	case 'n':
	    number = atoi(optarg);
	    break;
	case 'v':
	    verbose = 1;
	    break;
	case 'V':
	    print_version();
	    exit(EXIT_SUCCESS);
	    break;
	}

    }

    if (number < 1)
	number = 10;
    if (patlen < 1)
	patlen = 2;

    if (number < 0 || patlen < 0) {
	print_usage();
	exit(EXIT_SUCCESS);
    }

    init_data(patlen);

    if (optind < argc)
	for (; optind < argc; optind++)
	    read_data(argv[optind], patlen, filter);
    else
	read_data("", patlen, filter);

    print_stats();

    for (c = 1; c <= patlen && c <= size_data(); c++) {
	print_header_patterns(c);
	print_range_patterns(c);
	print_top_patterns(number, c);
	printf("\n");
    }

    if (bits)
	print_bits_stats();

    free_data(patlen);

    return EXIT_SUCCESS;
}
