/* 
 * Copyright (C) 1998,1999 Nikos Mavroyanopoulos 
 *
 * This library is free
 * software; you can redistribute it and/or modify it under the terms of the
 * GNU Library General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* $Id: mcrypt.c,v 1.8 1999/10/20 09:13:41 nmav Exp $ */

#ifndef LIBDEFS_H
#define LIBDEFS_H
#include <libdefs.h>
#endif
#include <bzero.h>
#include <swap.h>

#define MCRYPT_STRUCT_DEFINED
#include <mcrypt_int.h>
#include <blowfish.h>
#include <3-way.h>
#include <tripledes.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast-128.h>
#include <cast-256.h>
#include <tean.h>
#include <twofish.h>
#include <xmemory.h>
#include <rc2.h>
#include <enigma.h>
#include <saferplus.h>
#include <loki97.h>
#include <rijndael.h>
#include <serpent.h>
#include <arcfour.h>

MCRYPT init_mcrypt(const int mode, const int algorithm, void *key,
		   int lenofkey)
{
	MCRYPT thread;

	if (lenofkey == 0)
		return MCRYPT_FAILED;	/* error */

	thread = malloc(sizeof(struct MCRYPT_INT));

	if (mcrypt_get_key_size(algorithm) < lenofkey)
		lenofkey = mcrypt_get_key_size(algorithm);

	thread->algorithm_given = algorithm;
	thread->mode_given = mode;
	thread->keyword_given =
	    mxcalloc(1, mcrypt_get_key_size(algorithm));

	memmove(thread->keyword_given, key, lenofkey);


	switch (algorithm) {

	case MCRYPT_DES:
		thread->akey = mxcalloc(1, sizeof(DES_KEY));
		_mcrypt_des_set_key((void *) thread->akey,
				   (void *) thread->keyword_given, lenofkey);
		return thread;

	case MCRYPT_ENIGMA:
		thread->akey = mxcalloc(1, sizeof(CRYPT_KEY));
		_mcrypt_enigma_set_key((void *) thread->akey,
				   (void *) thread->keyword_given, lenofkey);
		return thread;

	case MCRYPT_SAFER_SK64:
		thread->akey =
		    mxcalloc(1, 1 +
			     SAFER_BLOCK_LEN * (1 +
						2 * SAFER_MAX_NOF_ROUNDS));

		_mcrypt_Safer_Init_Module();
		_mcrypt_Safer_Expand_Userkey((void *)
					     thread->keyword_given,
					     (void *)
					     thread->keyword_given,
					     SAFER_SK64_DEFAULT_NOF_ROUNDS,
					     1, thread->akey);
		/* 8 rounds and secure key schedule */
		return thread;

	case MCRYPT_CAST_128:
		thread->akey = mxcalloc(1, sizeof(struct cast_key));
		_mcrypt_cast_setkey((void *) thread->akey,
				    (void *) thread->keyword_given,
				    lenofkey);
		return thread;

	case MCRYPT_LOKI97:
		thread->akey = mxcalloc(1, 96 * sizeof(word32));
		_mcrypt_loki97_set_key((void *) thread->akey,
				       (void *) thread->keyword_given,
				       lenofkey);
		return thread;

	case MCRYPT_CAST_256:
		thread->akey = mxcalloc(1, 96 * sizeof(word32));
		_mcrypt_cast256_set_key((void *) thread->akey,
					(void *) thread->keyword_given,
					lenofkey);
		return thread;


	case MCRYPT_RIJNDAEL_128:
		thread->akey = mxcalloc(1, sizeof(RI));
		_mcrypt_rijndael_gentables();
		_mcrypt_rijndael_set_key((void *) thread->akey, 4,
					 lenofkey / 4,
					 (void *) thread->keyword_given);
		return thread;

	case MCRYPT_SERPENT:
		thread->akey = mxcalloc(1, sizeof(SERPENT_KEY));
		_mcrypt_serpent_set_key((void *) thread->akey,
					(void *) thread->keyword_given,
					lenofkey);
		return thread;

	case MCRYPT_RC2:
		thread->akey = mxcalloc(1, 64 * sizeof(word16));
		_mcrypt_rc2_keyschedule((void *) thread->akey,
					(void *) thread->keyword_given,
					lenofkey);
		return thread;

	case MCRYPT_ARCFOUR:
		thread->akey = mxcalloc(1, sizeof(arcfour_key));
		_mcrypt_arcfour_prepare_key((void *) thread->keyword_given,
					    lenofkey,
					    (void *) thread->akey);
		return thread;

	case MCRYPT_SAFER_SK128:
		_mcrypt_Safer_Init_Module();
		thread->akey =
		    mxcalloc(1, 1 +
			     SAFER_BLOCK_LEN * (1 +
						2 * SAFER_MAX_NOF_ROUNDS));

		_mcrypt_Safer_Expand_Userkey((void *)
					     thread->keyword_given,
					     (void *)
					     &(thread->keyword_given)[8],
					     SAFER_SK128_DEFAULT_NOF_ROUNDS,
					     1, thread->akey);

		/* 8 rounds and secure key schedule */

		return thread;

	case MCRYPT_TRIPLEDES:
		thread->akey = mxcalloc(1, sizeof(TRIPLEDES_KEY));
		_mcrypt_3des_set_key((void *) thread->akey,
				   (void *) thread->keyword_given, mcrypt_get_key_size(algorithm));

		return thread;

	case MCRYPT_SAFERPLUS:
		thread->akey = mxcalloc(1, sizeof(SPI));
		_mcrypt_saferplus_set_key((void *) thread->akey,
					  (void *) thread->keyword_given,
					  lenofkey);
		return thread;

	case MCRYPT_BLOWFISH:
		thread->akey = mxcalloc(1, sizeof(blf_ctx));
		if (lenofkey < 2) {
			_mcrypt_blf_key((void *) thread->akey,
					(void *) thread->keyword_given, 2);
		} else {
			_mcrypt_blf_key((void *) thread->akey,
					(void *) thread->keyword_given,
					lenofkey);
		}
		return thread;

	case MCRYPT_TWOFISH:
		thread->akey = mxcalloc(1, sizeof(TWI));
		_mcrypt_twofish_set_key((void *) thread->akey,
					(void *) thread->keyword_given,
					lenofkey);
		return thread;

	case MCRYPT_GOST:
		thread->akey = mxcalloc(1, 8*sizeof(word32));
                _mcrypt_gost_set_key((void *) thread->akey, (void *) thread->keyword_given, lenofkey);
                                                   
		return thread;

	case MCRYPT_THREEWAY:
	case MCRYPT_XTEA:
		return thread;

	default:
		return MCRYPT_FAILED;
	}

}


/* plaintext should be in block's size */
int mcrypt(MCRYPT thread, void *plaintext)
{

	switch (thread->algorithm_given) {

	case MCRYPT_SAFER_SK64:
	case MCRYPT_SAFER_SK128:
		_mcrypt_Safer_Encrypt_Block(plaintext, thread->akey);
		return 0;

	case MCRYPT_SAFERPLUS:
		_mcrypt_saferplus_encrypt((void *) thread->akey,
					  plaintext);
		return 0;

	case MCRYPT_CAST_128:
		_mcrypt_cast_encrypt((void *) thread->akey,
				     (void *) plaintext);
		return 0;

	case MCRYPT_LOKI97:
		_mcrypt_loki97_encrypt((void *) thread->akey,
				       (void *) plaintext);
		return 0;

	case MCRYPT_CAST_256:
		_mcrypt_cast256_encrypt((void *) thread->akey,
					(void *) plaintext);
		return 0;

	case MCRYPT_RC2:
		_mcrypt_rc2_encrypt((void *) thread->akey,
				    (void *) plaintext);
		return 0;

	case MCRYPT_RIJNDAEL_128:
		_mcrypt_rijndael_encrypt((void *) thread->akey,
					 (void *) plaintext);
		return 0;

	case MCRYPT_SERPENT:
		_mcrypt_serpent_encrypt((void *) thread->akey,
					(void *) plaintext);
		return 0;

	case MCRYPT_DES:
		_mcrypt_des_encrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_TRIPLEDES:
		_mcrypt_3des_encrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_BLOWFISH:
		_mcrypt_enblf((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_TWOFISH:
		_mcrypt_twofish_encrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_GOST:
		_mcrypt_gost_encrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_XTEA:
		_mcrypt_cl_enc_block((void *) thread->keyword_given,
				     plaintext);
		return 0;

	case MCRYPT_THREEWAY:
		_mcrypt_en3way(plaintext, (void *) thread->keyword_given);
		return 0;

	default:
		return 0;
	}

}


/* plaintext should be in block's size */
int mdecrypt(MCRYPT thread, void *plaintext)
{
	switch (thread->algorithm_given) {

	case MCRYPT_SAFER_SK64:
	case MCRYPT_SAFER_SK128:
		_mcrypt_Safer_Decrypt_Block(plaintext,
					    (void *) thread->akey);
		return 0;

	case MCRYPT_SAFERPLUS:
		_mcrypt_saferplus_decrypt((void *) thread->akey,
					  plaintext);
		return 0;

	case MCRYPT_CAST_128:
		_mcrypt_cast_decrypt((void *) thread->akey,
				     (void *) plaintext);
		return 0;

	case MCRYPT_LOKI97:
		_mcrypt_loki97_decrypt((void *) thread->akey,
				       (void *) plaintext);
		return 0;

	case MCRYPT_CAST_256:
		_mcrypt_cast256_decrypt((void *) thread->akey,
					(void *) plaintext);
		return 0;

	case MCRYPT_RC2:
		_mcrypt_rc2_decrypt((void *) thread->akey,
				    (void *) plaintext);
		return 0;

	case MCRYPT_RIJNDAEL_128:
		_mcrypt_rijndael_decrypt((void *) thread->akey,
					 (void *) plaintext);
		return 0;

	case MCRYPT_SERPENT:
		_mcrypt_serpent_decrypt((void *) thread->akey,
					(void *) plaintext);
		return 0;

	case MCRYPT_DES:
		_mcrypt_des_decrypt( (void*) thread->akey, plaintext);
		return 0;

	case MCRYPT_XTEA:
		_mcrypt_cl_dec_block((void *) thread->keyword_given,
				     plaintext);
		return 0;

	case MCRYPT_TRIPLEDES:
		_mcrypt_3des_decrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_TWOFISH:
		_mcrypt_twofish_decrypt((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_BLOWFISH:
		_mcrypt_deblf((void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_GOST:
		_mcrypt_gost_decrypt( (void *) thread->akey, plaintext);
		return 0;

	case MCRYPT_THREEWAY:
		_mcrypt_de3way(plaintext, (void *) thread->keyword_given);
		return 0;

	default:
		return 0;
	}
}


int end_mcrypt(MCRYPT td)
{

	if (td==MCRYPT_FAILED) return -1;
	
	mxfree(td->keyword_given,
	       mcrypt_get_key_size(td->algorithm_given));

	switch (td->algorithm_given) {

	case MCRYPT_RIJNDAEL_128:
		mxfree(td->akey, sizeof(RI));
		break;

	case MCRYPT_SERPENT:
		mxfree(td->akey, sizeof(SERPENT_KEY));
		break;


	case MCRYPT_DES:
		mxfree(td->akey, sizeof(DES_KEY));
		break;

	case MCRYPT_TRIPLEDES:
		mxfree(td->akey, sizeof(TRIPLEDES_KEY));
		break;

	case MCRYPT_RC2:
		mxfree(td->akey, 64 * sizeof(word16));
		break;

	case MCRYPT_ARCFOUR:
		mxfree(td->akey, sizeof(arcfour_key));
		break;

	case MCRYPT_ENIGMA:
		mxfree(td->akey, sizeof(CRYPT_KEY));
		break;

	case MCRYPT_TWOFISH:
		mxfree(td->akey, sizeof(TWI));
		break;

	case MCRYPT_BLOWFISH:
		mxfree(td->akey, sizeof(blf_ctx));
		break;

	case MCRYPT_GOST:
		mxfree(td->akey, 8*sizeof(word32));
		break;
		
	case MCRYPT_XTEA:
	case MCRYPT_THREEWAY:
		break;

	case MCRYPT_CAST_128:
		mxfree(td->akey, sizeof(struct cast_key));
		break;

	case MCRYPT_LOKI97:
		mxfree(td->akey, 96 * sizeof(word32));
		break;

	case MCRYPT_CAST_256:
		mxfree(td->akey, 96 * sizeof(word32));
		break;

	case MCRYPT_SAFER_SK64:
	case MCRYPT_SAFER_SK128:
		mxfree(td->akey,
		       1 + SAFER_BLOCK_LEN * (1 +
					      2 * SAFER_MAX_NOF_ROUNDS));
		break;

	default:
		return 0;
	}

	free(td);
	return 0;
}

/* ECB MODE */

MCRYPT init_mcrypt_ecb(const int algorithm, void *key, int lenofkey)
{
	MCRYPT x;

	x = init_mcrypt(MCRYPT_ECB, algorithm, key, lenofkey);

	return x;

}




int mcrypt_ecb(MCRYPT td, void *plaintext, int len)
{
	int j, blocksize;
	char *plain = plaintext;

	blocksize = mcrypt_get_block_size(td->algorithm_given);
	for (j = 0; j < len / blocksize; j++) {

		mcrypt(td, &plain[j * blocksize]);

	}
	return 0;
}



int mdecrypt_ecb(MCRYPT td, void *ciphertext, int len)
{
	int j, blocksize = mcrypt_get_block_size(td->algorithm_given);
	char *cipher = ciphertext;

	for (j = 0; j < len / blocksize; j++) {

		mdecrypt(td, &cipher[j * blocksize]);

	}

	return 0;
}



/* STREAM MODE */
MCRYPT init_mcrypt_stream(const int algorithm, void *key, int lenofkey)
{
	MCRYPT x;

	x = init_mcrypt(MCRYPT_STREAM, algorithm, key, lenofkey);

	return x;

}




int mcrypt_stream(MCRYPT td, void *plaintext, int len)
{
	switch (td->algorithm_given) {


	case MCRYPT_ENIGMA:
		_mcrypt_enigma_encrypt((void *) td->akey, plaintext, len);
		return 0;

	case MCRYPT_ARCFOUR:
		_mcrypt_arcfour((void *) plaintext, len,
				(void *) td->akey);
		return 0;
	default:
		return -1;
	}

}



int mdecrypt_stream(MCRYPT td, void *ciphertext, int len)
{
	switch (td->algorithm_given) {


	case MCRYPT_ENIGMA:
		_mcrypt_enigma_decrypt((void *) td->akey, ciphertext, len);
		return 0;

	case MCRYPT_ARCFOUR:
		_mcrypt_arcfour((void *) ciphertext, len,
				(void *) td->akey);
		return 0;
	default:
		return -1;
	}
}




/* CBC MODE */

int end_mcrypt_cbc(MCRYPT td)
{

	mxfree(td->previous_ciphertext,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->previous_cipher,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->previous_plaintext,
	       mcrypt_get_block_size(td->algorithm_given));
	return end_mcrypt(td);

}


MCRYPT init_mcrypt_cbc(const int algorithm, void *key, int lenofkey)
{

	MCRYPT x;

	x = init_mcrypt(MCRYPT_CBC, algorithm, key, lenofkey);

/* For cbc */
	if (x != MCRYPT_FAILED) {
		x->previous_ciphertext =
		    mxcalloc(1, mcrypt_get_block_size(algorithm));
		x->previous_cipher =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->previous_plaintext =
		    mxcalloc(1, mcrypt_get_block_size(algorithm));
	}
/* End cbc */

	return x;

}

MCRYPT init_mcrypt_cbc_iv(const int algorithm, void *key, int lenofkey,
			  void *IV)
{

	MCRYPT x;

	x = init_mcrypt(MCRYPT_CBC, algorithm, key, lenofkey);

	if (x != MCRYPT_FAILED) {
		x->previous_ciphertext =
		    mxcalloc(1, mcrypt_get_block_size(algorithm));
		x->previous_cipher =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->previous_plaintext =
		    mxcalloc(1, mcrypt_get_block_size(algorithm));

		memmove(x->previous_ciphertext, IV,
			mcrypt_get_block_size(algorithm));
		memmove(x->previous_plaintext, IV,
			mcrypt_get_block_size(algorithm));
/* End cbc */
	}

	return x;

}


int mcrypt_cbc(MCRYPT td, void *plaintext, int len)
{
	word32 *fplain = plaintext;
	word32 *plain;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);

	for (j = 0; j < len / blocksize; j++) {

		plain = &fplain[j * blocksize / sizeof(word32)];

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^= td->previous_ciphertext[i];
		}

		mcrypt(td, plain);


		/* Copy the ciphertext to prev_ciphertext */
		memmove(td->previous_ciphertext, plain, blocksize);
	}

	return 0;
}



int mdecrypt_cbc(MCRYPT td, void *ciphertext, int len)
{
	word32 *cipher;
	word32 *fcipher = ciphertext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);

	for (j = 0; j < len / blocksize; j++) {

		cipher = &fcipher[j * blocksize / sizeof(word32)];
		memmove(td->previous_cipher, cipher, blocksize);

		mdecrypt(td, cipher);
		for (i = 0; i < blocksize / sizeof(word32); i++) {
			cipher[i] ^= td->previous_plaintext[i];
		}

		/* Copy the ciphertext to prev_cipher */
		memmove(td->previous_plaintext, td->previous_cipher,
			blocksize);

	}

	return 0;
}



/* CFB MODE */

int end_mcrypt_cfb(MCRYPT td)
{

	mxfree(td->s_register, mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_s_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_sd_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->sd_register,
	       mcrypt_get_block_size(td->algorithm_given));

	return end_mcrypt(td);

}


MCRYPT init_mcrypt_cfb(const int algorithm, void *key, int lenofkey,
		       void *IV)
{
	MCRYPT x;

	x = init_mcrypt(MCRYPT_CFB, algorithm, key, lenofkey);

	if (x != MCRYPT_FAILED) {
/* For cfb */
		x->s_register = mxmalloc(mcrypt_get_block_size(algorithm));
		x->sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_s_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		memmove(x->sd_register, IV,
			mcrypt_get_block_size(algorithm));
		memmove(x->s_register, IV,
			mcrypt_get_block_size(algorithm));

/* End cfb */
	}

	return x;
}

int mcrypt_cfb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);;

	for (j = 0; j < len; j++) {

		memmove(td->enc_s_register, td->s_register, blocksize);

		mcrypt(td, td->enc_s_register);

		plain[j] ^= td->enc_s_register[0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			td->s_register[i] = td->s_register[i + 1];

		td->s_register[blocksize - 1] = plain[j];


	}

	return 0;

}


int mdecrypt_cfb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);;

	for (j = 0; j < len; j++) {

		memmove(td->enc_sd_register, td->sd_register, blocksize);

		mcrypt(td, td->enc_sd_register);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			td->sd_register[i] = td->sd_register[i + 1];

		td->sd_register[blocksize - 1] = plain[j];

		plain[j] ^= td->enc_sd_register[0];

	}

	return 0;
}


/* OFB MODE */

int end_mcrypt_ofb(MCRYPT td)
{

	mxfree(td->s_register, mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_s_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_sd_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->sd_register,
	       mcrypt_get_block_size(td->algorithm_given));

	return end_mcrypt(td);

}


MCRYPT init_mcrypt_ofb(const int algorithm, void *key, int lenofkey,
		       void *IV)
{
	MCRYPT x;

	x = init_mcrypt(MCRYPT_OFB, algorithm, key, lenofkey);


	if (x != MCRYPT_FAILED) {
/* For ofb */
		x->s_register = mxmalloc(mcrypt_get_block_size(algorithm));
		x->sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_s_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		memmove(x->sd_register, IV,
			mcrypt_get_block_size(algorithm));
		memmove(x->s_register, IV,
			mcrypt_get_block_size(algorithm));
/* End ofb */
	}


	return x;
}

int mcrypt_ofb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);;

	for (j = 0; j < len; j++) {

		memmove(td->enc_s_register, td->s_register, blocksize);

		mcrypt(td, td->enc_s_register);

		plain[j] ^= td->enc_s_register[0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			td->s_register[i] = td->s_register[i + 1];

		td->s_register[blocksize - 1] = td->enc_s_register[0];


	}

	return 0;

}


int mdecrypt_ofb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit ofb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);;

	for (j = 0; j < len; j++) {

		memmove(td->enc_sd_register, td->sd_register, blocksize);

		mcrypt(td, td->enc_sd_register);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			td->sd_register[i] = td->sd_register[i + 1];

		td->sd_register[blocksize - 1] = td->enc_sd_register[0];

		plain[j] ^= td->enc_sd_register[0];

	}

	return 0;
}



/* n-bit ofb */

int end_mcrypt_nofb(MCRYPT td)
{

	mxfree(td->s_register, mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_s_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->enc_sd_register,
	       mcrypt_get_block_size(td->algorithm_given));
	mxfree(td->sd_register,
	       mcrypt_get_block_size(td->algorithm_given));

	return end_mcrypt(td);

}


MCRYPT init_mcrypt_nofb(const int algorithm, void *key, int lenofkey,
			void *IV)
{
	MCRYPT x;

	x = init_mcrypt(MCRYPT_nOFB, algorithm, key, lenofkey);

	if (x != MCRYPT_FAILED) {
/* For nofb */
		x->s_register = mxmalloc(mcrypt_get_block_size(algorithm));
		x->sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_s_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		x->enc_sd_register =
		    mxmalloc(mcrypt_get_block_size(algorithm));
		memmove(x->sd_register, IV,
			mcrypt_get_block_size(algorithm));
		memmove(x->s_register, IV,
			mcrypt_get_block_size(algorithm));
/* End nofb */
	}

	return x;
}

int mcrypt_nofb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is n*blocksize bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(td->enc_s_register, td->s_register, blocksize);
		mcrypt(td, td->enc_s_register);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^=
			    ((word32 *) &
			     td->enc_s_register[0 +
						i * sizeof(word32)])[0];
		}
/* Put the new register */
		memmove(td->s_register, td->enc_s_register, blocksize);
	}
	return 0;
}


int mdecrypt_nofb(MCRYPT td, void *plaintext, int len)
{				/* plaintext is n bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(td->algorithm_given);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(td->enc_sd_register, td->sd_register, blocksize);
		mcrypt(td, td->enc_sd_register);

/* Shift the register */
		memmove(td->sd_register, td->enc_sd_register, blocksize);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^=
			    ((word32 *) &
			     td->enc_sd_register[0 +
						 i * sizeof(word32)])[0];
		}
	}

	return 0;
}



/* Generic -High level functions */

MCRYPT mcrypt_generic_init(const int mode, const int algorithm, void *key,
			   int lenofkey, void *IV)
{
	if (mcrypt_is_block_algorithm(algorithm) != 0) {
		switch (mode) {
		case MCRYPT_CBC:
			if (IV == NULL) {
				return init_mcrypt_cbc(algorithm, key,
						       lenofkey);
			} else {
				return init_mcrypt_cbc_iv(algorithm, key,
							  lenofkey, IV);
			}

		case MCRYPT_ECB:
			return init_mcrypt_ecb(algorithm, key, lenofkey);
		case MCRYPT_CFB:
			return init_mcrypt_cfb(algorithm, key, lenofkey,
					       IV);
		case MCRYPT_OFB:
			return init_mcrypt_ofb(algorithm, key, lenofkey,
					       IV);
		case MCRYPT_nOFB:
			return init_mcrypt_nofb(algorithm, key, lenofkey,
						IV);
		default:
			return MCRYPT_FAILED;
		}
	} else {
		if (mode == MCRYPT_STREAM) {
			return init_mcrypt_stream(algorithm, key,
						  lenofkey);
		} else {
			return MCRYPT_FAILED;
		}
	}


}

int mcrypt_generic(MCRYPT td, void *plaintext, int len)
{

	switch (td->mode_given) {
	case MCRYPT_CBC:
		return mcrypt_cbc(td, plaintext, len);
	case MCRYPT_ECB:
		return mcrypt_ecb(td, plaintext, len);
	case MCRYPT_CFB:
		return mcrypt_cfb(td, plaintext, len);
	case MCRYPT_OFB:
		return mcrypt_ofb(td, plaintext, len);
	case MCRYPT_nOFB:
		return mcrypt_nofb(td, plaintext, len);
	case MCRYPT_STREAM:
		return mcrypt_stream(td, plaintext, len);

	default:
		return -1;
	}

}

int mdecrypt_generic(MCRYPT td, void *ciphertext, int len)
{

	switch (td->mode_given) {
	case MCRYPT_CBC:
		return mdecrypt_cbc(td, ciphertext, len);
	case MCRYPT_ECB:
		return mdecrypt_ecb(td, ciphertext, len);
	case MCRYPT_CFB:
		return mdecrypt_cfb(td, ciphertext, len);
	case MCRYPT_OFB:
		return mdecrypt_ofb(td, ciphertext, len);
	case MCRYPT_nOFB:
		return mdecrypt_nofb(td, ciphertext, len);
	case MCRYPT_STREAM:
		return mdecrypt_stream(td, ciphertext, len);

	default:
		return -1;
	}

}

int mcrypt_generic_end(const MCRYPT td)
{

	switch (td->mode_given) {
	case MCRYPT_CBC:
		return end_mcrypt_cbc(td);
	case MCRYPT_ECB:
		return end_mcrypt_ecb(td);
	case MCRYPT_CFB:
		return end_mcrypt_cfb(td);
	case MCRYPT_OFB:
		return end_mcrypt_ofb(td);
	case MCRYPT_nOFB:
		return end_mcrypt_nofb(td);
	case MCRYPT_STREAM:
		return end_mcrypt_stream(td);
	default:
		return -1;
	}
}
