/* Twofish for GPG
 * By Matthew Skala <mskala@ansuz.sooke.bc.ca>, July 26, 1998
 * Minor changes to fit in mcrypt by Nikos Mavroyanopoulos <nmav@hellug.gr>
 *
 * This code is a "clean room" implementation, written from the paper
 * _Twofish: A 128-Bit Block Cipher_ by Bruce Schneier, John Kelsey,
 * Doug Whiting, David Wagner, Chris Hall, and Niels Ferguson, available
 * through http://www.counterpane.com/twofish.html
 *
 * For background information on multiplication in finite fields, used for
 * the matrix operations in the key schedule, see the book _Contemporary
 * Abstract Algebra_ by Joseph A. Gallian, especially chapter 22 in the
 * Third Edition.
 *
 * Only the 128-bit block size is supported at present.  This code is intended
 * for GNU C on a 32-bit system, but it should work almost anywhere.  Loops
 * are unrolled, precomputation tables are used, etc., for maximum speed at
 * some cost in memory consumption. */

#ifndef LIBDEFS_H
# define LIBDEFS_H
# include <libdefs.h>
#endif
#include <twofish.h>
#include <xmemory.h>

/* Prototype for the self-test function. */
static const char *selftest(void);

/* Macros used by the info function. */
#define FNCCAST_SETKEY(f)  ((int(*)(void*, word8*, unsigned))(f))
#define FNCCAST_CRYPT(f)   ((void(*)(void*, word8*, word8*))(f))

/* Tables moved in twofish.h */

/* These two tables are the q0 and q1 permutations, exactly as described in
 * the Twofish paper. */


/* Macro to perform one column of the RS matrix multiplication.  The
 * parameters a, b, c, and d are the four bytes of output; i is the index
 * of the key bytes, and w, x, y, and z, are the column of constants from
 * the RS matrix, preprocessed through the poly_to_exp table. */

#define CALC_S(a, b, c, d, i, w, x, y, z) \
   if (key[i]) { \
      tmp = poly_to_exp[key[i] - 1]; \
      (a) ^= exp_to_poly[tmp + (w)]; \
      (b) ^= exp_to_poly[tmp + (x)]; \
      (c) ^= exp_to_poly[tmp + (y)]; \
      (d) ^= exp_to_poly[tmp + (z)]; \
   }

/* Macros to calculate the key-dependent S-boxes using the S vector from
 * CALC_S.  CALC_SB_2 computes a single entry in all four S-boxes, where i
 * is the index of the entry to compute, and a and b are the index numbers
 * preprocessed through the q0 and q1 tables respectively.  CALC_SB is
 * simply a convenience to make the code shorter; it calls CALC_SB_2 four
 * times with consecutive indices from i to i+3, using the remaining
 * parameters two by two. */

#define CALC_SB_2(i, a, b) \
   ctx->s[0][i] = mds[0][q0[(a) ^ sa] ^ se]; \
   ctx->s[1][i] = mds[1][q0[(b) ^ sb] ^ sf]; \
   ctx->s[2][i] = mds[2][q1[(a) ^ sc] ^ sg]; \
   ctx->s[3][i] = mds[3][q1[(b) ^ sd] ^ sh]

#define CALC_SB(i, a, b, c, d, e, f, g, h) \
   CALC_SB_2 (i, a, b); CALC_SB_2 ((i)+1, c, d); \
   CALC_SB_2 ((i)+2, e, f); CALC_SB_2 ((i)+3, g, h)

/* Macros to calculate the whitening and round subkeys.  CALC_K_2 computes the
 * h() function for a given index (either 2i or 2i+1).	a and b are the index
 * preprocessed through q0 and q1 respectively; j is the index of the first
 * key byte to use.  CALC_K computes a pair of subkeys by calling CALC_K_2
 * twice, doing the Psuedo-Hadamard Transform, and doing the necessary
 * rotations.  Its parameters are: a, the array to write the results into,
 * j, the index of the first output entry, k and l, the preprocessed indices
 * for index 2i, and m and n, the preprocessed indices for index 2i+1. */

#define CALC_K_2(a, b, j) \
     mds[0][q0[a ^ key[(j) + 8]] ^ key[j]] \
   ^ mds[1][q0[b ^ key[(j) + 9]] ^ key[(j) + 1]] \
   ^ mds[2][q1[a ^ key[(j) + 10]] ^ key[(j) + 2]] \
   ^ mds[3][q1[b ^ key[(j) + 11]] ^ key[(j) + 3]]

#define CALC_K(a, j, k, l, m, n) \
   x = CALC_K_2 (k, l, 0); \
   y = CALC_K_2 (m, n, 4); \
   y = (y << 8) + (y >> 24); \
   x += y; y += x; ctx->a[j] = x; \
   ctx->a[(j) + 1] = (y << 9) + ( y >> 23)

/* Perform the key setup.  Note that this works *only* with 128-bit keys,
 * despite the API that makes it look like it might support other sizes. */

int
twofish_setkey (TWOFISH_context *ctx, const word8 *key, const unsigned keylen)
{
   /* Temporaries for CALC_K. */
   word32 x, y;

   /* The S vector used to key the S-boxes, split up into individual
    * bytes. */
   word8 sa = 0, sb = 0, sc = 0, sd = 0, se = 0, sf = 0, sg = 0, sh = 0;

   /* Temporary for CALC_S. */
   word8 tmp;

   /* Flags for self-test. */
   static int initialized = 0;
   static const char *selftest_failed=0;

   /* Check key length. */
   if( keylen != 16 )
       return -1;

   /* Do self-test if necessary. */
   if (!initialized) {
      initialized = 1;
      selftest_failed = selftest ();
      if( selftest_failed )
	fprintf(stderr, "%s\n", selftest_failed );
   }
   if( selftest_failed )
      return -1;

   /* Compute the S vector.  The magic numbers are the entries of the RS
    * matrix, preprocessed through poly_to_exp.  The numbers in the comments
    * are the original (polynomial form) matrix entries. */
   CALC_S (sa, sb, sc, sd, 0, 0x00, 0x2D, 0x01, 0x2D); /* 01 A4 02 A4 */
   CALC_S (sa, sb, sc, sd, 1, 0x2D, 0xA4, 0x44, 0x8A); /* A4 56 A1 55 */
   CALC_S (sa, sb, sc, sd, 2, 0x8A, 0xD5, 0xBF, 0xD1); /* 55 82 FC 87 */
   CALC_S (sa, sb, sc, sd, 3, 0xD1, 0x7F, 0x3D, 0x99); /* 87 F3 C1 5A */
   CALC_S (sa, sb, sc, sd, 4, 0x99, 0x46, 0x66, 0x96); /* 5A 1E 47 58 */
   CALC_S (sa, sb, sc, sd, 5, 0x96, 0x3C, 0x5B, 0xED); /* 58 C6 AE DB */
   CALC_S (sa, sb, sc, sd, 6, 0xED, 0x37, 0x4F, 0xE0); /* DB 68 3D 9E */
   CALC_S (sa, sb, sc, sd, 7, 0xE0, 0xD0, 0x8C, 0x17); /* 9E E5 19 03 */
   CALC_S (se, sf, sg, sh, 8, 0x00, 0x2D, 0x01, 0x2D); /* 01 A4 02 A4 */
   CALC_S (se, sf, sg, sh, 9, 0x2D, 0xA4, 0x44, 0x8A); /* A4 56 A1 55 */
   CALC_S (se, sf, sg, sh, 10, 0x8A, 0xD5, 0xBF, 0xD1); /* 55 82 FC 87 */
   CALC_S (se, sf, sg, sh, 11, 0xD1, 0x7F, 0x3D, 0x99); /* 87 F3 C1 5A */
   CALC_S (se, sf, sg, sh, 12, 0x99, 0x46, 0x66, 0x96); /* 5A 1E 47 58 */
   CALC_S (se, sf, sg, sh, 13, 0x96, 0x3C, 0x5B, 0xED); /* 58 C6 AE DB */
   CALC_S (se, sf, sg, sh, 14, 0xED, 0x37, 0x4F, 0xE0); /* DB 68 3D 9E */
   CALC_S (se, sf, sg, sh, 15, 0xE0, 0xD0, 0x8C, 0x17); /* 9E E5 19 03 */

   /* Compute the S-boxes.  The constants are indices of
    * S-box entries, preprocessed through q0 and q1. */
   CALC_SB (0, 0xA9, 0x75, 0x67, 0xF3, 0xB3, 0xC6, 0xE8, 0xF4);
   CALC_SB (4, 0x04, 0xDB, 0xFD, 0x7B, 0xA3, 0xFB, 0x76, 0xC8);
   CALC_SB (8, 0x9A, 0x4A, 0x92, 0xD3, 0x80, 0xE6, 0x78, 0x6B);
   CALC_SB (12, 0xE4, 0x45, 0xDD, 0x7D, 0xD1, 0xE8, 0x38, 0x4B);
   CALC_SB (16, 0x0D, 0xD6, 0xC6, 0x32, 0x35, 0xD8, 0x98, 0xFD);
   CALC_SB (20, 0x18, 0x37, 0xF7, 0x71, 0xEC, 0xF1, 0x6C, 0xE1);
   CALC_SB (24, 0x43, 0x30, 0x75, 0x0F, 0x37, 0xF8, 0x26, 0x1B);
   CALC_SB (28, 0xFA, 0x87, 0x13, 0xFA, 0x94, 0x06, 0x48, 0x3F);
   CALC_SB (32, 0xF2, 0x5E, 0xD0, 0xBA, 0x8B, 0xAE, 0x30, 0x5B);
   CALC_SB (36, 0x84, 0x8A, 0x54, 0x00, 0xDF, 0xBC, 0x23, 0x9D);
   CALC_SB (40, 0x19, 0x6D, 0x5B, 0xC1, 0x3D, 0xB1, 0x59, 0x0E);
   CALC_SB (44, 0xF3, 0x80, 0xAE, 0x5D, 0xA2, 0xD2, 0x82, 0xD5);
   CALC_SB (48, 0x63, 0xA0, 0x01, 0x84, 0x83, 0x07, 0x2E, 0x14);
   CALC_SB (52, 0xD9, 0xB5, 0x51, 0x90, 0x9B, 0x2C, 0x7C, 0xA3);
   CALC_SB (56, 0xA6, 0xB2, 0xEB, 0x73, 0xA5, 0x4C, 0xBE, 0x54);
   CALC_SB (60, 0x16, 0x92, 0x0C, 0x74, 0xE3, 0x36, 0x61, 0x51);
   CALC_SB (64, 0xC0, 0x38, 0x8C, 0xB0, 0x3A, 0xBD, 0xF5, 0x5A);
   CALC_SB (68, 0x73, 0xFC, 0x2C, 0x60, 0x25, 0x62, 0x0B, 0x96);
   CALC_SB (72, 0xBB, 0x6C, 0x4E, 0x42, 0x89, 0xF7, 0x6B, 0x10);
   CALC_SB (76, 0x53, 0x7C, 0x6A, 0x28, 0xB4, 0x27, 0xF1, 0x8C);
   CALC_SB (80, 0xE1, 0x13, 0xE6, 0x95, 0xBD, 0x9C, 0x45, 0xC7);
   CALC_SB (84, 0xE2, 0x24, 0xF4, 0x46, 0xB6, 0x3B, 0x66, 0x70);
   CALC_SB (88, 0xCC, 0xCA, 0x95, 0xE3, 0x03, 0x85, 0x56, 0xCB);
   CALC_SB (92, 0xD4, 0x11, 0x1C, 0xD0, 0x1E, 0x93, 0xD7, 0xB8);
   CALC_SB (96, 0xFB, 0xA6, 0xC3, 0x83, 0x8E, 0x20, 0xB5, 0xFF);
   CALC_SB (100, 0xE9, 0x9F, 0xCF, 0x77, 0xBF, 0xC3, 0xBA, 0xCC);
   CALC_SB (104, 0xEA, 0x03, 0x77, 0x6F, 0x39, 0x08, 0xAF, 0xBF);
   CALC_SB (108, 0x33, 0x40, 0xC9, 0xE7, 0x62, 0x2B, 0x71, 0xE2);
   CALC_SB (112, 0x81, 0x79, 0x79, 0x0C, 0x09, 0xAA, 0xAD, 0x82);
   CALC_SB (116, 0x24, 0x41, 0xCD, 0x3A, 0xF9, 0xEA, 0xD8, 0xB9);
   CALC_SB (120, 0xE5, 0xE4, 0xC5, 0x9A, 0xB9, 0xA4, 0x4D, 0x97);
   CALC_SB (124, 0x44, 0x7E, 0x08, 0xDA, 0x86, 0x7A, 0xE7, 0x17);
   CALC_SB (128, 0xA1, 0x66, 0x1D, 0x94, 0xAA, 0xA1, 0xED, 0x1D);
   CALC_SB (132, 0x06, 0x3D, 0x70, 0xF0, 0xB2, 0xDE, 0xD2, 0xB3);
   CALC_SB (136, 0x41, 0x0B, 0x7B, 0x72, 0xA0, 0xA7, 0x11, 0x1C);
   CALC_SB (140, 0x31, 0xEF, 0xC2, 0xD1, 0x27, 0x53, 0x90, 0x3E);
   CALC_SB (144, 0x20, 0x8F, 0xF6, 0x33, 0x60, 0x26, 0xFF, 0x5F);
   CALC_SB (148, 0x96, 0xEC, 0x5C, 0x76, 0xB1, 0x2A, 0xAB, 0x49);
   CALC_SB (152, 0x9E, 0x81, 0x9C, 0x88, 0x52, 0xEE, 0x1B, 0x21);
   CALC_SB (156, 0x5F, 0xC4, 0x93, 0x1A, 0x0A, 0xEB, 0xEF, 0xD9);
   CALC_SB (160, 0x91, 0xC5, 0x85, 0x39, 0x49, 0x99, 0xEE, 0xCD);
   CALC_SB (164, 0x2D, 0xAD, 0x4F, 0x31, 0x8F, 0x8B, 0x3B, 0x01);
   CALC_SB (168, 0x47, 0x18, 0x87, 0x23, 0x6D, 0xDD, 0x46, 0x1F);
   CALC_SB (172, 0xD6, 0x4E, 0x3E, 0x2D, 0x69, 0xF9, 0x64, 0x48);
   CALC_SB (176, 0x2A, 0x4F, 0xCE, 0xF2, 0xCB, 0x65, 0x2F, 0x8E);
   CALC_SB (180, 0xFC, 0x78, 0x97, 0x5C, 0x05, 0x58, 0x7A, 0x19);
   CALC_SB (184, 0xAC, 0x8D, 0x7F, 0xE5, 0xD5, 0x98, 0x1A, 0x57);
   CALC_SB (188, 0x4B, 0x67, 0x0E, 0x7F, 0xA7, 0x05, 0x5A, 0x64);
   CALC_SB (192, 0x28, 0xAF, 0x14, 0x63, 0x3F, 0xB6, 0x29, 0xFE);
   CALC_SB (196, 0x88, 0xF5, 0x3C, 0xB7, 0x4C, 0x3C, 0x02, 0xA5);
   CALC_SB (200, 0xB8, 0xCE, 0xDA, 0xE9, 0xB0, 0x68, 0x17, 0x44);
   CALC_SB (204, 0x55, 0xE0, 0x1F, 0x4D, 0x8A, 0x43, 0x7D, 0x69);
   CALC_SB (208, 0x57, 0x29, 0xC7, 0x2E, 0x8D, 0xAC, 0x74, 0x15);
   CALC_SB (212, 0xB7, 0x59, 0xC4, 0xA8, 0x9F, 0x0A, 0x72, 0x9E);
   CALC_SB (216, 0x7E, 0x6E, 0x15, 0x47, 0x22, 0xDF, 0x12, 0x34);
   CALC_SB (220, 0x58, 0x35, 0x07, 0x6A, 0x99, 0xCF, 0x34, 0xDC);
   CALC_SB (224, 0x6E, 0x22, 0x50, 0xC9, 0xDE, 0xC0, 0x68, 0x9B);
   CALC_SB (228, 0x65, 0x89, 0xBC, 0xD4, 0xDB, 0xED, 0xF8, 0xAB);
   CALC_SB (232, 0xC8, 0x12, 0xA8, 0xA2, 0x2B, 0x0D, 0x40, 0x52);
   CALC_SB (236, 0xDC, 0xBB, 0xFE, 0x02, 0x32, 0x2F, 0xA4, 0xA9);
   CALC_SB (240, 0xCA, 0xD7, 0x10, 0x61, 0x21, 0x1E, 0xF0, 0xB4);
   CALC_SB (244, 0xD3, 0x50, 0x5D, 0x04, 0x0F, 0xF6, 0x00, 0xC2);
   CALC_SB (248, 0x6F, 0x16, 0x9D, 0x25, 0x36, 0x86, 0x42, 0x56);
   CALC_SB (252, 0x4A, 0x55, 0x5E, 0x09, 0xC1, 0xBE, 0xE0, 0x91);

   /* Calculate whitening and round subkeys.  The constants are
    * indices of subkeys, preprocessed through q0 and q1. */
   CALC_K (w, 0, 0xA9, 0x75, 0x67, 0xF3);
   CALC_K (w, 2, 0xB3, 0xC6, 0xE8, 0xF4);
   CALC_K (w, 4, 0x04, 0xDB, 0xFD, 0x7B);
   CALC_K (w, 6, 0xA3, 0xFB, 0x76, 0xC8);
   CALC_K (k, 0, 0x9A, 0x4A, 0x92, 0xD3);
   CALC_K (k, 2, 0x80, 0xE6, 0x78, 0x6B);
   CALC_K (k, 4, 0xE4, 0x45, 0xDD, 0x7D);
   CALC_K (k, 6, 0xD1, 0xE8, 0x38, 0x4B);
   CALC_K (k, 8, 0x0D, 0xD6, 0xC6, 0x32);
   CALC_K (k, 10, 0x35, 0xD8, 0x98, 0xFD);
   CALC_K (k, 12, 0x18, 0x37, 0xF7, 0x71);
   CALC_K (k, 14, 0xEC, 0xF1, 0x6C, 0xE1);
   CALC_K (k, 16, 0x43, 0x30, 0x75, 0x0F);
   CALC_K (k, 18, 0x37, 0xF8, 0x26, 0x1B);
   CALC_K (k, 20, 0xFA, 0x87, 0x13, 0xFA);
   CALC_K (k, 22, 0x94, 0x06, 0x48, 0x3F);
   CALC_K (k, 24, 0xF2, 0x5E, 0xD0, 0xBA);
   CALC_K (k, 26, 0x8B, 0xAE, 0x30, 0x5B);
   CALC_K (k, 28, 0x84, 0x8A, 0x54, 0x00);
   CALC_K (k, 30, 0xDF, 0xBC, 0x23, 0x9D);

   return 0;
}

/* Macros to compute the g() function in the encryption and decryption
 * rounds.  G1 is the straight g() function; G2 includes the 8-bit
 * rotation for the high 32-bit word. */

#define G1(a) \
     (ctx->s[0][(a) & 0xFF]) ^ (ctx->s[1][((a) >> 8) & 0xFF]) \
   ^ (ctx->s[2][((a) >> 16) & 0xFF]) ^ (ctx->s[3][(a) >> 24])

#define G2(b) \
     (ctx->s[1][(b) & 0xFF]) ^ (ctx->s[2][((b) >> 8) & 0xFF]) \
   ^ (ctx->s[3][((b) >> 16) & 0xFF]) ^ (ctx->s[0][(b) >> 24])

/* Encryption and decryption Feistel rounds.  Each one calls the two g()
 * macros, does the PHT, and performs the XOR and the appropriate bit
 * rotations.  The parameters are the round number (used to select subkeys),
 * and the four 32-bit chunks of the text. */

#define ENCROUND(n, a, b, c, d) \
   x = G1 (a); y = G2 (b); \
   x += y; y += x + ctx->k[2 * (n) + 1]; \
   (c) ^= x + ctx->k[2 * (n)]; \
   (c) = ((c) >> 1) + ((c) << 31); \
   (d) = (((d) << 1)+((d) >> 31)) ^ y

#define DECROUND(n, a, b, c, d) \
   x = G1 (a); y = G2 (b); \
   x += y; y += x; \
   (d) ^= y + ctx->k[2 * (n) + 1]; \
   (d) = ((d) >> 1) + ((d) << 31); \
   (c) = (((c) << 1)+((c) >> 31)); \
   (c) ^= (x + ctx->k[2 * (n)])

/* Encryption and decryption cycles; each one is simply two Feistel rounds
 * with the 32-bit chunks re-ordered to simulate the "swap" */

#define ENCCYCLE(n) \
   ENCROUND (2 * (n), a, b, c, d); \
   ENCROUND (2 * (n) + 1, c, d, a, b)

#define DECCYCLE(n) \
   DECROUND (2 * (n) + 1, c, d, a, b); \
   DECROUND (2 * (n), a, b, c, d)

/* Macros to convert the input and output bytes into 32-bit words,
 * and simultaneously perform the whitening step.  INPACK packs word
 * number n into the variable named by x, using whitening subkey number m.
 * OUTUNPACK unpacks word number n from the variable named by x, using
 * whitening subkey number m. */

/* Added to make it compatible with bigendian machines. I use
 * 32 bit integers instead of 8 bit that the original code used.
 * changed by nmav@hellug.gr
 */
 
#ifndef WORDS_BIGENDIAN

# define INPACK(n, x, m) \
   x = in[n] ^ ctx->w[m]

# define OUTUNPACK(n, x, m) \
   x ^= ctx->w[m]; \
   out[(n)] = x; 
#else /* bigendian */

# define INPACK(n, x, m) \
   x = byteswap(in[n]) ^ ctx->w[m]

# define OUTUNPACK(n, x, m) \
   x ^= ctx->w[m]; \
   out[(n)] = byteswap(x); 
#endif 

/* Encrypt one block.  in and out may be the same. */

void
twofish_encrypt (const TWOFISH_context *ctx, void *g_out, void *g_in)
{
   /* The four 32-bit chunks of the text. */
   word32 *out=g_out;
   word32 *in=g_in;
   word32 a,b ,c ,d;

   /* Temporaries used by the round function. */
   word32 x, y;

   /* Input whitening and packing. */
   INPACK (0, a, 0);
   INPACK (1, b, 1);
   INPACK (2, c, 2);
   INPACK (3, d, 3);

   /* Encryption Feistel cycles. */
   ENCCYCLE (0);
   ENCCYCLE (1);
   ENCCYCLE (2);
   ENCCYCLE (3);
   ENCCYCLE (4);
   ENCCYCLE (5);
   ENCCYCLE (6);
   ENCCYCLE (7);

   /* Output whitening and unpacking. */
   OUTUNPACK (0, c, 4);
   OUTUNPACK (1, d, 5);
   OUTUNPACK (2, a, 6);
   OUTUNPACK (3, b, 7);

}

/* Decrypt one block.  in and out may be the same. */

void
twofish_decrypt (const TWOFISH_context *ctx, void *g_out, void *g_in)
{
   /* The four 32-bit chunks of the text. */
   word32 *out=g_out;
   word32 *in=g_in;
   word32 a, b, c, d;

   /* Temporaries used by the round function. */
   word32 x, y;

   /* Input whitening and packing. */
   INPACK (0, c, 4);
   INPACK (1, d, 5);
   INPACK (2, a, 6);
   INPACK (3, b, 7);

   /* Encryption Feistel cycles. */
   DECCYCLE (7);
   DECCYCLE (6);
   DECCYCLE (5);
   DECCYCLE (4);
   DECCYCLE (3);
   DECCYCLE (2);
   DECCYCLE (1);
   DECCYCLE (0);

   /* Output whitening and unpacking. */
   OUTUNPACK (0, a, 0);
   OUTUNPACK (1, b, 1);
   OUTUNPACK (2, c, 2);
   OUTUNPACK (3, d, 3);

}

/* Test a single encryption and decryption, as a sanity check. */

static const char*
selftest (void)
{
   TWOFISH_context ctx; /* Expanded key. */
   word8 scratch[16];	/* Encryption/decryption result buffer. */

   /* Test vector for single encryption/decryption.  Note that I am using
    * the vector from the Twofish paper's "known answer test", I=3, instead
    * of the all-0 vector from the "intermediate value test", because an
    * all-0 key would trigger all the special cases in the RS matrix multiply,
    * leaving the actual math untested. */
   static word8 plaintext[16] = {
      0xD4, 0x91, 0xDB, 0x16, 0xE7, 0xB1, 0xC3, 0x9E,
      0x86, 0xCB, 0x08, 0x6B, 0x78, 0x9F, 0x54, 0x19
   };
   static const word8 key[16] = {
      0x9F, 0x58, 0x9F, 0x5C, 0xF6, 0x12, 0x2C, 0x32,
      0xB6, 0xBF, 0xEC, 0x2F, 0x2A, 0xE8, 0xC3, 0x5A
   };
   static const word8 ciphertext[16] = {
      0x01, 0x9F, 0x98, 0x09, 0xDE, 0x17, 0x11, 0x85,
      0x8F, 0xAA, 0xC3, 0xA3, 0xBA, 0x20, 0xFB, 0xC3
   };

   twofish_setkey (&ctx, key, sizeof(key));
   twofish_encrypt (&ctx, scratch, plaintext);
   if (memcmp (scratch, ciphertext, sizeof (ciphertext)))
     return "Twofish test encryption failed.";
   twofish_decrypt (&ctx, scratch, scratch);
   if (memcmp (scratch, plaintext, sizeof (plaintext)))
     return "Twofish test decryption failed.";
   return NULL;
}

