/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef DEFINES_H
 #define DEFINES_H
 #include <defines.h>
#endif

#include <lcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>

static blf_ctx c; /* For Blowfish */
static char kn1[16][8]; /* For DES */
static char kn2[16][8];
static char kn3[16][8]; /* For tripleDES */
static unsigned char saferkey[1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS)];

int get_block_size(int algorithm)
{
  switch(algorithm) {
  
   case DES:
   case TripleDES:
   case BLOWFISH:
   case GOST:
   case SAFER64:
   case SAFER128:
      return 8;
      break;

   case ThreeWAY:
      return 12;
      break;

   default:
      return 0;
  }


}

int get_key_size(int algorithm) /* In bytes */
{
  switch(algorithm) {
  
   case DES:
   case SAFER64:
      return 8; /* Actually 7 but the algorithm here uses 8 */
      break;

   case TripleDES:
      return 24; 
      break;

   case BLOWFISH:
      return 56; 
      break;

   case GOST:
      return 32;
      break;

   case ThreeWAY:
      return 12;
      break;
  
   case SAFER128:
      return 16;
      break;

   default:
      return 0;
  }

}


int init_mcrypt(const int algorithm, void * key,const int lenofkey)
{

char *keyword1, *keyword2, *keyword3; /* For 3DES and safer*/
char *skey=key;

   switch(algorithm) {
  
   case DES:
      desinit(0);
      des_setkey(kn1, key);
      return 0;
      break;

   case SAFER64:
      keyword1=malloc(8);
      memmove(keyword1, key, 8);
      Safer_Init_Module();
      Safer_Expand_Userkey( (unsigned char*) keyword1,(unsigned char*) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey);
      /* 8 rounds and secure key schedule */
      free(keyword1);

      return 0;
      break; 

   case SAFER128:
      Safer_Init_Module();
      keyword1=malloc(8); keyword2=malloc(8);
      memmove(keyword1, skey, 8);
      memmove(keyword2, &skey[8], 8);      
      Safer_Expand_Userkey( (unsigned char*)keyword1, (unsigned char*)keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey);
      /* 8 rounds and secure key schedule */
      free(keyword1); free(keyword2);

      return 0;
      break;
      
   case TripleDES:
      desinit(0);
      keyword1=malloc(8); keyword2=malloc(8); keyword3=malloc(8);
      
      BreakToThree(key, lenofkey, keyword1, keyword2, keyword3);
      des_setkey(kn1, keyword1);
      des_setkey(kn2, keyword2);
      des_setkey(kn3, keyword3);
      free(keyword1); free(keyword2); free(keyword3);
      return 0;
      break;
       
   case BLOWFISH:
      blf_key(&c,key,lenofkey);
      return 0;
      break;

   case GOST:
      kboxinit();
      return 0;
      break;

   case ThreeWAY:
      return 0;
      break;

   default:
      return 0;
  }

}


/* plaintext should be in block's size */
int mcrypt(int algorithm, void *key, const int lenofkey, void *plaintext)
{

  switch(algorithm) {

   case SAFER64:
   case SAFER128:
      Safer_Encrypt_Block(plaintext, saferkey);
      return 0;
      break;
      
   case DES:
      endes (kn1, plaintext);
      return 0;
      break;
      
   case TripleDES:
      endes (kn1, plaintext);
      dedes (kn2, plaintext);
      endes (kn3, plaintext);
      return 0;
      break;

   case BLOWFISH:
      enblf(&c, plaintext);
      return 0;
      break;

   case GOST:
      gostcrypt(plaintext, key);
      return 0;
      break;

   case ThreeWAY:
      en3way(plaintext,key);
      return 0;
      break;

   default:
      return 0;
  }
}


/* plaintext should be in block's size */
int mdecrypt(int algorithm, void *key,const int lenofkey, void *plaintext)
{

  switch(algorithm) {
  
   case SAFER64:
   case SAFER128:
      Safer_Decrypt_Block(plaintext, saferkey);
      return 0;
      break;

   case DES:
      dedes (kn1, plaintext);
      return 0;
      break;
      
   case TripleDES:
      dedes (kn3, plaintext);
      endes (kn2, plaintext);
      dedes (kn1, plaintext);
      return 0;
      break;

   case BLOWFISH:
      deblf(&c, plaintext);
      return 0;
      break;

   case GOST:
      gostdecrypt(plaintext, key);
      return 0;
      break;

   case ThreeWAY:
      de3way(plaintext,key);
      return 0;
      break;

   default:
      return 0;
  }
}


int end_mcrypt(const int algorithm)
{

   switch(algorithm) {
  
   case DES:
   case TripleDES:
      desdone();
      break;
      
   case BLOWFISH:
   case GOST:
   case ThreeWAY:
   case SAFER64:
   case SAFER128:
      break;

   default:
      return 0;
  }

return 0;
}

