/*
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef DEFINES_H
 #define DEFINES_H
 #include <defines.h>
#endif

#include <lcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast.h>

#define MAX_THREADS 50 /* After that pointers are being reused */

/* I think now is re-entrant */

static int cur_thread=-1;
static struct cast_key* ckey=NULL; /* For cast */
static blf_ctx *c=NULL; /* For Blowfish */
static char **kn1=NULL; /* For DES */
static char **kn2=NULL;
static char **kn3=NULL; /* For tripleDES */
static unsigned char** saferkey=NULL; /* For safer64 and 128 */
static int finished_threads=0;

/* hold the key and the algorithm */
static char **keyword_given=NULL;
static int *algorithm_given=NULL;

int get_block_size(int algorithm)
{
  switch(algorithm) {
  
   case DES:
   case TripleDES:
   case BLOWFISH:
   case GOST:
   case SAFER64:
   case SAFER128:
   case CAST128:
      return 8;
      break;

   case ThreeWAY:
      return 12;
      break;

   default:
      return 0;
  }


}

int get_key_size(int algorithm) /* In bytes */
{
  switch(algorithm) {
  
   case DES:
   case SAFER64:
      return 8; /* Actually 7 but the algorithm here uses 8 */
      break;

   case TripleDES:
      return 24; 
      break;

   case BLOWFISH:
      return 56; 
      break;

   case GOST:
      return 32;
      break;

   case ThreeWAY:
      return 12;
      break;
  
   case SAFER128:
   case CAST128:
      return 16;
      break;

   default:
      return 0;
  }

}


int init_mcrypt(const int algorithm, void * key,const int lenofkey)
{

char *keyword1, *keyword2, *keyword3; /* For 3DES and safer*/
char *skey=NULL;

cur_thread++;

if (cur_thread < MAX_THREADS && finished_threads==0) {

  saferkey=realloc(saferkey, (cur_thread+1)*sizeof(char*));
  kn1=realloc(kn1, (cur_thread+1)*sizeof(char*));
  kn2=realloc(kn2, (cur_thread+1)*sizeof(char*));
  kn3=realloc(kn3, (cur_thread+1)*sizeof(char*));
  c=realloc(c, (cur_thread+1) * sizeof(blf_ctx));
  ckey=realloc(ckey, (cur_thread+1) * sizeof(struct cast_key));  
  keyword_given=realloc(keyword_given, (cur_thread+1)*sizeof(char*));
  algorithm_given=realloc(algorithm_given, (cur_thread+1)*sizeof(int));

}
else {
  cur_thread=0;
  finished_threads=1;
  while (keyword_given[cur_thread]!=NULL) {
   cur_thread++;
  }  
  if (cur_thread >= MAX_THREADS) return -1;
}

  keyword_given[cur_thread]=malloc(lenofkey);
  Bzero(keyword_given[cur_thread],lenofkey);
  memmove(keyword_given[cur_thread], key, lenofkey);
  algorithm_given[cur_thread]=algorithm;
  skey=keyword_given[cur_thread];

   switch(algorithm) {
  
   case DES:
      desinit(0);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      
      des_setkey( (void*) kn1[cur_thread], keyword_given[cur_thread]);
      return cur_thread;
      break;

   case SAFER64:
      keyword1=malloc(8);
      memmove(keyword1, keyword_given[cur_thread], 8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

      Safer_Init_Module();
      Safer_Expand_Userkey( (unsigned char*) keyword1,(unsigned char*) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1);

   case CAST128:
      cast_setkey( &ckey[cur_thread], (void*)keyword_given[cur_thread], lenofkey);
      return cur_thread;
      break; 

   case SAFER128:
      Safer_Init_Module();
      keyword1=malloc(8); keyword2=malloc(8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
      
      memmove(keyword1, skey, 8);
      memmove(keyword2, &skey[8], 8);
      Safer_Expand_Userkey( (unsigned char*)keyword1, (unsigned char*)keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1); free(keyword2);

      return cur_thread;
      break;
      
   case TripleDES:
      desinit(0);
      keyword1=malloc(8); keyword2=malloc(8); keyword3=malloc(8);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      kn2[cur_thread]=malloc(8*16*sizeof(char));
      kn3[cur_thread]=malloc(8*16*sizeof(char));
      
      BreakToThree(keyword_given[cur_thread], lenofkey, keyword1, keyword2, keyword3);
      des_setkey( (void*)kn1[cur_thread], keyword1);
      des_setkey( (void*)kn2[cur_thread], keyword2);
      des_setkey( (void*)kn3[cur_thread], keyword3);
      free(keyword1); free(keyword2); free(keyword3);

      return cur_thread;
      break;
       
   case BLOWFISH:
      blf_key(&c[cur_thread],keyword_given[cur_thread],lenofkey);
      return cur_thread;
      break;

   case GOST:
      kboxinit();
      return cur_thread;
      break;

   case ThreeWAY:
      return cur_thread;
      break;

   default:
      return -1;
  }

}


/* plaintext should be in block's size */
int mcrypt(int thread, void *plaintext)
{

  switch(algorithm_given[thread]) {

   case SAFER64:
   case SAFER128:
      Safer_Encrypt_Block(plaintext, saferkey[thread]);
      return 0;
      break;

   case CAST128:
      cast_encrypt(&ckey[thread], (void*) plaintext);
      return 0;
      break;
            
   case DES:
      endes ( (void*)kn1[thread], plaintext);
      return 0;
      break;
      
   case TripleDES:
      endes ( (void*)kn1[thread], plaintext);
      dedes ( (void*)kn2[thread], plaintext);
      endes ( (void*)kn3[thread], plaintext);
      return 0;
      break;

   case BLOWFISH:
      enblf(&c[thread], plaintext);
      return 0;
      break;

   case GOST:
      gostcrypt(plaintext, (void*) keyword_given[thread]);
      return 0;
      break;

   case ThreeWAY:
      en3way(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   default:
      return 0;
  }
}


/* plaintext should be in block's size */
int mdecrypt(int thread, void *plaintext)
{

  switch(algorithm_given[thread]) {
  
   case SAFER64:
   case SAFER128:
      Safer_Decrypt_Block(plaintext, (void*)saferkey[thread]);
      return 0;
      break;

   case CAST128:
      cast_decrypt(&ckey[thread], (void*) plaintext);
      return 0;
      break;

   case DES:
      dedes ( (char (*)[8])kn1[thread], plaintext);
      return 0;
      break;
      
   case TripleDES:
      dedes ( (void*)kn3[thread], plaintext);
      endes ( (void*)kn2[thread], plaintext);
      dedes ( (void*)kn1[thread], plaintext);
      return 0;
      break;

   case BLOWFISH:
      deblf(&c[thread], plaintext);
      return 0;
      break;

   case GOST:
      gostdecrypt(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   case ThreeWAY:
      de3way(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   default:
      return 0;
  }
}


int end_mcrypt(int td)
{

free(keyword_given[td]); 
keyword_given[td]=NULL;

   switch(algorithm_given[td]) {
  
   case DES:
	desdone();
	free(kn1[td]);
      break;

   case TripleDES:
	desdone();
	free(kn1[td]);
	free(kn2[td]);
	free(kn3[td]);
      break;
      
   case BLOWFISH:
	break;

   case GOST:
   case ThreeWAY:
	break;
   case CAST128:
        break;
   case SAFER64:
   case SAFER128:
	free(saferkey[td]);   
	break;
      break;

   default:
      return 0;
  }

return 0;
}

