/* Copyright (C) 1991, 1992, 1993 Free Software Foundation, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */
                
#ifndef LIBDEFS_H
 #define LIBDEFS_H
 #include <libdefs.h>
#endif

#include <swap.h>
#include <lcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast.h>
#include <tean.h>

#define MAX_THREADS 100 /* After that pointers are being reused */

/* I think now is re-entrant */
/* No the init_xxx functions are NOT */

static int cur_thread=-1;
static struct cast_key* ckey=NULL; /* For cast */
static blf_ctx *c=NULL; /* For Blowfish */
static char **kn1=NULL; /* For DES */
static char **kn2=NULL;
static char **kn3=NULL; /* For tripleDES */
static unsigned char** saferkey=NULL; /* For safer64 and 128 */
static int finished_threads=0;

/* For CBC mode */
static char **previous_ciphertext;
static char **previous_plaintext;
static char **previous_cipher;

/* For CFB mode */
static char **sd_register;
static char **s_register;
static char **enc_s_register;
static char **enc_sd_register;

/* hold the key and the algorithm */
static char **keyword_given=NULL;
static int *algorithm_given=NULL;

int get_block_size(int algorithm)
{
  switch(algorithm) {
  
   case DES:
   case TripleDES:
   case BLOWFISH:
   case GOST:
   case SAFER64:
   case SAFER128:
   case CAST128:
   case TEAN:
      return 8;
      break;

   case ThreeWAY:
      return 12;
      break;

   default:
      return 0;
  }


}

int get_key_size(int algorithm) /* In bytes */
{
  switch(algorithm) {
  
   case DES:
   case SAFER64:
      return 8; /* Actually 7 but the algorithm here uses 8 */
      break;

   case TripleDES:
      return 24; 
      break;

   case BLOWFISH:
      return 56; 
      break;

   case GOST:
      return 32;
      break;

   case ThreeWAY:
      return 12;
      break;
  
   case SAFER128:
   case CAST128:
   case TEAN:
      return 16;
      break;

   default:
      return 0;
  }

}


int init_mcrypt(const int algorithm, void * key,const int lenofkey)
{

char *keyword1, *keyword2, *keyword3; /* For 3DES and safer*/
char *skey=NULL;

cur_thread++;

if (cur_thread < MAX_THREADS && finished_threads==0) {

  saferkey=realloc(saferkey, (cur_thread+1)*sizeof(char*));
  kn1=realloc(kn1, (cur_thread+1)*sizeof(char*));
  kn2=realloc(kn2, (cur_thread+1)*sizeof(char*));
  kn3=realloc(kn3, (cur_thread+1)*sizeof(char*));
  c=realloc(c, (cur_thread+1) * sizeof(blf_ctx));
  ckey=realloc(ckey, (cur_thread+1) * sizeof(struct cast_key));  
  keyword_given=realloc(keyword_given, (cur_thread+1)*sizeof(char*));
  algorithm_given=realloc(algorithm_given, (cur_thread+1)*sizeof(int));
}
else {
  cur_thread=0;
  finished_threads=1;
  while (keyword_given[cur_thread]!=NULL && cur_thread <= MAX_THREADS) {
   cur_thread++;
  }  
  if (cur_thread >= MAX_THREADS) return -1;
}

  algorithm_given[cur_thread]=algorithm;
  keyword_given[cur_thread]=calloc(1, get_key_size(algorithm));
  memmove(keyword_given[cur_thread], key, lenofkey);
  skey=keyword_given[cur_thread];


   switch(algorithm) {
  
   case DES:
      desinit(0);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      
      des_setkey( (void*) kn1[cur_thread], keyword_given[cur_thread]);
      return cur_thread;
      break;

   case SAFER64:
      keyword1=malloc(8);
      memmove(keyword1, keyword_given[cur_thread], 8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

      Safer_Init_Module();
      Safer_Expand_Userkey( (unsigned char*) keyword1,(unsigned char*) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1);

   case CAST128:
      cast_setkey( &ckey[cur_thread], (void*)keyword_given[cur_thread], lenofkey);
      return cur_thread;
      break; 

   case SAFER128:
      Safer_Init_Module();
      keyword1=malloc(8); keyword2=malloc(8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
      
      memmove(keyword1, skey, 8);
      memmove(keyword2, &skey[8], 8);
      Safer_Expand_Userkey( (unsigned char*)keyword1, (unsigned char*)keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1); free(keyword2);

      return cur_thread;
      break;
      
   case TripleDES:
      desinit(0);
      keyword1=malloc(8); keyword2=malloc(8); keyword3=malloc(8);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      kn2[cur_thread]=malloc(8*16*sizeof(char));
      kn3[cur_thread]=malloc(8*16*sizeof(char));
      
      /* If it returns 1 or -1 the key is too small */

      if (BreakToThree(keyword_given[cur_thread], lenofkey, keyword1, keyword2, keyword3)!=0) {
        free(keyword1); free(keyword2); free(keyword3);
        end_mcrypt(cur_thread);
        return (-1);
      }
      des_setkey( (void*)kn1[cur_thread], keyword1);
      des_setkey( (void*)kn2[cur_thread], keyword2);
      des_setkey( (void*)kn3[cur_thread], keyword3);
      free(keyword1); free(keyword2); free(keyword3);

      return cur_thread;
      break;
       
   case BLOWFISH:
      if (lenofkey<2) return (-1); 
      blf_key(&c[cur_thread],keyword_given[cur_thread],lenofkey);
      return cur_thread;
      break;

   case GOST:
      kboxinit();
      return cur_thread;
      break;

   case ThreeWAY:
   case TEAN:
      return cur_thread;
      break;

   default:
      return -1;
  }

}


/* plaintext should be in block's size */
int mcrypt(int thread, void *plaintext)
{

  switch(algorithm_given[thread]) {

   case SAFER64:
   case SAFER128:
      Safer_Encrypt_Block(plaintext, saferkey[thread]);
      return 0;
      break;

   case CAST128:
      cast_encrypt(&ckey[thread], (void*) plaintext);
      return 0;
      break;
            
   case DES:
      endes ( (void*)kn1[thread], plaintext);
      return 0;
      break;
      
   case TripleDES:
      endes ( (void*)kn1[thread], plaintext);
      dedes ( (void*)kn2[thread], plaintext);
      endes ( (void*)kn3[thread], plaintext);
      return 0;
      break;

   case BLOWFISH:
      enblf(&c[thread], plaintext);
      return 0;
      break;

   case GOST:
      gostcrypt(plaintext, (void*) keyword_given[thread]);
      return 0;
      break;

   case TEAN:
      cl_enc_block((void*) keyword_given[thread], plaintext);
      return 0;
      break;

   case ThreeWAY:
      en3way(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   default:
      return 0;
  }

}


/* plaintext should be in block's size */
int mdecrypt(int thread, void *plaintext)
{

  switch(algorithm_given[thread]) {
  
   case SAFER64:
   case SAFER128:
      Safer_Decrypt_Block(plaintext, (void*)saferkey[thread]);
      return 0;
      break;

   case CAST128:
      cast_decrypt(&ckey[thread], (void*) plaintext);
      return 0;
      break;

   case DES:
      dedes ( (char (*)[8])kn1[thread], plaintext);
      return 0;
      break;

   case TEAN:
      cl_dec_block((void*) keyword_given[thread], plaintext);
      return 0;
      break;      

   case TripleDES:
      dedes ( (void*)kn3[thread], plaintext);
      endes ( (void*)kn2[thread], plaintext);
      dedes ( (void*)kn1[thread], plaintext);
      return 0;
      break;

   case BLOWFISH:
      deblf(&c[thread], plaintext);
      return 0;
      break;

   case GOST:
      gostdecrypt(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   case ThreeWAY:
      de3way(plaintext, (void*)keyword_given[thread]);
      return 0;
      break;

   default:
      return 0;
  }
}


int end_mcrypt(int td)
{

free(keyword_given[td]); 
keyword_given[td]=NULL;

   switch(algorithm_given[td]) {
  
   case DES:
	desdone();
	free(kn1[td]);
      break;

   case TripleDES:
	desdone();
	free(kn1[td]);
	free(kn2[td]);
	free(kn3[td]);
      break;
      
   case BLOWFISH:
   case TEAN:   
	break;

   case GOST:
   case ThreeWAY:
	break;
   case CAST128:
        break;
   case SAFER64:
   case SAFER128:
	free(saferkey[td]);   
	break;
      break;

   default:
      return 0;
  }

return 0;
}


/* ECB MODE */

int mcrypt_ecb(int td, void *plaintext, int len)
{
int j;
char *plain=plaintext;

 for (j=0; j<len/get_block_size(algorithm_given[td]); j++) {

  mcrypt(td, &plain[j*get_block_size(algorithm_given[td])]);

 }
  return 0;
}



int mdecrypt_ecb(int td, void *ciphertext, int len)
{
int j;
char *cipher=ciphertext;

 for (j=0; j<len/get_block_size(algorithm_given[td]); j++) {

    mdecrypt(td, &cipher[j*get_block_size(algorithm_given[td])]);

 }

  return 0;
}




/* CBC MODE */

int end_mcrypt_cbc(int td)
{

free(keyword_given[td]); 
keyword_given[td]=NULL;
free(previous_ciphertext[td]);
free(previous_cipher[td]);
free(previous_plaintext[td]);

   switch(algorithm_given[td]) {
  
   case DES:
	desdone();
	free(kn1[td]);
      break;

   case TripleDES:
	desdone();
	free(kn1[td]);
	free(kn2[td]);
	free(kn3[td]);
      break;
      
   case BLOWFISH:
   case TEAN:   
	break;

   case GOST:
   case ThreeWAY:
	break;
   case CAST128:
        break;
   case SAFER64:
   case SAFER128:
	free(saferkey[td]);   
	break;
      break;

   default:
      return 0;
  }

return 0;
}


int init_mcrypt_cbc(const int algorithm, void * key,const int lenofkey)
{

char *keyword1, *keyword2, *keyword3; /* For 3DES and safer*/
char *skey=NULL;

cur_thread++;

if (cur_thread < MAX_THREADS && finished_threads==0) {

  saferkey=realloc(saferkey, (cur_thread+1)*sizeof(char*));
  kn1=realloc(kn1, (cur_thread+1)*sizeof(char*));
  kn2=realloc(kn2, (cur_thread+1)*sizeof(char*));
  kn3=realloc(kn3, (cur_thread+1)*sizeof(char*));
/* for cbc */
  previous_ciphertext=realloc(previous_ciphertext, (cur_thread+1) * sizeof(char*));
  previous_cipher=realloc(previous_cipher, (cur_thread+1) * sizeof(char*));
  previous_plaintext=realloc(previous_plaintext, (cur_thread+1) * sizeof(char*));
/* end cbc */
  c=realloc(c, (cur_thread+1) * sizeof(blf_ctx));
  ckey=realloc(ckey, (cur_thread+1) * sizeof(struct cast_key));  
  keyword_given=realloc(keyword_given, (cur_thread+1)*sizeof(char*));
  algorithm_given=realloc(algorithm_given, (cur_thread+1)*sizeof(int));
}
else {
  cur_thread=0;
  finished_threads=1;
  while (keyword_given[cur_thread]!=NULL && cur_thread <= MAX_THREADS) {
   cur_thread++;
  }  
  if (cur_thread >= MAX_THREADS) return -1;
}

  algorithm_given[cur_thread]=algorithm;
  keyword_given[cur_thread]=calloc(1, get_key_size(algorithm));
  memmove(keyword_given[cur_thread], key, lenofkey);
  skey=keyword_given[cur_thread];

/* For cbc */
  previous_ciphertext[cur_thread]=calloc(1,get_block_size(algorithm_given[cur_thread]));
  previous_cipher[cur_thread]=malloc(get_block_size(algorithm_given[cur_thread]));
  previous_plaintext[cur_thread]=calloc(1,get_block_size(algorithm_given[cur_thread]));

/* End cbc */


   switch(algorithm) {
  
   case DES:
      desinit(0);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      
      des_setkey( (void*) kn1[cur_thread], keyword_given[cur_thread]);
      return cur_thread;
      break;

   case SAFER64:
      keyword1=malloc(8);
      memmove(keyword1, keyword_given[cur_thread], 8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

      Safer_Init_Module();
      Safer_Expand_Userkey( (unsigned char*) keyword1,(unsigned char*) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1);

   case CAST128:
      cast_setkey( &ckey[cur_thread], (void*)keyword_given[cur_thread], lenofkey);
      return cur_thread;
      break; 

   case SAFER128:
      Safer_Init_Module();
      keyword1=malloc(8); keyword2=malloc(8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
      
      memmove(keyword1, skey, 8);
      memmove(keyword2, &skey[8], 8);
      Safer_Expand_Userkey( (unsigned char*)keyword1, (unsigned char*)keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1); free(keyword2);

      return cur_thread;
      break;
      
   case TripleDES:
      desinit(0);
      keyword1=malloc(8); keyword2=malloc(8); keyword3=malloc(8);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      kn2[cur_thread]=malloc(8*16*sizeof(char));
      kn3[cur_thread]=malloc(8*16*sizeof(char));
      
      /* If it returns 1 or -1 the key is too small */

      if (BreakToThree(keyword_given[cur_thread], lenofkey, keyword1, keyword2, keyword3)!=0) {
        free(keyword1); free(keyword2); free(keyword3);
        end_mcrypt(cur_thread);
        return (-1);
      }
      des_setkey( (void*)kn1[cur_thread], keyword1);
      des_setkey( (void*)kn2[cur_thread], keyword2);
      des_setkey( (void*)kn3[cur_thread], keyword3);
      free(keyword1); free(keyword2); free(keyword3);

      return cur_thread;
      break;
       
   case BLOWFISH:
      if (lenofkey<2) return (-1); 
      blf_key(&c[cur_thread],keyword_given[cur_thread],lenofkey);
      return cur_thread;
      break;

   case GOST:
      kboxinit();
      return cur_thread;
      break;

   case ThreeWAY:
   case TEAN:
      return cur_thread;
      break;

   default:
      return -1;
  }

}

int mcrypt_cbc(int td, void *plaintext, int len)
{
  char* fplain=plaintext;
  char* plain;
  int i,j;

 for (j=0; j<len/get_block_size(algorithm_given[td]); j++) {

  plain=&fplain[j*get_block_size(algorithm_given[td])];
  
  for (i=0; i<get_block_size(algorithm_given[td]); i++) {
    plain[i] ^= previous_ciphertext[td][i];
  }

  mcrypt(td, plain);


  /* Copy the ciphertext to prev_ciphertext */
  memmove(previous_ciphertext[td], plain, get_block_size(algorithm_given[td]));
 }
 
  return 0;
}



int mdecrypt_cbc(int td, void *ciphertext, int len)
{
  char* cipher;
  char *fcipher=ciphertext;
  int i,j;

 for (j=0; j<len/get_block_size(algorithm_given[td]); j++) {

    cipher=&fcipher[j*get_block_size(algorithm_given[td])];
    memmove(previous_cipher[td], cipher, get_block_size(algorithm_given[td]));  
  
    mdecrypt(td, cipher);
    for (i=0; i<get_block_size(algorithm_given[td]); i++) {
      cipher[i] ^= previous_plaintext[td][i];
    }

    /* Copy the plaintext to prev_plaintext */
    memmove(previous_plaintext[td], previous_cipher[td], get_block_size(algorithm_given[td]));  

 }

  return 0;
}



/* CFB MODE */

int end_mcrypt_cfb(int td)
{

free(keyword_given[td]); 
keyword_given[td]=NULL;
free(s_register[td]);
free(enc_s_register[td]);
free(enc_sd_register[td]);
free(sd_register[td]);

   switch(algorithm_given[td]) {
  
   case DES:
	desdone();
	free(kn1[td]);
      break;

   case TripleDES:
	desdone();
	free(kn1[td]);
	free(kn2[td]);
	free(kn3[td]);
      break;
      
   case BLOWFISH:
   case TEAN:   
	break;

   case GOST:
   case ThreeWAY:
	break;
   case CAST128:
        break;
   case SAFER64:
   case SAFER128:
	free(saferkey[td]);   
	break;
      break;

   default:
      return 0;
  }

return 0;
}


int init_mcrypt_cfb(const int algorithm, void * key,const int lenofkey, void *IV)
{

char *keyword1, *keyword2, *keyword3; /* For 3DES and safer*/
char *skey=NULL;

cur_thread++;

if (cur_thread < MAX_THREADS && finished_threads==0) {

  saferkey=realloc(saferkey, (cur_thread+1)*sizeof(char*));
  kn1=realloc(kn1, (cur_thread+1)*sizeof(char*));
  kn2=realloc(kn2, (cur_thread+1)*sizeof(char*));
  kn3=realloc(kn3, (cur_thread+1)*sizeof(char*));
/* for cfb */
  s_register=realloc(s_register, (cur_thread+1) * sizeof(char*));
  enc_s_register=realloc(enc_s_register, (cur_thread+1) * sizeof(char*));  
  enc_sd_register=realloc(enc_sd_register, (cur_thread+1) * sizeof(char*));  
  sd_register=realloc(sd_register, (cur_thread+1) * sizeof(char*));
/* end cfb */
  c=realloc(c, (cur_thread+1) * sizeof(blf_ctx));
  ckey=realloc(ckey, (cur_thread+1) * sizeof(struct cast_key));  
  keyword_given=realloc(keyword_given, (cur_thread+1)*sizeof(char*));
  algorithm_given=realloc(algorithm_given, (cur_thread+1)*sizeof(int));
}
else {
  cur_thread=0;
  finished_threads=1;
  while (keyword_given[cur_thread]!=NULL && cur_thread <= MAX_THREADS) {
   cur_thread++;
  }  
  if (cur_thread >= MAX_THREADS) return -1;
}

  algorithm_given[cur_thread]=algorithm;
  keyword_given[cur_thread]=calloc(1, get_key_size(algorithm));
  memmove(keyword_given[cur_thread], key, lenofkey);
  skey=keyword_given[cur_thread];

/* For cfb */
  s_register[cur_thread]=malloc( get_block_size(algorithm_given[cur_thread]));
  sd_register[cur_thread]=malloc( get_block_size(algorithm_given[cur_thread]));
  enc_s_register[cur_thread]=malloc( get_block_size(algorithm_given[cur_thread]));
  enc_sd_register[cur_thread]=malloc( get_block_size(algorithm_given[cur_thread]));
  memmove(sd_register[cur_thread], IV, get_block_size(algorithm_given[cur_thread]));
  memmove(s_register[cur_thread], IV, get_block_size(algorithm_given[cur_thread]));  

/* End cfb */


   switch(algorithm) {
  
   case DES:
      desinit(0);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      
      des_setkey( (void*) kn1[cur_thread], keyword_given[cur_thread]);
      return cur_thread;
      break;

   case SAFER64:
      keyword1=malloc(8);
      memmove(keyword1, keyword_given[cur_thread], 8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

      Safer_Init_Module();
      Safer_Expand_Userkey( (unsigned char*) keyword1,(unsigned char*) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1);

   case CAST128:
      cast_setkey( &ckey[cur_thread], (void*)keyword_given[cur_thread], lenofkey);
      return cur_thread;
      break; 

   case SAFER128:
      Safer_Init_Module();
      keyword1=malloc(8); keyword2=malloc(8);
      saferkey[cur_thread]=malloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
      
      memmove(keyword1, skey, 8);
      memmove(keyword2, &skey[8], 8);
      Safer_Expand_Userkey( (unsigned char*)keyword1, (unsigned char*)keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[cur_thread]);
      /* 8 rounds and secure key schedule */
      free(keyword1); free(keyword2);

      return cur_thread;
      break;
      
   case TripleDES:
      desinit(0);
      keyword1=malloc(8); keyword2=malloc(8); keyword3=malloc(8);
      kn1[cur_thread]=malloc(8*16*sizeof(char));
      kn2[cur_thread]=malloc(8*16*sizeof(char));
      kn3[cur_thread]=malloc(8*16*sizeof(char));
      
      /* If it returns 1 or -1 the key is too small */

      if (BreakToThree(keyword_given[cur_thread], lenofkey, keyword1, keyword2, keyword3)!=0) {
        free(keyword1); free(keyword2); free(keyword3);
        end_mcrypt(cur_thread);
        return (-1);
      }
      des_setkey( (void*)kn1[cur_thread], keyword1);
      des_setkey( (void*)kn2[cur_thread], keyword2);
      des_setkey( (void*)kn3[cur_thread], keyword3);
      free(keyword1); free(keyword2); free(keyword3);

      return cur_thread;
      break;
       
   case BLOWFISH:
      if (lenofkey<2) return (-1); 
      blf_key(&c[cur_thread],keyword_given[cur_thread],lenofkey);
      return cur_thread;
      break;

   case GOST:
      kboxinit();
      return cur_thread;
      break;

   case ThreeWAY:
   case TEAN:
      return cur_thread;
      break;

   default:
      return -1;
  }

}

int mcrypt_cfb(int td, void *plaintext,int len) /* plaintext is 1 byte (8bit cfb) */
{
  char* plain=plaintext;
  int i,j;

 for (j=0;j<len;j++) {
  
  memmove(enc_s_register[td], s_register[td], get_block_size(algorithm_given[td]));
  
  mcrypt(td, enc_s_register[td]);

  plain[j] ^= enc_s_register[td][0];
  
/* Shift the register */
  for (i=0 ; i< (get_block_size(algorithm_given[td])-1) ; i++) 
    s_register[td][i]=s_register[td][i+1];
  
  s_register[td][get_block_size(algorithm_given[td])-1]=plain[j];


 }

  return 0;

}


int mdecrypt_cfb(int td, void *plaintext,int len) /* plaintext is 1 byte (8bit cfb) */
{
  char* plain=plaintext;
  int i,j;
  
 for (j=0;j<len;j++) {
 
  memmove(enc_sd_register[td], sd_register[td], get_block_size(algorithm_given[td]));

  mcrypt(td, enc_sd_register[td]);

/* Shift the register */
  for (i=0 ; i<(get_block_size(algorithm_given[td])-1) ; i++) 
    sd_register[td][i]=sd_register[td][i+1];
  
  sd_register[td][get_block_size(algorithm_given[td])-1]=plain[j];

  plain[j] ^= enc_sd_register[td][0];

 }

  return 0;
}
