/* Copyright (C) 1998,1999 Nikos Mavroyanopoulos This library is free
 * software; you can redistribute it and/or modify it under the terms of the
 * GNU Library General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef LIBDEFS_H
#define LIBDEFS_H
#include <libdefs.h>
#endif
#include <bzero.h>
#include <swap.h>
#include <mcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast.h>
#include <tean.h>
#include <twofish/aes.h>
#include <xmemory.h>
#include <rc2.h>
#include <rc6.h>
#include <idea.h>

#define MAX_THREADS 256		/* After that pointers are being reused */

/* I think now is re-entrant */

static int check_pointer[MAX_THREADS] =
{0};
static int used_all_threads = 0;

static int cur_thread = 0;	/* No threads */

static struct cast_key *ckey = NULL;	/* For cast */
static blf_ctx *c = NULL;	/* For Blowfish */

static cipherInstance *CI = NULL;	/* twofish */
static keyInstance *KI = NULL;


static char **kn1 = NULL;	/* For DES */
static char **kn2 = NULL;
static char **kn3 = NULL;	/* For tripleDES */
static word8 **saferkey = NULL;	/* For safer64 and 128 */
static word16 **rc2_key;	/* For rc2 */

#ifdef NON_FREE
static word16 **idea_ikey;
static word16 **idea_key;	/* For idea */

static word32 **rc6_skey;	/* For RC6 */
#endif

/* For CBC mode */
static word32 **previous_ciphertext;
static word32 **previous_plaintext;
static word32 **previous_cipher;

/* For CFB mode */
static word8 **sd_register;
static word8 **s_register;
static word8 **enc_s_register;
static word8 **enc_sd_register;



/* hold the key and the algorithm */
static word8 **keyword_given = NULL;
static int *algorithm_given = NULL;

int get_block_size(int algorithm)
{
	switch (algorithm) {

	case DES:
	case TripleDES:
	case BLOWFISH:
	case GOST:
	case SAFER64:
	case SAFER128:
	case CAST128:
	case TEAN:
	case RC2:
		return 8;
		break;

#ifdef NON_FREE
	case IDEA:
		return 8;
		break;

	case RC6:
		return 16;
		break;
#endif

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		return 16;
		break;

	case ThreeWAY:
		return 12;
		break;

	default:
		return 0;
	}


}

int get_key_size(int algorithm)
{				/* In bytes */
	switch (algorithm) {

	case DES:
	case SAFER64:
		return 8;	/* Actually 7 but the algorithm here uses 8 */
		break;

	case TripleDES:
		return 24;
		break;

	case BLOWFISH:
		return 56;
		break;

	case RC2:
		return 128;
		break;

#ifdef NON_FREE
	case RC6:
		return RC6_BKEY;	/* 32 */
		break;
#endif

	case TWOFISH192:
		return 24;
		break;

	case GOST:
	case TWOFISH256:
		return 32;
		break;

	case TWOFISH128:
		return 16;
		break;

	case ThreeWAY:
		return 12;
		break;

	case SAFER128:
	case CAST128:
	case TEAN:
#ifdef NON_FREE
	case IDEA:
#endif
		return 16;
		break;

	default:
		return 0;
	}

}


int init_mcrypt(const int algorithm, int thread, void *key, int lenofkey)
{

	char *keyword1, *keyword2, *keyword3;	/* For 3DES and safer */
	unsigned char *safer_skey = NULL;
	char twokey[68];
	int i, j;

        if (lenofkey==0) return -1;

	if (used_all_threads == 0) {
		saferkey = mxrealloc(saferkey, (thread + 1) * sizeof(char *));
		kn1 = mxrealloc(kn1, (thread + 1) * sizeof(char *));
		kn2 = mxrealloc(kn2, (thread + 1) * sizeof(char *));
		kn3 = mxrealloc(kn3, (thread + 1) * sizeof(char *));
		c = mxrealloc(c, (thread + 1) * sizeof(blf_ctx));
		CI = mxrealloc(CI, (thread + 1) * sizeof(cipherInstance));
		KI = mxrealloc(KI, (thread + 1) * sizeof(keyInstance));
		ckey = mxrealloc(ckey, (thread + 1) * sizeof(struct cast_key));
		keyword_given = mxrealloc(keyword_given, (thread + 1) * sizeof(char *));
		rc2_key = mxrealloc(rc2_key, (thread + 1) * sizeof(word16 *));

#ifdef NON_FREE
		rc6_skey = mxrealloc(rc6_skey, (thread + 1) * sizeof(word32 *));
		idea_key = mxrealloc(idea_key, (thread + 1) * sizeof(word16 *));
		idea_ikey = mxrealloc(idea_ikey, (thread + 1) * sizeof(word16 *));
#endif

		algorithm_given = mxrealloc(algorithm_given, (thread + 1) * sizeof(int));
	}
	if (get_key_size(algorithm) < lenofkey)
		lenofkey = get_key_size(algorithm);

	check_pointer[thread] = 1;
	algorithm_given[thread] = algorithm;
	keyword_given[thread] = mxcalloc(1, get_key_size(algorithm));
	memmove(keyword_given[thread], key, lenofkey);
	safer_skey = keyword_given[thread];


	switch (algorithm) {

	case DES:
		desinit(0);
		kn1[thread] = mxmalloc(8 * 16 * sizeof(char));
		des_setkey((void *) kn1[thread], (void *) keyword_given[thread]);
		return thread;
		break;

	case SAFER64:
		keyword1 = mxmalloc(8);
		memmove(keyword1, keyword_given[thread], 8);
		saferkey[thread] = mxmalloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

		Safer_Init_Module();
		Safer_Expand_Userkey((unsigned char *) keyword1, (unsigned char *) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[thread]);
		/* 8 rounds and secure key schedule */
		free(keyword1);

	case CAST128:
		cast_setkey(&ckey[thread], (void *) keyword_given[thread], lenofkey);
		return thread;
		break;

	case RC2:
		rc2_key[thread] = mxmalloc(64 * sizeof(word16));
		rc2_keyschedule(rc2_key[thread], (void *) keyword_given[thread], lenofkey, 1024);
		return thread;
		break;

#ifdef NON_FREE
	case IDEA:
		idea_key[thread] = mxmalloc(52 * sizeof(word16));
		idea_ikey[thread] = mxmalloc(52 * sizeof(word16));
		Idea_ExpandUserKey((void *) keyword_given[thread], idea_key[thread]);
		Idea_InvertKey(idea_key[thread], idea_ikey[thread]);
		return thread;
		break;

	case RC6:
		rc6_skey[thread] = mxmalloc(RC6_FACTOR * sizeof(word32));
		rc6_schedule((void *) keyword_given[thread], rc6_skey[thread]);
		return thread;
		break;
#endif

	case SAFER128:
		Safer_Init_Module();
		keyword1 = mxmalloc(8);
		keyword2 = mxmalloc(8);
		saferkey[thread] = mxmalloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

		memmove(keyword1, safer_skey, 8);
		memmove(keyword2, &safer_skey[8], 8);
		Safer_Expand_Userkey((unsigned char *) keyword1, (unsigned char *) keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[thread]);
		/* 8 rounds and secure key schedule */
		free(keyword1);
		free(keyword2);

		return thread;
		break;

	case TripleDES:
		desinit(0);
		keyword1 = mxcalloc(1, 8 * sizeof(char));
		keyword2 = mxcalloc(1, 8 * sizeof(char));
		keyword3 = mxcalloc(1, 8 * sizeof(char));
		kn1[thread] = mxmalloc(8 * 16 * sizeof(char));
		kn2[thread] = mxmalloc(8 * 16 * sizeof(char));
		kn3[thread] = mxmalloc(8 * 16 * sizeof(char));

		/* If it returns 1 or -1 the key is too small */

		if (BreakToThree(keyword_given[thread], lenofkey, keyword1, keyword2, keyword3) != 0) {
			free(keyword1);
			free(keyword2);
			free(keyword3);
			end_mcrypt(thread);
			return (-1);
		}
		des_setkey((void *) kn1[thread], keyword1);
		des_setkey((void *) kn2[thread], keyword2);
		des_setkey((void *) kn3[thread], keyword3);
		free(keyword1);
		free(keyword2);
		free(keyword3);

		return thread;
		break;

	case BLOWFISH:
		if (lenofkey < 2) {
			blf_key(&c[thread], (void *) keyword_given[thread], 2);
		} else {
			blf_key(&c[thread], (void *) keyword_given[thread], lenofkey);
		}
		return thread;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:

		Bzero(&KI[cur_thread], sizeof(keyInstance));
		Bzero(&CI[cur_thread], sizeof(cipherInstance));
		cipherInit(&CI[cur_thread], MODE_ECB, NULL);

		j = 0;
		for (i = 0; i < lenofkey; i++) {
			sprintf(&twokey[j], "%.2x", (unsigned char) keyword_given[cur_thread][i]);
			j += 2;
		}
		for (; i < get_key_size(algorithm_given[cur_thread]); i++) {
			sprintf(&twokey[j], "%.2x", 0);
			j += 2;
		}

		j = makeKey(&KI[cur_thread], DIR_ENCRYPT, get_key_size(algorithm_given[cur_thread]) * 8, twokey);
		if (j < 0) {
			return (-1);
		}
		return thread;
		break;

	case GOST:
		kboxinit();
		return thread;
		break;

	case ThreeWAY:
	case TEAN:
		return thread;
		break;

	default:
		return -1;
	}

}






/* plaintext should be in block's size */
int mcrypt(int thread, void *plaintext)
{

	switch (algorithm_given[thread]) {

	case SAFER64:
	case SAFER128:
		Safer_Encrypt_Block(plaintext, saferkey[thread]);
		return 0;
		break;

	case CAST128:
		cast_encrypt(&ckey[thread], (void *) plaintext);
		return 0;
		break;

	case RC2:
		rc2_encrypt(rc2_key[thread], (void *) plaintext, (void *) plaintext);
		return 0;
		break;

#ifdef NON_FREE
	case IDEA:
		Idea_Crypt((void *) plaintext, (void *) plaintext, idea_key[thread]);
		return 0;
		break;

	case RC6:
		rc6_encrypt((void *) plaintext, rc6_skey[thread]);
		return 0;
		break;
#endif

	case DES:
		endes((void *) kn1[thread], plaintext);
		return 0;
		break;

	case TripleDES:
		endes((void *) kn1[thread], plaintext);
		dedes((void *) kn2[thread], plaintext);
		endes((void *) kn3[thread], plaintext);
		return 0;
		break;

	case BLOWFISH:
		enblf(&c[thread], plaintext);
		return 0;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		blockEncrypt(&CI[thread], &KI[thread], plaintext, 16 * 8, plaintext);
		return 0;
		break;

	case GOST:
		gostcrypt(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	case TEAN:
		cl_enc_block((void *) keyword_given[thread], plaintext);
		return 0;
		break;

	case ThreeWAY:
		en3way(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	default:
		return 0;
	}

}


/* plaintext should be in block's size */
int mdecrypt(int thread, void *plaintext)
{
	switch (algorithm_given[thread]) {

	case SAFER64:
	case SAFER128:
		Safer_Decrypt_Block(plaintext, (void *) saferkey[thread]);
		return 0;
		break;

	case CAST128:
		cast_decrypt(&ckey[thread], (void *) plaintext);
		return 0;
		break;

	case RC2:
		rc2_decrypt(rc2_key[thread], (void *) plaintext, (void *) plaintext);
		return 0;
		break;

#ifdef NON_FREE
	case IDEA:
		Idea_Crypt((void *) plaintext, (void *) plaintext, idea_ikey[thread]);
		return 0;
		break;

	case RC6:
		rc6_decrypt((void *) plaintext, rc6_skey[thread]);
		return 0;
		break;
#endif

	case DES:
		dedes((char (*)[8]) kn1[thread], plaintext);
		return 0;
		break;

	case TEAN:
		cl_dec_block((void *) keyword_given[thread], plaintext);
		return 0;
		break;

	case TripleDES:
		dedes((void *) kn3[thread], plaintext);
		endes((void *) kn2[thread], plaintext);
		dedes((void *) kn1[thread], plaintext);
		return 0;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		blockDecrypt(&CI[thread], &KI[thread], plaintext, 16 * 8, plaintext);
		return 0;
		break;

	case BLOWFISH:
		deblf(&c[thread], plaintext);
		return 0;
		break;

	case GOST:
		gostdecrypt(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	case ThreeWAY:
		de3way(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	default:
		return 0;
	}
}


int end_mcrypt(int td)
{

	Bzero(keyword_given[td], get_key_size(algorithm_given[td]));
	free(keyword_given[td]);
	check_pointer[td] = 0;

	switch (algorithm_given[td]) {

	case DES:
		desdone();
		Bzero(kn1[td], 8 * 16 * sizeof(char));
		free(kn1[td]);
		break;

	case TripleDES:
		desdone();
		Bzero(kn1[td], 8 * 16 * sizeof(char));
		Bzero(kn2[td], 8 * 16 * sizeof(char));
		Bzero(kn3[td], 8 * 16 * sizeof(char));
		free(kn1[td]);
		free(kn2[td]);
		free(kn3[td]);
		break;

	case RC2:
		Bzero(rc2_key[td], 64 * sizeof(word16));
		free(rc2_key[td]);
		break;

#ifdef NON_FREE
	case IDEA:
		Bzero(idea_key[td], 52 * sizeof(word16));
		Bzero(idea_key[td], 52 * sizeof(word16));
		free(idea_key[td]);
		free(idea_ikey[td]);
		break;

	case RC6:
		Bzero(rc6_skey[td], RC6_FACTOR * sizeof(word32));
		free(rc6_skey[td]);
		break;
#endif

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		Bzero(&CI[td], sizeof(cipherInstance));
		Bzero(&KI[td], sizeof(keyInstance));
		break;

	case BLOWFISH:
		Bzero(&c[td], sizeof(blf_ctx));
		break;

	case TEAN:
		break;

	case GOST:
	case ThreeWAY:
	case CAST128:
		Bzero(&ckey[td], sizeof(struct cast_key));
		break;

	case SAFER64:
	case SAFER128:
		Bzero(saferkey[td], 1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
		free(saferkey[td]);
		break;
		break;

	default:
		return 0;
	}

	return 0;
}

/* ECB MODE */

int init_mcrypt_ecb(const int algorithm, void *key, int lenofkey)
{
	int x;

	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		return -1;
	}
	x = init_mcrypt(algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt(cur_thread);
		return -1;
	} else {
		return x;
	}

}




int mcrypt_ecb(int td, void *plaintext, int len)
{
	int j, blocksize = get_block_size(algorithm_given[td]);
	char *plain = plaintext;

	for (j = 0; j < len / blocksize; j++) {

		mcrypt(td, &plain[j * blocksize]);

	}
	return 0;
}



int mdecrypt_ecb(int td, void *ciphertext, int len)
{
	int j, blocksize = get_block_size(algorithm_given[td]);
	char *cipher = ciphertext;

	for (j = 0; j < len / blocksize; j++) {

		mdecrypt(td, &cipher[j * blocksize]);

	}

	return 0;
}




/* CBC MODE */

int end_mcrypt_cbc(int td)
{

	free(previous_ciphertext[td]);
	free(previous_cipher[td]);
	free(previous_plaintext[td]);

	return end_mcrypt(td);

}


int init_mcrypt_cbc(const int algorithm, void *key, int lenofkey)
{

	int x;

	cur_thread++;

	if (cur_thread < MAX_THREADS && check_pointer[cur_thread] == 0) {

/* for cbc */
		if (used_all_threads == 0) {
			previous_ciphertext = mxrealloc(previous_ciphertext, (cur_thread + 1) * sizeof(word32 *));
			previous_cipher = mxrealloc(previous_cipher, (cur_thread + 1) * sizeof(word32 *));
			previous_plaintext = mxrealloc(previous_plaintext, (cur_thread + 1) * sizeof(word32 *));
		}
/* end cbc */

	} else {
		cur_thread = 0;
		used_all_threads = 1;
		while (check_pointer[cur_thread] == 1 && cur_thread <= MAX_THREADS) {
			cur_thread++;
		}

		if (cur_thread >= MAX_THREADS) {
			return (-1);
		}
	}


/* For cbc */
	previous_ciphertext[cur_thread] = mxcalloc(1, get_block_size(algorithm));
	previous_cipher[cur_thread] = mxmalloc(get_block_size(algorithm));
	previous_plaintext[cur_thread] = mxcalloc(1, get_block_size(algorithm));

/* End cbc */

	x = init_mcrypt(algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cbc(cur_thread);
		return -1;
	} else {
		return x;
	}

}

int mcrypt_cbc(int td, void *plaintext, int len)
{
	word32 *fplain = plaintext;
	word32 *plain;
	int i, j, blocksize = get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		plain = &fplain[j * blocksize / 4];

		for (i = 0; i < blocksize / 4; i++) {
			plain[i] ^= previous_ciphertext[td][i];
		}

		mcrypt(td, plain);


		/* Copy the ciphertext to prev_ciphertext */
		memmove(previous_ciphertext[td], plain, blocksize);
	}

	return 0;
}



int mdecrypt_cbc(int td, void *ciphertext, int len)
{
	word32 *cipher;
	word32 *fcipher = ciphertext;
	int i, j, blocksize = get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		cipher = &fcipher[j * blocksize / 4];
		memmove(previous_cipher[td], cipher, blocksize);

		mdecrypt(td, cipher);
		for (i = 0; i < blocksize / 4; i++) {
			cipher[i] ^= previous_plaintext[td][i];
		}

		/* Copy the plaintext to prev_plaintext */
		memmove(previous_plaintext[td], previous_cipher[td], blocksize);

	}

	return 0;
}



/* CFB MODE */

int end_mcrypt_cfb(int td)
{

	free(s_register[td]);
	free(enc_s_register[td]);
	free(enc_sd_register[td]);
	free(sd_register[td]);

	return end_mcrypt(td);

}


int init_mcrypt_cfb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	cur_thread++;


	if (cur_thread < MAX_THREADS && check_pointer[cur_thread] == 0) {

/* for cfb */
		if (used_all_threads == 0) {
			s_register = mxrealloc(s_register, (cur_thread + 1) * sizeof(char *));
			enc_s_register = mxrealloc(enc_s_register, (cur_thread + 1) * sizeof(char *));
			enc_sd_register = mxrealloc(enc_sd_register, (cur_thread + 1) * sizeof(char *));
			sd_register = mxrealloc(sd_register, (cur_thread + 1) * sizeof(char *));
		}
/* end cfb */
	} else {
		cur_thread = 0;
		used_all_threads = 1;

		while (check_pointer[cur_thread] == 1 && cur_thread <= MAX_THREADS) {
			cur_thread++;
		}

		if (cur_thread >= MAX_THREADS) {
			return -1;
		}
	}


/* For cfb */
	s_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	sd_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	enc_s_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	enc_sd_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV, get_block_size(algorithm));
	memmove(s_register[cur_thread], IV, get_block_size(algorithm));

/* End cfb */

	x = init_mcrypt(algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cfb(cur_thread);
		return -1;
	} else {
		return x;
	}
}

int mcrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = plain[j];


	}

	return 0;

}


int mdecrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = plain[j];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}


/* OFB MODE */

int end_mcrypt_ofb(int td)
{

	free(s_register[td]);
	free(enc_s_register[td]);
	free(enc_sd_register[td]);
	free(sd_register[td]);

	return end_mcrypt(td);

}


int init_mcrypt_ofb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	cur_thread++;


	if (cur_thread < MAX_THREADS && check_pointer[cur_thread] == 0) {

/* for ofb */
		if (used_all_threads == 0) {
			s_register = mxrealloc(s_register, (cur_thread + 1) * sizeof(char *));
			enc_s_register = mxrealloc(enc_s_register, (cur_thread + 1) * sizeof(char *));
			enc_sd_register = mxrealloc(enc_sd_register, (cur_thread + 1) * sizeof(char *));
			sd_register = mxrealloc(sd_register, (cur_thread + 1) * sizeof(char *));
		}
/* end ofb */
	} else {
		cur_thread = 0;
		used_all_threads = 1;

		while (check_pointer[cur_thread] == 1 && cur_thread <= MAX_THREADS) {
			cur_thread++;
		}

		if (cur_thread >= MAX_THREADS) {
			return -1;
		}
	}


/* For cfb */
	s_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	sd_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	enc_s_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	enc_sd_register[cur_thread] = mxmalloc(get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV, get_block_size(algorithm));
	memmove(s_register[cur_thread], IV, get_block_size(algorithm));

/* End cfb */

	x = init_mcrypt(algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cfb(cur_thread);
		return -1;
	} else {
		return x;
	}
}

int mcrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = enc_s_register[td][0];


	}

	return 0;

}


int mdecrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = enc_sd_register[td][0];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}


/* Extra functions */

char *get_algorithms_name(int algorithm)
{

	char *x = mxmalloc(20);

	switch (algorithm) {
	case DES:
		strcpy(x, "DES");
		return x;
		break;

	case RC2:
		strcpy(x, "RC2");
		return x;
		break;

#ifdef NON_FREE
	case IDEA:
		strcpy(x, "IDEA");
		return x;
		break;

	case RC6:
		strcpy(x, "RC6");
		return x;
		break;
#endif

	case TripleDES:
		strcpy(x, "3DES");
		return x;
		break;

	case ThreeWAY:
		strcpy(x, "3-WAY");
		return x;
		break;

#ifdef MCRYPT2
	case CRYPT:
		strcpy(x, "Crypt");
		return x;
		break;

	case DES_COMPAT:
		strcpy(x, "DES-COMPAT");
		return x;
		break;
#endif

	case BLOWFISH:
		strcpy(x, "Blowfish");
		return x;
		break;

	case TWOFISH128:
		strcpy(x, "Twofish-128");
		return x;
		break;
	case TWOFISH192:
		strcpy(x, "Twofish-192");
		return x;
		break;
	case TWOFISH256:
		strcpy(x, "Twofish-256");
		return x;
		break;

	case GOST:
		strcpy(x, "GOST");
		return x;
		break;

	case SAFER64:
		strcpy(x, "SAFER-SK64");
		return x;
		break;

	case SAFER128:
		strcpy(x, "SAFER-SK128");
		return x;
		break;
	case CAST128:
		strcpy(x, "CAST-128");
		return x;
		break;
	case TEAN:
		strcpy(x, "xTEA");
		return x;
		break;
	}
	return NULL;

}


int is_ok_algorithm(int algorithm)
{

	switch (algorithm) {
	case DES:
	case RC2:
#ifdef NON_FREE
	case IDEA:
	case RC6:
#endif
	case TripleDES:
	case ThreeWAY:
#ifdef MCRYPT2
	case CRYPT:
	case DES_COMPAT:
#endif
	case BLOWFISH:
	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
	case GOST:
	case SAFER64:
	case SAFER128:
	case CAST128:
	case TEAN:
		return 0;
		break;

	default:
		return 1;
		break;

	}

}

int is_ok_mode(int mode)
{				/* if this mode exists returns 0 */

	switch (mode) {
	case OFB:
	case CFB:
	case CBC:
	case ECB:
		return 0;
		break;

	default:
		return 1;
		break;

	}
}
