/*  Mini-Crypt 2.2 program. It encrypts text or binary files using
 *  keywords. The keyword is NOT included in the encrypted file so
 *  there is no way to decrypt it, if you forgot it. 
 *  For a brief description of the algorithms read the README file and
 *  the man page.
 *                                     Nikos Mavroyanopoulos
 *                                         nmav@i-net.paiko.gr
 *
 *    Copyright (C) 1998,1999 Nikos Mavroyanopoulos
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* $Id: mcrypt.c,v 1.3 1999/10/15 21:52:29 nmav Exp $ */

#ifndef DEFINES_H
#define DEFINES_H
#include <defines.h>
#endif

#define STREAM_BUFFER 1024

#include <mcrypt.h>
#include <xmalloc.h>
#include <bzero.h>
#include <extra.h>
#include <bits.h>
#include "../libmcrypt/lib/mcrypt.h"
#include <unlink.h>
#include <crc32.h>
#include <locks.h>
#include <enigma.h>
#include <hex.h>
#include <keys.h>
#include <random.h>
#include <gaa.h>
#include <errors.h>
#include <environ.h>
#include <swap.h>
#include "../libmcrypt/lib/xmemory.h"
#include <des-compat.h>

static char rcsid[] = "$Id: mcrypt.c,v 1.3 1999/10/15 21:52:29 nmav Exp $";

char tmperr[128];
unsigned int bit_flag, hexflag = FALSE;
unsigned int stream_flag = FALSE;
unsigned int mode = MCRYPT_CBC;	/* 0:cbc 1:ecb */
unsigned int type = MCRYPT_TWOFISH_192;	/* default: twofish */
int nolock = FALSE;		/* Use locks by default */
int noecho = TRUE;		/* Echo asterisks by default */
int double_check = FALSE;	/* Do not double check for passwords when decrypting */
int quiet = TRUE;		/* silent by default */
int tmpi;
int unlink_flag = FALSE;
int bare_flag = FALSE, real_random_flag = FALSE;
int cleanDelete = FALSE;
#ifdef ZIP
int gzipflag = FALSE;
int bzipflag = FALSE;
#endif
int flush = FALSE;
char *outfile = 0, *tmpc;

void usage(void)
{

	fprintf(stdout,
		_
		("Usage: mcrypt [-dLFus87bhvrx] [-f keyfile] [-k key1 key2 ...] [-m mode] [-o keymode] [-a algorithm] [-c config_file] [file ...]\n\n"));

	fprintf(stdout,
		_
		("mcrypt encrypts/decrypts files using several symmetric algorithms.\n"));
	fprintf(stdout,
		_("Legal parameters are:\n" "\n"
		  " -d --decrypt                 decrypts\n"
		  " -c --config FILENAME         Use the configuration file FILENAME.\n"
		  "                              If not specified the .mcryptrc in your home\n"
		  "                              directory is used.\n"
		  " -a --algorithm ALGORITHM     specify the encryption and decryption algorithm.\n"
		  "                              Use the --list parameter to see the supported\n"
		  "                              algorithms.\n"));
	fprintf(stdout,
		_
		("                              (Default is twofish-192)\n"));

	fprintf(stdout,
		_
		(" -m --mode MODE               specify the mode. Use the --list parameter\n"
		 "	                      to see the supported modes. (Default is cbc)\n"
		 " -k --key KEY1, KEY2, ...     keyword(s) to use.\n"
		 " -f --keyfile FILENAME        file to read the keyword from.\n"
		 " -x --hex                     keyword in hex.\n"
		 " -o --keymode MODE            keyword mode. Can be 8bit, md5hash, or sha1hash.\n"
		 "                               (sha1hash is the default)\n"
		 " -u --unlink                  unlink the input file after encryption/decryption\n"
		 " -L --license                 displays license information.\n"
		 " -q --quiet                   suppress some non critical warnings.\n"
		 " -l --doublecheck             Double check for passphrase even if decrypting.\n"
		 "    --echo                    Echo asterisks when entering the password.\n"
		 "    --list                    Lists the supported algorithms and modes.\n"
		 "    --flush                   Immediately flush the output.\n"
		 " -V --verbose                 more information is displayed.\n"
		 " -b --bare                    Do not keep algorithm information in the encrypted\n"
		 "                              file.\n"
		 " -F --force                   forces output to stdout.\n"));

#ifdef HAVE_DEV_RANDOM
	fprintf(stdout,
		_
		(" -r --random                  Use real random data.\n"));
#endif

#ifdef GZIP
	fprintf(stdout,
		_
		(" -z --gzip                    use gzip to compress files before encryption.\n"));
#endif

#ifdef BZIP2
	fprintf(stdout,
		_
		(" -p --bzip2                   use bzip2 to compress files before encryption.\n"));
#endif

#ifndef NO_FCNTL_LOCK
	fprintf(stdout,
		_(" -n --nolock                  Do not lock files.\n"));
#endif

	fprintf(stdout, _("\n"
			  " -h --help                    prints this help\n"
			  " -v --version                 prints the version number\n"
			  "\n"
			  "Report bugs to mcrypt@i-net.paiko.gr.\n\n"));
}

void mcrypt_license()
{
	fprintf(stdout,
		_("\nCopyright (C) 1998,1999 Nikos Mavroyanopoulos\n"
		  "This program is free software; you can redistribute it and/or modify \n"
		  "it under the terms of the GNU General Public License as published by \n"
		  "the Free Software Foundation; either version 2 of the License, or \n"
		  "(at your option) any later version. \n" "\n"
		  "This program is distributed in the hope that it will be useful, \n"
		  "but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
		  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
		  "GNU General Public License for more details. \n" "\n"
		  "You should have received a copy of the GNU General Public License \n"
		  "along with this program; if not, write to the Free Software \n"
		  "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n"));
}

void mcrypt_version()
{

	fprintf(stderr, _("Mcrypt v.%s (%s-%s-%s)\n"), VERSION, T_CPU,
		T_VENDOR, T_OS);
	fprintf(stderr,
		_
		("Copyright (C) 1998,1999 Nikos Mavroyanopoulos (nmav@hellug.gr)\n"));

}

/*                              The main    */
int main(int argc, char **argv)
{

	short int ein = 0, din = 0, kin = 0, force = 0, return_val = 0;
	char *einfile = 0, *dinfile = 0, *keyword = 0, *modet = 0, *typet =
	    0;
	char **file, **keyfile = NULL, *cfile;
	int x, y, i, file_count = 0, keys = 0, used_algo = FALSE, v;
	gaainfo info;

#ifdef HAVE_SIGNAL_H
	Signal(SIGINT, shandler);
	Signal(SIGQUIT, shandler);
	Signal(SIGSEGV, shandler);
	Signal(SIGPIPE, shandler);
	Signal(SIGTERM, shandler);
	Signal(SIGHUP, shandler);
	Signal(SIGUSR1, SIG_IGN);
	Signal(SIGUSR2, SIG_IGN);
	Signal(SIGALRM, SIG_IGN);
#endif

#ifdef ENABLE_NLS
	setlocale(LC_MESSAGES, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif



	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(_("Error in the arguments.\n"));
	}
	if (info.config_file != NULL) {
		cfile = xmalloc(strlen(info.config_file));
		strcpy(cfile, info.config_file);
		if (check_file(cfile) != 1) {
			err_crit(_
				 ("Warning: Cannot access config file.\n"));
		}
	} else {
#ifdef HAVE_GETPWUID
		cfile = get_cfile(getuid(), ".mcryptrc");
#else
		cfile = xmalloc(11);
		strcpy(cfile, ".mcryptrc");
#endif
	}

	/* If config file exists */
	if (check_file(cfile) == 1) {
		if ((v = gaa_file(cfile, &info)) != -1) {
			/* gets options from file 'config' */
			err_quit(_("Error in the configuration file.\n"));
		}
	}
	free(cfile);


	if (check_env() == TRUE) {

		if (get_env_key() != NULL) {
			info.keysize = 1;
			free(info.keys);
			info.keys = get_env_key();
		}
		if (get_env_algo() != NULL) {
			free(info.algorithm);
			info.algorithm = get_env_algo();
		}
		if (get_env_mode() != NULL) {
			free(info.mode);
			info.mode = get_env_mode();
		}
		if (get_env_bit_mode() != 0) {
			info.kmode = get_env_bit_mode();
		}
	}
/* Check again the command line variables 
 * This will be uncommented when gaa is corrected.
 */

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(_("Error in the arguments.\n"));
	}
/* Examine how we were called */
	if (strstr(argv[0], "decrypt") != NULL) {
		Bzero(argv[0], strlen(argv[0]));
		strcpy(argv[0], "mdecrypt");
		din = TRUE;
		ein = FALSE;
	} else {
		if (strlen(argv[0]) > 5) {
			Bzero(argv[0], strlen(argv[0]));
			strcpy(argv[0], "mcrypt");
		}
		din = FALSE;
		ein = TRUE;	/* It will change by the arguments */
	}


	/* File pointers are as much as the file arguments */

	file = xmalloc((info.size) * sizeof(char *));

	if (info.ed_specified != 0) {
		din = info.din;
		ein = info.ein;
	}
	if (info.real_random_flag == TRUE) {
#ifdef HAVE_DEV_RANDOM
		real_random_flag = TRUE;
#else
		err_warn(_
			 ("Warning: This system does not support real random data.\n"));
#endif
	}
	force = info.force;
	bare_flag = info.bare_flag;
	unlink_flag = info.unlink_flag;
	quiet = info.quiet;
	hexflag = info.hexflag;
	nolock = info.nolock;
	noecho = info.noecho;
	double_check = info.double_check;
	flush = info.flush;
#ifdef ZIP
	gzipflag = info.gzipflag;
	bzipflag = info.bzipflag;
#endif

	if (info.kmode != NULL) {
		if ((strcasecmp(info.kmode, "4bit") == 0 \
		     ||strstr(info.kmode, "hash") != NULL) \
		    &&hexflag == TRUE) {
			err_quit(_
				 ("Hex mode and 4bit/hash modes are mutually exclusive.\n"));
		} else {
			if (strcasecmp(info.kmode, "8bit") == 0) {
				bit_flag = BITS8;
			} else {
				if (strcasecmp(info.kmode, "4bit") == 0) {
					bit_flag = BITS4;
				} else {
					if (strcasecmp
					    (info.kmode, "md5hash") == 0) {
						bit_flag = MD5HASH;
					} else {
						if (strcasecmp
						    (info.kmode,
						     "hash") != 0
						    && strcasecmp(info.
								  kmode,
								  "sha1hash")
						    != 0) {
							err_crit(_
								 ("Only 8bit, 4bit, md5hash and sha1hash key modes are supported. Assuming sha1hash mode.\n"));
						}
						bit_flag = SHA1HASH;	/* default */
					}
				}
			}
		}
	} else {
		bit_flag = SHA1HASH;	/* Default */
	}


	if (info.keyfile != NULL) {
		kin = 2;
		keyfile = read_key_file(info.keyfile, &keys);
		if (keyfile == NULL)
			kin = 0;
		if (keys == 0)
			kin = 0;
	} else {
		if (info.keys != NULL) {
			kin = 2;
			keyfile = xmalloc(info.keysize * sizeof(char *));
			keys = info.keysize;

			for (i = 0; i < info.keysize; i++) {
				keyfile[i] =
				    mxmalloc(strlen(info.keys[i]) + 10);
				strcpy(keyfile[i], info.keys[i]);
			}
		}
	}

	if (info.mode != NULL) {
		modet = xmalloc(strlen(info.mode) + 1);
		strcpy(modet, info.mode);
		mode = check_mode(modet);

		if (mode == -1) {
			mode = MCRYPT_CBC;	/* Default */
			sprintf(tmperr, _("No such mode: %s\n"), modet);
			err_crit(tmperr);
			err_crit(_("Assuming CBC.\n"));
		}
		free(modet);
	}
	if (info.algorithm != NULL) {
		used_algo = TRUE;
		typet = xmalloc(strlen(info.algorithm) + 1);
		strcpy(typet, info.algorithm);
		type = check_algo(typet);
		if (type == -1) {
			type = MCRYPT_TWOFISH_192;
			sprintf(tmperr, _("No such algorithm: %s\n"),
				typet);
			err_crit(tmperr);
			err_crit(_("Assuming TWOFISH-192.\n"));
		}
		free(typet);
		if (
		    (is_block_algorithm(type) != 0
		     && is_block_mode(mode) == 0)) {
			err_quit(_
				 ("This algorithm is incompatible with this mode.\n"));
		}
		if (is_block_algorithm(type) == 0
		    && is_block_mode(mode) != 0) {
			err_warn(_
				 ("This algorithm is incompatible with this mode.\n"));
			err_warn(_("Assuming STREAM mode.\n"));
			mode = MCRYPT_STREAM;
		}
	}
	if (info.keysize != 0 && check_env == FALSE)
		err_warn(_
			 ("Warning: It is insecure to specify keywords in the command line\n"));


#ifdef HAVE_UMASK
	umask(066);
#endif

/* For RAND32... Called only here */
#ifndef HAVE_DEV_RANDOM
	if (quiet <= TRUE)
		err_warn(_("Warning: Using pseudo random numbers\n"));
	srand(time(0) * getpid());
#endif


/* Normal startup */


/* ok now how many files were specified? */
	i = 0;
	file_count += (info.size);


	if (file_count == 0) {
		stream_flag = TRUE;
	}
	while (i < info.size) {
		file[i] = info.input[i];
		i++;
	}


	if (stream_flag == TRUE)
		file_count++;

/* Do as many times as many files we got */
	for (i = 0; i < file_count; i++) {

		if (i != 0) {
			if (outfile != NULL)
				free(outfile);
		}

		/* If keyword file specified choose the i-th keyword */

		if (kin == 2 && i <= (keys - 1)) {
			keyword = keyfile[i];
			if (i != 0) {
				mxfree(keyfile[i - 1], strlen(keyfile[i - 1]));	/* Free previous keyword */
			}
		}
#ifdef HAVE_STAT
		if (stream_flag == FALSE) {
			if (is_normal_file(file[i]) == FALSE) {
				sprintf(tmperr,
					_
					("%s: %s is not a regular file. Skipping...\n"),
					argv[0], file[i]);
				err_crit(tmperr);
				outfile = NULL;
				continue;	/* next */
			}
		}
#endif

#ifdef ZIP
		if (stream_flag == TRUE) {
			if (gzipflag == TRUE || bzipflag == TRUE) {
				err_quit(_
					 ("--bzip2 or --gzip parameters cannot be used when dealing with streams.\n"));
			}
		}
#endif

		/* Check how we were called */
		if (din == TRUE) {	/* decryption */
			if (stream_flag != TRUE)
				dinfile = file[i];
			if ((isatty(fileno((FILE *) (stdin))) == 1)
			    && (stream_flag == TRUE) && (force == 0)) {	/* not a tty */
				sprintf(tmperr,
					_
					("%s: Encrypted data will not be read from a terminal.\n"),
					argv[0]);
				err_crit(tmperr);
				err_quit(_
					 ("Redirect the input instead.\nUse the --help parameter for more help.\n"));
			}
		} else {	/* encryption */
			if (stream_flag != TRUE)
				einfile = file[i];
			if ((isatty(fileno((FILE *) (stdout))) == 1)
			    && (stream_flag == TRUE) && (force == 0)) {	/* not a tty */
				sprintf(tmperr,
					_
					("%s: Encrypted data will not be written to a terminal.\n"),
					argv[0]);
				err_crit(tmperr);
				err_quit(_
					 ("Redirect the output instead.\nUse the --help parameter for more help.\n"));
			}
		}


		/* If no streams make the extensions */
		if (stream_flag != TRUE) {

			if (din == TRUE) {

				y = strlen(dinfile);
				/* If the file has the .enc suffix, then remove it */

				if (dinfile[y - 1] == 'c'
				    && dinfile[y - 2] == 'n'
				    && dinfile[y - 3] == 'e'
				    && dinfile[y - 4] == '.') {

					outfile = xcalloc(y - 3, 1);
					strncat(outfile, dinfile, y - 4);
#ifdef HAVE_STAT
					/* But if it exists exit */
					if (check_file(outfile) != 0) {
						cleanDelete = FALSE;
						if (ask_overwrite
						    (argv[0],
						     outfile) == FALSE) {
							continue;
						} else {
							cleanDelete = TRUE;
						}
					} else {
						cleanDelete = TRUE;
					}
#endif
				} else {	/* append .dec */

					cleanDelete = TRUE;
					outfile = xcalloc(y + 5, 1);
					strncpy(outfile, dinfile, y);
					strcat(outfile, ".dec");


				}
			} else {	/* encryption- append .enc */
				outfile =
				    xcalloc(strlen(einfile) + 5 + 4, 1);
				strcpy(outfile, einfile);
				/* if file has already the .enc ignore it */
				if (strstr(outfile, ".enc") != NULL) {
					sprintf(tmperr,
						_
						("%s: file %s has the .enc suffix... skipping...\n"),
						argv[0], outfile);
					err_crit(tmperr);
					continue;
				}
#ifdef ZIP
				if (stream_flag == FALSE) {
					if (gzipflag == TRUE)
						strcat(outfile, ".gz");
					if (bzipflag == TRUE)
						strcat(outfile, ".bz2");
				}
#endif
				strcat(outfile, ".enc");
#ifdef HAVE_STAT
				/* But if it exists exit */
				if (check_file(outfile) != 0) {
					cleanDelete = FALSE;
					if (ask_overwrite(argv[0], outfile)
					    == FALSE) {
						continue;
					} else {
						cleanDelete = TRUE;
					}

				} else {
					cleanDelete = TRUE;
				}
#endif

			}
		} else {	/* if streams */
			outfile = xmalloc(7);
			strcpy(outfile, "stdout");
		}



/* Decrypt */
		if (din == TRUE) {

			if (bare_flag == FALSE) {
				x =
				    check_file_head(file[i], &type, &mode,
						    &bit_flag,
						    stream_flag);
				if (x != 0) {

					if (x == -1) {
						type = NO_TYPE;	/* Lock failed */
					} else {	/* Unix Crypt */
						if (used_algo == FALSE
						    && bare_flag == FALSE) {
							sprintf(tmperr,
								_
								("%s: no headers. Assuming encrypted file with Unix crypt.\n"),
								argv[0]);
							err_crit(tmperr);
							fflush(stderr);
							type = CRYPT;
						}
					}
				}
			}



			if (is_ok_mode(mode) == 0) {

				switch (type) {

				case CRYPT:
					x =
					    encrypt_ucrypt(dinfile,
							   outfile,
							   keyword);
					break;

				case DES_COMPAT:
					bare_flag = TRUE;
					x =
					    decrypt_desc(dinfile, outfile,
							 keyword);
					break;

				case NO_TYPE:
					x = 1;
					break;

				default:
					if (is_ok_algorithm(type) == 0) {
						x =
						    decrypt_general(type,
								    dinfile,
								    outfile,
								    keyword);
					} else {
						err_crit(_
							 ("Algorithm unsupported in this version\n"));
						x = 1;
					}
					break;

				}
			} else {
				x = 1;
				err_crit(_
					 ("Mode unsupported in this version\n"));
			}

			if (x == 0) {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was decrypted.\n"),
						dinfile);
					err_warn(tmperr);
				} else {
					sprintf(tmperr,
						_
						("Stdin was decrypted.\n"));
					err_warn(tmperr);
				}
#ifdef HAVE_STAT
# ifdef HAVE_UTIME_H
				if (stream_flag == FALSE)
					copyDate(dinfile, outfile);
# endif
#endif

				if (unlink_flag == TRUE
				    && stream_flag == FALSE) x =
					    u_unlink(dinfile);
				if (x != 0)
					perror("u_unlink");

			} else {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was NOT decrypted successfully.\n"),
						dinfile);
					err_crit(tmperr);
					if (x != -1) {
						remove(outfile);
					} else {
						x = 1;
					}
				} else {
					err_crit(_
						 ("Stdin was NOT decrypted successfully.\n"));
				}
			}

			return_val += x;
		}

/* Encrypt */
		if (ein == TRUE) {
			if (is_ok_mode(mode) == 0) {

				switch (type) {
				case CRYPT:
					x =
					    encrypt_ucrypt(einfile,
							   outfile,
							   keyword);
					break;

				case DES_COMPAT:
					bare_flag = TRUE;
					x =
					    encrypt_desc(einfile, outfile,
							 keyword);
					break;

				default:
					if (is_ok_algorithm(type) == 0) {
						x =
						    encrypt_general(type,
								    einfile,
								    outfile,
								    keyword);
					} else {
						err_crit(_
							 ("Algorithm unsupported in this version\n"));
						x = 1;
					}
					break;
				}
			} else {
				x = 1;
				err_crit(_
					 ("Mode unsupported in this version\n"));
			}

			if (x == 0) {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was encrypted.\n"),
						einfile);
					err_warn(tmperr);
				} else {
					sprintf(tmperr,
						_
						("Stdin was encrypted.\n"));
					err_warn(tmperr);
				}
#ifdef HAVE_STAT
#ifdef HAVE_UTIME_H
				if (stream_flag == FALSE)
					copyDate(einfile, outfile);
#endif
#endif
				if (unlink_flag == TRUE
				    && stream_flag == FALSE) x =
					    u_unlink(einfile);
				if (x != 0)
					perror("u_unlink");

			} else {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was NOT encrypted successfully.\n"),
						einfile);
					err_crit(tmperr);
					if (x != -1) {
						remove(outfile);
					} else {
						x = 1;
					}
				} else {
					err_crit(_
						 ("Stdin was NOT encrypted successfully.\n"));
				}

			}


			return_val += x;
		}
	}			/* the main loop */

/* Clear the last keyword used */
	if (keyword != NULL)
		Bzero(keyword, strlen(keyword));
	free(keyword);

/* Clear ALL arguments in the command line */
	for (i = 0; i < argc; i++) {
		Bzero(argv[i], strlen(argv[i]));
	}

	if (return_val != 0) {
		return 1;
	} else {
		return 0;
	}

}




/* UNIX CRYPT Encryption */

int encrypt_ucrypt(char *fromfile, char *tofile, char *key)
{
	word32 plaintext[2];	/* 64 bit */
	int ch = 0, how = 8;	/* how needs to be initialized to 8 */
	unsigned int len;
	char *keyword = mxmalloc(13);
	char *tmp;
	FILE *TOF;
	FILE *FROMF;
	char *command = NULL;


/* Crypt does not care about these 
 * It's purpose is to be compatible with the unix crypt which is not safe.
 * It isn't a block algorithm after all.
 */
	bit_flag = BITS8;


/* open files */
	if (stream_flag == TRUE) {
		FROMF = stdin;
		TOF = stdout;
	} else {
#ifdef ZIP
		if (gzipflag == FALSE && bzipflag == FALSE) {
#endif
			FROMF = fopen(fromfile, "rb");
			if (FROMF == NULL) {
				perror("fopen");
				return 1;
			}
			if (read_lock(fileno((FILE *) FROMF)) == -1)
				return 1;

			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;
#ifdef ZIP
		} else {
			/* command if using gzip */
			if (gzipflag == TRUE) {
				command =
				    malloc(strlen(GZIP) +
					   strlen(fromfile) + 5);
				strcpy(command, GZIP);
			}
			if (bzipflag == TRUE) {
				command =
				    malloc(strlen(BZIP2) +
					   strlen(fromfile) + 5);
				strcpy(command, BZIP2);
			}
			strcat(command, " -c ");
			strcat(command, fromfile);

			FROMF = popen(command, "r");

			if (FROMF == NULL) {
				perror("popen");

				return 1;
			}
			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;

		}
#endif
	}

	Bzero(keyword, 13);

	if (key == NULL) {
		if (stream_flag == FALSE) {
			sprintf(tmperr, _("File: %s\n"), fromfile);
			err_warn(tmperr);
		}
		tmp = get_password(8, bit_flag, DECRYPT, type, &len, NULL);
		if (tmp == NULL)
			return 1;
		memmove(keyword, tmp, len);
		Bzero(tmp, strlen(tmp));


	} else {
		if (strlen(key) > 8) {
			key[8] = '\0';
			err_warn(_
				 ("Warning: Reducing the size of the key\n"));
		}
		memmove(keyword, key, strlen(key));	/* Copy the 64 bits in key */
	}


	if (quiet == FALSE)
		show_mode(ENCRYPT, type, mode, tofile, bit_flag);


	if (unix_setup(keyword) == 1) {
		err_crit(_("Cannot encrypt key\n"));
		return 1;
	}			/* init the crypt */
	do {
/* else read the first 1 byte of the file and stores the to plaintext */

		how = fread(plaintext, 1, sizeof(plaintext), FROMF);
		if (how == 0) {
			if (feof(FROMF) != 0) {
				ch = 1;
			} else {
				perror("fread");
				return 1;
			}
		}
		unix_crypt(keyword, plaintext, how, 0);

		if (ch == 0)
			if (fwrite(plaintext, 1, how, TOF) < how) {
				perror("fwrite");
				return (1);
			}
		if (flush == TRUE)
			fflush(TOF);
	}
	while (ch == 0);

/* close files */
	if (stream_flag == FALSE) {
		fflush(FROMF);
		fflush(TOF);
		unlock(fileno((FILE *) TOF));
		if (gzipflag == FALSE && bzipflag == FALSE)
			unlock(fileno((FILE *) FROMF));
	}
	fclose(TOF);
	if (gzipflag == FALSE && bzipflag == FALSE) {
		fclose(FROMF);
	} else {
		pclose(FROMF);
	}

	mxfree(keyword, 13);

	return 0;
}






/* General Encryption */

int encrypt_general(int algorithm, char *fromfile, char *tofile, char *key)
{
	unsigned char counter = 0;
	char stream_buf[STREAM_BUFFER];
	word32 fcrc32;
	word32 *keyword;
	int i = 0, j = 0, ch = 0, how = 0;
	unsigned int len = 0;
	FILE *FROMF;
	int keysize = mcrypt_get_key_size(algorithm);
	int blocksize = mcrypt_get_block_size(algorithm);
	int td;
	char *command = NULL;
	char *buffer;
	word32 *IV = NULL;
	word32 *salt = NULL;

	word32 *ciphertext = xmalloc(blocksize);
	FILE *TOF = NULL;

	if ((bit_flag == SHA1HASH || bit_flag == MD5HASH)
	    && bare_flag == FALSE) {
		salt = xmalloc(SALT_SIZE);	/* 20 bytes salt */

		/* Fill the salt with random data */
		for (i = 0; i < SALT_SIZE / sizeof(word32); i++) {
			if (real_random_flag == TRUE) {

				salt[i] = SRAND32;
			} else {
				salt[i] = RAND32;
			}
		}
	}
	if (mode != MCRYPT_ECB || mode != MCRYPT_STREAM) {
		IV = mxmalloc(blocksize);
		/* Initialisation Vector */


/* Random number generator */

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			if (real_random_flag == TRUE) {
				IV[i] = SRAND32;
			} else {
				IV[i] = RAND32;
			}
		}

#ifdef DEBUG
		fprintf(stderr, "IV: ");
		for (i = 0; i < blocksize; i++) {
			fprintf(stderr, "%.2x.",
				((unsigned char *) IV)[i]);
		}
		fprintf(stderr, "\n");
#endif
	}

/* open files */
	if (stream_flag == TRUE) {
		FROMF = stdin;
		TOF = stdout;
	} else {
#ifdef ZIP
		if (gzipflag == FALSE && bzipflag == FALSE) {
#endif
			FROMF = fopen(fromfile, "rb");
			if (FROMF == NULL) {
				perror("fopen");
				return 1;
			}
			if (read_lock(fileno((FILE *) FROMF)) == -1)
				return 1;

			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;
#ifdef ZIP
		} else {
			/* command if using gzip */
			if (gzipflag == TRUE) {
				command =
				    malloc(strlen(GZIP) +
					   strlen(fromfile) + 5);
				strcpy(command, GZIP);
			}
			if (bzipflag == TRUE) {
				command =
				    malloc(strlen(BZIP2) +
					   strlen(fromfile) + 5);
				strcpy(command, BZIP2);
			}
			strcat(command, " -c ");
			strcat(command, fromfile);
			FROMF = popen(command, "r");

			if (FROMF == NULL) {
				perror("popen");
				return 1;
			}
			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;
		}
#endif
	}



	if (bare_flag == FALSE) {
		write_file_head(TOF, type, mode, bit_flag, salt);
		buffer = xcalloc(1, RESERVED);
		if (fwrite(buffer, 1, RESERVED, TOF) < RESERVED) {
			perror("fwrite");
			return -1;
		}
		/* 20 bytes reserved */
		free(buffer);
	}
	if (key == NULL) {
		if (stream_flag == FALSE) {
			sprintf(tmperr, _("File: %s\n"), fromfile);
			err_warn(tmperr);
		}
	}
/* Get keyword */
	if (key == NULL) {
		keyword =
		    fixkey(NULL, &len, bit_flag, hexflag, algorithm, quiet,
			   stream_flag, salt, ENCRYPT);
	} else {
		len = strlen(key);
		keyword =
		    fixkey(key, &len, bit_flag, hexflag, algorithm, quiet,
			   stream_flag, salt, ENCRYPT);
	}
	if (keyword == NULL)
		return 1;

	if (quiet == FALSE)
		show_mode(ENCRYPT, algorithm, mode, tofile, bit_flag);


	switch (mode) {
	case MCRYPT_ECB:
		td = init_mcrypt_ecb(algorithm, keyword, len);
		break;
	case MCRYPT_CBC:
		td = init_mcrypt_cbc(algorithm, keyword, len);
		break;
	case MCRYPT_CFB:
		td = init_mcrypt_cfb(algorithm, keyword, len, IV);
		break;
	case MCRYPT_OFB:
		td = init_mcrypt_ofb(algorithm, keyword, len, IV);
		break;
	case MCRYPT_nOFB:
		td = init_mcrypt_nofb(algorithm, keyword, len, IV);
		break;
	case MCRYPT_STREAM:
		td = init_mcrypt_stream(algorithm, keyword, len);
		break;
	default:
		return 1;
	}

	if (td < 0) {
		err_crit(_("Unknown algorithm error\n"));
		return 1;
	}
	if (bare_flag == FALSE)
		clear_crc32();


/* Encryption Starts here */

	switch (mode) {

	case MCRYPT_CBC:

		/* Put the IV into the file */
		mcrypt_cbc(td, IV, blocksize);
		if (fwrite(IV, 1, blocksize, TOF) < blocksize) {
			perror("fwrite");
			return 1;
		}
		if (flush == TRUE)
			fflush(TOF);
		do {
			/* read the first n bytes of the file and store to ciphertext */

			how = fread(ciphertext, 1, blocksize, FROMF);

			if (how < blocksize) {
				if (how == 0 && feof(FROMF) == 0) {
					perror("fread");
					return 1;
				}
				counter = ((unsigned char) how);
				ch = 1;
			}
			/* crc32 */
			if (bare_flag == FALSE)
				crc32(ciphertext, how);

			mcrypt_cbc(td, ciphertext, blocksize);


			if (fwrite(ciphertext, 1, blocksize, TOF) <
			    blocksize) {
				perror("fwrite");
				return 1;
			}
			if (flush == TRUE)
				fflush(TOF);

			if (ch == 1) {
				if (fwrite
				    (&counter, 1, sizeof(counter),
				     TOF) < sizeof(counter)) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(TOF);
			}
		}
		while (ch == 0);
		break;

	case MCRYPT_nOFB:

		/* Put the IV into the file */
		if (fwrite(IV, 1, blocksize, TOF) < blocksize) {
			perror("fwrite");
			return 1;
		}
		if (flush == TRUE)
			fflush(TOF);
		do {
			/* read the first n bytes of the file and store to ciphertext */

			how = fread(ciphertext, 1, blocksize, FROMF);

			if (how < blocksize) {
				if (how == 0 && feof(FROMF) == 0) {
					perror("fread");
					return 1;
				}
				counter = ((unsigned char) how);
				ch = 1;
			}
			/* crc32 */
			if (bare_flag == FALSE)
				crc32(ciphertext, how);

			mcrypt_nofb(td, ciphertext, blocksize);


			if (fwrite(ciphertext, 1, blocksize, TOF) <
			    blocksize) {
				perror("fwrite");
				return 1;
			}
			if (flush == TRUE)
				fflush(TOF);
			if (ch == 1) {
				if (fwrite
				    (&counter, 1, sizeof(counter),
				     TOF) < sizeof(counter)) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(TOF);
			}
		}
		while (ch == 0);
		break;


	case MCRYPT_CFB:

		/* Put the IV into the file */
		if (fwrite(IV, 1, blocksize, TOF) < blocksize) {
			perror("fwrite");
			return 1;
		}
		if (flush == TRUE)
			fflush(TOF);
		do {
			/* read 1 byte of the file and store to ciphertext */

			how =
			    fread(ciphertext, 1, sizeof(ciphertext),
				  FROMF);

			if (how < 1) {
				if (how == 0 && feof(FROMF) == 0) {
					perror("fread");
					return 1;
				}
				ch = 1;
			}
			/* crc32 */
			if (bare_flag == FALSE)
				crc32(ciphertext, how);
			mcrypt_cfb(td, ciphertext, how);

			if (ch == 0) {
				if (fwrite(ciphertext, 1, how, TOF) < how) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(TOF);
			}
		}
		while (ch == 0);
		break;

	case MCRYPT_OFB:

		/* Put the IV into the file */
		if (fwrite(IV, 1, blocksize, TOF) < blocksize) {
			perror("fwrite");
			return 1;
		}
		if (flush == TRUE)
			fflush(TOF);
		do {
			/* read 1 byte of the file and store to ciphertext */

			how =
			    fread(ciphertext, 1, sizeof(ciphertext),
				  FROMF);

			if (how < 1) {
				if (how == 0 && feof(FROMF) == 0) {
					perror("fread");
					return 1;
				}
				ch = 1;
			}
			/* crc32 */
			if (bare_flag == FALSE)
				crc32(ciphertext, how);
			mcrypt_ofb(td, ciphertext, how);

			if (ch == 0) {
				if (fwrite(ciphertext, 1, how, TOF) < how) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(TOF);
			}
		}
		while (ch == 0);
		break;


	case MCRYPT_ECB:
		do {

			/* read the first n bytes of the file and store to ciphertext */

			how = fread(ciphertext, 1, blocksize, FROMF);

			if (how < blocksize) {
				if (how == 0 && feof(FROMF) == 0) {
					perror("fread");
					return 1;
				}
				counter = ((unsigned char) how);
				ch = 1;
			}
			/* crc32 */
			if (bare_flag == FALSE)
				crc32(ciphertext, how);

			mcrypt_ecb(td, ciphertext, blocksize);


			if (fwrite(ciphertext, 1, blocksize, TOF) <
			    blocksize) {
				perror("fwrite");
				return 1;
			}
			if (ch == 1) {
				if (fwrite
				    (&counter, 1, sizeof(counter),
				     TOF) < sizeof(counter)) {
					perror("fwrite");
					return 1;
				}
			}
			if (flush == TRUE)
				fflush(TOF);
		}
		while (ch == 0);
		break;

	case MCRYPT_STREAM:
		do {

			/* read the first n bytes of the file and store to ciphertext */

			how = fread(stream_buf, 1, STREAM_BUFFER, FROMF);

			if (how != 0) {

				/* crc32 */
				if (bare_flag == FALSE)
					crc32(stream_buf, how);

				mcrypt_stream(td, stream_buf, how);

				if (fwrite(stream_buf, 1, how, TOF) < how) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(TOF);
			} else {
				ch = 1;
			}
		}
		while (ch == 0);
		break;


	default:
		return 1;
	}			/*  switch */

	if (bare_flag == FALSE) {	/* put crc32 */
#ifdef WORDS_BIGENDIAN
		fcrc32 = get_crc32();
#else
		fcrc32 = byteswap(get_crc32());
#endif

/* encrypt the crc32, but first put some random data there */
		if (is_block_algorithm(algorithm) != 0) {
			for (j = 0; j < blocksize / sizeof(word32); j++) {
				if (real_random_flag == TRUE) {
					ciphertext[j] = SRAND32;
				} else {
					ciphertext[j] = RAND32;
				}
			}
			memmove(ciphertext, &fcrc32, sizeof(fcrc32));
			mcrypt(td, ciphertext);

			if (fwrite(ciphertext, 1, blocksize, TOF) <
			    blocksize) {
				perror("fwrite");
				return 1;
			}

		} else {	/* stream */
			mcrypt_stream(td, &fcrc32, sizeof(fcrc32));
			if (fwrite(&fcrc32, 1, sizeof(fcrc32), TOF) <
			    sizeof(fcrc32)) {
				perror("fwrite");
				return 1;
			}
		}


		if (flush == TRUE)
			fflush(TOF);
	}

	/* if bare_flag==FALSE */
	/* End of copy */
	/* close and unlock files */
	fflush(TOF);
	unlock(fileno((FILE *) TOF));

	if (stream_flag == FALSE) {
		fflush(FROMF);
		fflush(TOF);
		if (gzipflag == FALSE && bzipflag == FALSE)
			unlock(fileno((FILE *) FROMF));
		unlock(fileno((FILE *) TOF));
	}
	fclose(TOF);

	if (gzipflag == FALSE && bzipflag == FALSE) {
		fclose(FROMF);
	} else {
		pclose(FROMF);
	}
/* Ready */


	switch (mode) {
	case MCRYPT_ECB:
		end_mcrypt_ecb(td);
		break;
	case MCRYPT_CBC:
		end_mcrypt_cbc(td);
		mxfree(IV, blocksize);
		break;
	case MCRYPT_CFB:
		end_mcrypt_cfb(td);
		mxfree(IV, blocksize);
		break;
	case MCRYPT_OFB:
		end_mcrypt_ofb(td);
		mxfree(IV, blocksize);
		break;
	case MCRYPT_nOFB:
		end_mcrypt_nofb(td);
		mxfree(IV, blocksize);
		break;
	case MCRYPT_STREAM:
		Bzero(stream_buf, STREAM_BUFFER);
		end_mcrypt_stream(td);
		break;
	}

	if ((bit_flag == SHA1HASH || bit_flag == MD5HASH)
	    && bare_flag == FALSE) {
		free(salt);
	}

	mxfree(keyword, keysize);
	free(ciphertext);


	return 0;
}




/* General Decryption */

int decrypt_general(int algorithm, char *fromfile, char *tofile, char *key)
{
	word32 fcrc32, newcrc32;
	char stream_buf[STREAM_BUFFER];
	char tmp_buf[STREAM_BUFFER];
	unsigned char counter = 0;
	int i = 0, ch = 0, start = 0, j;
	int keysize = mcrypt_get_key_size(algorithm);
	int blocksize = mcrypt_get_block_size(algorithm);
	int td, pid;
	char *tmp;

	word32 *ciphertext_old = xmalloc(blocksize);
	word32 *buffer = xmalloc(blocksize);
	word32 *ciphertext = xmalloc(blocksize);
	word32 *ciphertext2 = xmalloc(blocksize);
	word32 *ciphertext_orig = xmalloc(blocksize);
	word32 *keyword;
	word32 *salt = NULL;

	unsigned int len = 0;

	FILE *RTOF;
	FILE *FROMF;

	if (bare_flag == FALSE) {
		salt = xmalloc(SALT_SIZE);
	}
/* open files */
	if (stream_flag == TRUE) {
		FROMF = stdin;
		RTOF = stdout;
	} else {
		FROMF = fopen(fromfile, "rb");
		if (FROMF == NULL) {
			perror("fopen");
			return 1;
		}
		if (read_lock(fileno((FILE *) FROMF)) == -1)
			return 1;

		RTOF = fopen(tofile, "wb");
		if (RTOF == NULL) {
			perror("fopen");
			return -1;
		}
		if (write_lock(fileno((FILE *) RTOF)) == -1)
			return -1;

		if (bare_flag == FALSE)
			fseek(FROMF, 6, SEEK_CUR);
		/* In streams this has been done when checking for headers. */
	}

	if (bare_flag == FALSE) {
		/* Read salt */
		fread(salt, 1, SALT_SIZE, FROMF);	/* read salt */
		tmp = xmalloc(RESERVED);
		fread(tmp, 1, RESERVED, FROMF);	/* read reserved data */
		/* actually skip reserved bytes and go to crc32 */
		free(tmp);
	}
	if (quiet <= TRUE && stream_flag == FALSE) {
		sprintf(tmperr, _("File: %s\n"), fromfile);
		err_warn(tmperr);
	}
/* Get key */

	if (key == NULL) {
		keyword =
		    fixkey(NULL, &len, bit_flag, hexflag, algorithm, quiet,
			   stream_flag, salt, DECRYPT);
	} else {
		len = strlen(key);
		keyword =
		    fixkey(key, &len, bit_flag, hexflag, algorithm, quiet,
			   stream_flag, salt, DECRYPT);
	}
	if (keyword == NULL)
		return 1;


	if (quiet == FALSE)
		show_mode(DECRYPT, algorithm, mode, tofile, bit_flag);

	switch (mode) {
	case MCRYPT_CBC:
		td = init_mcrypt_cbc(algorithm, keyword, len);
		break;
	case MCRYPT_ECB:
		td = init_mcrypt_ecb(algorithm, keyword, len);
		break;
	case MCRYPT_CFB:
		td = 0;
		break;
	case MCRYPT_OFB:
		td = 0;
		break;
	case MCRYPT_nOFB:
		td = 0;
		break;
	case MCRYPT_STREAM:
		td = init_mcrypt_stream(algorithm, keyword, len);
		break;
	default:
		return 1;
	}

	if (td < 0) {
		err_crit(_("Unknown algorithm error\n"));
		return 1;
	}

	if (bare_flag == FALSE) {
		clear_crc32();
	}
	switch (mode) {

	case MCRYPT_CBC:
		/* decryption starts here */
		fread(buffer, 1, blocksize, FROMF);	/* Read the IV */

#ifdef DEBUG
		fprintf(stderr, "IV: ");
		for (i = 0; i < blocksize; i++) {
			fprintf(stderr, "%.2x.",
				((unsigned char *) buffer)[i]);
		}
		fprintf(stderr, "\n");
#endif

		fread(ciphertext, 1, blocksize, FROMF);

		do {

			i = fread(ciphertext2, 1, blocksize, FROMF);
			if (i != blocksize) {
				if (ferror(FROMF) != 0) {
					perror("fread");
					return 1;
				} else {	/* eof */
					if (i == 1) {
						ch = 1;
					} else {
						err_crit(_
							 ("Incorrect file type\n"));
						return 1;
					}

					if (bare_flag == FALSE) {
						memmove(&counter,
							ciphertext,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}

						/* CRC32 */
						for (j = 0; j < blocksize;
						     j++)
							((unsigned char *)
							 ciphertext)[j] =
							    ((unsigned
							      char *)
							     ciphertext)[j
									 +
									 1];
						((unsigned char *)
						 ciphertext)[blocksize -
							     1] =
						    ((unsigned char *)
						     ciphertext2)[0];
						mdecrypt(td, ciphertext);
					} else {
						memmove(&counter,
							ciphertext2,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}
						mdecrypt_cbc(td, buffer,
							     blocksize);
						if (fwrite
						    (buffer, 1,
						     (int) blocksize,
						     RTOF) < blocksize) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);
						memmove(buffer, ciphertext,
							blocksize);
					}

				}
			}
			/* Decrypt the ciphertext */
			mdecrypt_cbc(td, buffer, blocksize);

			if (ch == 1) {
				if (counter == 0)
					break;
				if (fwrite(buffer, 1, (int) counter, RTOF)
				    == 0) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(RTOF);
				/* CRC32 */
				if (bare_flag == FALSE)
					crc32(buffer, (int) counter);

			} else {
				if (start > 0) {
					if (fwrite
					    (buffer, 1, blocksize,
					     RTOF) < blocksize) {
						perror("fwrite");
						return 1;
					}
					if (flush == TRUE)
						fflush(RTOF);
					/* CRC32 */
					if (bare_flag == FALSE) {
						crc32(buffer, blocksize);
					}
				}
				start++;
				memmove(buffer, ciphertext, blocksize);
				memmove(ciphertext, ciphertext2,
					blocksize);
			}


		}
		while (ch == 0);
		break;

	case MCRYPT_nOFB:
		i = fread(buffer, 1, blocksize, FROMF);	/* Read the IV */
		td = init_mcrypt_nofb(algorithm, keyword, len, buffer);

		if (td < 0) {
			err_crit(_("Unknown algorithm error\n"));
			return 1;
		}

#ifdef DEBUG
		fprintf(stderr, "IV: ");
		for (i = 0; i < blocksize; i++) {
			fprintf(stderr, "%.2x.",
				((unsigned char *) buffer)[i]);
		}
		fprintf(stderr, "\n");
#endif

		fread(buffer, 1, blocksize, FROMF);
		fread(ciphertext, 1, blocksize, FROMF);

		do {

			i = fread(ciphertext2, 1, blocksize, FROMF);
			if (i != blocksize) {
				if (ferror(FROMF) != 0) {
					perror("fread");
					return 1;
				} else {	/* eof */
					if (i == 1) {
						ch = 1;
					} else {
						err_crit(_
							 ("Incorrect file type\n"));
						return 1;
					}

					if (bare_flag == FALSE) {
						memmove(&counter,
							ciphertext,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}

						/* CRC32 */
						for (j = 0; j < blocksize;
						     j++)
							((unsigned char *)
							 ciphertext)[j] =
							    ((unsigned
							      char *)
							     ciphertext)[j
									 +
									 1];
						((unsigned char *)
						 ciphertext)[blocksize -
							     1] =
						    ((unsigned char *)
						     ciphertext2)[0];
						mdecrypt(td, ciphertext);
					} else {
						memmove(&counter,
							ciphertext2,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}
						mdecrypt_nofb(td, buffer,
							      blocksize);
						if (fwrite
						    (buffer,
						     (int) blocksize, 1,
						     RTOF) == 0) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);
						memmove(buffer, ciphertext,
							blocksize);
					}

				}
			}
			/* Decrypt the ciphertext */
			mdecrypt_nofb(td, buffer, blocksize);

			if (ch == 1) {
				if (counter == 0)
					break;
				if (fwrite(buffer, (int) counter, 1, RTOF)
				    == 0) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(RTOF);
				/* CRC32 */
				if (bare_flag == FALSE)
					crc32(buffer, (int) counter);

			} else {
				if (fwrite(buffer, blocksize, 1, RTOF) ==
				    0) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(RTOF);
				/* CRC32 */
				if (bare_flag == FALSE) {
					crc32(buffer, blocksize);
				}
				memmove(buffer, ciphertext, blocksize);
				memmove(ciphertext, ciphertext2,
					blocksize);
			}


		}
		while (ch == 0);
		break;


	case MCRYPT_ECB:

		fread(ciphertext, 1, blocksize, FROMF);
		memmove(buffer, ciphertext, blocksize);

		do {

			i = fread(ciphertext2, 1, blocksize, FROMF);
			if (i != blocksize) {
				if (ferror(FROMF) != 0) {
					perror("fread");
					return 1;
				} else {	/* eof */
					if (i == 1) {
						ch = 1;
					} else {
						err_crit(_
							 ("Incorrect file type\n"));
						return 1;
					}

					if (bare_flag == FALSE) {
						memmove(&counter,
							ciphertext,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}

						/* CRC32 */
						for (j = 0; j < blocksize;
						     j++)
							((unsigned char *)
							 ciphertext)[j] =
							    ((unsigned
							      char *)
							     ciphertext)[j
									 +
									 1];
						((unsigned char *)
						 ciphertext)[blocksize -
							     1] =
						    ((unsigned char *)
						     ciphertext2)[0];
						mdecrypt(td, ciphertext);
					} else {
						memmove(&counter,
							ciphertext2,
							sizeof(counter));
						if (counter >= blocksize) {
							err_crit(_
								 ("Incorrect file type\n"));
							return 1;
						}
						mdecrypt_ecb(td, buffer,
							     blocksize);
						if (fwrite
						    (buffer,
						     (int) blocksize, 1,
						     RTOF) == 0) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);
						memmove(buffer, ciphertext,
							blocksize);
					}
				}
			}
			/* Decrypt the ciphertext */
			mdecrypt_ecb(td, buffer, blocksize);

			if (ch == 1) {
				if (counter == 0)
					break;
				if (fwrite(buffer, (int) counter, 1, RTOF)
				    == 0) {
					perror("fwrite");
					return 1;
				}
				if (flush == TRUE)
					fflush(RTOF);
				/* CRC32 */
				if (bare_flag == FALSE)
					crc32(buffer, (int) counter);

			} else {
				if (start > 0) {
					if (fwrite
					    (buffer, blocksize, 1,
					     RTOF) == 0) {
						perror("fwrite");
						return 1;
					}
					if (flush == TRUE)
						fflush(RTOF);
					/* CRC32 */
					if (bare_flag == FALSE) {
						crc32(buffer, blocksize);
					}
				}
				start++;
				memmove(buffer, ciphertext, blocksize);
				memmove(ciphertext, ciphertext2,
					blocksize);
			}


		}
		while (ch == 0);
		break;


	case MCRYPT_STREAM:
		start = 0;
		do {

			i = fread(stream_buf, 1, STREAM_BUFFER, FROMF);

			if (i == STREAM_BUFFER) {
				if (start == 1) {
					mdecrypt_stream(td,
							&tmp_buf[i - 4],
							4);
					if (bare_flag == FALSE) {
						crc32(&tmp_buf[i - 4], 4);
					}
					if (fwrite
					    (&tmp_buf[i - 4], 4, 1,
					     RTOF) == 0) {
						perror("fwrite");
						return 1;
					}
				}
				memmove(tmp_buf, stream_buf, i);

				/* Decrypt the ciphertext */
				mdecrypt_stream(td, stream_buf, i - 4);

				if (bare_flag == FALSE) {
					crc32(stream_buf, i - 4);
				}

				if (fwrite(stream_buf, i - 4, 1, RTOF) ==
				    0) {
					perror("fwrite");
					return 1;
				}

				if (flush == TRUE)
					fflush(RTOF);
				start = 1;
			} else {
				if (i >= 4) {
					mdecrypt_stream(td,
							&tmp_buf
							[STREAM_BUFFER -
							 4], 4);
					if (bare_flag == FALSE) {
						crc32(&tmp_buf
						      [STREAM_BUFFER - 4],
						      4);
					}
					if (fwrite
					    (&tmp_buf[STREAM_BUFFER - 4],
					     4, 1, RTOF) == 0) {
						perror("fwrite");
						return 1;
					}
					i -= 4;
					mdecrypt_stream(td, stream_buf, i);

					if (bare_flag == FALSE) {
						crc32(stream_buf, i);
					}

					if (fwrite(stream_buf, i, 1, RTOF)
					    == 0) {
						perror("fwrite");
						return 1;
					}
					if (bare_flag == FALSE) {
						mdecrypt_stream(td,
								&stream_buf
								[i], 4);
						memmove(&fcrc32,
							&stream_buf[i], 4);
					} else { /* bare flag */
							mdecrypt_stream(td,
							&stream_buf[i], 4);
							if (fwrite(&stream_buf[i], 4, 1, RTOF)
						    == 0) {
							perror("fwrite");
							return 1;
							}
					}
					if (flush == TRUE)
						fflush(RTOF);

				} else {	/* i=0,1,2,3 */
					mdecrypt_stream(td,
							&tmp_buf
							[STREAM_BUFFER -
							 4], i);
					if (bare_flag == FALSE) {
						crc32(&tmp_buf
						      [STREAM_BUFFER - 4],
						      i);
					}
					if (fwrite
					    (&tmp_buf[STREAM_BUFFER - 4],
					     i, 1, RTOF) == 0) {
						perror("fwrite");
						return 1;
					}
					if (bare_flag == FALSE) {
						memmove(&fcrc32,
							&tmp_buf
							[STREAM_BUFFER -
							 4 + i], 4 - i);
						memmove(&((char *) fcrc32)
							[4 - i],
							stream_buf, i);
						mdecrypt_stream(td,
								&fcrc32,
								4);
					}
				}
				ch = 1;
			}

		}
		while (ch == 0);
		break;

	case MCRYPT_CFB:

		i = fread(ciphertext, 1, blocksize, FROMF);	/* Read the IV */
		td = init_mcrypt_cfb(algorithm, keyword, len, ciphertext);

		if (td < 0) {
			err_crit(_("Unknown algorithm error\n"));
			return 1;
		}


		/* decryption starts here */
		/* fill the buffer */
		fread(buffer, 1, blocksize, FROMF);
		tmp = malloc(1);

		do {

			/* reads 1 byte of the file and stores the to ciphertext */

			i = fread(ciphertext, 1, 1, FROMF);

			if (i < 1) {
				if (ferror(FROMF) != 0) {
					perror("fread");
					return 1;
				} else {
					if (bare_flag == FALSE) {
						/* crc32 */
						mdecrypt(td, buffer);
						memmove(ciphertext, buffer,
							blocksize);
					} else {
						mdecrypt_cfb(td, buffer,
							     blocksize);
						if (fwrite
						    (buffer, 1, blocksize,
						     RTOF) < blocksize) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);
					}
				}
				ch = 1;
			}
			/* Decrypt the ciphertext */

			if (ch != 1) {
				tmp[0] = ((char *) buffer)[0];
				mdecrypt_cfb(td, &tmp[0], 1);
				rol_buf(buffer, blocksize, ciphertext);

				if (fwrite(&tmp[0], 1, 1, RTOF) == 0) {
					perror("fwrite");
					return 1;
				}
				/* CRC32 */
				if (bare_flag == FALSE)
					crc32(&tmp[0], 1);
			}
		}
		while (ch == 0);
		free(tmp);

		break;

	case MCRYPT_OFB:

		i = fread(ciphertext, 1, blocksize, FROMF);	/* Read the IV */
		td = init_mcrypt_cfb(algorithm, keyword, len, ciphertext);

		if (td < 0) {
			err_crit(_("Unknown algorithm error\n"));
			return 1;
		}


		/* decryption starts here */
		/* fill the buffer */
		fread(buffer, 1, blocksize, FROMF);
		tmp = malloc(1);

		do {

			/* reads 1 byte of the file and stores the to ciphertext */

			i = fread(ciphertext, 1, 1, FROMF);

			if (i < 1) {
				if (ferror(FROMF) != 0) {
					perror("fread");
					return 1;
				} else {
					if (bare_flag == FALSE) {
						/* crc32 */
						mdecrypt(td, buffer);
						memmove(ciphertext, buffer,
							blocksize);
					} else {
						mdecrypt_ofb(td, buffer,
							     blocksize);
						if (fwrite
						    (buffer, 1, blocksize,
						     RTOF) < blocksize) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);
					}
				}
				ch = 1;
			}
			/* Decrypt the ciphertext */

			if (ch != 1) {
				tmp[0] = ((char *) buffer)[0];
				mdecrypt_ofb(td, &tmp[0], 1);
				rol_buf(buffer, blocksize, ciphertext);

				if (fwrite(&tmp[0], 1, 1, RTOF) == 0) {
					perror("fwrite");
					return 1;
				}
				/* CRC32 */
				if (bare_flag == FALSE)
					crc32(&tmp[0], 1);
			}
		}
		while (ch == 0);
		free(tmp);

		break;


	default:
		return 1;

	}			/* switch */


/* close and unlock files */

	if (stream_flag == FALSE) {
		fflush(FROMF);
		fflush(RTOF);
		unlock(fileno((FILE *) FROMF));
		unlock(fileno((FILE *) RTOF));
	}
	fclose(RTOF);
	fclose(FROMF);

/* ready */

	switch (mode) {
	case MCRYPT_CBC:
		end_mcrypt_cbc(td);
		break;
	case MCRYPT_ECB:
		end_mcrypt_ecb(td);
		break;
	case MCRYPT_CFB:
		end_mcrypt_cfb(td);
		break;
	case MCRYPT_OFB:
		end_mcrypt_ofb(td);
		break;
	case MCRYPT_nOFB:
		end_mcrypt_nofb(td);
		break;
	case MCRYPT_STREAM:
		Bzero(stream_buf, STREAM_BUFFER);
		end_mcrypt_stream(td);
		break;
	}

	free(ciphertext_old);
	free(buffer);
	free(ciphertext_orig);

	if (bare_flag == FALSE) {
		if (is_block_algorithm(algorithm) == 1) {
			memmove(&fcrc32, ciphertext, sizeof(fcrc32));
		}
#ifndef WORDS_BIGENDIAN
		fcrc32 = byteswap(fcrc32);
#endif
	}
	free(ciphertext);

	mxfree(keyword, keysize);

	if (bare_flag == FALSE) {

		newcrc32 = get_crc32();
		if (newcrc32 == fcrc32) {
#ifdef ZIP
			if (stream_flag == FALSE
			    && (gzipflag == TRUE || bzipflag == TRUE)) {

				pid = fork();
				if (pid == 0) {
					if (gzipflag == TRUE) {
						err_warn(_
							 ("Decompressing the output file...\n"));
						i =
						    execlp(GZIP, GZIP,
							   "-d", tofile,
							   NULL);
						if (i == -1) {
							perror("exec");
							return 1;
						}

					}
					if (bzipflag == TRUE) {
						err_warn(_
							 ("Decompressing the output file...\n"));
						i =
						    execlp(BZIP2, BZIP2,
							   "-d", tofile,
							   NULL);
						if (i == -1) {
							perror("exec");
							return 1;
						}

					}
				}
#ifdef HAVE_WAITPID
				waitpid(pid, NULL, 0);
#endif
				if (bzipflag == TRUE) {
					tofile[strlen(tofile) -
					       strlen(".bz2")] = '\0';
				} else {
					tofile[strlen(tofile) -
					       strlen(".gz")] = '\0';
				}
			}
#endif
			return 0;	/* crc check passed */
		} else {
			err_crit(_("CRC32 check failed\n"));
			return 1;	/* CRC32s Do not match */
		}
	} else {		/* bare mode: no crc is stored here */
		return 0;
	}
}



int check_mode(char *chain)
{
	char *x;
	int i;

	if (strcasecmp("STREAM", chain) == 0)
		return MCRYPT_STREAM;

	for (i = 0; i < 10; i++) {
		if (is_ok_mode(i) == 0) {
			x = mcrypt_get_modes_name(i);
			if (strcasecmp(x, chain) == 0) {
				free(x);
				return i;
			}
			free(x);
		}
	}

	return -1;

}

int check_algo(char *chain)
{
	int i;
	char *y;



	for (i = 0; i < 255; i++) {
		if (is_ok_algorithm(i) == 0 || i == CRYPT
		    || i == DES_COMPAT) {
			y = get_algorithms_name_local(i);
			if (strcasecmp(y, chain) == 0) {
				free(y);
				return i;
			}
			free(y);
		}
	}

	return -1;
}




/* Rols the buffer by one byte and puts the value in the empty cell */

void rol_buf(void *buffer, int buffersize, void *value)
{
	char *buf = buffer;
	char *val = value;
	int i;

	for (i = 0; i < buffersize - 1; i++) {
		buf[i] = buf[i + 1];
	}
	buf[buffersize - 1] = val[0];
}

void print_list(void)
{
	int tmpi, tmpj;
	char *tmpc;

	fprintf(stdout, _("Supported Algorithms and modes:\n\n"));
	for (tmpi = 0; tmpi < 256; tmpi++) {
		tmpc = get_algorithms_name_local(tmpi);
		if (tmpc != NULL) {
			fprintf(stdout, " %s:", tmpc);
			free(tmpc);

			if (is_block_algorithm_local(tmpi) == TRUE) {
				for (tmpj = 0; tmpj < 256; tmpj++) {
					if (is_block_mode(tmpj) != 0) {
						tmpc =
						    mcrypt_get_modes_name
						    (tmpj);
						if (tmpc != NULL) {
							fprintf(stdout,
								" %s",
								tmpc);
							free(tmpc);
						}
					}
				}
				fprintf(stdout, "\n");
			} else {
				for (tmpj = 0; tmpj < 256; tmpj++) {
					if (is_block_mode(tmpj) == 0) {
						tmpc =
						    mcrypt_get_modes_name
						    (tmpj);
						if (tmpc != NULL) {
							fprintf(stdout,
								" %s",
								tmpc);
							free(tmpc);
						}
					}
				}
				fprintf(stdout, "\n");
			}

		}
	}

}
