/*  Mini-Crypt 2.2 program. It encrypts text or binary files using
 *  keywords. The keyword is NOT included in the encrypted file so
 *  there is no way to decrypt it, if you forgot it. 
 *  For a brief description of the algorithms read the README file and
 *  the man page.
 *                                     Nikos Mavroyanopoulos
 *                                         nmav@i-net.paiko.gr
 *
 *    Copyright (C) 1998,1999,2000 Nikos Mavroyanopoulos
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


/* $Id: mcrypt.c,v 1.25 2000/03/10 19:19:12 nikos Exp $ */

#ifndef DEFINES_H
#define DEFINES_H
#include <defines.h>
#endif
#include <mcrypt_int.h>
#include <xmalloc.h>
#include <extra.h>
#include <locks.h>
#include <keys.h>
#include <random.h>
#include <gaa.h>
#include <errors.h>
#include <environ.h>


static char rcsid[] =
    "$Id: mcrypt.c,v 1.25 2000/03/10 19:19:12 nikos Exp $";

char tmperr[128];
unsigned int stream_flag = FALSE;
char *keymode = NULL;
char *mode = NULL;
char *algorithm = NULL;
char *hash = NULL;
int keysize;
int nodelete = FALSE;		/* Delete by default */
int nolock = FALSE;		/* Use locks by default */
int noiv = FALSE;
int hash_algorithm = MHASH_MD5;
int noecho = TRUE;		/* Echo asterisks by default */
int double_check = FALSE;	/* Do not double check for passwords when decrypting */
int quiet = TRUE;		/* silent by default */
int tmpi;
int unlink_flag = FALSE;
int bare_flag = FALSE, real_random_flag = FALSE;
int cleanDelete = FALSE;
#ifdef ZIP
int gzipflag = FALSE;
int bzipflag = FALSE;
#endif
int flush = FALSE;
char *outfile = 0, *tmpc;
char *algorithms_directory = NULL;
char *modes_directory = NULL;

void usage(void)
{

	fprintf(stdout,
		_
		("Usage: mcrypt [-dLFus87bhvrx] [-f keyfile] [-k key1 key2 ...] [-m mode] [-o keymode] [-s keysize] [-a algorithm] [-c config_file] [file ...]\n\n"));

	fprintf(stdout,
		_
		("mcrypt encrypts/decrypts files using several symmetric algorithms.\n"));
	fprintf(stdout,
		_("Legal parameters are:\n" "\n"
		  " -d --decrypt                 decrypts\n"
		  " -c --config FILENAME         Use the configuration file FILENAME.\n"
		  "                              If not specified the .mcryptrc in your home\n"
		  "                              directory is used.\n"
		  " -a --algorithm ALGORITHM     specify the encryption and decryption algorithm.\n"
		  "                              Use the --list parameter to see the supported\n"
		  "                              algorithms.\n"
		  " --algorithms_directory DIR   specify the directory where the algorithms are.\n"));
	fprintf(stdout,
		_
		(" -m --mode MODE               specify the mode. Use the --list parameter\n"
		 "	                        to see the supported modes.\n"
		 " --modes_directory DIR        specify the directory where the modes are.\n"
		 " -k --key KEY1, KEY2, ...     keyword(s) to use.\n"
		 " -f --keyfile FILENAME        file to read the keyword from.\n"
		 " -o --keymode MODE            specify the keyword mode. Use the --keymodeslist\n"
		 "                               parameter to view all modes.\n"
		 " -h --hash ALGORITHM          specify the hash algorithm to be used, in order to compute\n"
		 "                               the checksum. Use the --hashlist parameter to\n"
		 "                               view all algorithms.\n"
		 " -s --keysize SIZE            Algorithm's key size. The default is the maximum\n"
		 "                               supported by the algorithm.\n"
		 " -u --unlink                  unlink the input file after encryption/decryption\n"
		 " -L --license                 displays license information.\n"
		 " -q --quiet                   suppress some non critical warnings.\n"
		 " -l --doublecheck             Double check for passphrase even if decrypting.\n"
		 "    --echo                    Echo asterisks when entering the password.\n"
		 "    --list                    Lists the supported algorithms and modes.\n"
		 "    --keymodeslist            Lists the supported key modes.\n"
		 "    --hashlist                Lists the supported hash algorithms.\n"
		 "    --flush                   Immediately flush the output.\n"
		 "    --nodelete                Does not delete the output file even if decryption\n"
		 "				fails.\n"
		 " -V --verbose                 more information is displayed.\n"
		 " -b --bare                    Do not keep algorithm information in the encrypted\n"
		 "                              file.\n"
		 " -F --force                   forces output to stdout.\n"));

#ifdef HAVE_DEV_RANDOM
	fprintf(stdout,
		_
		(" -r --random                  Use real random data.\n"));
#endif

#ifdef GZIP
	fprintf(stdout,
		_
		(" -z --gzip                    use gzip to compress files before encryption.\n"));
#endif

#ifdef BZIP2
	fprintf(stdout,
		_
		(" -p --bzip2                   use bzip2 to compress files before encryption.\n"));
#endif

#ifndef NO_FCNTL_LOCK
	fprintf(stdout,
		_(" -n --nolock                  Do not lock files.\n"));
#endif

	fprintf(stdout, _("\n"
			  " -h --help                    prints this help\n"
			  " -v --version                 prints the version number\n"
			  "\n"
			  "Report bugs to mcrypt@i-net.paiko.gr.\n\n"));
}

void mcrypt_license()
{
	fprintf(stdout,
		_("\nCopyright (C) 1998,1999,2000 Nikos Mavroyanopoulos\n"
		  "This program is free software; you can redistribute it and/or modify \n"
		  "it under the terms of the GNU General Public License as published by \n"
		  "the Free Software Foundation; either version 2 of the License, or \n"
		  "(at your option) any later version. \n" "\n"
		  "This program is distributed in the hope that it will be useful, \n"
		  "but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
		  "MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
		  "GNU General Public License for more details. \n" "\n"
		  "You should have received a copy of the GNU General Public License \n"
		  "along with this program; if not, write to the Free Software \n"
		  "Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n"));
}

int check_hash_algo(char *chain)
{
	int i;
	char *y;

	for (i = 0; i < 255; i++) {
		y = mhash_get_hash_name(i);
		if (y != NULL) {
			_tolow(y, strlen(y));
			if (strcmp(y, chain) == 0) {
				xfree(y);
				return i;
			}
			xfree(y);
		}
	}
	return -1;
}


#ifdef LIBMCRYPT_22
int check_algo(char *chain)
{
	int i;
	char *y;

	for (i = 0; i < 255; i++) {
		if (is_ok_algorithm(i) == 0) {
			y = mcrypt_get_algorithms_name(i);
			if (strcmp(y, chain) == 0) {
				xfree(y);
				return i;
			}
			xfree(y);
		}
	}
	return -1;
}

int check_mode(char *chain)
{
	int i;
	char *y;

	for (i = 0; i < 255; i++) {
		if (is_ok_mode(i) == 0) {
			y = mcrypt_get_modes_name(i);
			if (strcmp(y, chain) == 0) {
				xfree(y);
				return i;
			}
			xfree(y);
		}
	}
	return -1;
}
#endif

void mcrypt_version()
{

	fprintf(stderr, _("Mcrypt v.%s (%s-%s-%s)\n"), VERSION, T_CPU,
		T_VENDOR, T_OS);
	fprintf(stderr, _("Linked against libmcrypt v.%s\n"),
		LIBMCRYPT_VERSION);
	fprintf(stderr,
		_
		("Copyright (C) 1998,1999,2000 Nikos Mavroyanopoulos (nmav@hellug.gr)\n"));

}

/*                              The main    */
int main(int argc, char **argv)
{

	short int ein = 0, din = 0, kin = 0, force = 0, return_val = 0;
	char *einfile = 0, *dinfile = 0, *keyword = 0;
	char **file, **keyfile = NULL, *cfile;
	int x, y, i, file_count = 0, keys = 0, used_algo = FALSE, v, *siz;
	gaainfo info;

#ifdef HAVE_SIGNAL_H
	Signal(SIGINT, shandler);
	Signal(SIGQUIT, shandler);
	Signal(SIGSEGV, shandler);
	Signal(SIGPIPE, shandler);
	Signal(SIGTERM, shandler);
	Signal(SIGHUP, shandler);
	Signal(SIGUSR1, SIG_IGN);
	Signal(SIGUSR2, SIG_IGN);
	Signal(SIGALRM, SIG_IGN);
#endif

#ifdef ENABLE_NLS
	setlocale(LC_MESSAGES, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);
#endif



	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(_("Error in the arguments.\n"));
	}
	if (info.config_file != NULL) {
		cfile = xmalloc(strlen(info.config_file));
		strcpy(cfile, info.config_file);
		if (check_file(cfile) != 1) {
			err_crit(_
				 ("Warning: Cannot access config file.\n"));
		}
	} else {
#ifdef HAVE_GETPWUID
		cfile = get_cfile(getuid(), ".mcryptrc");
#else
		cfile = xmalloc(11);
		strcpy(cfile, ".mcryptrc");
#endif
	}

	/* If config file exists */
	if (check_file(cfile) == 1) {
		if ((v = gaa_file(cfile, &info)) != -1) {
			/* gets options from file 'config' */
			err_quit(_("Error in the configuration file.\n"));
		}
	}
	xfree(cfile);


	if (check_env() == TRUE) {

		if (get_env_key() != NULL) {
			info.keylen = 1;
			xfree(info.keys);
			info.keys = get_env_key();
		}
		if (get_env_algo() != NULL) {
			xfree(info.algorithm);
			info.algorithm = get_env_algo();
		}
		if (get_env_mode() != NULL) {
			xfree(info.mode);
			info.mode = get_env_mode();
		}
		if (get_env_bit_mode() != 0) {
			info.kmode = get_env_bit_mode();
		}
	}
/* Check again the command line variables 
 * This will be uncommented when gaa is corrected.
 */

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(_("Error in the arguments.\n"));
	}
/* Examine how we were called */
	if (strstr(argv[0], "decrypt") != NULL) {
		Bzero(argv[0], strlen(argv[0]));
		strcpy(argv[0], "mdecrypt");
		din = TRUE;
		ein = FALSE;
	} else {
		if (strlen(argv[0]) > 5) {
			Bzero(argv[0], strlen(argv[0]));
			strcpy(argv[0], "mcrypt");
		}
		din = FALSE;
		ein = TRUE;	/* It will change by the arguments */
	}


	/* File pointers are as much as the file arguments */

	file = xmalloc((info.size) * sizeof(char *));

	if (info.ed_specified != 0) {
		din = info.din;
		ein = info.ein;
	}
	if (info.real_random_flag == TRUE) {
#ifdef HAVE_DEV_RANDOM
		real_random_flag = TRUE;
#else
		err_warn(_
			 ("Warning: This system does not support real random data.\n"));
#endif
	}
	force = info.force;
	bare_flag = info.bare_flag;
	unlink_flag = info.unlink_flag;
	quiet = info.quiet;
	nolock = info.nolock;
	noecho = info.noecho;
	noiv = info.noiv;
	double_check = info.double_check;
	flush = info.flush;
	nodelete = info.nodelete;
#ifdef LIBMCRYPT_22
	keysize = 0;
#else
	keysize = info.keysize;
#endif
	algorithms_directory = info.algorithms_directory;
	modes_directory = info.modes_directory;
#ifdef ZIP
	gzipflag = info.gzipflag;
	bzipflag = info.bzipflag;
#endif

	if (info.kmode != NULL) {
		keymode = info.kmode;
	} else {
		keymode = strdup(DEFAULT_KEYMODE);	/* Default */
	}

	if (info.hash != NULL) {
		hash = xmalloc(strlen(info.hash));
		strcpy(hash, info.hash);
		hash_algorithm = check_hash_algo(hash);

		if (hash_algorithm < 0) {
			hash_algorithm = MHASH_CRC32;
			fprintf(stderr,
				_
				("The '%s' hash algorithm was not found.\n"),
				hash);
			fprintf(stderr,
				_
				("Use the --hashlist parameter to view all supported hashes.\n"));
			exit(1);
		}
	} else {
		hash = xmalloc(strlen("crc32") + 1);
		strcpy(hash, "crc32");
	}

	if (info.keyfile != NULL) {
		kin = 2;
		keyfile = read_key_file(info.keyfile, &keys);
		if (keyfile == NULL)
			kin = 0;
		if (keys == 0)
			kin = 0;
	} else {
		if (info.keys != NULL) {
			kin = 2;
			keyfile = xmalloc(info.keylen * sizeof(char *));
			keys = info.keylen;

			for (i = 0; i < info.keylen; i++) {
				keyfile[i] =
				    secure_xmalloc(strlen(info.keys[i]) +
						   10);
				strcpy(keyfile[i], info.keys[i]);
			}
		}
	}

	if (info.mode != NULL) {
		mode = xmalloc(strlen(info.mode) + 1);
		strcpy(mode, info.mode);
	} else {
		mode = strdup(DEFAULT_MODE);
	}
	if (info.algorithm != NULL) {
		used_algo = TRUE;
		algorithm = xmalloc(strlen(info.algorithm) + 1);
		strcpy(algorithm, info.algorithm);
#ifndef LIBMCRYPT_22
		i = mcrypt_is_block_algorithm(algorithm, algorithms_directory);

		if (i < 0) {
			fprintf(stderr, _("Error in algorithm '%s'.\n"),
				algorithm);
			exit(1);
		}

		if (i == 0) {
			y = mcrypt_is_block_algorithm_mode(mode, modes_directory);
			if (y >= 0) {
				mode =
				    xrealloc(mode, strlen("stream") + 1);
				strcpy(mode, "stream");
			} else {
				fprintf(stderr, _("Error in mode '%s'.\n"),
					mode);
				exit(1);
			}
		}

#else
		i = check_algo(algorithm);
		if (i < 0) {
			fprintf(stderr, _("Error in algorithm '%s'.\n"),
				algorithm);
			exit(1);
		}
		y = check_mode(mode);
		if (y < 0) {
			fprintf(stderr, _("Error in mode '%s'.\n"), mode);
			exit(1);
		}
		if (is_block_algorithm(i) == 0) {
			if (is_block_mode(y) != 0) {
				mode =
				    xrealloc(mode, strlen("stream") + 1);
				strcpy(mode, "stream");
			}
		}
#endif
	} else {
		algorithm = strdup(DEFAULT_ALGORITHM);
	}

	if (info.mode != NULL) {
#ifndef LIBMCRYPT_22
		y = mcrypt_is_block_algorithm_mode(mode, modes_directory);
		if (y < 0) {
			fprintf(stderr, _("Error in mode '%s'.\n"), mode);
			exit(1);
		}
#else
		y = check_mode(mode);
		if (y < 0) {
			fprintf(stderr, _("Error in mode '%s'.\n"), mode);
			exit(1);
		}
#endif
	}

#ifdef LIBMCRYPT_22
	i = check_algo(algorithm);
	i = keysize = mcrypt_get_key_size(i);

#endif

	if (info.keysize != 0) {
#ifndef LIBMCRYPT_22
		if (info.keysize > mcrypt_get_algo_key_size(algorithm, algorithms_directory)) {
			err_quit(_("The specified key size is too large for this algorithm.\n"));
		}

		siz = mcrypt_get_algo_supported_key_sizes( algorithm, algorithms_directory, &i);
		x=0;
		for (y=0;y<i;y++) {
			if (siz[y]==info.keysize) {
				x=1;
				break;
			}
		}
		if (x==0) {
			err_quit(_("The specified key size not valid for this algorithm.\n"));
		}
		free(siz);
#else
		err_warn(_("The key size parameter is ignored in that version of mcrypt.\n"));
#endif

	}

	if (info.keylen != 0 && check_env() == FALSE) {
		err_warn(_
			 ("Warning: It is insecure to specify keywords in the command line\n"));
	}
#ifdef HAVE_UMASK
	umask(066);
#endif

/* For RAND32... Called only here */
#ifndef HAVE_DEV_RANDOM
	if (quiet <= TRUE)
		err_warn(_("Warning: Using pseudo random numbers\n"));
	srand(time(0) * getpid());
#endif


/* Normal startup */


/* ok now how many files were specified? */
	i = 0;
	file_count += (info.size);


	if (file_count == 0) {
		stream_flag = TRUE;
	}
	while (i < info.size) {
		file[i] = info.input[i];
		i++;
	}


	if (stream_flag == TRUE)
		file_count++;

/* Do as many times as many files we got */
	for (i = 0; i < file_count; i++) {

		if (i != 0) {
			if (outfile != NULL)
				xfree(outfile);
		}

		/* If keyword file specified choose the i-th keyword */

		if (kin == 2 && i <= (keys - 1)) {
			keyword = keyfile[i];
			if (i != 0) {
				secure_xfree(keyfile[i - 1],
					     strlen(keyfile[i - 1]));	/* Free previous keyword */
			}
		}
#ifdef HAVE_STAT
		if (stream_flag == FALSE) {
			if (is_normal_file(file[i]) == FALSE) {
				sprintf(tmperr,
					_
					("%s: %s is not a regular file. Skipping...\n"),
					argv[0], file[i]);
				err_crit(tmperr);
				outfile = NULL;
				continue;	/* next */
			}
		}
#endif

#ifdef ZIP
		if (stream_flag == TRUE) {
			if (gzipflag == TRUE || bzipflag == TRUE) {
				err_quit(_
					 ("--bzip2 or --gzip parameters cannot be used when dealing with streams.\n"));
			}
		}
#endif

		/* Check how we were called */
		if (din == TRUE) {	/* decryption */
			if (stream_flag != TRUE)
				dinfile = file[i];
			if ((isatty(fileno((FILE *) (stdin))) == 1)
			    && (stream_flag == TRUE) && (force == 0)) {	/* not a tty */
				sprintf(tmperr,
					_
					("%s: Encrypted data will not be read from a terminal.\n"),
					argv[0]);
				err_crit(tmperr);
				err_quit(_
					 ("Redirect the input instead.\nUse the --help parameter for more help.\n"));
			}
		} else {	/* encryption */
			if (stream_flag != TRUE)
				einfile = file[i];
			if ((isatty(fileno((FILE *) (stdout))) == 1)
			    && (stream_flag == TRUE) && (force == 0)) {	/* not a tty */
				sprintf(tmperr,
					_
					("%s: Encrypted data will not be written to a terminal.\n"),
					argv[0]);
				err_crit(tmperr);
				err_quit(_
					 ("Redirect the output instead.\nUse the --help parameter for more help.\n"));
			}
		}


		/* If no streams make the extensions */
		if (stream_flag != TRUE) {

			if (din == TRUE) {

				y = strlen(dinfile);
				/* If the file has the .nc suffix, then remove it */

				if ((dinfile[y - 1] == 'c'
				     && dinfile[y - 2] == 'n'
				     && dinfile[y - 3] == '.')) {

					outfile = xcalloc(y - 2, 1);
					strncpy(outfile, dinfile, y - 3);
#ifdef HAVE_STAT
					/* But if it exists exit */
					if (check_file(outfile) != 0) {
						cleanDelete = FALSE;
						if (ask_overwrite
						    (argv[0],
						     outfile) == FALSE) {
							continue;
						} else {
							cleanDelete = TRUE;
						}
					} else {
						cleanDelete = TRUE;
					}
#endif
				} else {	/* append .dc */

					cleanDelete = TRUE;
					outfile = xcalloc(y + 5, 1);
					strncpy(outfile, dinfile, y);
					strcat(outfile, ".dc");


				}
			} else {	/* encryption- append .nc */
				outfile =
				    xcalloc(strlen(einfile) + 5 + 4, 1);
				strcpy(outfile, einfile);
				/* if file has already the .nc ignore it */
				if (strstr(outfile, ".nc") != NULL) {
					sprintf(tmperr,
						_
						("%s: file %s has the .nc suffix... skipping...\n"),
						argv[0], outfile);
					err_crit(tmperr);
					continue;
				}
#ifdef ZIP
				if (stream_flag == FALSE) {
					if (gzipflag == TRUE)
						strcat(outfile, ".gz");
					if (bzipflag == TRUE)
						strcat(outfile, ".bz2");
				}
#endif
				strcat(outfile, ".nc");
#ifdef HAVE_STAT
				/* But if it exists exit */
				if (check_file(outfile) != 0) {
					cleanDelete = FALSE;
					if (ask_overwrite(argv[0], outfile)
					    == FALSE) {
						continue;
					} else {
						cleanDelete = TRUE;
					}

				} else {
					cleanDelete = TRUE;
				}
#endif

			}
		} else {	/* if streams */
			outfile = xmalloc(7);
			strcpy(outfile, "stdout");
		}



/* Decrypt */
		if (din == TRUE) {
			x =
			    decrypt_general(algorithm, dinfile, outfile,
					    keyword);

			if (x == 0) {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was decrypted.\n"),
						dinfile);
					err_warn(tmperr);
				} else {
					sprintf(tmperr,
						_
						("Stdin was decrypted.\n"));
					err_warn(tmperr);
				}
#ifdef HAVE_STAT
# ifdef HAVE_UTIME_H
				if (stream_flag == FALSE)
					copyDate(dinfile, outfile);
# endif
#endif

				if (unlink_flag == TRUE
				    && stream_flag == FALSE) x =
					    unlink(dinfile);
				if (x != 0)
					perror("unlink");

			} else {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was NOT decrypted successfully.\n"),
						dinfile);
					err_crit(tmperr);
					if (x != -1) {
						if (nodelete == FALSE)
							remove(outfile);
					} else {
						x = 1;
					}
				} else {
					err_crit(_
						 ("Stdin was NOT decrypted successfully.\n"));
				}
			}

			return_val += x;
		}

/* Encrypt */
		if (ein == TRUE) {
			x =
			    encrypt_general(algorithm, einfile, outfile,
					    keyword);

			if (x == 0) {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was encrypted.\n"),
						einfile);
					err_warn(tmperr);
				} else {
					sprintf(tmperr,
						_
						("Stdin was encrypted.\n"));
					err_warn(tmperr);
				}
#ifdef HAVE_STAT
#ifdef HAVE_UTIME_H
				if (stream_flag == FALSE)
					copyDate(einfile, outfile);
#endif
#endif
				if (unlink_flag == TRUE
				    && stream_flag == FALSE) x =
					    unlink(einfile);
				if (x != 0)
					perror("unlink");

			} else {
				if (stream_flag == FALSE) {
					sprintf(tmperr,
						_
						("File %s was NOT encrypted successfully.\n"),
						einfile);
					err_crit(tmperr);
					if (x != -1) {
						if (nodelete == FALSE)
							remove(outfile);
					} else {
						x = 1;
					}
				} else {
					err_crit(_
						 ("Stdin was NOT encrypted successfully.\n"));
				}

			}


			return_val += x;
		}
	}			/* the main loop */

/* Clear the last keyword used */
	if (keyword != NULL)
		Bzero(keyword, strlen(keyword));
	xfree(keyword);

/* Clear ALL arguments in the command line */
	for (i = 0; i < argc; i++) {
		Bzero(argv[i], strlen(argv[i]));
	}

	if (return_val != 0) {
		return 1;
	} else {
		return 0;
	}

}


int make_mult(int num, int d)
{
	int i = 0;

	while (i < num) {
		i += d;
	}
	return i;
}



/* General Encryption */

int
encrypt_general(char *algorithm, char *fromfile, char *tofile, char *key)
{
	int full_blocks = 0, rest_blocks = 0;
	char *fcrc32 = NULL;
	word32 *keyword;
#ifdef LIBMCRYPT_22
	int algo, _mode;
#endif
	hashid tm = 0;
	int i = 0, how = 0;
	unsigned int len = 0;
	FILE *FROMF;
	int keylen, salt_size;
	int blocksize;
	int td, tc, pad_size;
	char *command = NULL;
	keygenid ki;
	word32 *IV = NULL;
	word32 *salt = NULL;
	int hash_size = mhash_get_block_size(hash_algorithm);

	word8 *ciphertext, *ciphertext_pad = NULL;
	FILE *TOF = NULL;

#ifndef LIBMCRYPT_22
	td =
	    mcrypt_module_open(algorithm, algorithms_directory, mode,
			       modes_directory);

	if (td < 0) {
		mcrypt_perror(td);
		return 1;
	}
#else
	algo = check_algo(algorithm);
	_mode = check_mode(mode);
	if (algo < 0 || _mode < 0)
		return 1;

#endif

#ifndef LIBMCRYPT_22
	if (keysize == 0)
		keysize = mcrypt_get_key_size(td);

	keylen = keysize;
	blocksize = mcrypt_get_block_size(td);
#else
	if (keysize == 0)
		keysize = mcrypt_get_key_size(algo);

	keylen = keysize;
	blocksize = mcrypt_get_block_size(algo);
#endif


	ciphertext = xmalloc(blocksize * BUFFER_SIZE);

	if (bare_flag == FALSE) {
		salt = xmalloc(SALT_SIZE);	/* 20 bytes salt */

		/* Fill the salt with random data */
		for (i = 0; i < SALT_SIZE / sizeof(word32); i++) {
			if (real_random_flag == TRUE) {

				salt[i] = SRAND32;
			} else {
				salt[i] = RAND32;
			}
		}
	}
#ifndef LIBMCRYPT_22
	IV = secure_xmalloc(mcrypt_get_iv_size(td));
/* Random number generator */
	for (i = 0; i < mcrypt_get_iv_size(td) / sizeof(word32); i++) {
#else
	IV = secure_xmalloc(mcrypt_get_block_size(algo));
/* Random number generator */
	for (i = 0; i < mcrypt_get_block_size(algo) / sizeof(word32); i++) {
#endif
		if (noiv == FALSE) {
			if (real_random_flag == TRUE) {
				IV[i] = SRAND32;
			} else {
				IV[i] = RAND32;
			}
		} else {
			IV[i] = 0;
		}
	}

#ifdef DEBUG
	fprintf(stderr, "IV: ");
	for (i = 0; i < blocksize; i++) {
		fprintf(stderr, "%.2x.", ((unsigned char *) IV)[i]);
	}
	fprintf(stderr, "\n");
#endif

/* open files */
	if (stream_flag == TRUE) {
		FROMF = stdin;
		TOF = stdout;
	} else {
#ifdef ZIP
		if (gzipflag == FALSE && bzipflag == FALSE) {
#endif
			FROMF = fopen(fromfile, "rb");
			if (FROMF == NULL) {
				perror("fopen");
				return 1;
			}
			if (read_lock(fileno((FILE *) FROMF)) == -1)
				return 1;

			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;
#ifdef ZIP
		} else {
			/* command if using gzip */
			if (gzipflag == TRUE) {
				command =
				    malloc(strlen(GZIP) +
					   strlen(fromfile) + 5);
				strcpy(command, GZIP);
			}
			if (bzipflag == TRUE) {
				command =
				    malloc(strlen(BZIP2) +
					   strlen(fromfile) + 5);
				strcpy(command, BZIP2);
			}
			strcat(command, " -c ");
			strcat(command, fromfile);
			FROMF = popen(command, "r");

			if (FROMF == NULL) {
				perror("popen");
				return 1;
			}
			TOF = fopen(tofile, "wb");
			if (TOF == NULL) {
				perror("fopen");
				return -1;
			}
			if (write_lock(fileno((FILE *) TOF)) == -1)
				return -1;
		}
#endif
	}



	ki = _which_algo(keymode);
	if (ki == -1) {
		fprintf(stderr, _("Error in keymode '%s'.\n"), keymode);
		return 1;
	}
	if (mhash_keygen_uses_salt(ki) == 1) {
		salt_size = mhash_get_keygen_salt_size(ki);
		if (salt_size == 0)
			salt_size = SALT_SIZE;
	} else {
		salt_size = 0;
	}

	if (bare_flag == FALSE) {
		write_file_head(TOF, algorithm, mode, keymode, &keysize,
				salt, salt_size);
	}
#ifndef LIBMCRYPT_22
	if (mcrypt_mode_has_iv(td) == 1 && noiv == FALSE) {
		write_iv(TOF, IV, mcrypt_get_iv_size(td));
	}
#else
	if (mcrypt_mode_has_iv(_mode) == 1 && noiv == FALSE) {
		write_iv(TOF, IV, mcrypt_get_block_size(algo));
	}
#endif

	if (quiet == FALSE)
		fprintf(stderr,
			_
			("Algorithm: %s\nKey size: %d\nMode: %s\nKeyword mode: %s\nDigest: %s\n"),
			algorithm, keysize, mode, keymode, hash);

	if (key == NULL) {
		if (stream_flag == FALSE) {
			sprintf(tmperr, _("File: %s\n"), fromfile);
			err_warn(tmperr);
		}
	}
/* Get keyword */
	if (key == NULL) {
		keyword =
		    fixkey(NULL, &len, keymode, keysize, quiet,
			   stream_flag, salt, salt_size, ENCRYPT);
	} else {
		len = strlen(key);
		keyword =
		    fixkey(key, &len, keymode, keysize, quiet,
			   stream_flag, salt, salt_size, ENCRYPT);
	}
	if (keyword == NULL)
		return 1;



#ifndef LIBMCRYPT_22
	tc = mcrypt_generic_init(td, keyword, len, IV);
	if (tc < 0) {
		mcrypt_perror(tc);
		return 1;
	}
#else
	td = mcrypt_generic_init(_mode, algo, keyword, len, IV);
	if (td < 0) {
		return 1;
	}
#endif

	if (bare_flag == FALSE)
		tm = mhash_init(hash_algorithm);


/* Encryption Starts here */
	/* read the first n bytes of the file and store to ciphertext */
	for (;;) {
		how = fread(ciphertext, 1, blocksize * BUFFER_SIZE, FROMF);

		if (how < BUFFER_SIZE * blocksize) {
			if (ferror(FROMF) != 0) {
				perror("fread");
				return 1;
			}
			if (bare_flag == FALSE)
				mhash(tm, ciphertext, how);

			if (is_block_mode(td) == 1) {
				rest_blocks = how % blocksize;
				full_blocks = how / blocksize;

				mcrypt_generic(td, ciphertext,
					       full_blocks * blocksize);

				if (fwrite
				    (ciphertext, 1,
				     (full_blocks * blocksize),
				     TOF) < (full_blocks * blocksize)) {
					perror("fwrite");
					return 1;
				}
			} else {	/* stream mode */
				mcrypt_generic(td, ciphertext, how);
				if (fwrite(ciphertext, 1, how, TOF) < how) {
					perror("fwrite");
					return 1;
				}
			}
			if (flush == TRUE)
				fflush(TOF);
			break;
		}
		/* crc32 */
		if (bare_flag == FALSE)
			mhash(tm, ciphertext, how);
		mcrypt_generic(td, ciphertext, how);

		if (fwrite(ciphertext, 1, how, TOF) < blocksize) {
			perror("fwrite");
			return 1;
		}
		if (flush == TRUE)
			fflush(TOF);
	}

	if (bare_flag == FALSE)
		fcrc32 = mhash_end(tm);

	if (bare_flag == FALSE) {
		pad_size =
		    make_mult(rest_blocks + hash_size + 1, blocksize);

		/* in case of a stream cipher no padding is needed */
		if (is_block_mode(td) == 0) {
			pad_size = hash_size;
		}
	} else {		/* bare_flag==TRUE */
		pad_size = blocksize;
		if (is_block_mode(td) == 0) {
			pad_size = 0;
		}
	}

	if (pad_size > 0)
		ciphertext_pad = xmalloc(pad_size);

	if (is_block_mode(td) == 1) {
		memmove(ciphertext_pad,
			&ciphertext[full_blocks * blocksize], rest_blocks);
		if (bare_flag == FALSE) {
			memmove(&ciphertext_pad[rest_blocks], fcrc32,
				hash_size);
			ciphertext_pad[pad_size - 1] =
			    (unsigned char) blocksize - (pad_size -
							 hash_size -
							 rest_blocks);
		} else {
			ciphertext_pad[pad_size - 1] =
			    (unsigned char) rest_blocks;
		}
	} else {
		if (bare_flag == FALSE)
			memmove(ciphertext_pad, fcrc32, hash_size);
	}

	if (pad_size > 0) {
		mcrypt_generic(td, ciphertext_pad, pad_size);
		if (fwrite(ciphertext_pad, 1, pad_size, TOF) < pad_size) {
			perror("fwrite");
			return 1;
		}
	}


	if (flush == TRUE)
		fflush(TOF);

	/* if bare_flag==FALSE */
	/* End of copy */
	/* close and unlock files */
	fflush(TOF);
	unlock(fileno((FILE *) TOF));

	if (stream_flag == FALSE) {
		fflush(FROMF);
		fflush(TOF);

#ifdef ZIP
		if (gzipflag == FALSE && bzipflag == FALSE)
#endif
			unlock(fileno((FILE *) FROMF));
		unlock(fileno((FILE *) TOF));
	}
	fclose(TOF);

#ifdef ZIP
	if (gzipflag == FALSE && bzipflag == FALSE) {
#endif
		fclose(FROMF);
#ifdef ZIP
	} else {
		pclose(FROMF);
	}
#endif
/* Ready */

	mcrypt_generic_end(td);

	if (bare_flag == FALSE) {
		xfree(salt);
	}

	secure_xfree(keyword, keylen);
	xfree(ciphertext);


	return 0;
}



/* General Decryption */

int
decrypt_general(char *algorithm, char *fromfile, char *tofile, char *key)
{
	char *fcrc32, *newcrc32;
	hashid tm = 0;
	char tmp_buf[BUFFER_SIZE];
	int how = 0;
	int i = 0, j;
	int blocksize, crcsize, salt_size;
	int td, pid, buf_block, start;
	word32 *IV = NULL;
	word8 *ciphertext;
	word8 *ciphertext_old, *ciphertext_p_old;
	word32 *keyword;
	word32 *salt;
	char local_algorithm[50];
	char local_mode[50];
	char local_keymode[50];
	char local_salt[100];
	unsigned int len = 0;
	int hash_size, pad_size;
#ifdef LIBMCRYPT_22
	int algo, _mode;
#endif
	FILE *RTOF;
	FILE *FROMF;


/* open files */
	if (stream_flag == TRUE) {
		FROMF = stdin;
		RTOF = stdout;
	} else {
		FROMF = fopen(fromfile, "rb");
		if (FROMF == NULL) {
			perror("fopen");
			return 1;
		}
		if (read_lock(fileno((FILE *) FROMF)) == -1)
			return 1;

		RTOF = fopen(tofile, "wb");
		if (RTOF == NULL) {
			perror("fopen");
			return -1;
		}
		if (write_lock(fileno((FILE *) RTOF)) == -1)
			return -1;

	}


	if (bare_flag == FALSE) {
		salt = xmalloc(SALT_SIZE);
		if (check_file_head
		    (FROMF, local_algorithm, local_mode, local_keymode,
		     &keysize, local_salt, &salt_size) != 0) {
			err_crit(_("No valid file headers found.\n"));
			return 1;
		}
		salt = xmalloc(salt_size);
		memmove(salt, local_salt, salt_size);
		algorithm = xmalloc(strlen(local_algorithm)+1);
		strcpy(algorithm, local_algorithm);
		mode = xmalloc(strlen(local_mode)+1);
		strcpy(mode, local_mode);
		keymode = xmalloc(strlen(local_keymode)+1);
		strcpy(keymode, local_keymode);
	} else {
		salt = NULL;
	}

	hash_size = mhash_get_block_size(hash_algorithm);

#ifdef LIBMCRYPT_22
	algo = check_algo(algorithm);
	_mode = check_mode(mode);
	if (algo < 0 || _mode < 0)
		return 1;

	if (keysize == 0)
		keysize = mcrypt_get_key_size(algo);

	blocksize = mcrypt_get_block_size(algo);

#else
	td =
	    mcrypt_module_open(algorithm, algorithms_directory, mode,
			       modes_directory);
	if (td < 0) {
		mcrypt_perror(td);
		return 1;
	}

	if (keysize == 0)
		keysize = mcrypt_get_key_size(td);


	blocksize = mcrypt_get_block_size(td);

#endif

	ciphertext = xmalloc(blocksize * BUFFER_SIZE);
	ciphertext_old = xmalloc(blocksize);

	if (quiet == FALSE)
		fprintf(stderr,
			_
			("Algorithm: %s\nKeysize: %d\nMode: %s\nKeyword mode: %s\n"),
			algorithm, keysize, mode, keymode);



	if (quiet <= TRUE && stream_flag == FALSE) {
		sprintf(tmperr, _("File: %s\n"), fromfile);
		err_warn(tmperr);
	}
/* Get key */

	if (key == NULL) {
		keyword =
		    fixkey(NULL, &len, keymode, keysize, quiet,
			   stream_flag, salt, salt_size, DECRYPT);
	} else {
		len = strlen(key);
		keyword =
		    fixkey(key, &len, keymode, keysize, quiet,
			   stream_flag, salt, salt_size, DECRYPT);
	}
	if (keyword == NULL)
		return 1;

#ifndef LIBMCRYPT_22
	if (mcrypt_mode_has_iv(td) == 1) {
		if (noiv == FALSE) {
			IV = read_iv(FROMF, mcrypt_get_iv_size(td));
		} else {
			IV = xcalloc(1, mcrypt_get_iv_size(td));
		}
	} else {
		IV = NULL;
	}
#else
	if (mcrypt_mode_has_iv(_mode) == 1) {
		if (noiv == FALSE) {
			IV = read_iv(FROMF, mcrypt_get_block_size(algo));
		} else {
			IV = xcalloc(1, mcrypt_get_block_size(algo));
		}
	} else {
		IV = NULL;
	}
#endif
#ifdef DEBUG
	fprintf(stderr, "IV: ");
	for (i = 0; i < mcrypt_get_iv_size(td); i++) {
		fprintf(stderr, "%.2x.", ((unsigned char *) IV)[i]);
	}
	fprintf(stderr, "\n");
#endif


#ifndef LIBMCRYPT_22
	j = mcrypt_generic_init(td, keyword, len, IV);
	if (j < 0) {
		mcrypt_perror(j);
		return 1;
	}
#else
	td = mcrypt_generic_init(_mode, algo, keyword, len, IV);
	if (td < 0) {
		return 1;
	}
#endif

	if (bare_flag == FALSE) {
		tm = mhash_init(hash_algorithm);
	}

	crcsize = hash_size;
	fcrc32 = xmalloc(crcsize);

	/* decryption starts here */

	if (bare_flag == FALSE) {
		pad_size = make_mult(blocksize + hash_size, blocksize);

		/* in case of a stream cipher no padding is needed */
		if (is_block_mode(td) == 0) {
			pad_size = hash_size;
		}

		ciphertext_p_old = xmalloc(pad_size);
	} else {
		pad_size = 0;
		ciphertext_p_old = NULL;
	}

	if (is_block_mode(td) == 1) {
		if (bare_flag == FALSE) {
			how =
			    fread(ciphertext, 1, blocksize * BUFFER_SIZE,
				  FROMF);
			if (pad_size > how)
				pad_size -= blocksize;
			how -= pad_size;
			memmove(ciphertext_p_old, &ciphertext[how],
				pad_size);
			mdecrypt_generic(td, ciphertext, how);

			for (;;) {

				if (how !=
				    blocksize * BUFFER_SIZE - pad_size) {
					if (ferror(FROMF) != 0) {
						perror("fread");
						return 1;
					}

					if (how >= 0) {
						if (how > 0) {
							mhash(tm,
							      ciphertext,
							      how);
							fwrite(ciphertext,
							       1, how,
							       RTOF);
						}
						mdecrypt_generic(td,
								 ciphertext_p_old,
								 pad_size);

						how =
						    blocksize -
						    ciphertext_p_old
						    [pad_size - 1];
						if (how > blocksize) {
							err_warn(_
								 ("Corrupted file.\n"));
							return 1;
						}

						mhash(tm, ciphertext_p_old,
						      pad_size - how -
						      hash_size);
						fwrite(ciphertext_p_old, 1,
						       pad_size - how -
						       hash_size, RTOF);
						if (flush == TRUE)
							fflush(RTOF);

						memmove(fcrc32,
							&ciphertext_p_old
							[pad_size - how -
							 hash_size],
							hash_size);
					} else {
						fprintf(stderr,
							_
							("Unexpected error\n"),
							how);
						return 1;
					}
					break;
				}

				mhash(tm, ciphertext, how);

				if (fwrite(ciphertext, 1, how, RTOF) == 0) {
					perror("fwrite");
					return 1;
				}

				if (flush == TRUE)
					fflush(RTOF);

				memmove(ciphertext, ciphertext_p_old,
					pad_size);
				how =
				    fread(&ciphertext[pad_size], 1,
					  blocksize * BUFFER_SIZE -
					  pad_size, FROMF);
				mdecrypt_generic(td, ciphertext, how);
				memmove(ciphertext_p_old, &ciphertext[how],
					pad_size);

			}
		} else {	/* bare flag == TRUE */

			buf_block = BUFFER_SIZE * blocksize;
			start = 0;
			for (;;) {

				how =
				    fread(ciphertext, 1, buf_block, FROMF);

				if (how == buf_block && start == 1) {
					if (fwrite
					    (ciphertext_old, 1, blocksize,
					     RTOF) == 0) {
						perror("fwrite");
						return 1;
					}
					if (flush == TRUE)
						fflush(RTOF);

				}

				mdecrypt_generic(td, ciphertext, how);

				if (how != buf_block) {
					if (ferror(FROMF) != 0) {
						perror("fread");
						return 1;
					}

					if (how % blocksize != 0) {
						err_crit(_
							 ("Corrupted file.\n"));
						return 1;
					}
					if (how == 0) {
						if (start != 0) {
							if (fwrite
							    (ciphertext_old,
							     1,
							     ciphertext_old
							     [blocksize -
							      1],
							     RTOF) !=
							    ciphertext_old
							    [blocksize -
							     1]) {
								perror
								    ("fwrite");
								return 1;
							}
						}
						if (flush == TRUE)
							fflush(RTOF);

					} else {	/* how > 0 > blocksize */
						if (start != 0) {
							if (fwrite
							    (ciphertext_old,
							     1, blocksize,
							     RTOF) == 0) {
								perror
								    ("fwrite");
								return 1;
							}
						}
						if (fwrite
						    (ciphertext, 1,
						     how - blocksize,
						     RTOF) !=
						    how - blocksize) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);

						if (fwrite
						    (&ciphertext
						     [how - blocksize], 1,
						     ciphertext[how - 1],
						     RTOF) !=
						    ciphertext[how - 1]) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);

					}
					break;
				}

				memmove(ciphertext_old,
					&ciphertext[buf_block - blocksize],
					blocksize);

				start = 1;

				if (fwrite
				    (ciphertext, 1, buf_block - blocksize,
				     RTOF) == 0) {
					perror("fwrite");
					return 1;
				}

				if (flush == TRUE)
					fflush(RTOF);

			}

		}
	} else {		/* stream */
		if (bare_flag == FALSE) {
			for (;;) {
				how =
				    fread(ciphertext, 1, BUFFER_SIZE,
					  FROMF);
				mdecrypt_generic(td, ciphertext, how);

				if (how != BUFFER_SIZE) {
					if (ferror(FROMF) != 0) {
						perror("fread");
						return 1;
					}
					if (how < hash_size) {
						memmove(fcrc32,
							&tmp_buf
							[BUFFER_SIZE -
							 hash_size + how],
							hash_size - how);
						memmove(&fcrc32
							[hash_size - how],
							ciphertext, how);
					} else {
						memmove(fcrc32,
							&ciphertext[how -
								    hash_size],
							hash_size);
					}

					if (how > hash_size) {
						mhash(tm, ciphertext,
						      how - hash_size);
						if (fwrite
						    (ciphertext, 1,
						     how - hash_size,
						     RTOF) !=
						    how - hash_size) {
							perror("fwrite");
							return 1;
						}
						if (flush == TRUE)
							fflush(RTOF);

					}
					break;
				}

				memmove(tmp_buf, ciphertext, BUFFER_SIZE);
				mhash(tm, ciphertext, BUFFER_SIZE);

				if (fwrite
				    (ciphertext, 1, BUFFER_SIZE,
				     RTOF) != BUFFER_SIZE) {
					perror("fwrite");
					return 1;
				}

				if (flush == TRUE)
					fflush(RTOF);

			}
		} else {	/* bare flag == TRUE */

			for (;;) {

				how =
				    fread(ciphertext, 1, BUFFER_SIZE,
					  FROMF);
				mdecrypt_generic(td, ciphertext, how);

				if (how != BUFFER_SIZE) {
					if (ferror(FROMF) != 0) {
						perror("fread");
						return 1;
					}
					if (fwrite
					    (ciphertext, 1, how,
					     RTOF) != how) {
						perror("fwrite");
						return 1;
					}
					if (flush == TRUE)
						fflush(RTOF);
					break;
				}

				if (fwrite
				    (ciphertext, 1, BUFFER_SIZE,
				     RTOF) != BUFFER_SIZE) {
					perror("fwrite");
					return 1;
				}

				if (flush == TRUE)
					fflush(RTOF);

			}
		}
	}


/* close and unlock files */

	if (stream_flag == FALSE) {
		fflush(FROMF);
		fflush(RTOF);
		unlock(fileno((FILE *) FROMF));
		unlock(fileno((FILE *) RTOF));
	}
	fclose(RTOF);
	fclose(FROMF);

/* ready */

	if (pad_size > 0) {
		Bzero(ciphertext_p_old, pad_size);
		xfree(ciphertext_p_old);
	}
	Bzero(ciphertext_old, blocksize);
	Bzero(ciphertext, blocksize * BUFFER_SIZE);

	xfree(ciphertext_old);
	xfree(ciphertext);

	Bzero(tmp_buf, BUFFER_SIZE);
	secure_xfree(keyword, keysize);

	if (bare_flag == FALSE) {
		newcrc32 = mhash_end(tm);
		if (is_block_mode(td) == 1) {
			if (hash_size > blocksize) {
				hash_size = blocksize;
			}
		}
		if (memcmp(newcrc32, fcrc32, hash_size) == 0) {
#ifdef ZIP
			if (stream_flag == FALSE
			    && (gzipflag == TRUE || bzipflag == TRUE)) {

				pid = fork();
				if (pid == 0) {
					if (gzipflag == TRUE) {
						err_warn(_
							 ("Decompressing the output file...\n"));
						i =
						    execlp(GZIP, GZIP,
							   "-d", tofile,
							   NULL);
						if (i == -1) {
							perror("exec");
							return 1;
						}

					}
					if (bzipflag == TRUE) {
						err_warn(_
							 ("Decompressing the output file...\n"));
						i =
						    execlp(BZIP2, BZIP2,
							   "-d", tofile,
							   NULL);
						if (i == -1) {
							perror("exec");
							return 1;
						}

					}
				}
#ifdef HAVE_WAITPID
				waitpid(pid, NULL, 0);
#endif
				if (bzipflag == TRUE) {
					tofile[strlen(tofile) -
					       strlen(".bz2")] = '\0';
				} else {
					tofile[strlen(tofile) -
					       strlen(".gz")] = '\0';
				}
			}
#endif
			/* crc check passed */

		} else {
			fprintf(stderr, _("%s check failed\n"),
				mhash_get_hash_name(hash_algorithm));
			return 1;	/* CRC32s Do not match */
		}
	}

	mcrypt_generic_end(td);

	return 0;


}




/* Rols the buffer by one byte and puts the value in the empty cell */

void rol_buf(void *buffer, int buffersize, void *value)
{
	char *buf = buffer;
	char *val = value;
	int i;

	for (i = 0; i < buffersize - 1; i++) {
		buf[i] = buf[i + 1];
	}
	buf[buffersize - 1] = val[0];
}

#ifndef LIBMCRYPT_22
void print_list(void)
{
	int td;
	int i, imax;
	int j, jmax;
	int start;
	char **names;
	char **modes;

	names = mcrypt_list_algorithms(algorithms_directory, &jmax);
	modes = mcrypt_list_modes(modes_directory, &imax);

	for (j = 0; j < jmax; j++) {
		printf("%s ", names[j]);
		start = 0;

		for (i = 0; i < imax; i++) {
			td =
			    mcrypt_module_open(names[j],
					       algorithms_directory,
					       modes[i], modes_directory);

			if (td > 0) {
				if (start == 0)
					printf("(%d): ",
					       mcrypt_get_key_size(td));
				printf("%s ", modes[i]);
				mcrypt_module_close(td);
				start = 1;
			}

		}
		printf("\n");
	}
	mcrypt_free_p(names, jmax);
	mcrypt_free_p(modes, imax);
}
#else
void print_list(void)
{
	int tmpi, tmpj;
	char *tmpc;

	fprintf(stdout, _("Supported Algorithms and modes:\n\n"));
	for (tmpi = 0; tmpi < 256; tmpi++) {
		tmpc = mcrypt_get_algorithms_name(tmpi);
		if (tmpc != NULL) {
			fprintf(stdout, " %s:", tmpc);
			xfree(tmpc);

			if (is_block_algorithm(tmpi) == TRUE) {
				for (tmpj = 0; tmpj < 256; tmpj++) {
					if (is_block_mode(tmpj) != 0) {
						tmpc =
						    mcrypt_get_modes_name
						    (tmpj);
						if (tmpc != NULL) {
							fprintf(stdout,
								" %s",
								tmpc);
							xfree(tmpc);
						}
					}
				}
				fprintf(stdout, "\n");
			} else {
				for (tmpj = 0; tmpj < 256; tmpj++) {
					if (is_block_mode(tmpj) == 0) {
						tmpc =
						    mcrypt_get_modes_name
						    (tmpj);
						if (tmpc != NULL) {
							fprintf(stdout,
								" %s",
								tmpc);
							xfree(tmpc);
						}
					}
				}
				fprintf(stdout, "\n");
			}

		}
	}

}


#endif

void print_keylist(void)
{

	printf("asis\n");
	printf("mcrypt-md5\n");
	printf("mcrypt-sha1\n");
	printf("hex\n");
	printf("pkdes\n");
	printf("s2k-simple-md5\n");
	printf("s2k-simple-sha1\n");
	printf("s2k-salted-md5\n");
	printf("s2k-salted-sha1\n");
	printf("s2k-isalted-md5\n");
	printf("s2k-isalted-sha1\n");
}

void print_hashlist(void)
{
	int tmpi, tmpj;
	char *tmpc;

	fprintf(stdout, _("Supported Hash Algorithms:\n"));
	for (tmpi = 0; tmpi < 256; tmpi++) {
		tmpc = mhash_get_hash_name(tmpi);
		if (tmpc != NULL) {
			_tolow(tmpc, strlen(tmpc));
			fprintf(stdout, "%s\n", tmpc);
			xfree(tmpc);
		}
	}
}
