/* Copyright (C) 1998,1999 Nikos Mavroyanopoulos
 * This library is free
 * software; you can redistribute it and/or modify it under the terms of the
 * GNU Library General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifndef LIBDEFS_H
#define LIBDEFS_H
#include <libdefs.h>
#endif
#include <bzero.h>
#include <swap.h>
#include <mcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast-128.h>
#include <cast-256.h>
#include <tean.h>
#include <aes.h>
#include <xmemory.h>
#include <rc2.h>
#include <rc6.h>
#include <idea.h>
#include <saferplus.h>
#include <loki97.h>

/* I think now is re-entrant */

static int init_check_pointer=0;
static int check_pointer[MAX_THREADS];
static int used_all_threads = 0;

#ifdef HAVE_LIBPTHREAD
# include <pthread.h>
 static int cur_thread = 0;
 pthread_mutex_t counter_cur_thread = PTHREAD_MUTEX_INITIALIZER;
# define LOCK pthread_mutex_lock(&counter_cur_thread)
# define UNLOCK pthread_mutex_unlock(&counter_cur_thread)
#else
# define LOCK 
# define UNLOCK 
 static int cur_thread = 0;	/* No threads */
#endif


static char **ckey = NULL;
static char **c = NULL;     /* For Blowfish */
static char **CI = NULL;     /* twofish */
static char **KI = NULL;
static char **saferplus_key=NULL;
static char **loki97_key=NULL;

static char **kn1 = NULL;	/* For DES */
static char **kn2 = NULL;
static char **kn3 = NULL;	/* For tripleDES */
static word8 **saferkey = NULL;	/* For safer64 and 128 */
static word8 **rc2_key=NULL;	/* For rc2 */
static char **cast256_key=NULL; /* cast-256 */

#ifdef NON_FREE
static char **idea_ikey;
static char **idea_key;	/* For idea */

static char **rc6_skey;	/* For RC6 */
#endif

/* For CBC mode */
static word32 **previous_ciphertext=NULL;
static word32 **previous_plaintext=NULL;
static word32 **previous_cipher=NULL;

/* For CFB mode */
static word8 **sd_register=NULL;
static word8 **s_register=NULL;
static word8 **enc_s_register=NULL;
static word8 **enc_sd_register=NULL;



/* hold the key and the algorithm */
static word8 **keyword_given = NULL;
static int *algorithm_given = NULL;
static int *mode_given = NULL;


int init_mcrypt(const int mode, const int algorithm, int thread, void *key, int lenofkey)
{

	char *keyword1, *keyword2, *keyword3;	/* For 3DES and safer */
	unsigned char *safer_skey = NULL;
	char twokey[68];
	int i, j;

        if (!init_check_pointer) {
		init_check_pointer=1;
        	for (i=0;i<sizeof(check_pointer);i++) {
        	       check_pointer[i]=0;
        	}
        }

	if (lenofkey == 0)
		return -1;	/* error */

	if (used_all_threads == 0) {
		saferkey = mxrealloc(saferkey, (thread + 1) * sizeof(char *));
		kn1 = mxrealloc(kn1, (thread + 1) * sizeof(char *));
		kn2 = mxrealloc(kn2, (thread + 1) * sizeof(char *));
		kn3 = mxrealloc(kn3, (thread + 1) * sizeof(char *));
		c = mxrealloc(c, (thread + 1) * sizeof(char*));
		CI = mxrealloc(CI, (thread + 1) * sizeof(char*));
		KI = mxrealloc(KI, (thread + 1) * sizeof(char*));
		ckey = mxrealloc(ckey, (thread + 1) * sizeof(word8*));		
		keyword_given = mxrealloc(keyword_given, (thread + 1) * sizeof(word8 *));
		rc2_key = mxrealloc(rc2_key, (thread + 1) * sizeof(char*));
		cast256_key = mxrealloc( cast256_key, (thread + 1) * sizeof(char*));
		saferplus_key = mxrealloc( saferplus_key, (thread + 1) * sizeof(char*));
		loki97_key = mxrealloc (loki97_key, (thread + 1) * sizeof(char*));
#ifdef NON_FREE
		rc6_skey = mxrealloc(rc6_skey, (thread + 1) * sizeof(char*));
		idea_key = mxrealloc(idea_key, (thread + 1) * sizeof(char*));
		idea_ikey = mxrealloc(idea_ikey, (thread + 1) * sizeof(char*));
#endif

		algorithm_given = realloc(algorithm_given, (thread + 1) * sizeof(int));
		mode_given = realloc(mode_given, (thread + 1) * sizeof(int));

	}
	if (mcrypt_get_key_size(algorithm) < lenofkey)
		lenofkey = mcrypt_get_key_size(algorithm);

	check_pointer[thread] = 1;
	algorithm_given[thread] = algorithm;
	mode_given[thread] = mode;
	keyword_given[thread] = mxcalloc(1, mcrypt_get_key_size(algorithm));

	memmove(keyword_given[thread], key, lenofkey);

	safer_skey = keyword_given[thread];


	switch (algorithm) {

	case DES:
		_mcrypt_desinit(0);
		kn1[thread] = mxmalloc(8 * 16 * sizeof(char));
		_mcrypt_des_setkey((void *) kn1[thread], (void *) keyword_given[thread]);
		return thread;
		break;

	case SAFER64:
		keyword1 = mxmalloc(8);
		memmove(keyword1, keyword_given[thread], 8);
		saferkey[thread] = mxmalloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

		_mcrypt_Safer_Init_Module();
		_mcrypt_Safer_Expand_Userkey((unsigned char *) keyword1, (unsigned char *) keyword1, SAFER_SK64_DEFAULT_NOF_ROUNDS, 1, saferkey[thread]);
		/* 8 rounds and secure key schedule */
		mxfree(keyword1, 8);

	case CAST128:
                ckey[thread] = mxmalloc( sizeof(struct cast_key));
		_mcrypt_cast_setkey( (void*) ckey[thread], (void *) keyword_given[thread], lenofkey);
		return thread;
		break;

	case LOKI97:
		loki97_key[thread] = mxmalloc(96*sizeof(word32));
		_mcrypt_loki97_set_key( (void*)loki97_key[thread], (void*) keyword_given[thread], lenofkey);
		return thread;
		break;
		
	case CAST256:
                cast256_key[thread] = mxmalloc(96*sizeof(word32));
                _mcrypt_cast256_set_key( (void*)cast256_key[thread], (void *) keyword_given[thread], lenofkey);
		return thread;
		break;

	case RC2:
		rc2_key[thread] = mxcalloc(1, 64 * sizeof(word16));
		_mcrypt_rc2_keyschedule( (void*)rc2_key[thread], (void *) keyword_given[thread], lenofkey);
		return thread;
		break;

#ifdef NON_FREE
	case IDEA:
		idea_key[thread] = mxmalloc(52 * sizeof(word16));
		idea_ikey[thread] = mxmalloc(52 * sizeof(word16));
		_mcrypt_Idea_ExpandUserKey((void *) keyword_given[thread], (void*)idea_key[thread]);
		_mcrypt_Idea_InvertKey( (void*) idea_key[thread], (void*) idea_ikey[thread]);
		return thread;
		break;

	case RC6:
		rc6_skey[thread] = mxmalloc(44*sizeof(word32));
		_mcrypt_rc6_set_key( (void*)rc6_skey[thread], (void *) keyword_given[thread], lenofkey);

		return thread;
		break;
#endif

	case SAFER128:
		_mcrypt_Safer_Init_Module();
		keyword1 = mxmalloc(8);
		keyword2 = mxmalloc(8);
		saferkey[thread] = mxmalloc(1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));

		memmove(keyword1, safer_skey, 8);
		memmove(keyword2, &safer_skey[8], 8);
		_mcrypt_Safer_Expand_Userkey((unsigned char *) keyword1, (unsigned char *) keyword2, SAFER_SK128_DEFAULT_NOF_ROUNDS, 1, saferkey[thread]);
		/* 8 rounds and secure key schedule */
		mxfree(keyword1, 8);
		mxfree(keyword2, 8);

		return thread;
		break;

	case TripleDES:
		_mcrypt_desinit(0);
		keyword1 = mxcalloc(1, 8 * sizeof(char));
		keyword2 = mxcalloc(1, 8 * sizeof(char));
		keyword3 = mxcalloc(1, 8 * sizeof(char));

		kn1[thread] = mxmalloc(8 * 16 * sizeof(char));
		kn2[thread] = mxmalloc(8 * 16 * sizeof(char));
		kn3[thread] = mxmalloc(8 * 16 * sizeof(char));

		/* If it returns 1 or -1 the key is too small */

		if (BreakToThree(keyword_given[thread], lenofkey, keyword1, keyword2, keyword3) != 0) {
			mxfree(keyword1, 8);
			mxfree(keyword2, 8);
			mxfree(keyword3, 8);
			end_mcrypt(thread);
			return (-1);
		}
		_mcrypt_des_setkey((void *) kn1[thread], keyword1);
		_mcrypt_des_setkey((void *) kn2[thread], keyword2);
		_mcrypt_des_setkey((void *) kn3[thread], keyword3);
		mxfree(keyword1, 8);
		mxfree(keyword2, 8);
		mxfree(keyword3, 8);

		return thread;
		break;

	case SAFERPLUS:
		saferplus_key[thread] = mxmalloc( sizeof(SPI));
		_mcrypt_saferplus_set_key( (void*)saferplus_key[thread], (void *) keyword_given[thread], lenofkey);
		return thread;
		break;
		
	case BLOWFISH448:
	case BLOWFISH128:
	case BLOWFISH192:
	case BLOWFISH256:
                c[cur_thread] = mxmalloc( sizeof(blf_ctx));
		if (lenofkey < 2) {
			_mcrypt_blf_key((void*)c[thread], (void *) keyword_given[thread], 2);
		} else {
			_mcrypt_blf_key((void*)c[thread], (void *) keyword_given[thread], lenofkey);
		}
		return thread;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:

                CI[cur_thread] = mxcalloc(1, sizeof(cipherInstance));
                KI[cur_thread] = mxcalloc(1, sizeof(keyInstance));

		_mcrypt_cipherInit((void*)CI[cur_thread], MODE_ECB, NULL);

		j = 0;
		for (i = 0; i < lenofkey; i++) {
			sprintf(&twokey[j], "%.2x", (unsigned char) keyword_given[cur_thread][i]);
			j += 2;
		}
		for (; i < mcrypt_get_key_size(algorithm_given[cur_thread]); i++) {
			sprintf(&twokey[j], "%.2x", 0);
			j += 2;
		}

		j = _mcrypt_makeKey((void*)KI[cur_thread], DIR_ENCRYPT, mcrypt_get_key_size(algorithm_given[cur_thread]) * 8, twokey);
		if (j < 0) {
			return (-1);
		}
		return thread;
		break;

	case GOST:
		_mcrypt_kboxinit();
		return thread;
		break;

	case ThreeWAY:
	case TEAN:
		return thread;
		break;

	default:
		return -1;
	}

}






/* plaintext should be in block's size */
int mcrypt(int thread, void *plaintext)
{

	switch (algorithm_given[thread]) {

	case SAFER64:
	case SAFER128:
		_mcrypt_Safer_Encrypt_Block(plaintext, saferkey[thread]);
		return 0;
		break;

	case SAFERPLUS:
		_mcrypt_saferplus_encrypt( (void*)saferplus_key[thread], plaintext);
		return 0;
		break;
		
	case CAST128:
		_mcrypt_cast_encrypt((void*)ckey[thread], (void *) plaintext);
		return 0;
		break;

	case LOKI97:
		_mcrypt_loki97_encrypt((void*)loki97_key[thread], (void *) plaintext);
		return 0;
		break;
		
	case CAST256:
		_mcrypt_cast256_encrypt((void*)cast256_key[thread], (void *) plaintext);
		return 0;
		break;

	case RC2:
		_mcrypt_rc2_encrypt( (void*)rc2_key[thread], (void *) plaintext);
		return 0;
		break;

#ifdef NON_FREE
	case IDEA:
		_mcrypt_Idea_Crypt((void *) plaintext, (void *) plaintext, (void*)idea_key[thread]);
		return 0;
		break;

	case RC6:
		_mcrypt_rc6_encrypt( (void*)rc6_skey[thread], (void *) plaintext);
		return 0;
		break;
#endif

	case DES:
		_mcrypt_endes((void *) kn1[thread], plaintext);
		return 0;
		break;

	case TripleDES:
		_mcrypt_endes((void *) kn1[thread], plaintext);
		_mcrypt_dedes((void *) kn2[thread], plaintext);
		_mcrypt_endes((void *) kn3[thread], plaintext);
		return 0;
		break;

	case BLOWFISH448:
	case BLOWFISH128:
	case BLOWFISH192:
	case BLOWFISH256:
		_mcrypt_enblf((void*)c[thread], plaintext);
		return 0;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		_mcrypt_blockEncrypt((void*)CI[thread], (void*)KI[thread], plaintext, 16 * 8, plaintext);
		return 0;
		break;

	case GOST:
		_mcrypt_gostcrypt(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	case TEAN:
		_mcrypt_cl_enc_block((void *) keyword_given[thread], plaintext);
		return 0;
		break;

	case ThreeWAY:
		_mcrypt_en3way(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	default:
		return 0;
	}

}


/* plaintext should be in block's size */
int mdecrypt(int thread, void *plaintext)
{
	switch (algorithm_given[thread]) {

	case SAFER64:
	case SAFER128:
		_mcrypt_Safer_Decrypt_Block(plaintext, (void *) saferkey[thread]);
		return 0;
		break;

	case SAFERPLUS:
		_mcrypt_saferplus_decrypt( (void*)saferplus_key[thread], plaintext);
		return 0;
		break;


	case CAST128:
		_mcrypt_cast_decrypt((void*)ckey[thread], (void *) plaintext);
		return 0;
		break;

	case LOKI97:
		_mcrypt_loki97_decrypt((void*)loki97_key[thread], (void *) plaintext);
		return 0;
		break;

	case CAST256:
		_mcrypt_cast256_decrypt((void*)cast256_key[thread], (void *) plaintext);
		return 0;
		break;

	case RC2:
		_mcrypt_rc2_decrypt( (void*) rc2_key[thread], (void *) plaintext);
		return 0;
		break;

#ifdef NON_FREE
	case IDEA:
		_mcrypt_Idea_Crypt((void *) plaintext, (void *) plaintext, (void*) idea_ikey[thread]);
		return 0;
		break;

	case RC6:
		_mcrypt_rc6_decrypt( (void*) rc6_skey[thread], (void *) plaintext);
		return 0;
		break;
#endif

	case DES:
		_mcrypt_dedes((char (*)[8]) kn1[thread], plaintext);
		return 0;
		break;

	case TEAN:
		_mcrypt_cl_dec_block((void *) keyword_given[thread], plaintext);
		return 0;
		break;

	case TripleDES:
		_mcrypt_dedes((void *) kn3[thread], plaintext);
		_mcrypt_endes((void *) kn2[thread], plaintext);
		_mcrypt_dedes((void *) kn1[thread], plaintext);
		return 0;
		break;

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		_mcrypt_blockDecrypt((void*)CI[thread], (void*)KI[thread], plaintext, 16 * 8, plaintext);
		return 0;
		break;

	case BLOWFISH448:
	case BLOWFISH128:
	case BLOWFISH192:
	case BLOWFISH256:
		_mcrypt_deblf((void*)c[thread], plaintext);
		return 0;
		break;

	case GOST:
		_mcrypt_gostdecrypt(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	case ThreeWAY:
		_mcrypt_de3way(plaintext, (void *) keyword_given[thread]);
		return 0;
		break;

	default:
		return 0;
	}
}


int end_mcrypt(int td)
{

	mxfree(keyword_given[td], mcrypt_get_key_size(algorithm_given[td]));
	check_pointer[td] = 0;

	switch (algorithm_given[td]) {

	case DES:
		_mcrypt_desdone();
		mxfree(kn1[td], 8 * 16 * sizeof(char));
		break;

	case TripleDES:
		_mcrypt_desdone();
		mxfree(kn1[td], 8 * 16 * sizeof(char));
		mxfree(kn2[td], 8 * 16 * sizeof(char));
		mxfree(kn3[td], 8 * 16 * sizeof(char));
		break;

	case RC2:
		mxfree(rc2_key[td], 64 * sizeof(word16));
		break;

#ifdef NON_FREE
	case IDEA:
		mxfree(idea_key[td], 52 * sizeof(word16));
		mxfree(idea_ikey[td], 52 * sizeof(word16));
		break;

	case RC6:
		mxfree(rc6_skey[td], 44*sizeof(word32));
		break;
#endif

	case TWOFISH128:
	case TWOFISH192:
	case TWOFISH256:
		mxfree(CI[td], sizeof(cipherInstance));
		mxfree(KI[td], sizeof(keyInstance));
		break;

	case BLOWFISH448:
	case BLOWFISH128:
	case BLOWFISH192:
	case BLOWFISH256:
		mxfree(c[td], sizeof(blf_ctx));
		break;

	case TEAN:
	case GOST:
	case ThreeWAY:
	        break;
	        
	case CAST128:
		mxfree(ckey[td], sizeof(struct cast_key));
		break;

	case LOKI97:
		mxfree(loki97_key[td], 96*sizeof(word32));
		break;
		                
	case CAST256:
		mxfree(cast256_key[td], 96*sizeof(word32));
		break;

	case SAFER64:
	case SAFER128:
		mxfree(saferkey[td], 1 + SAFER_BLOCK_LEN * (1 + 2 * SAFER_MAX_NOF_ROUNDS));
		break;
		break;

	default:
		return 0;
	}

	return 0;
}

/* ECB MODE */

int init_mcrypt_ecb(const int algorithm, void *key, int lenofkey)
{
	int x;

	LOCK;
	
	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	}
	x = init_mcrypt(ECB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}




int mcrypt_ecb(int td, void *plaintext, int len)
{
	int j, blocksize = mcrypt_get_block_size(algorithm_given[td]);
	char *plain = plaintext;

	for (j = 0; j < len / blocksize; j++) {

		mcrypt(td, &plain[j * blocksize]);

	}
	return 0;
}



int mdecrypt_ecb(int td, void *ciphertext, int len)
{
	int j, blocksize = mcrypt_get_block_size(algorithm_given[td]);
	char *cipher = ciphertext;

	for (j = 0; j < len / blocksize; j++) {

		mdecrypt(td, &cipher[j * blocksize]);

	}

	return 0;
}




/* CBC MODE */

int end_mcrypt_cbc(int td)
{

	mxfree(previous_ciphertext[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(previous_cipher[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(previous_plaintext[td], mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_cbc(const int algorithm, void *key, int lenofkey)
{

	int x;


	LOCK;

	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
                if (used_all_threads==0) {
			previous_ciphertext = mxrealloc(previous_ciphertext, (cur_thread + 1) * sizeof(char*));
			previous_cipher = mxrealloc(previous_cipher, (cur_thread + 1) * sizeof(char*));
			previous_plaintext = mxrealloc(previous_plaintext, (cur_thread + 1) * sizeof(char*));
		}
	}



/* For cbc */
	previous_ciphertext[cur_thread] = mxcalloc(1, mcrypt_get_block_size(algorithm));
	previous_cipher[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	previous_plaintext[cur_thread] = mxcalloc(1, mcrypt_get_block_size(algorithm));
/* End cbc */

	x = init_mcrypt(CBC, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cbc(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}

int init_mcrypt_cbc_iv(const int algorithm, void *key, int lenofkey, void *IV)
{

	int x, tmpi=MAX_THREADS;

	LOCK;

	if (cur_thread >= MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
		tmpi--;
		if (tmpi<0) break;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
                if (used_all_threads==0) {
			previous_ciphertext = mxrealloc(previous_ciphertext, (cur_thread + 1) * sizeof(char*));
			previous_cipher = mxrealloc(previous_cipher, (cur_thread + 1) * sizeof(char*));
			previous_plaintext = mxrealloc(previous_plaintext, (cur_thread + 1) * sizeof(char*));
		}
	}


	previous_ciphertext[cur_thread] = mxcalloc(1, mcrypt_get_block_size(algorithm));
	previous_cipher[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	previous_plaintext[cur_thread] = mxcalloc(1, mcrypt_get_block_size(algorithm));

	memmove(previous_ciphertext[cur_thread], IV, mcrypt_get_block_size(algorithm));
	memmove(previous_plaintext[cur_thread], IV, mcrypt_get_block_size(algorithm));
/* End cbc */

	x = init_mcrypt(CBC, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cbc(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}


int mcrypt_cbc(int td, void *plaintext, int len)
{
	word32 *fplain = plaintext;
	word32 *plain;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		plain = &fplain[j * blocksize / sizeof(word32)];

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^= previous_ciphertext[td][i];
		}

		mcrypt(td, plain);


		/* Copy the ciphertext to prev_ciphertext */
		memmove(previous_ciphertext[td], plain, blocksize);
	}

	return 0;
}



int mdecrypt_cbc(int td, void *ciphertext, int len)
{
	word32 *cipher;
	word32 *fcipher = ciphertext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		cipher = &fcipher[j * blocksize / sizeof(word32)];
		memmove(previous_cipher[td], cipher, blocksize);

		mdecrypt(td, cipher);
		for (i = 0; i < blocksize / sizeof(word32); i++) {
			cipher[i] ^= previous_plaintext[td][i];
		}

		/* Copy the ciphertext to prev_cipher */
		memmove(previous_plaintext[td], previous_cipher[td], blocksize);

	}

	return 0;
}



/* CFB MODE */

int end_mcrypt_cfb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td], mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_cfb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
                if (used_all_threads==0) {
			s_register = mxrealloc(s_register, (cur_thread + 1) * sizeof(char *));
			enc_s_register = mxrealloc(enc_s_register, (cur_thread + 1) * sizeof(char *));
			enc_sd_register = mxrealloc(enc_sd_register, (cur_thread + 1) * sizeof(char *));
			sd_register = mxrealloc(sd_register, (cur_thread + 1) * sizeof(char *));
		}
	}


/* For cfb */
	s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV, mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV, mcrypt_get_block_size(algorithm));

/* End cfb */

	x = init_mcrypt(CFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cfb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = plain[j];


	}

	return 0;

}


int mdecrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = plain[j];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}


/* OFB MODE */

int end_mcrypt_ofb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td], mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_ofb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
                if (used_all_threads==0) {
			s_register = mxrealloc(s_register, (cur_thread + 1) * sizeof(char *));
			enc_s_register = mxrealloc(enc_s_register, (cur_thread + 1) * sizeof(char *));
			enc_sd_register = mxrealloc(enc_sd_register, (cur_thread + 1) * sizeof(char *));
			sd_register = mxrealloc(sd_register, (cur_thread + 1) * sizeof(char *));
		}
	}


/* For ofb */
	s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV, mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV, mcrypt_get_block_size(algorithm));

/* End ofb */

	x = init_mcrypt(OFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_ofb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = enc_s_register[td][0];


	}

	return 0;

}


int mdecrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit ofb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = enc_sd_register[td][0];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}



/* n-bit ofb */

int end_mcrypt_nofb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td], mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_nofb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
                if (used_all_threads==0) {
			s_register = mxrealloc(s_register, (cur_thread + 1) * sizeof(char *));
			enc_s_register = mxrealloc(enc_s_register, (cur_thread + 1) * sizeof(char *));
			enc_sd_register = mxrealloc(enc_sd_register, (cur_thread + 1) * sizeof(char *));
			sd_register = mxrealloc(sd_register, (cur_thread + 1) * sizeof(char *));
		}
	}


/* For nofb */
	s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] = mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV, mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV, mcrypt_get_block_size(algorithm));

/* End nofb */

	x = init_mcrypt(nOFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_nofb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_nofb(int td, void *plaintext, int len)
{				/* plaintext is n*blocksize bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(enc_s_register[td], s_register[td], blocksize);
		mcrypt(td, enc_s_register[td]);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^= ((word32 *) & enc_s_register[td][0 + i * sizeof(word32)])[0];
		}
/* Put the new register */
		memmove(s_register[td], enc_s_register[td], blocksize);
	}
	return 0;
}


int mdecrypt_nofb(int td, void *plaintext, int len)
{				/* plaintext is n bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(enc_sd_register[td], sd_register[td], blocksize);
		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		memmove(sd_register[td], enc_sd_register[td], blocksize);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^= ((word32 *) & enc_sd_register[td][0 + i * sizeof(word32)])[0];
		}
	}

	return 0;
}



/* Generic -High level functions */

int mcrypt_generic_init(const int mode, const int algorithm, void *key, int lenofkey, void *IV)
{

	switch (mode) {
	case CBC:
		if (IV == NULL) {
			return init_mcrypt_cbc(algorithm, key, lenofkey);
		} else {
			return init_mcrypt_cbc_iv(algorithm, key, lenofkey, IV);
		}
		break;
	case ECB:
		return init_mcrypt_ecb(algorithm, key, lenofkey);
		break;
	case CFB:
		return init_mcrypt_cfb(algorithm, key, lenofkey, IV);
		break;
	case OFB:
		return init_mcrypt_ofb(algorithm, key, lenofkey, IV);
		break;
	case nOFB:
		return init_mcrypt_nofb(algorithm, key, lenofkey, IV);
		break;
	default:
		return -1;
	}
}

int mcrypt_generic(int td, void *plaintext, int len)
{

	switch (mode_given[td]) {
	case CBC:
		return mcrypt_cbc(td, plaintext, len);
		break;
	case ECB:
		return mcrypt_ecb(td, plaintext, len);
		break;
	case CFB:
		return mcrypt_cfb(td, plaintext, len);
		break;
	case OFB:
		return mcrypt_ofb(td, plaintext, len);
		break;
	case nOFB:
		return mcrypt_nofb(td, plaintext, len);
		break;
	default:
		return -1;
	}

}

int mdecrypt_generic(int td, void *ciphertext, int len)
{

	switch (mode_given[td]) {
	case CBC:
		return mdecrypt_cbc(td, ciphertext, len);
		break;
	case ECB:
		return mdecrypt_ecb(td, ciphertext, len);
		break;
	case CFB:
		return mdecrypt_cfb(td, ciphertext, len);
		break;
	case OFB:
		return mdecrypt_ofb(td, ciphertext, len);
		break;
	case nOFB:
		return mdecrypt_nofb(td, ciphertext, len);
		break;
	default:
		return -1;
	}

}

int mcrypt_generic_end(const int td)
{

	switch (mode_given[td]) {
	case CBC:
		return end_mcrypt_cbc(td);
		break;
	case ECB:
		return end_mcrypt_ecb(td);
		break;
	case CFB:
		return end_mcrypt_cfb(td);
		break;
	case OFB:
		return end_mcrypt_ofb(td);
		break;
	case nOFB:
		return end_mcrypt_nofb(td);
		break;
	default:
		return -1;
	}
}

