
/* Copyright (C) 1998,1999 Nikos Mavroyanopoulos 
 * This library is free
 * software; you can redistribute it and/or modify it under the terms of the
 * GNU Library General Public License as published by the Free Software
 * Foundation; either version 2 of the License, or (at your option) any
 * later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* $Id: mcrypt.c,v 1.8 1999/10/20 09:13:41 nmav Exp $ */

#ifndef LIBDEFS_H
#define LIBDEFS_H
#include <libdefs.h>
#endif
#include <bzero.h>
#include <swap.h>
#include <mcrypt.h>
#include <blowfish.h>
#include <3-way.h>
#include <gost.h>
#include <des.h>
#include <safer.h>
#include <cast-128.h>
#include <cast-256.h>
#include <tean.h>
#include <twofish.h>
#include <xmemory.h>
#include <rc2.h>
#include <rc6.h>
#include <idea.h>
#include <saferplus.h>
#include <loki97.h>
#include <rijndael.h>
#include <serpent.h>
#include <arcfour.h>
#include <enigma.h>

/* I think now is re-entrant */

static int init_check_pointer = 0;
static int check_pointer[MAX_THREADS];
static int used_all_threads = 0;

#ifdef HAVE_LIBPTHREAD
#include <pthread.h>
static int cur_thread = 0;
pthread_mutex_t counter_cur_thread = PTHREAD_MUTEX_INITIALIZER;
#define LOCK pthread_mutex_lock(&counter_cur_thread)
#define UNLOCK pthread_mutex_unlock(&counter_cur_thread)
#else
#define LOCK
#define UNLOCK
static int cur_thread = 0;	/* No threads */
#endif



/* Up to three keys for an algorithm */
static word8 **akey = NULL;
static word8 **bkey = NULL;
static word8 **ckey = NULL;

/* For CBC mode */
static word32 **previous_ciphertext = NULL;
static word32 **previous_plaintext = NULL;
static word32 **previous_cipher = NULL;

/* For CFB mode */
static word8 **sd_register = NULL;
static word8 **s_register = NULL;
static word8 **enc_s_register = NULL;
static word8 **enc_sd_register = NULL;


/* hold the key and the algorithm */
static word8 **keyword_given = NULL;
static int *algorithm_given = NULL;
static int *mode_given = NULL;


int init_mcrypt(const int mode, const int algorithm, int thread, void *key,
		int lenofkey)
{
	int i;

	if (!init_check_pointer) {
		init_check_pointer = 1;
		for (i = 0; i < MAX_THREADS; i++) {
			check_pointer[i] = 0;
		}
	}
	if (lenofkey == 0)
		return -1;	/* error */

	if (used_all_threads == 0) {
		akey = mxrealloc(akey, (thread + 1) * sizeof(word8 *));
		bkey = mxrealloc(bkey, (thread + 1) * sizeof(word8 *));
		ckey = mxrealloc(ckey, (thread + 1) * sizeof(word8 *));

		keyword_given =
		    mxrealloc(keyword_given,
			      (thread + 1) * sizeof(word8 *));

		algorithm_given =
		    realloc(algorithm_given, (thread + 1) * sizeof(int));

		mode_given =
		    realloc(mode_given, (thread + 1) * sizeof(int));

	}
	if (mcrypt_get_key_size(algorithm) < lenofkey)
		lenofkey = mcrypt_get_key_size(algorithm);

	check_pointer[thread] = 1;
	algorithm_given[thread] = algorithm;
	mode_given[thread] = mode;
	keyword_given[thread] =
	    mxcalloc(1, mcrypt_get_key_size(algorithm));

	memmove(keyword_given[thread], key, lenofkey);


	switch (algorithm) {

	case MCRYPT_DES:
		_mcrypt_desinit(0);
		akey[thread] = mxcalloc(1, 8 * 16 * sizeof(char));
		_mcrypt_des_setkey((void *) akey[thread],
				   (void *) keyword_given[thread]);
		return thread;

	case MCRYPT_SAFER_64:
		akey[thread] =
		    mxcalloc(1, 1 +
			     SAFER_BLOCK_LEN * (1 +
						2 * SAFER_MAX_NOF_ROUNDS));

		_mcrypt_Safer_Init_Module();
		_mcrypt_Safer_Expand_Userkey((void *)
					     keyword_given[thread],
					     (void *)
					     keyword_given[thread],
					     SAFER_SK64_DEFAULT_NOF_ROUNDS,
					     1, akey[thread]);
		/* 8 rounds and secure key schedule */
		return thread;

	case MCRYPT_CAST_128:
		akey[thread] = mxcalloc(1, sizeof(struct cast_key));
		_mcrypt_cast_setkey((void *) akey[thread],
				    (void *) keyword_given[thread],
				    lenofkey);
		return thread;

	case MCRYPT_LOKI97:
		akey[thread] = mxcalloc(1, 96 * sizeof(word32));
		_mcrypt_loki97_set_key((void *) akey[thread],
				       (void *) keyword_given[thread],
				       lenofkey);
		return thread;

	case MCRYPT_CAST_256:
		akey[thread] = mxcalloc(1, 96 * sizeof(word32));
		_mcrypt_cast256_set_key((void *) akey[thread],
					(void *) keyword_given[thread],
					lenofkey);
		return thread;


	case MCRYPT_RIJNDAEL_128:
	case MCRYPT_RIJNDAEL_192:
	case MCRYPT_RIJNDAEL_256:
		akey[thread] = mxcalloc(1, sizeof(RI));
		_mcrypt_rijndael_gentables();
		_mcrypt_rijndael_set_key((void *) akey[thread], 4,
					 lenofkey / 4,
					 (void *) keyword_given[thread]);
		return thread;

	case MCRYPT_SERPENT_128:
	case MCRYPT_SERPENT_192:
	case MCRYPT_SERPENT_256:
		akey[thread] = mxcalloc(1, sizeof(SERPENT_KEY));
		_mcrypt_serpent_set_key((void *) akey[thread],
				       (void *) keyword_given[thread], lenofkey);
		return thread;

	case MCRYPT_RC2_128:
	case MCRYPT_RC2_1024:
	case MCRYPT_RC2_256:
		akey[thread] = mxcalloc(1, 64 * sizeof(word16));
		_mcrypt_rc2_keyschedule((void *) akey[thread],
					(void *) keyword_given[thread],
					lenofkey);
		return thread;

#ifdef NON_FREE

	case MCRYPT_IDEA:
		akey[thread] = mxcalloc(1, 52 * sizeof(word16));
		bkey[thread] = mxcalloc(1, 52 * sizeof(word16));
		_mcrypt_Idea_ExpandUserKey((void *) keyword_given[thread],
					   (void *) akey[thread]);
		_mcrypt_Idea_InvertKey((void *) akey[thread],
				       (void *) bkey[thread]);
		return thread;

	case MCRYPT_RC6_128:
	case MCRYPT_RC6_192:
	case MCRYPT_RC6_256:
		akey[thread] = mxcalloc(1, 44 * sizeof(word32));
		_mcrypt_rc6_set_key((void *) akey[thread],
				    (void *) keyword_given[thread],
				    lenofkey);
		return thread;

	case MCRYPT_ARCFOUR:
		akey[thread] = mxcalloc(1, sizeof(arcfour_key));
		_mcrypt_arcfour_prepare_key( (void *) keyword_given[thread],
				    lenofkey, (void *) akey[thread]);
		return thread;

#endif
	case MCRYPT_CRYPT:
		akey[thread] = mxcalloc(1, sizeof(CRYPT_KEY));
		_mcrypt_set_key_crypt( (void *) akey[thread], (void *) keyword_given[thread],
				    lenofkey);
		return thread;


	case MCRYPT_SAFER_128:
		_mcrypt_Safer_Init_Module();
		akey[thread] =
		    mxcalloc(1, 1 +
			     SAFER_BLOCK_LEN * (1 +
						2 * SAFER_MAX_NOF_ROUNDS));

		_mcrypt_Safer_Expand_Userkey((void *)
					     keyword_given[thread],
					     (void *)
					     &(keyword_given[thread])[8],
					     SAFER_SK128_DEFAULT_NOF_ROUNDS,
					     1, akey[thread]);

		/* 8 rounds and secure key schedule */

		return thread;

	case MCRYPT_3DES:
		_mcrypt_desinit(0);

		akey[thread] = mxcalloc(1, 8 * 16 * sizeof(char));
		bkey[thread] = mxcalloc(1, 8 * 16 * sizeof(char));
		ckey[thread] = mxcalloc(1, 8 * 16 * sizeof(char));

		/* If it returns 1 or -1 the key is too small */

		_mcrypt_des_setkey((void *) akey[thread],
				   (void *) keyword_given[thread]);
		_mcrypt_des_setkey((void *) bkey[thread],
				   (void *) &(keyword_given[thread])[8]);
		_mcrypt_des_setkey((void *) ckey[thread],
				   (void *) &(keyword_given[thread])[16]);

		return thread;

	case MCRYPT_SAFERPLUS:
		akey[thread] = mxcalloc(1, sizeof(SPI));
		_mcrypt_saferplus_set_key((void *) akey[thread],
					  (void *) keyword_given[thread],
					  lenofkey);
		return thread;

	case MCRYPT_BLOWFISH_448:
	case MCRYPT_BLOWFISH_128:
	case MCRYPT_BLOWFISH_192:
	case MCRYPT_BLOWFISH_256:
		akey[thread] = mxcalloc(1, sizeof(blf_ctx));
		if (lenofkey < 2) {
			_mcrypt_blf_key((void *) akey[thread],
					(void *) keyword_given[thread], 2);
		} else {
			_mcrypt_blf_key((void *) akey[thread],
					(void *) keyword_given[thread],
					lenofkey);
		}
		return thread;

	case MCRYPT_TWOFISH_128:
	case MCRYPT_TWOFISH_192:
	case MCRYPT_TWOFISH_256:
		akey[thread] = mxcalloc(1, sizeof(TWI));
		_mcrypt_twofish_set_key((void *) akey[cur_thread],
					(void *) keyword_given[thread],
					lenofkey);
		return thread;

	case MCRYPT_GOST:
		_mcrypt_kboxinit();
		return thread;

	case MCRYPT_3WAY:
	case MCRYPT_XTEA:
		return thread;

	default:
		return -1;
	}

}






/* plaintext should be in block's size */
int mcrypt(int thread, void *plaintext)
{

	switch (algorithm_given[thread]) {

	case MCRYPT_SAFER_64:
	case MCRYPT_SAFER_128:
		_mcrypt_Safer_Encrypt_Block(plaintext, akey[thread]);
		return 0;

	case MCRYPT_SAFERPLUS:
		_mcrypt_saferplus_encrypt((void *) akey[thread],
					  plaintext);
		return 0;

	case MCRYPT_CAST_128:
		_mcrypt_cast_encrypt((void *) akey[thread],
				     (void *) plaintext);
		return 0;

	case MCRYPT_LOKI97:
		_mcrypt_loki97_encrypt((void *) akey[thread],
				       (void *) plaintext);
		return 0;

	case MCRYPT_CAST_256:
		_mcrypt_cast256_encrypt((void *) akey[thread],
					(void *) plaintext);
		return 0;

	case MCRYPT_RC2_128:
	case MCRYPT_RC2_256:
	case MCRYPT_RC2_1024:
		_mcrypt_rc2_encrypt((void *) akey[thread],
				    (void *) plaintext);
		return 0;

#ifdef NON_FREE
	case MCRYPT_IDEA:
		_mcrypt_Idea_Crypt((void *) plaintext, (void *) plaintext,
				   (void *) akey[thread]);
		return 0;

	case MCRYPT_RC6_128:
	case MCRYPT_RC6_192:
	case MCRYPT_RC6_256:
		_mcrypt_rc6_encrypt((void *) akey[thread],
				    (void *) plaintext);
		return 0;


#endif

	case MCRYPT_RIJNDAEL_128:
	case MCRYPT_RIJNDAEL_192:
	case MCRYPT_RIJNDAEL_256:
		_mcrypt_rijndael_encrypt((void *) akey[thread],
					 (void *) plaintext);
		return 0;

	case MCRYPT_SERPENT_128:
	case MCRYPT_SERPENT_192:
	case MCRYPT_SERPENT_256:
		_mcrypt_serpent_encrypt((void *) akey[thread],
					(void *) plaintext);
		return 0;

	case MCRYPT_DES:
		_mcrypt_endes((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_3DES:
		_mcrypt_endes((void *) akey[thread], plaintext);
		_mcrypt_dedes((void *) bkey[thread], plaintext);
		_mcrypt_endes((void *) ckey[thread], plaintext);
		return 0;

	case MCRYPT_BLOWFISH_448:
	case MCRYPT_BLOWFISH_128:
	case MCRYPT_BLOWFISH_192:
	case MCRYPT_BLOWFISH_256:
		_mcrypt_enblf((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_TWOFISH_128:
	case MCRYPT_TWOFISH_192:
	case MCRYPT_TWOFISH_256:
		_mcrypt_twofish_encrypt((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_GOST:
		_mcrypt_gostcrypt(plaintext,
				  (void *) keyword_given[thread]);
		return 0;

	case MCRYPT_XTEA:
		_mcrypt_cl_enc_block((void *) keyword_given[thread],
				     plaintext);
		return 0;

	case MCRYPT_3WAY:
		_mcrypt_en3way(plaintext, (void *) keyword_given[thread]);
		return 0;

	default:
		return 0;
	}

}


/* plaintext should be in block's size */
int mdecrypt(int thread, void *plaintext)
{
	switch (algorithm_given[thread]) {

	case MCRYPT_SAFER_64:
	case MCRYPT_SAFER_128:
		_mcrypt_Safer_Decrypt_Block(plaintext,
					    (void *) akey[thread]);
		return 0;

	case MCRYPT_SAFERPLUS:
		_mcrypt_saferplus_decrypt((void *) akey[thread],
					  plaintext);
		return 0;

	case MCRYPT_CAST_128:
		_mcrypt_cast_decrypt((void *) akey[thread],
				     (void *) plaintext);
		return 0;

	case MCRYPT_LOKI97:
		_mcrypt_loki97_decrypt((void *) akey[thread],
				       (void *) plaintext);
		return 0;

	case MCRYPT_CAST_256:
		_mcrypt_cast256_decrypt((void *) akey[thread],
					(void *) plaintext);
		return 0;

	case MCRYPT_RC2_128:
	case MCRYPT_RC2_256:
	case MCRYPT_RC2_1024:
		_mcrypt_rc2_decrypt((void *) akey[thread],
				    (void *) plaintext);
		return 0;

#ifdef NON_FREE
	case MCRYPT_IDEA:
		_mcrypt_Idea_Crypt((void *) plaintext, (void *) plaintext,
				   (void *) bkey[thread]);
		return 0;

	case MCRYPT_RC6_128:
	case MCRYPT_RC6_192:
	case MCRYPT_RC6_256:
		_mcrypt_rc6_decrypt((void *) akey[thread],
				    (void *) plaintext);
		return 0;

#endif

	case MCRYPT_RIJNDAEL_128:
	case MCRYPT_RIJNDAEL_192:
	case MCRYPT_RIJNDAEL_256:
		_mcrypt_rijndael_decrypt((void *) akey[thread],
					 (void *) plaintext);
		return 0;

	case MCRYPT_SERPENT_128:
	case MCRYPT_SERPENT_192:
	case MCRYPT_SERPENT_256:
		_mcrypt_serpent_decrypt((void *) akey[thread],
					(void *) plaintext);
		return 0;

	case MCRYPT_DES:
		_mcrypt_dedes((char (*)[8]) akey[thread], plaintext);
		return 0;

	case MCRYPT_XTEA:
		_mcrypt_cl_dec_block((void *) keyword_given[thread],
				     plaintext);
		return 0;

	case MCRYPT_3DES:
		_mcrypt_dedes((void *) ckey[thread], plaintext);
		_mcrypt_endes((void *) bkey[thread], plaintext);
		_mcrypt_dedes((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_TWOFISH_128:
	case MCRYPT_TWOFISH_192:
	case MCRYPT_TWOFISH_256:
		_mcrypt_twofish_decrypt((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_BLOWFISH_448:
	case MCRYPT_BLOWFISH_128:
	case MCRYPT_BLOWFISH_192:
	case MCRYPT_BLOWFISH_256:
		_mcrypt_deblf((void *) akey[thread], plaintext);
		return 0;

	case MCRYPT_GOST:
		_mcrypt_gostdecrypt(plaintext,
				    (void *) keyword_given[thread]);
		return 0;

	case MCRYPT_3WAY:
		_mcrypt_de3way(plaintext, (void *) keyword_given[thread]);
		return 0;

	default:
		return 0;
	}
}


int end_mcrypt(int td)
{

	mxfree(keyword_given[td],
	       mcrypt_get_key_size(algorithm_given[td]));
	check_pointer[td] = 0;

	switch (algorithm_given[td]) {

	case MCRYPT_RIJNDAEL_128:
	case MCRYPT_RIJNDAEL_192:
	case MCRYPT_RIJNDAEL_256:
		mxfree(akey[td], sizeof(RI));
		break;

	case MCRYPT_SERPENT_128:
	case MCRYPT_SERPENT_192:
	case MCRYPT_SERPENT_256:
		mxfree(akey[td], sizeof(SERPENT_KEY));
		break;


	case MCRYPT_DES:
		_mcrypt_desdone();
		mxfree(akey[td], 8 * 16 * sizeof(char));
		break;

	case MCRYPT_3DES:
		_mcrypt_desdone();
		mxfree(akey[td], 8 * 16 * sizeof(char));
		mxfree(bkey[td], 8 * 16 * sizeof(char));
		mxfree(ckey[td], 8 * 16 * sizeof(char));
		break;

	case MCRYPT_RC2_128:
	case MCRYPT_RC2_256:
	case MCRYPT_RC2_1024:
		mxfree(akey[td], 64 * sizeof(word16));
		break;

#ifdef NON_FREE
	case MCRYPT_IDEA:
		mxfree(akey[td], 52 * sizeof(word16));
		mxfree(bkey[td], 52 * sizeof(word16));
		break;

	case MCRYPT_RC6_128:
	case MCRYPT_RC6_192:
	case MCRYPT_RC6_256:
		mxfree(akey[td], 44 * sizeof(word32));
		break;

	case MCRYPT_ARCFOUR:
		mxfree(akey[td], sizeof(arcfour_key));
		break;
#endif
	case MCRYPT_CRYPT:
		mxfree(akey[td], sizeof(CRYPT_KEY));
		break;
		
	case MCRYPT_TWOFISH_128:
	case MCRYPT_TWOFISH_192:
	case MCRYPT_TWOFISH_256:
		mxfree(akey[td], sizeof(TWI));
		break;

	case MCRYPT_BLOWFISH_448:
	case MCRYPT_BLOWFISH_128:
	case MCRYPT_BLOWFISH_192:
	case MCRYPT_BLOWFISH_256:
		mxfree(akey[td], sizeof(blf_ctx));
		break;

	case MCRYPT_XTEA:
	case MCRYPT_GOST:
	case MCRYPT_3WAY:
		break;

	case MCRYPT_CAST_128:
		mxfree(akey[td], sizeof(struct cast_key));
		break;

	case MCRYPT_LOKI97:
		mxfree(akey[td], 96 * sizeof(word32));
		break;

	case MCRYPT_CAST_256:
		mxfree(akey[td], 96 * sizeof(word32));
		break;

	case MCRYPT_SAFER_64:
	case MCRYPT_SAFER_128:
		mxfree(akey[td],
		       1 + SAFER_BLOCK_LEN * (1 +
					      2 * SAFER_MAX_NOF_ROUNDS));
		break;

	default:
		return 0;
	}

	return 0;
}

/* ECB MODE */

int init_mcrypt_ecb(const int algorithm, void *key, int lenofkey)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	}
	x = init_mcrypt(MCRYPT_ECB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}




int mcrypt_ecb(int td, void *plaintext, int len)
{
	int j, blocksize;
	char *plain = plaintext;

	blocksize = mcrypt_get_block_size(algorithm_given[td]);
	for (j = 0; j < len / blocksize; j++) {

		mcrypt(td, &plain[j * blocksize]);

	}
	return 0;
}



int mdecrypt_ecb(int td, void *ciphertext, int len)
{
	int j, blocksize = mcrypt_get_block_size(algorithm_given[td]);
	char *cipher = ciphertext;

	for (j = 0; j < len / blocksize; j++) {

		mdecrypt(td, &cipher[j * blocksize]);

	}

	return 0;
}



/* STREAM MODE */
int init_mcrypt_stream (const int algorithm, void *key, int lenofkey)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	}
	x = init_mcrypt(MCRYPT_STREAM, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}




int mcrypt_stream(int td, void *plaintext, int len)
{
	switch (algorithm_given[td]) {

#ifdef NON_FREE	
	case MCRYPT_ARCFOUR:
		_mcrypt_arcfour( (void *) plaintext, len, (void *) akey[td]);
		return 0;
#endif
	case MCRYPT_CRYPT:
		_mcrypt_encrypt_crypt( (void*) akey[td], (void *) plaintext, len);
		return 0;
	default:
		return -1;
	}

}



int mdecrypt_stream(int td, void *ciphertext, int len)
{
	switch (algorithm_given[td]) {

#ifdef NON_FREE	
	case MCRYPT_ARCFOUR:
		_mcrypt_arcfour( (void *) ciphertext, len, (void *) akey[td]);
		return 0;
#endif
	case MCRYPT_CRYPT:
		_mcrypt_decrypt_crypt( (void*) akey[td], (void *) ciphertext, len);
		return 0;
	default:
		return -1;
	}
}




/* CBC MODE */

int end_mcrypt_cbc(int td)
{

	mxfree(previous_ciphertext[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(previous_cipher[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(previous_plaintext[td],
	       mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_cbc(const int algorithm, void *key, int lenofkey)
{

	int x;


	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
		if (used_all_threads == 0) {
			previous_ciphertext =
			    mxrealloc(previous_ciphertext,
				      (cur_thread + 1) * sizeof(char *));
			previous_cipher =
			    mxrealloc(previous_cipher,
				      (cur_thread + 1) * sizeof(char *));
			previous_plaintext =
			    mxrealloc(previous_plaintext,
				      (cur_thread + 1) * sizeof(char *));
		}
	}



/* For cbc */
	previous_ciphertext[cur_thread] =
	    mxcalloc(1, mcrypt_get_block_size(algorithm));
	previous_cipher[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	previous_plaintext[cur_thread] =
	    mxcalloc(1, mcrypt_get_block_size(algorithm));
/* End cbc */

	x = init_mcrypt(MCRYPT_CBC, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cbc(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}

int init_mcrypt_cbc_iv(const int algorithm, void *key, int lenofkey,
		       void *IV)
{

	int x, tmpi = MAX_THREADS;

	LOCK;

	if (cur_thread >= MAX_THREADS) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
		tmpi--;
		if (tmpi < 0)
			break;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
		if (used_all_threads == 0) {
			previous_ciphertext =
			    mxrealloc(previous_ciphertext,
				      (cur_thread + 1) * sizeof(char *));
			previous_cipher =
			    mxrealloc(previous_cipher,
				      (cur_thread + 1) * sizeof(char *));
			previous_plaintext =
			    mxrealloc(previous_plaintext,
				      (cur_thread + 1) * sizeof(char *));
		}
	}


	previous_ciphertext[cur_thread] =
	    mxcalloc(1, mcrypt_get_block_size(algorithm));
	previous_cipher[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	previous_plaintext[cur_thread] =
	    mxcalloc(1, mcrypt_get_block_size(algorithm));

	memmove(previous_ciphertext[cur_thread], IV,
		mcrypt_get_block_size(algorithm));
	memmove(previous_plaintext[cur_thread], IV,
		mcrypt_get_block_size(algorithm));
/* End cbc */

	x = init_mcrypt(MCRYPT_CBC, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cbc(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}

}


int mcrypt_cbc(int td, void *plaintext, int len)
{
	word32 *fplain = plaintext;
	word32 *plain;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		plain = &fplain[j * blocksize / sizeof(word32)];

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^= previous_ciphertext[td][i];
		}

		mcrypt(td, plain);


		/* Copy the ciphertext to prev_ciphertext */
		memmove(previous_ciphertext[td], plain, blocksize);
	}

	return 0;
}



int mdecrypt_cbc(int td, void *ciphertext, int len)
{
	word32 *cipher;
	word32 *fcipher = ciphertext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {

		cipher = &fcipher[j * blocksize / sizeof(word32)];
		memmove(previous_cipher[td], cipher, blocksize);

		mdecrypt(td, cipher);
		for (i = 0; i < blocksize / sizeof(word32); i++) {
			cipher[i] ^= previous_plaintext[td][i];
		}

		/* Copy the ciphertext to prev_cipher */
		memmove(previous_plaintext[td], previous_cipher[td],
			blocksize);

	}

	return 0;
}



/* CFB MODE */

int end_mcrypt_cfb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_cfb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
		if (used_all_threads == 0) {
			s_register =
			    mxrealloc(s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_s_register =
			    mxrealloc(enc_s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_sd_register =
			    mxrealloc(enc_sd_register,
				      (cur_thread + 1) * sizeof(char *));
			sd_register =
			    mxrealloc(sd_register,
				      (cur_thread + 1) * sizeof(char *));
		}
	}


/* For cfb */
	s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));

/* End cfb */

	x = init_mcrypt(MCRYPT_CFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_cfb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = plain[j];


	}

	return 0;

}


int mdecrypt_cfb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = plain[j];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}


/* OFB MODE */

int end_mcrypt_ofb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_ofb(const int algorithm, void *key, int lenofkey, void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
		if (used_all_threads == 0) {
			s_register =
			    mxrealloc(s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_s_register =
			    mxrealloc(enc_s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_sd_register =
			    mxrealloc(enc_sd_register,
				      (cur_thread + 1) * sizeof(char *));
			sd_register =
			    mxrealloc(sd_register,
				      (cur_thread + 1) * sizeof(char *));
		}
	}


/* For ofb */
	s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));

/* End ofb */

	x = init_mcrypt(MCRYPT_OFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_ofb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit cfb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_s_register[td], s_register[td], blocksize);

		mcrypt(td, enc_s_register[td]);

		plain[j] ^= enc_s_register[td][0];

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			s_register[td][i] = s_register[td][i + 1];

		s_register[td][blocksize - 1] = enc_s_register[td][0];


	}

	return 0;

}


int mdecrypt_ofb(int td, void *plaintext, int len)
{				/* plaintext is 1 byte (8bit ofb) */
	char *plain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);;

	for (j = 0; j < len; j++) {

		memmove(enc_sd_register[td], sd_register[td], blocksize);

		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		for (i = 0; i < (blocksize - 1); i++)
			sd_register[td][i] = sd_register[td][i + 1];

		sd_register[td][blocksize - 1] = enc_sd_register[td][0];

		plain[j] ^= enc_sd_register[td][0];

	}

	return 0;
}



/* n-bit ofb */

int end_mcrypt_nofb(int td)
{

	mxfree(s_register[td], mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_s_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(enc_sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));
	mxfree(sd_register[td],
	       mcrypt_get_block_size(algorithm_given[td]));

	return end_mcrypt(td);

}


int init_mcrypt_nofb(const int algorithm, void *key, int lenofkey,
		     void *IV)
{
	int x;

	LOCK;

	if (cur_thread == MAX_THREADS-1) {
		cur_thread = 0;
		used_all_threads = 1;
	}
	while (check_pointer[cur_thread] == 1 && cur_thread < MAX_THREADS) {
		cur_thread++;
	}

	if (cur_thread >= MAX_THREADS) {
		UNLOCK;
		return -1;
	} else {
		if (used_all_threads == 0) {
			s_register =
			    mxrealloc(s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_s_register =
			    mxrealloc(enc_s_register,
				      (cur_thread + 1) * sizeof(char *));
			enc_sd_register =
			    mxrealloc(enc_sd_register,
				      (cur_thread + 1) * sizeof(char *));
			sd_register =
			    mxrealloc(sd_register,
				      (cur_thread + 1) * sizeof(char *));
		}
	}


/* For nofb */
	s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_s_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	enc_sd_register[cur_thread] =
	    mxmalloc(mcrypt_get_block_size(algorithm));
	memmove(sd_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));
	memmove(s_register[cur_thread], IV,
		mcrypt_get_block_size(algorithm));

/* End nofb */

	x = init_mcrypt(MCRYPT_nOFB, algorithm, cur_thread, key, lenofkey);

	if (x < 0) {
		end_mcrypt_nofb(cur_thread);
		UNLOCK;
		return -1;
	} else {
		UNLOCK;
		return x;
	}
}

int mcrypt_nofb(int td, void *plaintext, int len)
{				/* plaintext is n*blocksize bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(enc_s_register[td], s_register[td], blocksize);
		mcrypt(td, enc_s_register[td]);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^=
			    ((word32 *) &
			     enc_s_register[td][0 +
						i * sizeof(word32)])[0];
		}
/* Put the new register */
		memmove(s_register[td], enc_s_register[td], blocksize);
	}
	return 0;
}


int mdecrypt_nofb(int td, void *plaintext, int len)
{				/* plaintext is n bytes (nbit ofb) */
	word32 *plain, *fplain = plaintext;
	int i, j, blocksize = mcrypt_get_block_size(algorithm_given[td]);

	for (j = 0; j < len / blocksize; j++) {
		plain = &fplain[j * blocksize / sizeof(word32)];
		memmove(enc_sd_register[td], sd_register[td], blocksize);
		mcrypt(td, enc_sd_register[td]);

/* Shift the register */
		memmove(sd_register[td], enc_sd_register[td], blocksize);

		for (i = 0; i < blocksize / sizeof(word32); i++) {
			plain[i] ^=
			    ((word32 *) &
			     enc_sd_register[td][0 +
						 i * sizeof(word32)])[0];
		}
	}

	return 0;
}



/* Generic -High level functions */

int mcrypt_generic_init(const int mode, const int algorithm, void *key,
			int lenofkey, void *IV)
{
	if (is_block_algorithm(algorithm) != 0) {
		switch (mode) {
		case MCRYPT_CBC:
			if (IV == NULL) {
				return init_mcrypt_cbc(algorithm, key,
						       lenofkey);
			} else {
				return init_mcrypt_cbc_iv(algorithm, key,
							  lenofkey, IV);
			}

		case MCRYPT_ECB:
			return init_mcrypt_ecb(algorithm, key, lenofkey);
		case MCRYPT_CFB:
			return init_mcrypt_cfb(algorithm, key, lenofkey,
					       IV);
		case MCRYPT_OFB:
			return init_mcrypt_ofb(algorithm, key, lenofkey,
					       IV);
		case MCRYPT_nOFB:
			return init_mcrypt_nofb(algorithm, key, lenofkey,
						IV);
		default:
			return -1;
		}
	} else {
		if (mode==MCRYPT_STREAM) {
	        	return init_mcrypt_stream(algorithm, key, lenofkey);
	        } else {
	        	return -1;
	        }
	}


}

int mcrypt_generic(int td, void *plaintext, int len)
{

	switch (mode_given[td]) {
	case MCRYPT_CBC:
		return mcrypt_cbc(td, plaintext, len);
	case MCRYPT_ECB:
		return mcrypt_ecb(td, plaintext, len);
	case MCRYPT_CFB:
		return mcrypt_cfb(td, plaintext, len);
	case MCRYPT_OFB:
		return mcrypt_ofb(td, plaintext, len);
	case MCRYPT_nOFB:
		return mcrypt_nofb(td, plaintext, len);
	case MCRYPT_STREAM:
		return mcrypt_stream(td, plaintext, len);
		
	default:
		return -1;
	}

}

int mdecrypt_generic(int td, void *ciphertext, int len)
{

	switch (mode_given[td]) {
	case MCRYPT_CBC:
		return mdecrypt_cbc(td, ciphertext, len);
	case MCRYPT_ECB:
		return mdecrypt_ecb(td, ciphertext, len);
	case MCRYPT_CFB:
		return mdecrypt_cfb(td, ciphertext, len);
	case MCRYPT_OFB:
		return mdecrypt_ofb(td, ciphertext, len);
	case MCRYPT_nOFB:
		return mdecrypt_nofb(td, ciphertext, len);
	case MCRYPT_STREAM:
		return mdecrypt_stream(td, ciphertext, len);

	default:
		return -1;
	}

}

int mcrypt_generic_end(const int td)
{

	switch (mode_given[td]) {
	case MCRYPT_CBC:
		return end_mcrypt_cbc(td);
	case MCRYPT_ECB:
		return end_mcrypt_ecb(td);
	case MCRYPT_CFB:
		return end_mcrypt_cfb(td);
	case MCRYPT_OFB:
		return end_mcrypt_ofb(td);
	case MCRYPT_nOFB:
		return end_mcrypt_nofb(td);
	case MCRYPT_STREAM:
		return end_mcrypt_stream(td);
		
	default:
		return -1;
	}
}
