/*    Secure hash
 *
 *    Copyright (C) 1998,1999 Nikos Mavroyanopoulos
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <defines.h>
#include <mhash.h>
#include <shash.h>
#include <stdio.h>
#include <stdlib.h>
#include <gaa.h>
#include <errors.h>
#include <extra.h>
#include <locks.h>
#include <random.h>
#include <environ.h>

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#define MAX_KEY 127
#define MAX_NAME 4096
#define HASH_DEFAULT MHASH_SHA1

int quiet = 0;
int nolock = FALSE;
char msg[400];
int binary = FALSE;
int total = 0;
int nosalt = FALSE;
int double_check = FALSE;
int hmac=0;

int main(int argc, char **argv)
{

	void *hash;

	gaainfo info;
	char *cfile, **file = NULL;
	int file_count = 0, v, i, x = 0, use_password = 0, plen = 0;
	char *tmp;
	char *tmp2 = NULL;
	char *password = NULL;
	word32 rand;
	hashid algorithm = MHASH_MD5;

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit("Error in the arguments.\n");
	}

	if (info.config_file != NULL) {
		cfile = malloc(strlen(info.config_file));
		strcpy(cfile, info.config_file);
		if (check_file(cfile) != 1) {
			err_crit(
				 ("Warning: Cannot access config file.\n"));
		}
	} else {
#ifdef HAVE_GETPWUID
		cfile = get_cfile(getuid(), ".shashrc");
#else
		cfile = malloc(9);
		strcpy(cfile, ".shashrc");
#endif
	}

	/* If config file exists */
	if (check_file(cfile) == 1) {
		if ((v = gaa_file(cfile, &info)) != -1) {
			/* gets options from file 'config' */
			err_quit("Error in the configuration file.\n");
		}
	}
	free(cfile);

	if (check_env() == TRUE) {
		if (get_env_key() != NULL) {
			free(info.keys);
			info.keys = get_env_key();
			info.hmac = 1;
		}
		if (get_env_bit_mode() != 0) {
			info.kmode = get_env_bit_mode();
		}
	}

/* Check again the command line variables */

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(("Error in the arguments.\n"));
	}

	quiet = info.quiet;
	nosalt = info.nosalt;
	nolock = info.nolock;
	hmac = info.hmac;
	double_check = info.double_check;

	if (info.text == TRUE) {
		binary = FALSE;
	}

	if (info.binary == TRUE) {
		binary = TRUE;
	}

	if (info.algorithm != NULL) {

		algorithm = check_algo(info.algorithm);
		if (algorithm == -1) {
			err_crit("No such algorithm, default is MD5.\n");
			algorithm = MHASH_MD5;
		}

	} else {
		algorithm = MHASH_MD5;
	}


	if (hmac == 1 || info.keys != NULL) {
		if (mhash_get_hash_pblock(algorithm) == 0) {
			tmp = mhash_get_hash_name(algorithm);
			sprintf(msg,
				"You cannot use HMAC with the %s algorithm.\n",
				tmp);
			free(tmp);
			err_crit(msg);
			return 1;
		}

		if (info.kmode != NULL) {
			if (strcasecmp(info.kmode, "hash") != 0 &&
			    strcasecmp(info.kmode, "8bit") != 0 &&
			    strcasecmp(info.kmode, "hex") != 0) {
				err_crit("No such mode. Assumed hash.\n");
			}
		}
		password =
		    (void *) Getpass(algorithm, "Enter keyword:", &plen,
				     info.keys, info.kmode);
#ifdef HAVE_MLOCK
		mlock(password, plen);
#endif
		use_password = 1;


	}



	file = malloc((info.size) * sizeof(char *));

	i = 0;
	file_count = (info.size);

	while (i < info.size) {
		file[i] = info.input[i];
		i++;
	}

	if (file_count == 0) {	/* streams for check */
		file = malloc(sizeof(char *));
		file[0] = NULL;
		file_count++;
	}

	if (file[0] != NULL && info.check == FALSE) {
		tmp = mhash_get_hash_name(algorithm);
		if (hmac == 1) {
			strcpy(msg, "HMAC");

			if (nosalt == FALSE) {
				rand = get_rand32();
				printf("# %s %s %s\n", tmp, msg, "SALT");

				password = realloc(password, plen + 4);
				plen += 4;
				
			} else {
				printf("# %s %s\n", tmp, msg);
			}

		} else {
			strcpy(msg, "HASH");
			printf("# %s %s\n", tmp, msg);
		}
		free(tmp);
	}

	for (i = 0; i < file_count; i++) {
		if (info.check == 0) {
			if (nosalt == FALSE && hmac == 1) {
				rand = get_rand32();
				tmp2 = bin2print(&rand, 4);

				memmove(&password[plen-4], &rand, 4);
			}

			hash = sumfile(file[i], algorithm, password, plen);
			if (hash != NULL) {
				if (nosalt == FALSE && hmac == 1) {
					printf("%s ", tmp2);
					free(tmp2);
				}
				view(file[i], hash,
				     mhash_get_block_size(algorithm));
				free(hash);
			}

		} else {

			x =
			    checkfile(file[i], algorithm, password, plen,
				      info.kmode);

		}
	}

	if (password != NULL)
		bzero(password, plen);
#ifdef HAVE_MLOCK
	munlock(password, plen);
#endif
	free(password);

	return 0;

}

hashid check_algo(char *chain)
{
	hashid i;
	char *y;

	for (i = 0; i < 255; i++) {
		y = mhash_get_hash_name(i);
		if (y == NULL)
			continue;
		if (strcasecmp(y, chain) == 0) {
			free(y);
			return i;
		}
		free(y);
	}

	return -1;
}

void view(char *desc, void *x, int sizeofx)
{
	int j;
	unsigned char *y = x;


	for (j = 0; j < sizeofx; j++) {
		printf("%.2x", y[j]);
	}
	if (binary == TRUE) {
		if (desc != NULL) {
			printf(" *%s", desc);
		}
	} else {
		if (desc != NULL) {
			printf("  %s", desc);
		}
	}
	printf("\n");

}

void *sumfile(char *filename, hashid algorithm, char *password, int plen)
{
	FILE *file;
	char buffer[4096];
	MHASH td;
	int x;


	if (filename != NULL) {
		if (binary == TRUE) {
			file = fopen(filename, "rb");
		} else {
			file = fopen(filename, "r");
		}

		if (file == NULL) {
			sprintf(msg, "%s: No such file.\n", filename);
			err_crit(msg);
			return NULL;
		}

		if (read_lock(fileno(file)) == -1) {
			fclose(file);
			return NULL;
		}
	} else {
		file = stdin;

		if (file == NULL) {
			sprintf(msg, "stdin: Error.\n");
			err_crit(msg);
			return NULL;
		}

	}

	if (password == NULL) {
		td = mhash_init(algorithm);
	} else {

		td =
		    hmac_mhash_init(algorithm, password, plen,
				    mhash_get_hash_pblock(algorithm));
	}

	if (td < 0) {
		sprintf(msg, "Internal library error.\n");
		err_crit(msg);
		return NULL;
	}

	for (;;) {
		x = fread(buffer, 1, 4096, file);
		if (x == 0) {
			if (ferror(file) == 0) {
				break;
			} else {
				sprintf(msg, "%s: Error in file.\n",
					filename);
				err_crit(msg);
				fclose(file);
				return NULL;
			}
		}
		mhash(td, &buffer, x);
	}

	unlock(fileno(file));
	fclose(file);

	if (password == NULL) {
		return mhash_end(td);
	} else {
		return hmac_mhash_end(td);
	}
}

int hash_list()
{
	hashid i;
	char *y;

	for (i = 0; i < 255; i++) {
		y = mhash_get_hash_name(i);
		if (y == NULL)
			continue;
		printf("%s\n", y);
		free(y);
	}

	exit(0);
}

int checkfile(char *filename, hashid algorithm, char *keyword, int plen,
	      char *kmode)
{
	FILE *file;
	int counter = 0, counter_failed = 0;
	int j, x, block;
	unsigned char *sumtmp;
	char *linebuf;
	int line_storage;
	char *buffer, *password = NULL;
	char *buffer2;
	char *hexbuffer;
	char *tmp;
	char *salt = NULL;

	block = mhash_get_block_size(algorithm);


	buffer = malloc(block * 2 + 1);
	buffer2 = malloc(MAX_NAME + 1);
	hexbuffer = malloc(block * 2 + 1);

	if ((block * 2 + MAX_NAME) > 1024) {
		line_storage = (block * 2) + MAX_NAME + 1;
	} else {
		line_storage = 1024 + 1;
	}

	linebuf = malloc(line_storage);


	if (filename != NULL) {
		file = fopen(filename, "rb");
		if (file == NULL) {
			sprintf(msg, "%s: No such file.\n", filename);
			err_crit(msg);
			return 0;
		}

	} else {
		file = stdin;
		if (file == NULL) {
			sprintf(msg, "stdin: Error.\n");
			err_crit(msg);
			return 0;
		}

	}

	if (filename != NULL) {
		if (read_lock(fileno(file)) == -1) {
			sprintf(msg, "%s: File is locked.\n", filename);
			err_crit(msg);
			return 0;
		}
	}


	x = 1;

	for (;;) {

		fgets(linebuf, line_storage, file);

		if (linebuf == NULL)
			break;
		if (feof(file) != 0)
			break;
		if (linebuf[0] == '#' && linebuf[1] == ' ') {

			if (sscanf(linebuf, "# %s %s", buffer, buffer2) !=
			    0) {
				algorithm = check_algo(buffer);

				if (strcasecmp(buffer2, "hmac") == 0) {	/* hmac */
					hmac=1;
					if (sscanf(linebuf, "# %s %s %s",
						   buffer, buffer2,
						   hexbuffer) == 2) {
						nosalt = TRUE;
					} else {
						nosalt = FALSE;
					}


					if (keyword == NULL) {
						password =
						    Getpass
						    (algorithm,
						     "Enter keyword:",
						     &plen, keyword,
						     kmode);



					} else {
						password = malloc(plen);
						memmove(password, keyword,
							plen);
					}

					if (nosalt == FALSE && hmac == 1) {
						password =
						    realloc(password,
							    plen + 4);

						plen += 4;
					}
#ifdef HAVE_MLOCK
					mlock(password, plen);
#endif

				} else {
					hmac=0;
				}

				block = mhash_get_block_size(algorithm);

				buffer = realloc(buffer, block * 2 + 1);
				buffer2 = realloc(buffer2, MAX_NAME + 1);
				hexbuffer = realloc(hexbuffer, block*2 + 1);

				if ((block * 2 + MAX_NAME) > 1024) {
					line_storage =
					    (block * 2) + MAX_NAME + 1;
				} else {
					line_storage = 1024 + 1;
				}
				linebuf = realloc(linebuf, line_storage);
			}

			continue;
		}

		/* checks so it does not overflow */
		if (strlen(linebuf) > (block * 2) + MAX_NAME + 2)
			continue;
		if (strlen(linebuf) < block * 2)
			continue;

		bzero(buffer2, block * 2);
		
		if (nosalt == FALSE && hmac == 1) {
			if (sscanf
			    (linebuf, "%s %s  %s\n", hexbuffer, buffer,
			     buffer2) < 2) {
				if (sscanf
				    (linebuf, "%s %s *%s\n", hexbuffer,
				     buffer, buffer2) < 2)
					continue;
			}

			salt = print2bin(hexbuffer, strlen(hexbuffer));

if (salt!=NULL) {
			memmove( &password[plen-4], salt, 4);
			free(salt);
		}

		} else {
			if (sscanf(linebuf, "%s  %s\n", buffer, buffer2) ==
			    0) {
				if (sscanf
				    (linebuf, "%s *%s\n", buffer,
				     buffer2) == 0)
					continue;
			}
		}

		sumtmp = sumfile(buffer2, algorithm, password, plen);

		bzero(hexbuffer, block * 2 +1);

		if (sumtmp != NULL) {
			for (j = 0; j < block; j++) {
				sprintf(&hexbuffer[j * 2], "%.2x",
					sumtmp[j]);
			}


			if (strcmp(hexbuffer, buffer) == 0) {
				sprintf(msg, "%s: OK\n", buffer2);
				err_warn(msg);
				counter++;

			} else {
				tmp = mhash_get_hash_name(algorithm);
				sprintf(msg,
					"%s check failed for '%s'\n",
					tmp, buffer2);
				free(tmp);
				err_crit(msg);
				counter_failed++;
			}
			free(sumtmp);
		}

	}

	free(buffer);
	free(buffer2);
	free(hexbuffer);

	if (filename != NULL)
		unlock(fileno(file));
	fclose(file);

	tmp = mhash_get_hash_name(algorithm);
	sprintf(msg, "%d of %d file(s) failed %s check.\n",
		counter_failed, counter_failed + counter, tmp);
	err_warn(msg);
	if (password != NULL && counter == 0)
		err_crit("Probably wrong password.\n");
	free(tmp);

	if (password!=NULL) { 
		bzero(password, plen);
#ifdef HAVE_MLOCK
		munlock(password, plen);
#endif
		free(password);
	}

	return counter;
}

void hash_license()
{
	fprintf(stdout, ("\nCopyright (C) 1999 Nikos Mavroyanopoulos\n\
This program is free software; you can redistribute it and/or modify \n\
it under the terms of the GNU General Public License as published by \n\
the Free Software Foundation; either version 2 of the License, or \n\
(at your option) any later version. \n\
\n\
This program is distributed in the hope that it will be useful, \n\
but WITHOUT ANY WARRANTY; without even the implied warranty of \n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n\
GNU General Public License for more details. \n\
\n\
You should have received a copy of the GNU General Public License \n\
along with this program; if not, write to the Free Software \n\
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n"));
}

void hash_version()
{
	fprintf(stderr, ("shash v.%s (%s-%s-%s)\n"), VERSION, T_CPU,
		T_VENDOR, T_OS);
	fprintf(stderr,
		("Copyright (C) 1999 Nikos Mavroyanopoulos (nmav@hellug.gr)\n"));

} void *hash_key(char *key, int keylen)
{
	hashid td;

	td = mhash_init(HASH_DEFAULT);
	mhash(td, key, keylen);
	return mhash_end(td);

}
char *bin2print(void *x, int sizeofx)
{
	int j;
	unsigned char *y = x;
	char *retval = calloc(1, sizeofx * 2 + 1);
	char tmp[3];

	tmp[2] = '\0';

	for (j = 0; j < sizeofx; j++) {
		sprintf(tmp, "%.2x", y[j]);
		strcat(retval, tmp);
	}
	return retval;

}

void *print2bin(void *x, int sizeofx)
{
	int j;
	unsigned char *y = x;
	word8 *retval = calloc(1, sizeofx / 2);
	char tmp[3];

	if (sizeofx % 2 != 0)
		return NULL;
	if (check_hex(x, sizeofx) == -1)
		return NULL;


	tmp[2] = '\0';

	for (j = 0; j < sizeofx; j += 2) {
		memmove(tmp, &y[j], 2);
		retval[j / 2] = strtol(tmp, (char **) NULL, 16);
	} return retval;

}

void *Getpass(hashid algorithm, char *key, int *keylen, void *pkey,
	      char *kmode)
{

	char *tmpkey;
	char *password, *userkey;

	userkey = malloc(MAX_KEY + 1);
#ifdef HAVE_MLOCK
	mlock(userkey, MAX_KEY);
#endif
	if (pkey == NULL) {
		tmpkey = (void *) getpass("Enter password:");
		strncpy(userkey, tmpkey, MAX_KEY);
		bzero(tmpkey, strlen(tmpkey));
		if (double_check == TRUE) {
			tmpkey = (void *) getpass("Retype password:");
			if (strcmp(tmpkey, userkey) != 0)
				err_quit("Passwords do not match.\n");
			bzero(tmpkey, strlen(tmpkey));
		}
	} else {
		strncpy(userkey, pkey, MAX_KEY);
	}


	if (strlen((char *) userkey) == 0)
		err_quit("Error in keyword.\n");

	if (kmode != NULL) {
		if (strcasecmp(kmode, "8bit") == 0) {
			password = malloc(MAX_KEY + 1);
			strcpy(password, userkey);
			*keylen = strlen(password);

		} else {
			if (strcasecmp(kmode, "hex") == 0) {
				password =
				    print2bin(userkey, strlen(userkey));

				if (password == NULL)
					err_quit
					    ("Error in the key format.\n");
				*keylen = strlen(userkey) / 2;

			} else {
				password =
				    hash_key(userkey, strlen(userkey));
				*keylen =
				    mhash_get_block_size(HASH_DEFAULT);

			}
		}

	} else {
		/* hash is the default */
		password = hash_key(userkey, strlen(userkey));
		*keylen = mhash_get_block_size(HASH_DEFAULT);

	}

	bzero(userkey, strlen(userkey));
#ifdef HAVE_MLOCK
	munlock(userkey, MAX_KEY);
#endif
	free(userkey);

	return password;

}

int check_hex(char *given_chain, int len)
{
	int i;

	for (i = 0; i < len; i++)
		if (isxdigit(given_chain[i]) == 0)
			return -1;

	return 0;
}
