#include "libdefs.h"
#include "mhash.h"
#include "keygen.h"

#define KEYGEN_ENTRY(name, uses_hash_algorithm, uses_count, uses_salt, salt_size) \
        { #name, name, uses_hash_algorithm, uses_count, uses_salt, salt_size }

typedef struct mhash_keygen_entry {
	char *name;
	keygenid id;
	int uses_hash_algorithm;
	int uses_count;
	int uses_salt;
	size_t salt_size;
} mhash_keygen_entry;

static mhash_keygen_entry keygen_algorithms[] = {
	KEYGEN_ENTRY(KEYGEN_ASIS, 0, 0, 0, 0),
	KEYGEN_ENTRY(KEYGEN_HEX, 0, 0, 0, 0),
	KEYGEN_ENTRY(KEYGEN_MCRYPT, 1, 0, 1, 0),
	KEYGEN_ENTRY(KEYGEN_S2K_SIMPLE, 1, 0, 0, 0),
	KEYGEN_ENTRY(KEYGEN_S2K_SALTED, 1, 0, 1, 8),
	KEYGEN_ENTRY(KEYGEN_S2K_ISALTED, 1, 1, 1, 8),
	{0}
};

#define KEYGEN_LOOP(b) \
        mhash_keygen_entry *p; \
                for(p = keygen_algorithms; p->name != NULL; p++) { b ; }

#define KEYGEN_ALG_LOOP(a) \
                        KEYGEN_LOOP( if(p->id == type) { a; break; } )

int mhash_keygen_uses_hash_algorithm(keygenid type)
{
	int ret = 0;
	KEYGEN_ALG_LOOP(ret = p->uses_hash_algorithm);
	return ret;
}


int mhash_keygen_uses_count(keygenid type)
{
	int ret = 0;
	KEYGEN_ALG_LOOP(ret = p->uses_count);
	return ret;
}


int mhash_keygen_uses_salt(keygenid type)
{
	int ret = 0;
	KEYGEN_ALG_LOOP(ret = p->uses_salt);
	return ret;
}


size_t mhash_keygen_count(void)
{
	size_t count = 0;
	KEYGEN_LOOP(count++);
	return count;
}

size_t mhash_get_keygen_salt_size(keygenid type)
{
	size_t ret = 0;

	KEYGEN_ALG_LOOP(ret = p->salt_size);
	return ret;
}

char *mhash_get_keygen_name(hashid type)
{
	char *ret = NULL;

	/* avoid prefix */
	KEYGEN_ALG_LOOP(ret = strdup(p->name + sizeof("KEYGEN_") - 1));

	return ret;
}



int mhash_keygen(keygenid algorithm, hashid opt_algorithm,
		 unsigned long count, void *keyword, int keysize,
		 void *salt, int saltsize, unsigned char *password,
		 int passwordlen)
{

	switch (algorithm) {

	case KEYGEN_MCRYPT:
		_mhash_gen_key_mcrypt(opt_algorithm, keyword, keysize,
				      salt, saltsize, password,
				      passwordlen);
		break;
	case KEYGEN_ASIS:
		_mhash_gen_key_asis(keyword, keysize, password,
				    passwordlen);
		break;
	case KEYGEN_HEX:
		_mhash_gen_key_hex(keyword, keysize, password,
				   passwordlen);
		break;
	case KEYGEN_S2K_SIMPLE:
		_mhash_gen_key_s2k_simple(opt_algorithm, keyword, keysize,
					  password, passwordlen);
		break;
	case KEYGEN_S2K_SALTED:
		_mhash_gen_key_s2k_salted(opt_algorithm, keyword, keysize,
					  salt, saltsize, password,
					  passwordlen);
		break;
	case KEYGEN_S2K_ISALTED:
		_mhash_gen_key_s2k_isalted(opt_algorithm, count, keyword,
					   keysize, salt, saltsize,
					   password, passwordlen);
		break;
	default:
		return -1;
	}

	return 0;
}
