/*    SHASH, a program to compute secure hashes for files.
 *
 *    Copyright (C) 1998,1999 Nikos Mavroyanopoulos
 *
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the Free Software
 *    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <defines.h>
#include <mhash.h>
#include <shash.h>
#include <stdio.h>
#include <stdlib.h>
#include <gaa.h>
#include <errors.h>
#include <extra.h>
#include <locks.h>
#include <random.h>
#include <environ.h>
#include <time.h>

#ifdef HAVE_CTYPE_H
#include <ctype.h>
#endif

#define MAX_KEY 127
#define MAX_NAME 4096

int quiet = 0;
int nolock = FALSE;
char msg[400];
word32 salt[2];			/* 8 bytes */
int set_salt = 0;
int binary = FALSE;
int total = 0;
int nosalt = FALSE;
int double_check = FALSE;
int hmac = 0;
int show_time = 0;

int main(int argc, char **argv)
{

	void *hash;
	gaainfo info;
	char *cfile, **file = NULL;
	int file_count = 0, v, i, x = 0, use_password = 0, plen = 0;
	char *tmp;
	char *tmp2 = NULL;
	char *password = NULL;
	char *key;
	word32 newlen;
	int keylen;
	hashid algorithm = MHASH_MD5;
	keygenid keymode=KEYGEN_S2K_SALTED;

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit("Error in the arguments.\n");
	}

	if (info.config_file != NULL) {
		cfile = malloc(strlen(info.config_file));
		strcpy(cfile, info.config_file);
		if (check_file(cfile) != 1) {
			err_crit(
				 ("Warning: Cannot access config file.\n"));
		}
	} else {
#ifdef HAVE_GETPWUID
		cfile = get_cfile(getuid(), ".shashrc");
#else
		cfile = malloc(9);
		strcpy(cfile, ".shashrc");
#endif
	}

	/* If config file exists */
	if (check_file(cfile) == 1) {
		if ((v = gaa_file(cfile, &info)) != -1) {
			/* gets options from file 'config' */
			err_quit("Error in the configuration file.\n");
		}
	}
	free(cfile);

	if (check_env() == TRUE) {
		if (get_env_key() != NULL) {
			free(info.keys);
			info.keys = get_env_key();
			info.hmac = 1;
		}
		if (get_env_bit_mode() != 0) {
			info.kmode = get_env_bit_mode();
		}
	}

/* Check again the command line variables */

	if ((v = gaa(argc, argv, &info)) != -1) {
		err_quit(("Error in the arguments.\n"));
	}

	quiet = info.quiet;
	nosalt = info.nosalt;
	nolock = info.nolock;
	hmac = info.hmac;
	double_check = info.double_check;
	show_time = info.show_time;

	if (info.text == TRUE) {
		binary = FALSE;
	}

	if (info.binary == TRUE) {
		binary = TRUE;
	}

	if (info.algorithm != NULL) {

		algorithm = check_algo(info.algorithm);
		if (algorithm == -1) {
			err_crit("No such algorithm, default is MD5.\n");
			algorithm = MHASH_MD5;
		}

	} else {
		algorithm = MHASH_MD5;
	}


	if (hmac == 1 || info.keys != NULL) {
		if (mhash_get_hash_pblock(algorithm) == 0) {
			tmp = mhash_get_hash_name(algorithm);
			sprintf(msg,
				"You cannot use HMAC with the %s algorithm.\n",
				tmp);
			free(tmp);
			err_crit(msg);
			return 1;
		}

		if (info.kmode != NULL) {
			if (keygen_algorithm_is_ok(info.kmode) != 1) {
				err_crit
				    ("No such mode. Assumed s2k_salted.\n");
				if ( nosalt==TRUE) nosalt=FALSE;
				info.kmode=NULL;
			} else {
				keymode=find_keygen_algorithm(info.kmode);
				if ( mhash_keygen_uses_salt(keymode)==0 ) {
					nosalt=TRUE;
				} else {
					nosalt=FALSE;
				}
			}
		} else {
			if ( nosalt==TRUE) nosalt=FALSE;
		}
		password =
		    (void *) Getpass(algorithm, "Enter keyword:", &plen,
				     info.keys, info.kmode);
#ifdef HAVE_MLOCK
		mlock(password, plen);
#endif
		use_password = 1;


	}



	file = malloc((info.size) * sizeof(char *));

	i = 0;
	file_count = (info.size);

	while (i < info.size) {
		file[i] = info.input[i];
		i++;
	}

	if (file_count == 0) {	/* streams for check */
		file = malloc(sizeof(char *));
		file[0] = NULL;
		file_count++;
	}

	if (file[0] != NULL && info.check == FALSE) {
		tmp = mhash_get_hash_name(algorithm);
		tmp2 = mhash_get_keygen_name(keymode);
		if (hmac == 1) {
			strcpy(msg, "HMAC");

			if (nosalt == FALSE) {
				printf("# %s %s %s %s\n", tmp, msg, tmp2, "SALTED");
			} else {
				printf("# %s %s %s\n", tmp, msg, tmp2);
			}

		} else {
			strcpy(msg, "HASH");
			printf("# %s %s\n", tmp, msg);
		}
		free(tmp);
		free(tmp2);
	}

	for (i = 0; i < file_count; i++) {
		if (info.check == 0) {
			if (nosalt == FALSE && hmac == 1) {
				salt[0] = get_rand32();
				salt[1] = get_rand32();
				tmp2 =
				    bin2hex((void *) &salt, 8,
					    (size_t *) & newlen);
				key = pass2key(algorithm, info.kmode, &keylen, password, plen);
				if (key==NULL) err_quit("Error in key mode\n");
				hash = sumfile(file[i], algorithm, key, keylen);
			} else {
				if (nosalt==TRUE) {
					set_salt=1;
					key = pass2key(algorithm, info.kmode, &keylen, password, plen);
					if (key==NULL) err_quit("Error in key mode\n");
					hash = sumfile(file[i], algorithm, key, keylen);
				} else {
					hash = sumfile(file[i], algorithm, password, plen);				}
			}
	

			if (hash != NULL) {
				if (nosalt == FALSE && hmac == 1) {
					printf("%s ", tmp2);
					free(tmp2);
				}
				view(file[i], hash,
				     mhash_get_block_size(algorithm));
				free(hash);
			}

		} else {

			x =
			    checkfile(file[i], algorithm, password, plen,
				      info.kmode);

		}
	}

	if (password != NULL)
		bzero(password, plen);
#ifdef HAVE_MLOCK
	munlock(password, plen);
#endif
	free(password);

	return 0;

}

hashid check_algo(char *chain)
{
	hashid i;
	char *y;

	for (i = 0; i < 255; i++) {
		y = mhash_get_hash_name(i);
		if (y == NULL)
			continue;
		if (strcasecmp(y, chain) == 0) {
			free(y);
			return i;
		}
		free(y);
	}

	return -1;
}

void view(char *desc, void *x, int sizeofx)
{
	int j;
	unsigned char *y = x;


	for (j = 0; j < sizeofx; j++) {
		printf("%.2x", y[j]);
	}
	if (binary == TRUE) {
		if (desc != NULL) {
			printf(" *%s", desc);
		}
	} else {
		if (desc != NULL) {
			printf("  %s", desc);
		}
	}
	printf("\n");

}


#ifdef HAVE_FTIME
# include <sys/timeb.h>
#else
# include <time.h>
#endif

/* Accepts time elapsed in milliseconds and file size in bytes.
 */
static void show_stats( time_t ms_time, size_t file_size) {
time_t time = ms_time/1000;
time_t millitm = ms_time % 1000;
int show = 0;

	fprintf(stderr, "Time needed: ");
	
	if (time/3600 > 0) {
		fprintf(stderr, "%lu hours", time/3600);
		show = 1;
	}
	
	if ((time/60)-( (time/3600)*60 ) > 0) {
		if (show>0) fprintf(stderr, ", ");
		fprintf( stderr, "%lu minutes", (time/60)-(time/3600) );
		show = 1;
	}

	if (time - ((time/60)+((time/3600)*60)*60 ) > 0) {
		if (show>0) fprintf(stderr, ", ");
		fprintf( stderr, "%lu seconds", time-((time/60)+(time/3600)) );
		show = 1;
	}
	
	if (millitm > 0) {
		if (show>0) fprintf(stderr, ", ");
		fprintf( stderr, "%lu milliseconds", millitm);
		show = 1;
	}
	
	if (show==0)
		fprintf(stderr, "Wow, no time counted!\n");
	else {
		fprintf(stderr, ".\n");
	
		if (file_size > 1000000) {
			fprintf(stderr, "Hashed %.3f MB, at a %.3f MB/sec rate (1 MB == 1000000 Bytes).\n", 
				(float)( (double)file_size/(double)1000000),
				(float)( (( (double) file_size/ (double)1000000)/ (double)ms_time)*(double)1000));
		} else { /* KB */
			fprintf(stderr, "Hashed %.3f KB, at a %.3f KB/sec rate (1 KB == 1000 Bytes).\n", 
				(float)( (double)file_size/(double)1000),
				(float)( (( (double) file_size/ (double)1000)/ (double)ms_time)*(double)1000));
		}
	}
}



#define BUFFER_SIZE 32*1024
void *sumfile(char *filename, hashid algorithm, char *password, int plen)
{
	FILE *file;
	char buffer[BUFFER_SIZE];
	MHASH td;
	int x;
#ifdef HAVE_FTIME
	struct timeb start_time;
	struct timeb end_time;
#else
	time_t start_time;
	time_t end_time;
#endif
	size_t file_sum;


	if (filename != NULL) {
		if (binary == TRUE) {
			file = fopen(filename, "rb");
		} else {
			file = fopen(filename, "r");
		}

		if (file == NULL) {
			sprintf(msg, "%s: No such file.\n", filename);
			err_crit(msg);
			return NULL;
		}

		if (read_lock(fileno(file)) == -1) {
			fclose(file);
			return NULL;
		}
	} else {
		file = stdin;

		if (file == NULL) {
			sprintf(msg, "stdin: Error.\n");
			err_crit(msg);
			return NULL;
		}

	}

	if (password == NULL) {
		td = mhash_init(algorithm);
	} else {

		td =
		    mhash_hmac_init(algorithm, password, plen,
				    mhash_get_hash_pblock(algorithm));
	}

	if (td == MHASH_FAILED) {
		sprintf(msg, "Internal library error.\n");
		fclose(file);
		err_crit(msg);
		return NULL;
	}

#ifdef HAVE_FTIME
	ftime(&start_time);
#else
	start_time = time(0);
#endif
	
	file_sum = 0;
	
	for (;;) {
		clearerr( file);
		x = fread(buffer, 1, BUFFER_SIZE, file);
		file_sum += x;

		if (x == 0) {
			if (ferror(file)!=0) {
				fprintf(stderr, "Error in file\n");
				fclose(file);
				return NULL;
			}
			break;
		}
		mhash(td, &buffer, x);
	}

#ifdef HAVE_FTIME
	ftime(&end_time);
#else	
	end_time = time(0);
#endif
	
	unlock(fileno(file));
	fclose(file);

	if (show_time!=0) {
#ifdef HAVE_FTIME
		end_time.time*=1000;
		end_time.time+=end_time.millitm;

		start_time.time*=1000;
		start_time.time+=start_time.millitm;
		
		end_time.time -= start_time.time;
		
		start_time.time = end_time.time;
		
		end_time.millitm = end_time.time % 1000;

		show_stats( end_time.time, file_sum);

#else
		end_time -= start_time;

		show_stats( end_time, file_sum);

#endif
	}

	if (password == NULL) {
		return mhash_end(td);
	} else {
		return mhash_hmac_end(td);
	}
}

int hash_list()
{
	hashid i;
	char *y;

	for (i = 0; i < 255; i++) {
		y = mhash_get_hash_name(i);
		if (y == NULL)
			continue;
		printf("%s\n", y);
		free(y);
	}

	exit(0);
}

int checkfile(char *filename, hashid algorithm, char *keyword, int plen,
	      char *kmode)
{
	FILE *file;
	int counter = 0, counter_failed = 0;
	int j, x, block;
	unsigned char *sumtmp;
	char *linebuf;
	int line_storage;
	char *buffer, *password = NULL;
	char *buffer2, *keymode_buffer;
	char *hexbuffer;
	char *tmp;
	char *key = NULL;
	int keylen;

	block = mhash_get_block_size(algorithm);


	buffer = malloc(block * 2 + 1);
	buffer2 = malloc(MAX_NAME + 1);
	keymode_buffer = malloc(MAX_NAME + 1);
	hexbuffer = malloc(block * 2 + 1);

	if ((block * 2 + MAX_NAME) > 1024) {
		line_storage = (block * 2) + MAX_NAME + 1;
	} else {
		line_storage = 1024 + 1;
	}

	linebuf = malloc(line_storage);


	if (filename != NULL) {
		file = fopen(filename, "rb");
		if (file == NULL) {
			sprintf(msg, "%s: No such file.\n", filename);
			err_crit(msg);
			return 0;
		}

	} else {
		file = stdin;
		if (file == NULL) {
			sprintf(msg, "stdin: Error.\n");
			err_crit(msg);
			return 0;
		}

	}

	if (filename != NULL) {
		if (read_lock(fileno(file)) == -1) {
			sprintf(msg, "%s: File is locked.\n", filename);
			err_crit(msg);
			return 0;
		}
	}


	x = 1;

	for (;;) {

		fgets(linebuf, line_storage, file);

		if (linebuf == NULL)
			break;
		if (feof(file) != 0)
			break;
		if (linebuf[0] == '#' && linebuf[1] == ' ') {

			if (sscanf(linebuf, "# %s %s", buffer, buffer2) !=
			    0) {
				algorithm = check_algo(buffer);

				if (strcasecmp(buffer2, "hmac") == 0) {	/* hmac */
					hmac = 1;
					if (sscanf(linebuf, "# %s %s %s %s",
						   buffer, buffer2, keymode_buffer,
						   hexbuffer) == 3) {
						nosalt = TRUE;
					} else {
						nosalt = FALSE;
					}

					if (keyword == NULL) {
						password =
						    Getpass
						    (algorithm,
						     "Enter keyword:",
						     &plen, keyword,
						     "asis");


					} else {
						password = malloc(plen);
						memmove(password, keyword,
							plen);
					}


#ifdef HAVE_MLOCK
					mlock(password, plen);
#endif

				} else {
					hmac = 0;
				}

				block = mhash_get_block_size(algorithm);

				buffer = realloc(buffer, block * 2 + 1);
				buffer2 = realloc(buffer2, MAX_NAME + 1);
				hexbuffer =
				    realloc(hexbuffer, block * 2 + 1);

				if ((block * 2 + MAX_NAME) > 1024) {
					line_storage =
					    (block * 2) + MAX_NAME + 1;
				} else {
					line_storage = 1024 + 1;
				}
				linebuf = realloc(linebuf, line_storage);
			}

			continue;
		}

		/* checks so it does not overflow */
		if (strlen(linebuf) > (block * 2) + MAX_NAME + 2)
			continue;
		if (strlen(linebuf) < block * 2)
			continue;

		bzero(buffer2, block * 2);

		if (nosalt == FALSE && hmac == 1) {
			if (sscanf
			    (linebuf, "%s %s  %s\n", hexbuffer, buffer,
			     buffer2) < 2) {
				if (sscanf
				    (linebuf, "%s %s *%s\n", hexbuffer,
				     buffer, buffer2) < 2)
					continue;
			}

			if (strlen(hexbuffer) > 16) {
				err_crit("Error in file format\n");
				return 1;
			}

			hex2bin(hexbuffer, salt, strlen(hexbuffer));
			key = pass2key(algorithm, keymode_buffer, &keylen, password, plen);

		} else {
			if (sscanf(linebuf, "%s  %s\n", buffer, buffer2) ==
			    0) {
				if (sscanf
				    (linebuf, "%s *%s\n", buffer,
				     buffer2) == 0)
					continue;
			}

			set_salt=1;
			key = pass2key(algorithm, keymode_buffer, &keylen, password, plen);
		}

		sumtmp = sumfile(buffer2, algorithm, key, keylen);

		bzero(hexbuffer, block * 2 + 1);

		if (sumtmp != NULL) {
			for (j = 0; j < block; j++) {
				sprintf(&hexbuffer[j * 2], "%.2x",
					sumtmp[j]);
			}


			if (strcmp(hexbuffer, buffer) == 0) {
				sprintf(msg, "%s: OK\n", buffer2);
				err_warn(msg);
				counter++;

			} else {
				tmp = mhash_get_hash_name(algorithm);
				sprintf(msg,
					"%s check failed for '%s'\n",
					tmp, buffer2);
				free(tmp);
				err_crit(msg);
				counter_failed++;
			}
			free(sumtmp);
		}

	}

	free(buffer);
	free(buffer2);
	free(hexbuffer);

	if (filename != NULL)
		unlock(fileno(file));
	fclose(file);

	tmp = mhash_get_hash_name(algorithm);
	sprintf(msg, "%d of %d file(s) failed %s check.\n",
		counter_failed, counter_failed + counter, tmp);
	err_warn(msg);
	if (password != NULL && counter == 0)
		err_crit("Probably wrong password.\n");
	free(tmp);

	if (password != NULL) {
		bzero(password, plen);
#ifdef HAVE_MLOCK
		munlock(password, plen);
#endif
		free(password);
	}

	return counter;
}

void hash_license()
{
	fprintf(stdout, ("\nCopyright (C) 1999 Nikos Mavroyanopoulos\n"
"This program is free software; you can redistribute it and/or modify \n"
"it under the terms of the GNU General Public License as published by \n"
"the Free Software Foundation; either version 2 of the License, or \n"
"(at your option) any later version. \n"
"\n"
"This program is distributed in the hope that it will be useful, \n"
"but WITHOUT ANY WARRANTY; without even the implied warranty of \n"
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the \n"
"GNU General Public License for more details. \n"
"\n"
"You should have received a copy of the GNU General Public License \n"
"along with this program; if not, write to the Free Software \n"
"Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.\n\n"));
}

void hash_version()
{
	fprintf(stderr, ("shash v.%s (%s-%s-%s)\n"), VERSION, T_CPU,
		T_VENDOR, T_OS);
	fprintf(stderr,
		("Copyright (C) 1999 Nikos Mavroyanopoulos (nmav@hellug.gr)\n"));

} 


void *Getpass(hashid algorithm, char *key, int *keylen, void *pkey,
	      char *kmode)
{

	char *tmpkey;
	char *password, *userkey;

	userkey = malloc(MAX_KEY + 1);
#ifdef HAVE_MLOCK
	mlock(userkey, MAX_KEY);
#endif

	if (pkey == NULL) {
		tmpkey = (void *) getpass("Enter password:");
		strncpy(userkey, tmpkey, MAX_KEY);
		bzero(tmpkey, strlen(tmpkey));
		if (double_check == TRUE) {
			tmpkey = (void *) getpass("Retype password:");
			if (strcmp(tmpkey, userkey) != 0)
				err_quit("Passwords do not match.\n");
			bzero(tmpkey, strlen(tmpkey));
		}
	} else {
		strncpy(userkey, pkey, MAX_KEY);
	}


	if (strlen((char *) userkey) == 0)
		err_quit("Error in keyword.\n");


	*keylen=strlen(userkey);
	password=calloc(1, *keylen);
	memmove(password, userkey, *keylen);
	free(userkey);
	return password;

}

int check_hex(char *given_chain, int len)
{
	int i;

	for (i = 0; i < len; i++)
		if (isxdigit(given_chain[i]) == 0)
			return -1;

	return 0;
}

void list_keygen_algorithms(void)
{
	int i;
	char *name;

	printf("Key Generation algorithms supported: \n");
	for (i = 0; i < mhash_keygen_count(); i++) {
		name = mhash_get_keygen_name(i);
		printf("%s\n", name);
		free(name);
	}
}

int keygen_algorithm_is_ok(char *algo)
{
	int i;
	char *name;

	for (i = 0; i < mhash_keygen_count(); i++) {
		name = mhash_get_keygen_name(i);
		if (strcasecmp(name, algo) == 0) {
			free(name);
			return 1;
		}
		free(name);
	}
	return 0;
}

static char hexconvtab[] = "0123456789abcdef";

static char *bin2hex(const unsigned char *old, const size_t oldlen,
		     size_t * newlen)
{
	unsigned char *new = NULL;
	int i, j;

	new = malloc(oldlen * 2 * sizeof(char) + 1);
	if (!new) {
		return (char *) new;
	}

	for (i = j = 0; i < oldlen; i++) {
		new[j++] = hexconvtab[old[i] >> 4];
		new[j++] = hexconvtab[old[i] & 15];
	}
	new[j] = '\0';

	if (newlen)
		*newlen = oldlen * 2 * sizeof(char);

	return (char *) new;
}


void hex2bin(void *src, void *dst, int sizeofsrc)
{
	int j;
	char *source = src;
	char *dest = dst;
	char tmp[3];

	bzero(dst, sizeofsrc / 2);

	if (sizeofsrc % 2 != 0)
		return;
	if (check_hex(source, sizeofsrc) == -1)
		return;

	tmp[2] = '\0';

	for (j = 0; j < sizeofsrc; j += 2) {
		memmove(tmp, &source[j], 2);
		dest[j / 2] = strtol(tmp, (char **) NULL, 16);
	}
	return;

}

void *pass2key(hashid algorithm, char *kmode, int *keylen, void *password, int plen)
{

	char *key;
	keygenid id = 0;
	hashid algo = 0;
	void* local_salt=salt;
	int salt_size=8;

	if (plen == 0)
		return NULL;

	if (kmode!=NULL) {
		id = find_keygen_algorithm(kmode);
	} else {
		if (set_salt==1) return NULL;
		id = KEYGEN_S2K_SALTED;
		algo = algorithm;
		/* hash is the default */
		*keylen = 20;
		key = calloc(1, *keylen);
		if (mhash_keygen(id, algo, 0, key, *keylen, salt, 8,
				 (word8 *) password, (int) plen) < 0)
			return NULL;

		return key;
	}

	
	if (set_salt==1 && mhash_keygen_uses_salt(id)==1) return NULL;
	if (set_salt==0 && mhash_keygen_uses_salt(id)==0) return NULL;

	*keylen=20; /* 20 bytes key */
	if (strcasecmp(kmode, "asis")==0) *keylen=plen;
	if (strcasecmp(kmode, "hex")==0) *keylen=plen/2;
	
	algo=algorithm;

		key = calloc(1, *keylen);
		if (set_salt==1) {
			local_salt=NULL;
			salt_size=0;
		}
		if (mhash_keygen(id, algo, 0, key, *keylen, local_salt, salt_size,
				 (word8 *) password, plen) < 0)
			return NULL;

	return key;

}


int find_keygen_algorithm(char *algo)
{
	int i;
	char *name;

	if (algo==NULL) return -1;
	
	for (i = 0; i < mhash_keygen_count(); i++) {
		name = mhash_get_keygen_name(i);
		if (name==NULL) continue;
		if (strcasecmp(name, algo) == 0) {
			free(name);
			return i;
		}
		free(name);
	}
	return -1;
}
