/* snefru.c */

/* Snefru is a cryptographic hash function invented by Ralph Merkle
 * which supports 128-bit and 256-bit output. It was named after the
 * Egyptian Pharaoh Sneferu, continuing the tradition of the Khufu and
 * Khafre block ciphers.
 *
 * Snefru is a public-domain one-way hash function. For more information see
 * http://www.funet.fi/pub/crypt/mirrors/idea.sec.dsi.unimi.it/rpub.cl.msu.edu/
 *   crypt/docs/merkle-khufu-khafre-snefru.txt.gz
 *
 * This code was written by B. Poettering in 2004 for the mhash library. It
 * is in the public domain. 
 */

#include "libdefs.h"

#ifdef ENABLE_SNEFRU

#include "mhash_snefru.h"


#define R 8  // number of rounds

static __const mutils_word32 SBOX[R * 256 * 2]= {
  0x64f9001bl, 0xfeddcdf6l, 0x7c8ff1e2l, 0x11d71514l, 0x8b8c18d3l, 0xdddf881el,
  0x6eab5056l, 0x88ced8e1l, 0x49148959l, 0x69c56fd5l, 0xb7994f03l, 0x0fbcee3el,
  0x3c264940l, 0x21557e58l, 0xe14b3fc2l, 0x2e5cf591l, 0xdceff8cel, 0x092a1648l,
  0xbe812936l, 0xff7b0c6al, 0xd5251037l, 0xafa448f1l, 0x7dafc95al, 0x1ea69c3fl,
  0xa417abe7l, 0x5890e423l, 0xb0cb70c0l, 0xc85025f7l, 0x244d97e3l, 0x1ff3595fl,
  0xc4ec6396l, 0x59181e17l, 0xe635b477l, 0x354e7dbfl, 0x796f7753l, 0x66eb52ccl,
  0x77c3f995l, 0x32e3a927l, 0x80ccaed6l, 0x4e2be89dl, 0x375bbd28l, 0xad1a3d05l,
  0x2b1b42b3l, 0x16c44c71l, 0x4d54bfa8l, 0xe57ddc7al, 0xec6d8144l, 0x5a71046bl,
  0xd8229650l, 0x87fc8f24l, 0xcbc60e09l, 0xb6390366l, 0xd9f76092l, 0xd393a70bl,
  0x1d31a08al, 0x9cd971c9l, 0x5c1ef445l, 0x86fab694l, 0xfdb44165l, 0x8eaafcbel,
  0x4bcac6ebl, 0xfb7a94e5l, 0x5789d04el, 0xfa13cf35l, 0x236b8da9l, 0x4133f000l,
  0x6224261cl, 0xf412f23bl, 0xe75e56a4l, 0x30022116l, 0xbaf17f1fl, 0xd09872f9l,
  0xc1a3699cl, 0xf1e802aal, 0x0dd145dcl, 0x4fdce093l, 0x8d8412f0l, 0x6cd0f376l,
  0x3de6b73dl, 0x84ba737fl, 0xb43a30f2l, 0x44569f69l, 0x00e4eacal, 0xb58de3b0l,
  0x959113c8l, 0xd62efee9l, 0x90861f83l, 0xced69874l, 0x2f793ceel, 0xe8571c30l,
  0x483665d1l, 0xab07b031l, 0x914c844fl, 0x15bf3be8l, 0x2c3f2a9al, 0x9eb95fd4l,
  0x92e7472dl, 0x2297cc5bl, 0xee5f2782l, 0x5377b562l, 0xdb8ebbcfl, 0xf961deddl,
  0xc59b5c60l, 0x1bd3910dl, 0x26d206adl, 0xb28514d8l, 0x5ecf6b52l, 0x7fea78bbl,
  0x504879acl, 0xed34a884l, 0x36e51d3cl, 0x1753741dl, 0x8c47caedl, 0x9d0a40efl,
  0x3145e221l, 0xda27eb70l, 0xdf730ba3l, 0x183c8789l, 0x739ac0a6l, 0x9a58dfc6l,
  0x54b134c1l, 0xac3e242el, 0xcc493902l, 0x7b2dda99l, 0x8f15bc01l, 0x29fd38c7l,
  0x27d5318fl, 0x604aaff5l, 0xf29c6818l, 0xc38aa2ecl, 0x1019d4c3l, 0xa8fb936el,
  0x20ed7b39l, 0x0b686119l, 0x89a0906fl, 0x1cc7829el, 0x9952ef4bl, 0x850e9e8cl,
  0xcd063a90l, 0x67002f8el, 0xcfac8cb7l, 0xeaa24b11l, 0x988b4e6cl, 0x46f066dfl,
  0xca7eec08l, 0xc7bba664l, 0x831d17bdl, 0x63f575e6l, 0x9764350el, 0x47870d42l,
  0x026ca4a2l, 0x8167d587l, 0x61b6adabl, 0xaa6564d2l, 0x70da237bl, 0x25e1c74al,
  0xa1c901a0l, 0x0eb0a5dal, 0x7670f741l, 0x51c05aeal, 0x933dfa32l, 0x0759ff1al,
  0x56010ab8l, 0x5fdecb78l, 0x3f32edf8l, 0xaebedbb9l, 0x39f8326dl, 0xd20858c5l,
  0x9b638be4l, 0xa572c80al, 0x28e0a19fl, 0x432099fcl, 0x3a37c3cdl, 0xbf95c585l,
  0xb392c12al, 0x6aa707d7l, 0x52f66a61l, 0x12d483b1l, 0x96435b5el, 0x3e75802bl,
  0x3ba52b33l, 0xa99f51a5l, 0xbda1e157l, 0x78c2e70cl, 0xfcae7ce0l, 0xd1602267l,
  0x2affac4dl, 0x4a510947l, 0x0ab2b83al, 0x7a04e579l, 0x340dfd80l, 0xb916e922l,
  0xe29d5e9bl, 0xf5624af4l, 0x4ca9d9afl, 0x6bbd2cfel, 0xe3b7f620l, 0xc2746e07l,
  0x5b42b9b6l, 0xa06919bcl, 0xf0f2c40fl, 0x72217ab5l, 0x14c19df3l, 0xf3802dael,
  0xe094beb4l, 0xa2101affl, 0x0529575dl, 0x55cdb27cl, 0xa33bddb2l, 0x6528b37dl,
  0x740c05dbl, 0xe96a62c4l, 0x40782846l, 0x6d30d706l, 0xbbf48e2cl, 0xbce2d3del,
  0x049e37fal, 0x01b5e634l, 0x2d886d8dl, 0x7e5a2e7el, 0xd7412013l, 0x06e90f97l,
  0xe45d3ebal, 0xb8ad3386l, 0x13051b25l, 0x0c035354l, 0x71c89b75l, 0xc638fbd0l,
  0x197f11a1l, 0xef0f08fbl, 0xf8448651l, 0x38409563l, 0x452f4443l, 0x5d464d55l,
  0x03d8764cl, 0xb1b8d638l, 0xa70bba2fl, 0x94b3d210l, 0xeb6692a7l, 0xd409c2d9l,
  0x68838526l, 0xa6db8a15l, 0x751f6c98l, 0xde769a88l, 0xc9ee4668l, 0x1a82a373l,
  0x0896aa49l, 0x42233681l, 0xf62c55cbl, 0x9f1c5404l, 0xf74fb15cl, 0xc06e4312l,
  0x6ffe5d72l, 0x8aa8678bl, 0x337cd129l, 0x8211cefdl, 0x074a1d09l, 0x52a10e5al,
  0x9275a3f8l, 0x4b82506cl, 0x37df7e1bl, 0x4c78b3c5l, 0xcefab1dal, 0xf472267el,
  0xb63045f6l, 0xd66a1fc0l, 0x400298e3l, 0x27e60c94l, 0x87d2f1b8l, 0xdf9e56ccl,
  0x45cd1803l, 0x1d35e098l, 0xcce7c736l, 0x03483bf1l, 0x1f7307d7l, 0xc6e8f948l,
  0xe613c111l, 0x3955c6ffl, 0x1170ed7cl, 0x8e95da41l, 0x99c31bf4l, 0xa4da8021l,
  0x7b5f94fbl, 0xdd0da51fl, 0x6562aa77l, 0x556bcb23l, 0xdb1bacc6l, 0x798040b9l,
  0xbfe5378fl, 0x731d55e6l, 0xdaa5bfeel, 0x389bbc60l, 0x1b33fba4l, 0x9c567204l,
  0x36c26c68l, 0x77ee9d69l, 0x8aeb3e88l, 0x2d50b5cel, 0x9579e790l, 0x42b13cfcl,
  0x33fbd32bl, 0xee0503a7l, 0xb5862824l, 0x15e41eadl, 0xc8412ef7l, 0x9d441275l,
  0x2fcec582l, 0x5ff483b7l, 0x8f3931dfl, 0x2e5d2a7bl, 0x49467bf9l, 0x0653dea9l,
  0x2684ce35l, 0x7e655e5cl, 0xf12771d8l, 0xbb15cc67l, 0xab097ca1l, 0x983dcf52l,
  0x10ddf026l, 0x21267f57l, 0x2c58f6b4l, 0x31043265l, 0x0bab8c01l, 0xd5492099l,
  0xacaae619l, 0x944ce54al, 0xf2d13d39l, 0xadd3fc32l, 0xcda08a40l, 0xe2b0d451l,
  0x9efe08ael, 0xb9d50fd2l, 0xea5cd7fdl, 0xc9a749ddl, 0x13ea2253l, 0x832debaal,
  0x24be640fl, 0xe03e926al, 0x29e01cdel, 0x8bf59f18l, 0x0f9d00b6l, 0xe1238b46l,
  0x1e7d8e34l, 0x93619adbl, 0x76b32f9fl, 0xbd972cecl, 0xe31fa976l, 0xa68fbb10l,
  0xfb3ba49dl, 0x8587c41dl, 0xa5add1d0l, 0xf3cf84bfl, 0xd4e11150l, 0xd9ffa6bcl,
  0xc3f6018cl, 0xaef10572l, 0x74a64b2fl, 0xe7dc9559l, 0x2aae35d5l, 0x5b6f587fl,
  0xa9e353fel, 0xca4fb674l, 0x04ba24a8l, 0xe5c6875fl, 0xdcbc6266l, 0x6bc5c03fl,
  0x661eef02l, 0xed740babl, 0x058e34e4l, 0xb7e946cfl, 0x88698125l, 0x72ec48edl,
  0xb11073a3l, 0xa13485ebl, 0xa2a2429cl, 0xfa407547l, 0x50b76713l, 0x5418c37dl,
  0x96192da5l, 0x170bb04bl, 0x518a021el, 0xb0ac13d1l, 0x0963fa2al, 0x4a6e10e1l,
  0x58472bdcl, 0xf7f8d962l, 0x979139eal, 0x8d856538l, 0xc0997042l, 0x48324d7al,
  0x447623cbl, 0x8cbbe364l, 0x6e0c6b0el, 0xd36d63b0l, 0x3f244c84l, 0x3542c971l,
  0x2b228dc1l, 0xcb0325bbl, 0xf8c0d6e9l, 0xde11066bl, 0xa8649327l, 0xfc31f83el,
  0x7dd80406l, 0xf916dd61l, 0xd89f79d3l, 0x615144c2l, 0xebb45d31l, 0x28002958l,
  0x56890a37l, 0xf05b3808l, 0x123ae844l, 0x86839e16l, 0x914b0d83l, 0xc506b43cl,
  0xcf3cba5el, 0x7c60f5c9l, 0x22deb2a0l, 0x5d9c2715l, 0xc77ba0efl, 0x4f45360bl,
  0xc1017d8bl, 0xe45adc29l, 0xa759909bl, 0x412cd293l, 0xd7d796b1l, 0x00c8ff30l,
  0x23a34a80l, 0x4ec15c91l, 0x714e78b5l, 0x47b9e42el, 0x78f3ea4dl, 0x7f078f5bl,
  0x346c593al, 0xa3a87a1al, 0x9bcbfe12l, 0x3d439963l, 0xb2ef6d8el, 0xb8d46028l,
  0x6c2fd5cal, 0x62675256l, 0x01f2a2f3l, 0xbc96ae0al, 0x709a8920l, 0xb4146e87l,
  0x6308b9e2l, 0x64bda7bal, 0xafed6892l, 0x6037f2a2l, 0xf52969e0l, 0x0adb43a6l,
  0x82811400l, 0x90d0bdf0l, 0x19c9549el, 0x203f6a73l, 0x1accaf4fl, 0x89714e6dl,
  0x164d4705l, 0x67665f07l, 0xec206170l, 0x0c2182b2l, 0xa02b9c81l, 0x53289722l,
  0xf6a97686l, 0x140e4179l, 0x9f778849l, 0x9a88e15dl, 0x25cadb54l, 0xd157f36fl,
  0x32a421c3l, 0xb368e98al, 0x5a92cd0dl, 0x757aa8d4l, 0xc20ac278l, 0x08b551c7l,
  0x849491e8l, 0x4dc75ad6l, 0x697c33bel, 0xbaf0ca33l, 0x46125b4el, 0x59d677b3l,
  0x30d9c8f2l, 0xd0af860cl, 0x1c7fd0fal, 0xfe0ff72cl, 0x5c8d6f43l, 0x57fdec3bl,
  0x6ab6ad97l, 0xd22adf89l, 0x18171785l, 0x02bfe22dl, 0x6db80917l, 0x80b216afl,
  0xe85e4f9al, 0x7a1c306el, 0x6fc49bf5l, 0x3af7a11cl, 0x81e215e7l, 0x68363fcdl,
  0x3e9357c8l, 0xef52fd55l, 0x3b8bab4cl, 0x3c8cf495l, 0xbefceebdl, 0xfd25b714l,
  0xc498d83dl, 0x0d2e1a8dl, 0xe9f966acl, 0x0e387445l, 0x435419e5l, 0x5e7ebec4l,
  0xaa90b8d9l, 0xff1a3a96l, 0x4a8fe4e3l, 0xf27d99cdl, 0xd04a40cal, 0xcb5ff194l,
  0x3668275al, 0xff4816bel, 0xa78b394cl, 0x4c6be9dbl, 0x4eec38d2l, 0x4296ec80l,
  0xcdce96f8l, 0x888c2f38l, 0xe75508f5l, 0x7b916414l, 0x060aa14al, 0xa214f327l,
  0xbe608dafl, 0x1ebbdec2l, 0x61f98ce9l, 0xe92156fel, 0x4f22d7a3l, 0x3f76a8d9l,
  0x559a4b33l, 0x38ad2959l, 0xf3f17e9el, 0x85e1ba91l, 0xe5eba6fbl, 0x73dcd48cl,
  0xf5c3ff78l, 0x481b6058l, 0x8a3297f7l, 0x8f1f3bf4l, 0x93785ab2l, 0x477a4a5bl,
  0x6334eb5dl, 0x6d251b2el, 0x74a9102dl, 0x07e38ffal, 0x915c9c62l, 0xccc275eal,
  0x6be273ecl, 0x3ebddd70l, 0xd895796cl, 0xdc54a91bl, 0xc9afdf81l, 0x23633f73l,
  0x275119b4l, 0xb19f6b67l, 0x50756e22l, 0x2bb152e2l, 0x76ea46a2l, 0xa353e232l,
  0x2f596ad6l, 0x0b1edb0bl, 0x02d3d9a4l, 0x78b47843l, 0x64893e90l, 0x40f0caadl,
  0xf68d3ad7l, 0x46fd1707l, 0x1c9c67efl, 0xb5e086del, 0x96ee6ca6l, 0x9aa34774l,
  0x1ba4f48al, 0x8d01abfdl, 0x183ee1f6l, 0x5ff8aa7al, 0x17e4faael, 0x303983b0l,
  0x6c08668bl, 0xd4ac4382l, 0xe6c5849fl, 0x92fefb53l, 0xc1cac4cel, 0x43501388l,
  0x441118cfl, 0xec4fb308l, 0x53a08e86l, 0x9e0fe0c5l, 0xf91c1525l, 0xac45be05l,
  0xd7987cb5l, 0x49ba1487l, 0x57938940l, 0xd5877648l, 0xa958727fl, 0x58dfe3c3l,
  0xf436cf77l, 0x399e4d11l, 0xf0a5bfa9l, 0xef61a33bl, 0xa64cac60l, 0x04a8d0bal,
  0x030dd572l, 0xb83d320fl, 0xcab23045l, 0xe366f2f0l, 0x815d008dl, 0xc897a43al,
  0x1d352df3l, 0xb9cc571dl, 0x8bf38744l, 0x72209092l, 0xeba124ebl, 0xfb99ce5el,
  0x3bb94293l, 0x28da549cl, 0xaab8a228l, 0xa4197785l, 0x33c70296l, 0x25f6259bl,
  0x5c85da21l, 0xdf15bdeel, 0x15b7c7e8l, 0xe2abef75l, 0xfcc19bc1l, 0x417ff868l,
  0x14884434l, 0x62825179l, 0xc6d5c11cl, 0x0e4705dcl, 0x22700de0l, 0xd3d2af18l,
  0x9be822a0l, 0x35b669f1l, 0xc42bb55cl, 0x0a801252l, 0x115bf0fcl, 0x3cd7d856l,
  0xb43f5f9dl, 0xc2306516l, 0xa1231c47l, 0xf149207el, 0x5209a795l, 0x34b3ccd8l,
  0x67aefe54l, 0x2c83924el, 0x6662cbacl, 0x5eedd161l, 0x84e681aal, 0x5d57d26bl,
  0xfa465cc4l, 0x7e3ac3a8l, 0xbf7c0cc6l, 0xe18a9aa1l, 0xc32f0a6fl, 0xb22cc00dl,
  0x3d280369l, 0x994e554fl, 0x68f480d3l, 0xadcff5e6l, 0x3a8eb265l, 0x83269831l,
  0xbd568a09l, 0x4bc8ae6al, 0x69f56d2bl, 0x0f17eac8l, 0x772eb6c7l, 0x9f41343cl,
  0xab1d0742l, 0x826a6f50l, 0xfea2097cl, 0x1912c283l, 0xce185899l, 0xe4444839l,
  0x2d8635d5l, 0x65d0b1ffl, 0x865a7f17l, 0x326d9fb1l, 0x59e52820l, 0x0090ade1l,
  0x753c7149l, 0x9ddd8b98l, 0xa5a691dal, 0x0d0382bbl, 0x8904c930l, 0x086cb000l,
  0x6e69d3bdl, 0x24d4e7a7l, 0x05244fd0l, 0x101a5e0cl, 0x6a947dcbl, 0xe840f77bl,
  0x7d0c5003l, 0x7c370f1fl, 0x805245edl, 0xe05e3d3fl, 0x7906880el, 0xbabfcd35l,
  0x1a7ec697l, 0x8c052324l, 0x0c6ec8dfl, 0xd129a589l, 0xc7a75b02l, 0x12d81de7l,
  0xd9be2a66l, 0x1f4263abl, 0xde73fdb6l, 0x2a00680al, 0x56649e36l, 0x3133ed55l,
  0x90fa0bf2l, 0x2910a02al, 0x949d9d46l, 0xa0d1dcddl, 0xcfc9b7d4l, 0xd2677be5l,
  0x95cb36b3l, 0x13cd9410l, 0xdbf73313l, 0xb7c6e8c0l, 0xf781414bl, 0x510b016dl,
  0xb0de1157l, 0xd6b0f62cl, 0xbb074eccl, 0x7f1395b7l, 0xee792cf9l, 0xea6fd63el,
  0x5bd6938el, 0xaf02fc64l, 0xdab57ab8l, 0x8edb3784l, 0x8716318fl, 0x164d1a01l,
  0x26f26141l, 0xb372e6b9l, 0xf8fc2b06l, 0x7ac00e04l, 0x3727b89al, 0x97e9bca5l,
  0x9c2a742fl, 0xbc3b1f7dl, 0x7165b471l, 0x609b4c29l, 0x20925351l, 0x5ae72112l,
  0x454be5d1l, 0xc0ffb95fl, 0xdd0ef919l, 0x6f2d70c9l, 0x0974c5bfl, 0x98aa6263l,
  0x01d91e4dl, 0x2184bb6el, 0x70c43c1el, 0x4d435915l, 0xae7b8523l, 0xb6fb06bcl,
  0x5431ee76l, 0xfdbc5d26l, 0xed77493dl, 0xc5712ee4l, 0xa8380437l, 0x2eef261al,
  0x5a79392bl, 0xb8af32c2l, 0x41f7720al, 0x833a61ecl, 0x13dfedacl, 0xc4990bc4l,
  0xdc0f54bcl, 0xfedd5e88l, 0x80da1881l, 0x4dea1afdl, 0xfd402cc6l, 0xae67cc7al,
  0xc5238525l, 0x8ea01254l, 0xb56b9bd5l, 0x862fbd6dl, 0xac8575d3l, 0x6fba3714l,
  0xda7ebf46l, 0x59cd5238l, 0x8ac9dbfel, 0x353729fcl, 0xe497d7f2l, 0xc3ab84e0l,
  0xf05a114bl, 0x7b887a75l, 0xedc603ddl, 0x5e6fe680l, 0x2c84b399l, 0x884eb1dal,
  0x1cb8c8bfl, 0xaa51098al, 0xc862231cl, 0x8bac2221l, 0x21b387e5l, 0x208a430dl,
  0x2a3f0f8bl, 0xa5ff9cd2l, 0x6012a2eal, 0x147a9ee7l, 0xf62a501dl, 0xb4b2e51al,
  0x3ef3484cl, 0xc0253c59l, 0x2b82b536l, 0x0aa9696bl, 0xbe0c109bl, 0xc70b7929l,
  0xce3e8a19l, 0x2f66950el, 0x459f1c2cl, 0xe68fb93dl, 0xa3c3ff3el, 0x62b45c62l,
  0x300991cbl, 0x01914c57l, 0x7f7bc06al, 0x182831f5l, 0xe7b74bcal, 0xfa50f6d0l,
  0x523caa61l, 0xe3a7cf05l, 0xe9e41311l, 0x280a21d1l, 0x6a4297e1l, 0xf24dc67el,
  0xfc3189e6l, 0xb72bf34fl, 0x4b1e67afl, 0x543402cel, 0x79a59867l, 0x0648e02al,
  0x00a3ac17l, 0xc6208d35l, 0x6e7f5f76l, 0xa45bb4bel, 0xf168fa63l, 0x3f4125f3l,
  0xf311406fl, 0x02706565l, 0xbfe58022l, 0x0cfcfdd9l, 0x0735a7f7l, 0x8f049092l,
  0xd98edc27l, 0xf5c5d55cl, 0xe0f201dbl, 0x0dcafc9al, 0x7727fb79l, 0xaf43abf4l,
  0x26e938c1l, 0x401b26a6l, 0x900720fal, 0x2752d97bl, 0xcff1d1b3l, 0xa9d9e424l,
  0x42db99abl, 0x6cf8be5fl, 0xe82cebe3l, 0x3afb733bl, 0x6b734eb6l, 0x1036414al,
  0x975f667cl, 0x049d6377l, 0xba587c60l, 0xb1d10483l, 0xde1aefccl, 0x1129d055l,
  0x72051e91l, 0x6946d623l, 0xf9e86ea7l, 0x48768c00l, 0xb0166c93l, 0x9956bbf0l,
  0x1f1f6d84l, 0xfb15e18el, 0x033b495dl, 0x56e3362el, 0x4f44c53cl, 0x747cba51l,
  0x89d37872l, 0x5d9c331bl, 0xd2ef9fa8l, 0x254917f8l, 0x1b106f47l, 0x37d75553l,
  0xb3f053b0l, 0x7dccd8efl, 0xd30eb802l, 0x5889f42dl, 0x610206d7l, 0x1a7d34a1l,
  0x92d87dd8l, 0xe5f4a315l, 0xd1cf0e71l, 0xb22dfe45l, 0xb901e8ebl, 0x0fc0ce5el,
  0x2efa60c9l, 0x2de74290l, 0x36d0c906l, 0x381c70e4l, 0x4c6da5b5l, 0x3d81a682l,
  0x7e381f34l, 0x396c4f52l, 0x95ad5901l, 0x1db50c5al, 0x29982e9el, 0x1557689fl,
  0x3471ee42l, 0xd7e2f7c0l, 0x8795a1e2l, 0xbc324d8dl, 0xe224c3c8l, 0x12837e39l,
  0xcdee3d74l, 0x7ad2143fl, 0x0e13d40cl, 0x78bd4a68l, 0xa2eb194dl, 0xdb9451f9l,
  0x859b71dcl, 0x5c4f5b89l, 0xca14a8a4l, 0xef92f003l, 0x16741d98l, 0x33aa4444l,
  0x9e967fbbl, 0x092e3020l, 0xd86a35b8l, 0x8cc17b10l, 0xe1bf08ael, 0x55693fc5l,
  0x7680ad13l, 0x1e6546e8l, 0x23b6e7b9l, 0xee77a4b2l, 0x08ed0533l, 0x44fd2895l,
  0xb6393b69l, 0x05d6cacfl, 0x9819b209l, 0xecbbb72fl, 0x9a75779cl, 0xeaec0749l,
  0x94a65aeel, 0xbdf52dc3l, 0xd6a25d04l, 0x82008e4el, 0xa6de160fl, 0x9b036afbl,
  0x228b3a66l, 0x5fb10a70l, 0xcc338b58l, 0x5378a9dfl, 0xc908bca9l, 0x4959e25bl,
  0x46909a97l, 0x66ae8f6el, 0xdd0683e9l, 0x65f994b4l, 0x6426cda5l, 0xc24b8840l,
  0x32539da0l, 0x63175650l, 0xd0c815ffl, 0x50cbc41el, 0xf7c774a3l, 0x31b0c231l,
  0x8d0d8116l, 0x24bef16cl, 0xd555d256l, 0xdf47ea8cl, 0x6d21eccdl, 0xa887a012l,
  0x84542aedl, 0xa7b9c1bdl, 0x914c1bb1l, 0xa0d5b67dl, 0x438ce937l, 0x7030f873l,
  0x71f6b0c7l, 0x574576bal, 0xf8bc4541l, 0x9c61d348l, 0x1960579dl, 0x17c4daadl,
  0x96a4cb0bl, 0xc193f2f6l, 0x756eafa2l, 0x7c1d2f94l, 0xf4fe2b43l, 0xcb86e33al,
  0xebd4c728l, 0x9d18ae64l, 0x9fe13e30l, 0x3ce0f5del, 0xaba1f985l, 0xaddc2718l,
  0x68ce6278l, 0xd45e241fl, 0xa15c82b7l, 0x3b2293d4l, 0x739edd32l, 0x674a6bf1l,
  0x5b5d587fl, 0x4772deaal, 0x4a63968fl, 0x0be68686l, 0x513d6426l, 0x939a4787l,
  0xbba89296l, 0x4ec20007l, 0x818d0d08l, 0xff64dfd6l, 0xcb2297cbl, 0xdb48a144l,
  0xa16cbe4bl, 0xbbea1d6cl, 0x5af6b6b7l, 0x8a8110b6l, 0xf9236ef9l, 0xc98f83e6l,
  0x0f9c65b8l, 0x252d4a89l, 0xa497f068l, 0xa5d7ed2dl, 0x94c22845l, 0x9da1c8c4l,
  0xe27c2e2el, 0x6e8ba2b4l, 0xc3dd17fbl, 0x498cd482l, 0x0dfe6a9fl, 0xb0705829l,
  0x9a1e6dc1l, 0xf829717cl, 0x07bb8e3al, 0xda3c0b02l, 0x1af82fc7l, 0x73b70955l,
  0x7a04379cl, 0x5ee20a28l, 0x83712ae5l, 0xf4c47c6dl, 0xdf72ba56l, 0xd794858dl,
  0x8c0cf709l, 0x18f0f390l, 0xb6c69b35l, 0xbf2f01dbl, 0x2fa74dcal, 0xd0cd9127l,
  0xbde66cecl, 0x3deebd46l, 0x57c88fc3l, 0xcee1406fl, 0x0066385al, 0xf3c3444fl,
  0x3a79d5d5l, 0x75751eb9l, 0x3e7f8185l, 0x521c2605l, 0xe1aaab6el, 0x38ebb80fl,
  0xbee7e904l, 0x61cb9647l, 0xea54904el, 0x05ae00e4l, 0x2d7ac65fl, 0x087751a1l,
  0xdcd82915l, 0x0921ee16l, 0xdd86d33bl, 0xd6bd491al, 0x40fbadf0l, 0x4232cbd2l,
  0x33808d10l, 0x39098c42l, 0x193f3199l, 0x0bc1e47al, 0x4a82b149l, 0x02b65a8al,
  0x104cdc8el, 0x24a8f52cl, 0x685c6077l, 0xc79f95c9l, 0x1d11fe50l, 0xc08dafcdl,
  0x7b1a9a03l, 0x1c1f11d8l, 0x84250e7fl, 0x979db248l, 0xebdc0501l, 0xb9553395l,
  0xe3c05ea8l, 0xb1e51c4cl, 0x13b0e681l, 0x3b407766l, 0x36db3087l, 0xee17c9fcl,
  0x6c53ecf2l, 0xadccc58fl, 0xc427660bl, 0xefd5867dl, 0x9b6d54a5l, 0x6ff1aeffl,
  0x8e787952l, 0x9e2bffe0l, 0x8761d034l, 0xe00bdbadl, 0xae99a8d3l, 0xcc03f6e2l,
  0xfd0ed807l, 0x0e508ae3l, 0xb74182abl, 0x4349245dl, 0xd120a465l, 0xb246a641l,
  0xaf3b7ab0l, 0x2a6488bbl, 0x4b3a0d1fl, 0xe7c7e58cl, 0x3faff2ebl, 0x90445ffdl,
  0xcf38c393l, 0x995d07e7l, 0xf24f1b36l, 0x356f6891l, 0x6d6ebcbel, 0x8da9e262l,
  0x50fd520el, 0x5bca9e1el, 0x37472cf3l, 0x69075057l, 0x7ec5fdedl, 0x0cab892al,
  0xfb2412bal, 0x1728debfl, 0xa000a988l, 0xd843ce79l, 0x042e20ddl, 0x4fe8f853l,
  0x56659c3cl, 0x2739d119l, 0xa78a6120l, 0x80960375l, 0x70420611l, 0x85e09f78l,
  0xabd17e96l, 0x1b513eafl, 0x1e01eb63l, 0x26ad2133l, 0xa890c094l, 0x7613cf60l,
  0x817e781bl, 0xa39113d7l, 0xe957fa58l, 0x4131b99el, 0x28b1efdal, 0x66acfba7l,
  0xff68944al, 0x77a44fd1l, 0x7f331522l, 0x59ffb3fal, 0xa6df935bl, 0xfa12d9dfl,
  0xc6bf6f3fl, 0x89520cf6l, 0x659edd6al, 0x544da739l, 0x8b052538l, 0x7c30ea21l,
  0xc2345525l, 0x15927fb2l, 0x144a436bl, 0xba107b8bl, 0x1219ac97l, 0x06730432l,
  0x31831ab3l, 0xc55a5c24l, 0xaa0fcd3el, 0xe5606be8l, 0x5c88f19bl, 0x4c0841eel,
  0x1fe37267l, 0x11f9c4f4l, 0x9f1b9dael, 0x864e76d0l, 0xe637c731l, 0xd97d23a6l,
  0x32f53d5cl, 0xb8161980l, 0x93fa0f84l, 0xcaef0870l, 0x8874487el, 0x98f2cc73l,
  0x645fb5c6l, 0xcd853659l, 0x2062470dl, 0x16ede8e9l, 0x6b06dab5l, 0x78b43900l,
  0xfc95b786l, 0x5d8e7de1l, 0x465b5954l, 0xfe7ba014l, 0xf7d23f7bl, 0x92bc8b18l,
  0x03593592l, 0x55cef4f7l, 0x74b27317l, 0x79de1fc2l, 0xc8a0bfbdl, 0x229398ccl,
  0x62a602cel, 0xbcb94661l, 0x5336d206l, 0xd2a375fel, 0x6a6ab483l, 0x4702a5a4l,
  0xa2e9d73dl, 0x23a2e0f1l, 0x9189140al, 0x581d18dcl, 0xb39a922bl, 0x82356212l,
  0xd5f432a9l, 0xd356c2a3l, 0x5f765b4dl, 0x450afcc8l, 0x4415e137l, 0xe8ecdfbcl,
  0xed0de3eal, 0x60d42b13l, 0xf13df971l, 0x71fc5da2l, 0xc1455340l, 0xf087742fl,
  0xf55e5751l, 0x67b3c1f8l, 0xac6b8774l, 0x7dcfaaacl, 0x95983bc0l, 0x489bb0b1l,
  0x2c184223l, 0x964b6726l, 0x2bd3271cl, 0x72266472l, 0xded64530l, 0x0a2aa343l,
  0xd4f716a0l, 0xb4dad6d9l, 0x2184345el, 0x512c990cl, 0x29d92d08l, 0x2ebe709al,
  0x01144c69l, 0x34584b9dl, 0xe4634ed6l, 0xecc963cfl, 0x3c6984aal, 0x4ed056efl,
  0x9ca56976l, 0x8f3e80d4l, 0xb5bae7c5l, 0x30b5caf5l, 0x63f33a64l, 0xa9e4bbdel,
  0xf6b82298l, 0x4d673c1dl, 0x4b4f1121l, 0xba183081l, 0xc784f41fl, 0xd17d0bacl,
  0x083d2267l, 0x37b1361el, 0x3581ad05l, 0xfda2f6bcl, 0x1e892cddl, 0xb56d3c3al,
  0x32140e46l, 0x138d8aabl, 0xe14773d4l, 0x5b0e71dfl, 0x5d1fe055l, 0x3fb991d3l,
  0xf1f46c71l, 0xa325988cl, 0x10f66e80l, 0xb1006348l, 0x726a9f60l, 0x3b67f8bal,
  0x4e114ef4l, 0x05c52115l, 0x4c5ca11cl, 0x99e1efd8l, 0x471b83b3l, 0xcbf7e524l,
  0x43ad82f5l, 0x690ca93bl, 0xfaa61bb2l, 0x12a832b5l, 0xb734f943l, 0xbd22aea7l,
  0x88fec626l, 0x5e80c3e7l, 0xbe3eaf5el, 0x44617652l, 0xa5724475l, 0xbb3b9695l,
  0x7f3fee8fl, 0x964e7debl, 0x518c052dl, 0x2a0bbc2bl, 0xc2175f5cl, 0x9a7b3889l,
  0xa70d8d0cl, 0xeaccdd29l, 0xcccd6658l, 0x34bb25e6l, 0xb8391090l, 0xf651356fl,
  0x52987c9el, 0x0c16c1cdl, 0x8e372d3cl, 0x2fc6ebbdl, 0x6e5da3e3l, 0xb0e27239l,
  0x5f685738l, 0x45411786l, 0x067f65f8l, 0x61778b40l, 0x81ab2e65l, 0x14c8f0f9l,
  0xa6b7b4cel, 0x4036eaecl, 0xbf62b00al, 0xecfd5e02l, 0x045449a6l, 0xb20afd28l,
  0x2166d273l, 0x0d13a863l, 0x89508756l, 0xd51a7530l, 0x2d653f7al, 0x3cdbdbc3l,
  0x80c9df4fl, 0x3d5812d9l, 0x53fbb1f3l, 0xc0f185c0l, 0x7a3c3d7el, 0x68646410l,
  0x857607a0l, 0x1d12622el, 0x97f33466l, 0xdb4c9917l, 0x6469607cl, 0x566e043dl,
  0x79ef1edbl, 0x2c05898dl, 0xc9578e25l, 0xcd380101l, 0x46e04377l, 0x7d1cc7a9l,
  0x6552b837l, 0x20192608l, 0xb97500c5l, 0xed296b44l, 0x368648b4l, 0x62995cd5l,
  0x82731400l, 0xf9aebd8bl, 0x3844c0c7l, 0x7c2de794l, 0x33a1a770l, 0x8ae528c2l,
  0x5a2be812l, 0x1f8f4a07l, 0x2b5ed7cal, 0x937eb564l, 0x6fda7e11l, 0xe49b5d6cl,
  0xb4b3244el, 0x18aa53a4l, 0x3a061334l, 0x4d6067a3l, 0x83ba5868l, 0x9bdf4dfel,
  0x7449f261l, 0x709f8450l, 0xcad133cbl, 0xde941c3fl, 0xf52ae484l, 0x781d77edl,
  0x7e4395f0l, 0xae103b59l, 0x922331bbl, 0x42ce50c8l, 0xe6f08153l, 0xe7d941d0l,
  0x5028ed6bl, 0xb3d2c49bl, 0xad4d9c3el, 0xd201fb6el, 0xa45bd5bel, 0xffcb7f4bl,
  0x579d7806l, 0xf821bb5bl, 0x59d592adl, 0xd0be0c31l, 0xd4e3b676l, 0x0107165al,
  0x0fe939d2l, 0x49bcaafdl, 0x55ffcfe5l, 0x2ec1f783l, 0xf39a09a5l, 0x3eb42772l,
  0x19b55a5dl, 0x024a0679l, 0x8c83b3f7l, 0x8642ba1dl, 0xacacd9eal, 0x87d352c4l,
  0x60931f45l, 0xa05f97d7l, 0x1cecd42cl, 0xe2fcc87bl, 0xb60f94e2l, 0x67a34b0bl,
  0xfcdd40c9l, 0x0b150a27l, 0xd3ee9e04l, 0x582e29e9l, 0x4ac22b41l, 0x6ac4e1b8l,
  0xbccaa51al, 0x237af30el, 0xebc3b709l, 0xc4a59d19l, 0x284bc98al, 0xe9d41a93l,
  0x6bfa2018l, 0x73b2d651l, 0x11f9a2fal, 0xce09bff1l, 0x41a470aal, 0x25888f22l,
  0x77e754e8l, 0xf7330d8el, 0x158eab16l, 0xc5d68842l, 0xc685a6f6l, 0xe5b82fdel,
  0x09ea3a96l, 0x6dde1536l, 0x4fa919dal, 0x26c0be9fl, 0x9eed6f69l, 0xf05555f2l,
  0xe06fc285l, 0x9cd76d23l, 0xaf452a92l, 0xefc74cb7l, 0x9d6b4732l, 0x8be408eel,
  0x22401d0dl, 0xee6c459dl, 0x7587cb82l, 0xe8746862l, 0x5cbdde87l, 0x98794278l,
  0x31afb94dl, 0xc11e0f2fl, 0x30e8fc2al, 0xcf3261efl, 0x1a3023e1l, 0xaa2f86cfl,
  0xf202e24al, 0x8d08dcffl, 0x764837c6l, 0xa26374ccl, 0x9f7c3e88l, 0x949cc57dl,
  0xdd26a07fl, 0xc39efab0l, 0xc8f879a1l, 0xdce67bb9l, 0xf4b0a435l, 0x912c9ae0l,
  0xd85603e4l, 0x953a9bbfl, 0xfb8290d6l, 0x0aebcd5fl, 0x16206a9al, 0x6c787a14l,
  0xd9a0f16al, 0x29bf4f74l, 0x8f8bce91l, 0x0e5a9354l, 0xab038cb1l, 0x1b8ad11bl,
  0xe327ff49l, 0x0053da20l, 0x90cf51dcl, 0xda92fe6dl, 0x0390ca47l, 0xa8958097l,
  0xa9dc5bafl, 0x3931e3c1l, 0x840446b6l, 0x63d069fbl, 0xd7460299l, 0x7124ecd1l,
  0x0791e613l, 0x485918fcl, 0xd635d04cl, 0xdf96ac33l, 0x66f2d303l, 0x247056ael,
  0xa1a7b2a8l, 0x27d8cc9cl, 0x17b6e998l, 0x7bf5590fl, 0xfe97f557l, 0x5471d8a2l,
  0x83a327a1l, 0x9f379f51l, 0x40a7d007l, 0x11307423l, 0x224587c1l, 0xac27d63bl,
  0x3b7e64eal, 0x2e1cbfa6l, 0x09996000l, 0x03bc0e2cl, 0xd4c4478al, 0x4542e0abl,
  0xfeda26d4l, 0xc1d10fcbl, 0x8252f596l, 0x4494eb5cl, 0xa362f314l, 0xf5ba81fdl,
  0x75c3a376l, 0x4ca214cal, 0xe164deddl, 0x5088fa97l, 0x4b0930e0l, 0x2fcfb7e8l,
  0x33a6f4b2l, 0xc7e94211l, 0x2d66c774l, 0x43be8bael, 0xc663d445l, 0x908eb130l,
  0xf4e3be15l, 0x63b9d566l, 0x529396b5l, 0x1e1be743l, 0x4d5ff63fl, 0x985e4a83l,
  0x71ab9df7l, 0xc516c6f5l, 0x85c19ab4l, 0x1f4daee4l, 0xf2973431l, 0xb713dc5el,
  0x3f2e159al, 0xc824da16l, 0x06bf376al, 0xb2fe23ecl, 0xe39b1c22l, 0xf1eecb5fl,
  0x08e82d52l, 0x565686c2l, 0xab0aea93l, 0xfd47219fl, 0xebdbabd7l, 0x2404a185l,
  0x8c7312b9l, 0xa8f2d828l, 0x0c8902dal, 0x65b42b63l, 0xc0bbef62l, 0x4e3e4cefl,
  0x788f8018l, 0xee1ebab7l, 0x93928f9dl, 0x683d2903l, 0xd3b60689l, 0xafcb0ddcl,
  0x88a4c47al, 0xf6dd9c3dl, 0x7ea5fca0l, 0x8a6d7244l, 0xbe11f120l, 0x04ff91b8l,
  0x8d2dc8c0l, 0x27f97fdbl, 0x7f9e1f47l, 0x1734f0c7l, 0x26f3ed8el, 0x0df8f2bfl,
  0xb0833d9el, 0xe420a4e5l, 0xa423cae6l, 0x95616772l, 0x9ae6c049l, 0x075941f2l,
  0xd8e12812l, 0x000f6f4fl, 0x3c0d6b05l, 0x6cef921cl, 0xb82bc264l, 0x396cb008l,
  0x5d608a6fl, 0x6d7782c8l, 0x186550aal, 0x6b6fec09l, 0x28e70b13l, 0x57ce5688l,
  0xecd3af84l, 0x23335a95l, 0x91f40cd2l, 0x7b6a3b26l, 0xbd32b3b6l, 0x3754a6fbl,
  0x8ed088f0l, 0xf867e87cl, 0x20851746l, 0x6410f9c6l, 0x35380442l, 0xc2ca10a7l,
  0x1adea27fl, 0x76bddd79l, 0x92742cf4l, 0x0e98f7eel, 0x164e931dl, 0xb9c835b3l,
  0x69060a99l, 0xb44c531el, 0xfa7b66fel, 0xc98a5b53l, 0x7d95aae9l, 0x302f467bl,
  0x74b811del, 0xf3866abdl, 0xb5b3d32dl, 0xfc3157a4l, 0xd251fe19l, 0x0b5d8eacl,
  0xda71ffd5l, 0x47ea05a3l, 0x05c6a9e1l, 0xca0ee958l, 0x9939034dl, 0x25dc5edfl,
  0x79083cb1l, 0x86768450l, 0xcf757d6dl, 0x5972b6bcl, 0xa78d59c9l, 0xc4ad8d41l,
  0x2a362ad3l, 0xd1179991l, 0x601407ffl, 0xdcf50917l, 0x587069d0l, 0xe0821ed6l,
  0xdbb59427l, 0x73911a4bl, 0x7c904fc3l, 0x844afb92l, 0x6f8c955dl, 0xe8c0c5bbl,
  0xb67ab987l, 0xa529d96cl, 0xf91f7181l, 0x618b1b06l, 0xe718bb0cl, 0x8bd7615bl,
  0xd5a93a59l, 0x54aef81bl, 0x772136e3l, 0xce44fd9cl, 0x10cda57el, 0x87d66e0bl,
  0x3d798967l, 0x1b2c1804l, 0x3edfbd68l, 0x15f6e62bl, 0xef68b854l, 0x3896db35l,
  0x12b7b5e2l, 0xcb489029l, 0x9e4f98a5l, 0x62eb77a8l, 0x217c24a2l, 0x964152f6l,
  0x49b2080al, 0x53d23ee7l, 0x48fb6d69l, 0x1903d190l, 0x9449e494l, 0xbf6e7886l,
  0xfb356cfal, 0x3a261365l, 0x424bc1ebl, 0xa1192570l, 0x019ca782l, 0x9d3f7e0el,
  0x9c127575l, 0xedf02039l, 0xad57bccel, 0x5c153277l, 0x81a84540l, 0xbcaa7356l,
  0xccd59b60l, 0xa62a629bl, 0xa25ccd10l, 0x2b5b65cfl, 0x1c535832l, 0x55fd4e3al,
  0x31d9790dl, 0xf06bc37dl, 0x4afc1d71l, 0xaeed5533l, 0xba461634l, 0xbb694b78l,
  0x5f3a5c73l, 0x6a3c764al, 0x8fb0cca9l, 0xf725684cl, 0x4fe5382fl, 0x1d0163afl,
  0x5aa07a8fl, 0xe205a8edl, 0xc30bad38l, 0xff22cf1fl, 0x72432e2el, 0x32c2518bl,
  0x3487ce4el, 0x7ae0ac02l, 0x709fa098l, 0x0a3b395al, 0x5b4043f8l, 0xa9e48c36l,
  0x149a8521l, 0xd07dee6bl, 0x46acd2f3l, 0x8958dffcl, 0xb3a1223cl, 0xb11d31c4l,
  0xcd7f4d3el, 0x0f28e3adl, 0xe5b100bel, 0xaac54824l, 0xe9c9d7bal, 0x9bd47001l,
  0x80f149b0l, 0x66022f0fl, 0x020c4048l, 0x6efa192al, 0x67073f8dl, 0x13ec7bf9l,
  0x3655011al, 0xe6afe157l, 0xd9845f6el, 0xdecc4425l, 0x511ae2ccl, 0xdf81b4d8l,
  0xd7809e55l, 0xd6d883d9l, 0x2cc7978cl, 0x5e787cc5l, 0xdd0033d1l, 0xa050c937l,
  0x97f75dcdl, 0x299de580l, 0x41e2b261l, 0xea5a54f1l, 0x7e672590l, 0xbea513bbl,
  0x2c906fe6l, 0x86029c2bl, 0x55dc4f74l, 0x0553398el, 0x63e09647l, 0xcafd0babl,
  0x264c37dfl, 0x8272210fl, 0x67afa669l, 0x12d98a5fl, 0x8cab23c4l, 0x75c68bd1l,
  0xc3370470l, 0x33f37f4el, 0x283992ffl, 0xe73a3a67l, 0x1032f283l, 0xf5ad9fc2l,
  0x963f0c5dl, 0x664fbc45l, 0x202ba41cl, 0xc7c02d80l, 0x54731e84l, 0x8a1085f5l,
  0x601d80fbl, 0x2f968e55l, 0x35e96812l, 0xe45a8f78l, 0xbd7de662l, 0x3b6e6eadl,
  0x8097c5efl, 0x070b6781l, 0xb1e508f3l, 0x24e4fae3l, 0xb81a7805l, 0xec0fc918l,
  0x43c8774bl, 0x9b2512a9l, 0x2b05ad04l, 0x32c2536fl, 0xedf236e0l, 0x8bc4b0cfl,
  0xbaceb837l, 0x4535b289l, 0x0d0e94c3l, 0xa5a371d0l, 0xad695a58l, 0x39e3437dl,
  0x9186bffcl, 0x21038c3bl, 0x0aa9dff9l, 0x5d1f06cel, 0x62def8a4l, 0xf740a2b4l,
  0xa2575868l, 0x682683c1l, 0xdbb30facl, 0x61fe1928l, 0x468a6511l, 0xc61cd5f4l,
  0xe54d9800l, 0x6b98d7f7l, 0x8418b6a5l, 0x5f09a5d2l, 0x90b4e80bl, 0x49b2c852l,
  0x69f11c77l, 0x17412b7el, 0x7f6fc0edl, 0x56838dccl, 0x6e9546a2l, 0xd0758619l,
  0x087b9b9al, 0xd231a01dl, 0xaf46d415l, 0x097060fdl, 0xd920f657l, 0x882d3f9fl,
  0x3ae7c3c9l, 0xe8a00d9bl, 0x4fe67ebel, 0x2ef80eb2l, 0xc1916b0cl, 0xf4dffea0l,
  0xb97eb3ebl, 0xfdff84ddl, 0xff8b14f1l, 0xe96b0572l, 0xf64b508cl, 0xae220a6el,
  0x4423ae5al, 0xc2bece5el, 0xde27567cl, 0xfc935c63l, 0x47075573l, 0xe65b27f0l,
  0xe121fd22l, 0xf2668753l, 0x2debf5d7l, 0x8347e08dl, 0xac5eda03l, 0x2a7cebe9l,
  0x3fe8d92el, 0x23542fe4l, 0x1fa7bd50l, 0xcf9b4102l, 0x9d0dba39l, 0x9cb8902al,
  0xa7249d8bl, 0x0f6d667al, 0x5ebfa9ecl, 0x6a594df2l, 0x79600938l, 0x023b7591l,
  0xea2c79c8l, 0xc99d07eal, 0x64cb5ee1l, 0x1a9cab3dl, 0x76db9527l, 0xc08e012fl,
  0x3dfb481al, 0x872f22e7l, 0x2948d15cl, 0xa4782c79l, 0x6f50d232l, 0x78f0728al,
  0x5a87aab1l, 0xc4e2c19cl, 0xee767387l, 0x1b2a1864l, 0x7b8d10d3l, 0xd1713161l,
  0x0eeac456l, 0xd8799e06l, 0xb645b548l, 0x4043cb65l, 0xa874fb29l, 0x4b12d030l,
  0x7d687413l, 0x18ef9a1fl, 0xd7631d4cl, 0x5829c7dal, 0xcdfa30fal, 0xc5084bb0l,
  0x92cd20e2l, 0xd4c16940l, 0x03283ec0l, 0xa917813fl, 0x9a587d01l, 0x70041f8fl,
  0xdc6ab1dcl, 0xddaee3d5l, 0x31829742l, 0x198c022dl, 0x1c9eafcbl, 0x5bbc6c49l,
  0xd3d3293al, 0x16d50007l, 0x04bb8820l, 0x3c5c2a41l, 0x37ee7af8l, 0x8eb04025l,
  0x9313ecbal, 0xbffc4799l, 0x8955a744l, 0xef85d633l, 0x504499a7l, 0xa6ca6a86l,
  0xbb3d3297l, 0xb34a8236l, 0x6dccbe4fl, 0x06143394l, 0xce19fc7bl, 0xccc3c6c6l,
  0xe36254ael, 0x77b7eda1l, 0xa133dd9el, 0xebf9356al, 0x513ccf88l, 0xe2a1b417l,
  0x972ee5bdl, 0x853824cdl, 0x5752f4eel, 0x6c1142e8l, 0x3ea4f309l, 0xb2b5934al,
  0xdfd628aal, 0x59acea3el, 0xa01eb92cl, 0x389964bcl, 0xda305dd4l, 0x019a59b7l,
  0x11d2ca93l, 0xfaa6d3b9l, 0x4e772ecal, 0x72651776l, 0xfb4e5b0el, 0xa38f91a8l,
  0x1d0663b5l, 0x30f4f192l, 0xb50051b6l, 0xb716ccb3l, 0x4abd1b59l, 0x146c5f26l,
  0xf134e2del, 0x00f67c6cl, 0xb0e1b795l, 0x98aa4ec7l, 0x0cc73b34l, 0x654276a3l,
  0x8d1ba871l, 0x740a5216l, 0xe0d01a23l, 0x9ed161d6l, 0x9f36a324l, 0x993ebb7fl,
  0xfeb9491bl, 0x365ddcdbl, 0x810cffc5l, 0x71ec0382l, 0x2249e7bfl, 0x48817046l,
  0xf3a24a5bl, 0x4288e4d9l, 0x0bf5c243l, 0x257fe151l, 0x95b64c0dl, 0x4164f066l,
  0xaaf7db08l, 0x73b1119dl, 0x8f9f7bb8l, 0xd6844596l, 0xf07a34a6l, 0x53943d0al,
  0xf9dd166dl, 0x7a8957afl, 0xf8ba3ce5l, 0x27c9621el, 0x5cdae910l, 0xc8518998l,
  0x941538fel, 0x136115d8l, 0xaba8443cl, 0x4d01f931l, 0x34edf760l, 0xb45f266bl,
  0xd5d4de14l, 0x52d8ac35l, 0x15cfd885l, 0xcbc5cd21l, 0x4cd76d4dl, 0x7c80ef54l,
  0xbc92ee75l, 0x1e56a1f6l, 0xbaa20b6cl, 0x9ffbad26l, 0xe1f7d738l, 0x794aec8dl,
  0xc9e9cf3cl, 0x8a9a7846l, 0xc57c4685l, 0xb9a92fedl, 0x29cb141fl, 0x52f9ddb7l,
  0xf68ba6bcl, 0x19ccc020l, 0x4f584aaal, 0x3bf6a596l, 0x003b7cf7l, 0x54f0ce9al,
  0xa7ec4303l, 0x46cf0077l, 0x78d33aa1l, 0x215247d9l, 0x74bcdf91l, 0x08381d30l,
  0xdac43e40l, 0x64872531l, 0x0beffe5fl, 0xb317f457l, 0xaebb12dal, 0xd5d0d67bl,
  0x7d75c6b4l, 0x42a6d241l, 0x1502d0a9l, 0x3fd97fffl, 0xc6c3ed28l, 0x81868d0al,
  0x92628bc5l, 0x86679544l, 0xfd1867afl, 0x5ca3ea61l, 0x568d5578l, 0x4a2d71f4l,
  0x43c9d549l, 0x8d95de2bl, 0x6e5c74a0l, 0x9120ffc7l, 0x0d05d14al, 0xa93049d3l,
  0xbfa80e17l, 0xf4096810l, 0x043f5ef5l, 0xa673b4f1l, 0x6d780298l, 0xa4847783l,
  0x5ee726fbl, 0x9934c281l, 0x220a588cl, 0x384e240fl, 0x933d5c69l, 0x39e5ef47l,
  0x26e8b8f3l, 0x4c1c6212l, 0x8040f75dl, 0x074b7093l, 0x6625a8d7l, 0x36298945l,
  0x76285088l, 0x651d37c3l, 0x24f5274dl, 0xdbca3dabl, 0x186b7ee1l, 0xd80f8182l,
  0x14210c89l, 0x943a3075l, 0x4e6e11c4l, 0x4d7e6badl, 0xf05064c8l, 0x025dcd97l,
  0x4bc10302l, 0x7cede572l, 0x8f90a970l, 0xab88eebal, 0xb5998029l, 0x5124d839l,
  0xb0eeb6a3l, 0x89ddabdcl, 0xe8074d76l, 0xa1465223l, 0x32518cf2l, 0x9d39d4ebl,
  0xc0d84524l, 0xe35e6ea8l, 0x7abf3804l, 0x113e2348l, 0x9ae6069dl, 0xb4dfdabbl,
  0xa8c5313fl, 0x23ea3f79l, 0x530e36a2l, 0xa5fd228bl, 0x95d1d350l, 0x2b14cc09l,
  0x40042956l, 0x879d05ccl, 0x2064b9cal, 0xacaca40el, 0xb29c846el, 0x9676c9e3l,
  0x752b7b8al, 0x7be2bcc2l, 0x6bd58f5el, 0xd48f4c32l, 0x606835e4l, 0x9cd7c364l,
  0x2c269b7al, 0x3a0d079cl, 0x73b683fel, 0x45374f1el, 0x10afa242l, 0x577f8666l,
  0xddaa10f6l, 0xf34f561cl, 0x3d355d6bl, 0xe47048ael, 0xaa13c492l, 0x050344fdl,
  0x2aab5151l, 0xf5b26ae5l, 0xed919a59l, 0x5ac67900l, 0xf1cde380l, 0x0c79a11bl,
  0x351533fcl, 0xcd4d8e36l, 0x1f856005l, 0x690b9fddl, 0xe736dccfl, 0x1d47bf6al,
  0x7f66c72al, 0x85f21b7fl, 0x983cbdb6l, 0x01ebbebfl, 0x035f3b99l, 0xeb111f34l,
  0x28cefdc6l, 0x5bfc9ecdl, 0xf22eacb0l, 0x9e41cbb2l, 0xe0f8327cl, 0x82e3e26fl,
  0xfc43fc86l, 0xd0ba66dfl, 0x489ef2a7l, 0xd9e0c81dl, 0x68690d52l, 0xcc451367l,
  0xc2232e16l, 0xe95a7335l, 0x0fdae19bl, 0xff5b962cl, 0x97596527l, 0xc46db333l,
  0x3ed4c562l, 0xc14c9d9el, 0x5d6faa21l, 0x638e940dl, 0xf9316d58l, 0x47b3b0eal,
  0x30ffcad2l, 0xce1bba7dl, 0x1e6108e6l, 0x2e1ea33dl, 0x507bf05bl, 0xfafef94bl,
  0xd17de8e2l, 0x5598b214l, 0x1663f813l, 0x17d25a2dl, 0xeefa5ff9l, 0x582f4e37l,
  0x12128773l, 0xfef17ab8l, 0x06005322l, 0xbb32bbc9l, 0x8c898508l, 0x592c15f0l,
  0xd38a4054l, 0x4957b7d6l, 0xd2b891dbl, 0x37bd2d3el, 0x34ad20cbl, 0x622288e9l,
  0x2dc7345al, 0xafb416c0l, 0x1cf459b1l, 0xdc7739fal, 0x0a711a25l, 0x13e18a0cl,
  0x5f72af4cl, 0x6ac8db11l, 0xbe53c18el, 0x1aa569b9l, 0xef551ea4l, 0xa02a429fl,
  0xbd16e790l, 0x7eb9171al, 0x77d693d8l, 0x8e06993al, 0x9bde7560l, 0xe5801987l,
  0xc37a09bel, 0xb8db76acl, 0xe2087294l, 0x6c81616dl, 0xb7f30fe7l, 0xbc9b82bdl,
  0xfba4e4d4l, 0xc7b1012fl, 0xa20c043bl, 0xde9febd0l, 0x2f9297cel, 0xe610aef8l,
  0x70b06f19l, 0xc86ae00bl, 0x0e01988fl, 0x41192ae0l, 0x448c1cb5l, 0xadbe92eel,
  0x7293a007l, 0x1b54b5b3l, 0xd61f63d1l, 0xeae40a74l, 0x61a72b55l, 0xec83a7d5l,
  0x88942806l, 0x90a07da5l, 0xd7424b95l, 0x67745b4el, 0xa31a1853l, 0xca6021efl,
  0xdfb56c4fl, 0xcbc2d915l, 0x3c48e918l, 0x8bae3c63l, 0x6f659c71l, 0xf8b754c1l,
  0x2782f3del, 0xf796f168l, 0x71492c84l, 0x33c0f5a6l, 0x3144f6ecl, 0x25dc412el,
  0xb16c5743l, 0x83a1fa7el, 0x0997b101l, 0xb627e6e8l, 0xcf33905cl, 0x8456fb65l,
  0xb29bea74l, 0xc35da605l, 0x305c1ca3l, 0xd2e9f5bcl, 0x6fd5bff4l, 0xff347703l,
  0xfc45b163l, 0xf498e068l, 0xb71229fcl, 0x81acc3fbl, 0x78538a8bl, 0x984ecf81l,
  0xa5da47a4l, 0x8f259eefl, 0x6475dc65l, 0x081865b9l, 0x49e14a3cl, 0x19e66079l,
  0xd382e91bl, 0x5b109794l, 0x3f9f81e1l, 0x4470a388l, 0x41601abel, 0xaaf9f407l,
  0x8e175ef6l, 0xed842297l, 0x893a4271l, 0x1790839al, 0xd566a99el, 0x6b417deel,
  0x75c90d23l, 0x715edb31l, 0x723553f7l, 0x9afb50c9l, 0xfbc5f600l, 0xcd3b6a4el,
  0x97ed0fbal, 0x29689aecl, 0x63135c8el, 0xf0e26c7el, 0x0692ae7fl, 0xdbb208ffl,
  0x2ede3e9bl, 0x6a65bebdl, 0xd40867e9l, 0xc954afc5l, 0x73b08201l, 0x7ffdf809l,
  0x1195c24fl, 0x1ca5adcal, 0x74bd6d1fl, 0xb393c455l, 0xcadfd3fal, 0x99f13011l,
  0x0ebca813l, 0x60e791b8l, 0x6597ac7al, 0x18a7e46bl, 0x09cb49d3l, 0x0b27df6dl,
  0xcfe52f87l, 0xcef66837l, 0xe6328035l, 0xfa87c592l, 0x37baff93l, 0xd71fcc99l,
  0xdcab205cl, 0x4d7a5638l, 0x48012510l, 0x62797558l, 0xb6cf1fe5l, 0xbc311834l,
  0x9c2373acl, 0x14ec6175l, 0xa439cbdfl, 0x54afb0eal, 0xd686960bl, 0xfdd0d47bl,
  0x7b063902l, 0x8b78bac3l, 0x26c6a4d5l, 0x5c0055b6l, 0x2376102el, 0x0411783el,
  0x2aa3f1cdl, 0x51fc6ea8l, 0x701ce243l, 0x9b2a0abbl, 0x0ad93733l, 0x6e80d03dl,
  0xaf6295d1l, 0xf629896fl, 0xa30b0648l, 0x463d8dd4l, 0x963f84cbl, 0x01ff94f8l,
  0x8d7fefdcl, 0x553611c0l, 0xa97c1719l, 0xb96af759l, 0xe0e3c95el, 0x0528335bl,
  0x21fe5925l, 0x821a5245l, 0x807238b1l, 0x67f23db5l, 0xea6b4eabl, 0x0da6f985l,
  0xab1bc85al, 0xef8c90e4l, 0x4526230el, 0x38eb8b1cl, 0x1b91cd91l, 0x9fce5f0cl,
  0xf72cc72bl, 0xc64f2617l, 0xdaf7857dl, 0x7d373cf1l, 0x28eaedd7l, 0x203887d0l,
  0xc49a155fl, 0xa251b3b0l, 0xf2d47ae3l, 0x3d9ef267l, 0x4a94ab2fl, 0x7755a222l,
  0x0205e329l, 0xc28fa7a7l, 0xaec1fe51l, 0x270f164cl, 0x8c6d01bfl, 0x53b5bc98l,
  0xc09d3febl, 0x834986ccl, 0x4309a12cl, 0x578b2a96l, 0x3bb74b86l, 0x69561b4al,
  0x037e32f3l, 0xde335b08l, 0xc5156be0l, 0xe7ef09adl, 0x93b834c7l, 0xa7719352l,
  0x59302821l, 0xe3529d26l, 0xf961da76l, 0xcb142c44l, 0xa0f3b98dl, 0x76502457l,
  0x945a414bl, 0x078eeb12l, 0xdff8de69l, 0xeb6c8c2dl, 0xbda90c4dl, 0xe9c44d16l,
  0x168dfd66l, 0xad64763bl, 0xa65fd764l, 0x95a29c06l, 0x32d7713fl, 0x40f0b277l,
  0x224af08fl, 0x004cb5e8l, 0x92574814l, 0x8877d827l, 0x3e5b2d04l, 0x68c2d5f2l,
  0x86966273l, 0x1d433adal, 0x8774988al, 0x3c0e0bfel, 0xddad581dl, 0x2fd654edl,
  0x0f4769fdl, 0xc181ee9dl, 0x5fd88f61l, 0x341dbb3al, 0x528543f9l, 0xd92235cfl,
  0x1ea82eb4l, 0xb5cd790fl, 0x91d24f1el, 0xa869e6c2l, 0x61f474d2l, 0xcc205addl,
  0x0c7bfba9l, 0xbf2b0489l, 0xb02d72d8l, 0x2b46ece6l, 0xe4dcd90al, 0xb8a11440l,
  0xee8a63b7l, 0x854dd1a1l, 0xd1e00583l, 0x42b40e24l, 0x9e8964del, 0xb4b35d78l,
  0xbec76f6el, 0x24b9c620l, 0xd8d399a6l, 0x5adb2190l, 0x2db12730l, 0x3a5866afl,
  0x58c8fadbl, 0x5d8844e7l, 0x8a4bf380l, 0x15a01d70l, 0x79f5c028l, 0x66be3b8cl,
  0xf3e42b53l, 0x56990039l, 0x2c0c3182l, 0x5e16407cl, 0xecc04515l, 0x6c440284l,
  0x4cb6701al, 0x13bfc142l, 0x9d039f6al, 0x4f6e92c8l, 0xa1407c62l, 0x8483a095l,
  0xc70ae1c4l, 0xe20213a2l, 0xbacafc41l, 0x4ecc12b3l, 0x4bee3646l, 0x1fe807ael,
  0x25217f9cl, 0x35dde5f5l, 0x7a7dd6cel, 0xf89cce50l, 0xac07b718l, 0x7e73d2c6l,
  0xe563e76cl, 0x123ca536l, 0x3948ca56l, 0x9019dd49l, 0x10aa88d9l, 0xc82451e2l,
  0x473eb6d6l, 0x506fe854l, 0xe8bb03a5l, 0x332f4c32l, 0xfe1e1e72l, 0xb1ae572al,
  0x7c0d7bc1l, 0xe1c37eb2l, 0xf542aa60l, 0xf1a48ea0l, 0xd067b89fl, 0xbbfa195dl,
  0x1a049b0dl, 0x315946aal, 0x36d1b447l, 0x6d2ebdf0l, 0x0d188a6dl, 0x12cea0dbl,
  0x7e63740el, 0x6a444821l, 0x253d234fl, 0x6ffc6597l, 0x94a6bdefl, 0x33ee1b2fl,
  0x0a6c00c0l, 0x3aa336b1l, 0x5af55d17l, 0x265fb3dcl, 0x0e89cf4dl, 0x0786b008l,
  0xc80055b8l, 0x6b17c3cel, 0x72b05a74l, 0xd21a8d78l, 0xa6b70840l, 0xfe8eae77l,
  0xed69565cl, 0x55e1bcf4l, 0x585c2f60l, 0xe06f1a62l, 0xad67c0cdl, 0x7712af88l,
  0x9cc26acal, 0x1888053dl, 0x37eb853el, 0x9215abd7l, 0xde30adfcl, 0x1f1038e6l,
  0x70c51c8al, 0x8d586c26l, 0xf72bdd90l, 0x4dc3ce15l, 0x68eaeefal, 0xd0e9c8b9l,
  0x200f9c44l, 0xddd141bal, 0x024bf1d3l, 0x0f64c9d4l, 0xc421e9e9l, 0x9d11c14cl,
  0x9a0dd9e4l, 0x5f92ec19l, 0x1b980df0l, 0x1dcc4542l, 0xb8fe8c56l, 0x0c9c9167l,
  0x4e81eb49l, 0xca368f27l, 0xe3603b37l, 0xea08acccl, 0xac516992l, 0xc34f513bl,
  0x804d100dl, 0x6edca4c4l, 0xfc912939l, 0x29d219b0l, 0x278aaa3cl, 0x4868da7dl,
  0x54e890b7l, 0xb46d735al, 0x514589aal, 0xd6c630afl, 0x4980dfe8l, 0xbe3ccc55l,
  0x59d41202l, 0x650c078bl, 0xaf3a9e7bl, 0x3ed9827al, 0x9e79fc6el, 0xaadbfbael,
  0xc5f7d803l, 0x3daf7f50l, 0x67b4f465l, 0x73406e11l, 0x39313f8cl, 0x8a6e6686l,
  0xd8075f1fl, 0xd3cbfed1l, 0x69c7e49cl, 0x930581e0l, 0xe4b1a5a8l, 0xbbc45472l,
  0x09ddbf58l, 0xc91d687el, 0xbdbffda5l, 0x88c08735l, 0xe9e36bf9l, 0xdb5ea9b6l,
  0x95559404l, 0x08f432fbl, 0xe24ea281l, 0x64663579l, 0x000b8010l, 0x7914e7d5l,
  0x32fd0473l, 0xd1a7f0a4l, 0x445ab98el, 0xec72993fl, 0xa29a4d32l, 0xb77306d8l,
  0xc7c97cf6l, 0x7b6ab645l, 0xf5ef7adfl, 0xfb2e15f7l, 0xe747f757l, 0x5e944354l,
  0x234a2669l, 0x47e46359l, 0x9b9d11a9l, 0x40762cedl, 0x56f1de98l, 0x11334668l,
  0x890a9a70l, 0x1a296113l, 0xb3bd4af5l, 0x163b7548l, 0xd51b4f84l, 0xb99b2abcl,
  0x3cc1dc30l, 0xa9f0b56cl, 0x812272b2l, 0x0b233a5fl, 0xb650dbf2l, 0xf1a0771bl,
  0x36562b76l, 0xdc037b0fl, 0x104c97ffl, 0xc2ec98d2l, 0x90596f22l, 0x28b6620bl,
  0xdf42b212l, 0xfdbc4243l, 0xf3fb175el, 0x4a2d8b00l, 0xe8f3869bl, 0x30d69bc3l,
  0x853714c8l, 0xa7751d2el, 0x31e56deal, 0xd4840b0cl, 0x9685d783l, 0x068c9333l,
  0x8fba032cl, 0x76d7bb47l, 0x6d0ee22bl, 0xb546794bl, 0xd971b894l, 0x8b09d253l,
  0xa0ad5761l, 0xee77ba06l, 0x46359f31l, 0x577cc7ecl, 0x52825efdl, 0xa4beed95l,
  0x9825c52al, 0xeb48029al, 0xbaae59f8l, 0xcf490ee1l, 0xbc990164l, 0x8ca49dfel,
  0x4f38a6e7l, 0x2ba98389l, 0x8228f538l, 0x199f64acl, 0x01a1cac5l, 0xa8b51641l,
  0x5ce72d01l, 0x8e5df26bl, 0x60f28e1el, 0xcd5be125l, 0xe5b376bfl, 0x1c8d3116l,
  0x7132cbb3l, 0xcb7ae320l, 0xc0fa5366l, 0xd7653e34l, 0x971c88c2l, 0xc62c7dd0l,
  0x34d0a3dal, 0x868f6709l, 0x7ae6fa8fl, 0x22bbd523l, 0x66cd3d5bl, 0x1ef9288dl,
  0xf9cf58c1l, 0x5b784e80l, 0x7439a191l, 0xae134c36l, 0x9116c463l, 0x2e9e1396l,
  0xf8611f3al, 0x2d2f3307l, 0x247f37ddl, 0xc1e2ff9dl, 0x43c821e5l, 0x05ed5cabl,
  0xef74e80al, 0x4cca6028l, 0xf0ac3cbdl, 0x5d874b29l, 0x6c62f6a6l, 0x4b2a2ef3l,
  0xb1aa2087l, 0x62a5d0a3l, 0x0327221cl, 0xb096b4c6l, 0x417ec693l, 0xaba840d6l,
  0x789725ebl, 0xf4b9e02dl, 0xe6e00975l, 0xcc04961al, 0x63f624bbl, 0x7fa21ecbl,
  0x2c01ea7fl, 0xb2415005l, 0x2a8bbeb5l, 0x83b2b14el, 0xa383d1a7l, 0x5352f96al,
  0x043ecdadl, 0xce1918a1l, 0xfa6be6c9l, 0x50def36fl, 0xf6b80ce2l, 0x4543ef7cl,
  0x9953d651l, 0xf257955dl, 0x87244914l, 0xda1e0a24l, 0xffda4785l, 0x14d327a2l,
  0x3b93c29fl, 0x840684b4l, 0x61ab71a0l, 0x9f7b784al, 0x2fd570cfl, 0x15955bdel,
  0x38f8d471l, 0x3534a718l, 0x133fb71dl, 0x3fd80f52l, 0x4290a8bel, 0x75ff44c7l,
  0xa554e546l, 0xe1023499l, 0xbf2652e3l, 0x7d20399el, 0xa1df7e82l, 0x177092eel,
  0x217dd3f1l, 0x7c1ff8d9l, 0x12113f2el, 0xbfbd0785l, 0xf11793fbl, 0xa5bff566l,
  0x83c7b0e5l, 0x72fb316bl, 0x75526a9al, 0x41e0e612l, 0x7156ba09l, 0x53ce7deel,
  0x0aa26881l, 0xa43e0d7dl, 0x3da73ca3l, 0x182761edl, 0xbd5077ffl, 0x56db4aa0l,
  0xe792711cl, 0xf0a4eb1dl, 0x7f878237l, 0xec65c4e8l, 0x08dc8d43l, 0x0f8ce142l,
  0x8258abdal, 0xf4154e16l, 0x49dec2fdl, 0xcd8d5705l, 0x6c2c3a0fl, 0x5c12bb88l,
  0xeff3cdb6l, 0x2c89ed8cl, 0x7beba967l, 0x2a142157l, 0xc6d0836fl, 0xb4f97e96l,
  0x6931e969l, 0x514e6c7cl, 0xa7792600l, 0x0bbbf780l, 0x59671bbdl, 0x0707b676l,
  0x37482d93l, 0x80af1479l, 0x3805a60dl, 0xe1f4cac1l, 0x580b3074l, 0x30b8d6cel,
  0x05a304bel, 0xd176626dl, 0xebca97f3l, 0xbb201f11l, 0x6a1afe23l, 0xffaa86e4l,
  0x62b4da49l, 0x1b6629f5l, 0xf5d9e092l, 0xf37f3dd1l, 0x619bd45bl, 0xa6ec8e4fl,
  0x29c80939l, 0x0c7c0c34l, 0x9cfe6e48l, 0xe65fd3acl, 0x73613b65l, 0xb3c669f9l,
  0xbe2e8a9el, 0x286f9678l, 0x5797fd13l, 0x99805d75l, 0xcfb641c5l, 0xa91074bal,
  0x6343af47l, 0x6403cb46l, 0x8894c8dbl, 0x2663034cl, 0x3c40dc5el, 0x00995231l,
  0x96789aa2l, 0x2efde4b9l, 0x7dc195e1l, 0x547dadd5l, 0x06a8ea04l, 0xf2347a63l,
  0x5e0dc6f7l, 0x8462dfc2l, 0x1e6b2c3cl, 0x9bd275b3l, 0x91d419e2l, 0xbcefd17el,
  0xb9003924l, 0xd07e7320l, 0xdef0495cl, 0xc36ad00el, 0x1785b1abl, 0x92e20bcfl,
  0xb139f0e9l, 0x675bb9a1l, 0xaecfa4afl, 0x132376cbl, 0xe84589d3l, 0x79a05456l,
  0xa2f860bcl, 0x1ae4f8b5l, 0x20df4db4l, 0xa1e1428bl, 0x3bf60a1al, 0x27ff7bf1l,
  0xcb44c0e7l, 0xf7f587c4l, 0x1f3b9b21l, 0x94368f01l, 0x856e23a4l, 0x6f93de3fl,
  0x773f5bbfl, 0x8b22056el, 0xdf41f654l, 0xb8246ff4l, 0x8d57bff2l, 0xd57167eal,
  0xc5699f22l, 0x40734ba7l, 0x5d5c2772l, 0x033020a8l, 0xe30a7c4dl, 0xadc40fd6l,
  0x76353441l, 0x5aa5229bl, 0x81516590l, 0xda49f14el, 0x4fa672a5l, 0x4d9fac5fl,
  0x154be230l, 0x8a7a5cc0l, 0xce3d2f84l, 0xcca15514l, 0x5221360cl, 0xaf0fb81el,
  0x5bdd5873l, 0xf6825f8fl, 0x1113d228l, 0x70ad996cl, 0x93320051l, 0x60471c53l,
  0xe9ba567bl, 0x3a462ae3l, 0x5f55e72dl, 0x1d3c5ad7l, 0xdcfc45ecl, 0x34d812efl,
  0xfa96ee1bl, 0x369d1ef8l, 0xc9b1a189l, 0x7c1d3555l, 0x50845edcl, 0x4bb31877l,
  0x8764a060l, 0x8c9a9415l, 0x230e1a3al, 0xb05e9133l, 0x242b9e03l, 0xa3b99db7l,
  0xc2d7fb0al, 0x3333849dl, 0xd27278d4l, 0xb5d3efa6l, 0x78ac28adl, 0xc7b2c135l,
  0x0926ecf0l, 0xc1374c91l, 0x74f16d98l, 0x2274084al, 0x3f6d9cfal, 0x7ac0a383l,
  0xb73aff1fl, 0x3909a23dl, 0x9f1653ael, 0x4e2f3e71l, 0xca5ab22al, 0xe01e3858l,
  0x90c5a7ebl, 0x3e4a17dfl, 0xaa987fb0l, 0x488bbd62l, 0xb625062bl, 0x2d776bb8l,
  0x43b5fc08l, 0x1490d532l, 0xd6d12495l, 0x44e89845l, 0x2fe60118l, 0x9d9ef950l,
  0xac38133el, 0xd3864329l, 0x017b255al, 0xfdc2dd26l, 0x256851e6l, 0x318e7086l,
  0x2bfa4861l, 0x89eac706l, 0xee5940c6l, 0x68c3bc2fl, 0xe260334bl, 0x98da90bbl,
  0xf818f270l, 0x4706d897l, 0x212d3799l, 0x4cf7e5d0l, 0xd9c9649fl, 0xa85db5cdl,
  0x35e90e82l, 0x6b881152l, 0xab1c02c7l, 0x46752b02l, 0x664f598el, 0x45ab2e64l,
  0xc4cdb4b2l, 0xba42107fl, 0xea2a808al, 0x971bf3del, 0x4a54a836l, 0x4253aeccl,
  0x1029be68l, 0x6dcc9225l, 0xe4bca56al, 0xc0ae50b1l, 0x7e011d94l, 0xe59c162cl,
  0xd8e5c340l, 0xd470fa0bl, 0xb2be79ddl, 0xd783889cl, 0x1cede8f6l, 0x8f4c817al,
  0xddb785c9l, 0x860232d8l, 0x198aaad9l, 0xa0814738l, 0x3219cffcl, 0x169546d2l,
  0xfc0cb759l, 0x55911510l, 0x04d5cec3l, 0xed08cc3bl, 0x0d6cf427l, 0xc8e38ccal,
  0x0eeee3fel, 0x9ee7d7c8l, 0xf9f24fa9l, 0xdb04b35dl, 0x9ab0c9e0l, 0x651f4417l,
  0x028f8b07l, 0x6e28d9aal, 0xfba96319l, 0x8ed66687l, 0xfecbc58dl, 0x954ddb44l,
  0x7b0bdffel, 0x865d16b1l, 0x49a058c0l, 0x97abaa3fl, 0xcaacc75dl, 0xaba6c17dl,
  0xf8746f92l, 0x6f48aeedl, 0x8841d4b5l, 0xf36a146al, 0x73c390abl, 0xe6fb558fl,
  0x87b1019el, 0x26970252l, 0x246377b2l, 0xcbf676ael, 0xf923db06l, 0xf7389116l,
  0x14c81a90l, 0x83114eb4l, 0x8b137559l, 0x95a86a7al, 0xd5b8da8cl, 0xc4df780el,
  0x5a9cb3e2l, 0xe44d4062l, 0xe8dc8ef6l, 0x9d180845l, 0x817ad18bl, 0xc286c85bl,
  0x251f20del, 0xee6d5933l, 0xf6edef81l, 0xd4d16c1el, 0xc94a0c32l, 0x8437fd22l,
  0x3271ee43l, 0x42572aeel, 0x5f91962al, 0x1c522d98l, 0x59b23f0cl, 0xd86b8804l,
  0x08c63531l, 0x2c0d7a40l, 0xb97c4729l, 0x04964df9l, 0x13c74a17l, 0x5878362fl,
  0x4c808cd6l, 0x092cb1e0l, 0x6df02885l, 0xa0c2105el, 0x8aba9e68l, 0x64e03057l,
  0xe5d61325l, 0x0e43a628l, 0x16dbd62bl, 0x2733d90bl, 0x3ae57283l, 0xc0c1052cl,
  0x4b6fb620l, 0x37513953l, 0xfc898bb3l, 0x471b179fl, 0xdf6e66b8l, 0xd32142f5l,
  0x9b30fafcl, 0x4ed92549l, 0x105c6d99l, 0x4acd69ffl, 0x2b1a27d3l, 0x6bfcc067l,
  0x6301a278l, 0xad36e6f2l, 0xef3ff64el, 0x56b3cadbl, 0x0184bb61l, 0x17beb9fdl,
  0xfaec6109l, 0xa2e1ffa1l, 0x2fd224f8l, 0x238f5be6l, 0x8f8570cfl, 0xaeb5f25al,
  0x4f1d3e64l, 0x4377eb24l, 0x1fa45346l, 0xb2056386l, 0x52095e76l, 0xbb7b5adcl,
  0x3514e472l, 0xdde81e6el, 0x7acea9c4l, 0xac15cc48l, 0x71c97d93l, 0x767f941cl,
  0x911052a2l, 0xffea09bfl, 0xfe3ddcf0l, 0x15ebf3aal, 0x9235b8bcl, 0x75408615l,
  0x9a723437l, 0xe1a1bd38l, 0x33541b7el, 0x1bdd6856l, 0xb307e13el, 0x90814bb0l,
  0x51d7217bl, 0x0bb92219l, 0x689f4500l, 0xc568b01fl, 0x5df3d2d7l, 0x3c0ecd0dl,
  0x2a0244c8l, 0x852574e8l, 0xe72f23a9l, 0x8e26ed02l, 0x2d92cbddl, 0xdabc0458l,
  0xcdf5feb6l, 0x9e4e8dccl, 0xf4f1e344l, 0x0d8c436dl, 0x4427603bl, 0xbdd37fdal,
  0x80505f26l, 0x8c7d2b8el, 0xb73273c5l, 0x397362eal, 0x618a3811l, 0x608bfb88l,
  0x06f7d714l, 0x212e4677l, 0x28efceadl, 0x076c0371l, 0x36a3a4d9l, 0x5487b455l,
  0x3429a365l, 0x65d467acl, 0x78ee7eebl, 0x99bf12b7l, 0x4d129896l, 0x772a5601l,
  0xcce284c7l, 0x2ed85c21l, 0xd099e8a4l, 0xa179158al, 0x6ac0ab1al, 0x299a4807l,
  0xbe67a58dl, 0xdc19544al, 0xb8949b54l, 0x8d315779l, 0xb6f849c1l, 0x53c5ac34l,
  0x66de92a5l, 0xf195dd13l, 0x318d3a73l, 0x301ec542l, 0x0cc40da6l, 0xf253ade4l,
  0x467ee566l, 0xea5585ecl, 0x3baf19bbl, 0x7de9f480l, 0x79006e7cl, 0xa9b7a197l,
  0xa44bd8f1l, 0xfb2ba739l, 0xec342fd4l, 0xed4fd32dl, 0x3d1789bal, 0x400f5d7fl,
  0xc798f594l, 0x4506a847l, 0x034c0a95l, 0xe2162c9dl, 0x55a9cfd0l, 0x692d832el,
  0xcf9db2cal, 0x5e2287e9l, 0xd2610ef3l, 0x1ae7ecc2l, 0x48399ca0l, 0xa7e4269bl,
  0x6ee3a0afl, 0x7065bfe1l, 0xa6ffe708l, 0x2256804cl, 0x7476e21bl, 0x41b0796cl,
  0x7c243b05l, 0x000a950fl, 0x1858416bl, 0xf5a53c89l, 0xe9fef823l, 0x3f443275l,
  0xe0cbf091l, 0x0af27b84l, 0x3ebb0f27l, 0x1de6f7f4l, 0xc31c29f7l, 0xb166de3dl,
  0x12932ec3l, 0x9c0c0674l, 0x5cda81b9l, 0xd1bd9d12l, 0xaffd7c82l, 0x8962bca7l,
  0xa342c4a8l, 0x62457151l, 0x82089f03l, 0xeb49c670l, 0x5b5f6530l, 0x7e28bad2l,
  0x20880ba3l, 0xf0faafcdl, 0xce82b56fl, 0x0275335cl, 0xc18e8afbl, 0xde601d69l,
  0xba9b820al, 0xc8a2be4fl, 0xd7cac335l, 0xd9a73741l, 0x115e974dl, 0x7f5ac21dl,
  0x383bf9c6l, 0xbcaeb75fl, 0xfd0350cel, 0xb5d06b87l, 0x9820e03cl, 0x72d5f163l,
  0xe3644fc9l, 0xa5464c4bl, 0x57048fcbl, 0x9690c9dfl, 0xdbf9eafal, 0xbff4649al,
  0x053c00e3l, 0xb4b61136l, 0x67593dd1l, 0x503ee960l, 0x9fb4993al, 0x19831810l,
  0xc670d518l, 0xb05b51d8l, 0x0f3a1ce5l, 0x6caa1f9cl, 0xaacc31bel, 0x949ed050l,
  0x1ead07e7l, 0xa8479abdl, 0xd6cffcd5l, 0x936993efl, 0x472e91cbl, 0x5444b5b6l,
  0x62be5861l, 0x1be102c7l, 0x63e4b31el, 0xe81f71b7l, 0x9e2317c9l, 0x39a408ael,
  0x518024f4l, 0x1731c66fl, 0x68cbc918l, 0x71fb0c9el, 0xd03b7fddl, 0x7d6222ebl,
  0x9057eda3l, 0x1a34a407l, 0x8cc2253dl, 0xb6f6979dl, 0x835675dcl, 0xf319be9fl,
  0xbe1cd743l, 0x4d32fee4l, 0x77e7d887l, 0x37e9ebfdl, 0x15f851e8l, 0x23dc3706l,
  0x19d78385l, 0xbd506933l, 0xa13ad4a6l, 0x913f1a0el, 0xdde560b9l, 0x9a5f0996l,
  0xa65a0435l, 0x48d34c4dl, 0xe90839a7l, 0x8abba54el, 0x6fd13ce1l, 0xc7eebd3cl,
  0x0e297602l, 0x58b9bbb4l, 0xef7901e6l, 0x64a28a62l, 0xa509875al, 0xf8834442l,
  0x2702c709l, 0x07353f31l, 0x3b39f665l, 0xf5b18b49l, 0x4010ae37l, 0x784de00bl,
  0x7a1121e9l, 0xde918ed3l, 0xc8529dcdl, 0x816a5d05l, 0x02ed8298l, 0x04e3dd84l,
  0xfd2bc3e2l, 0xaf167089l, 0x96af367el, 0xa4da6232l, 0x18ff7325l, 0x05f9a9f1l,
  0x4fefb9f9l, 0xcd94eaa5l, 0xbfaa5069l, 0xa0b8c077l, 0x60d86f57l, 0xfe71c813l,
  0x29ebd2c8l, 0x4ca86538l, 0x6bf1a030l, 0xa237b88al, 0xaa8af41dl, 0xe1f7b6ecl,
  0xe214d953l, 0x33057879l, 0x49caa736l, 0xfa45cff3l, 0xc063b411l, 0xba7e27d0l,
  0x31533819l, 0x2a004ac1l, 0x210efc3fl, 0x2646885el, 0x66727dcfl, 0x9d7fbf54l,
  0xa8dd0ea8l, 0x3447cacel, 0x3f0c14dbl, 0xb8382aacl, 0x4ace3539l, 0x0a518d51l,
  0x95178981l, 0x35aee2cal, 0x73f0f7e3l, 0x94281140l, 0x59d0e523l, 0xd292cb88l,
  0x565d1b27l, 0x7ec8fbafl, 0x069af08dl, 0xc127fd24l, 0x0bc77b10l, 0x5f03e7efl,
  0x453e99bal, 0xeed9ff7fl, 0x87b55215l, 0x7915ab4cl, 0xd389a358l, 0x5e75ce6dl,
  0x28d655c0l, 0xdad26c73l, 0x2e2510ffl, 0x9fa7eeccl, 0x1d0629c3l, 0xdc9c9c46l,
  0x2d67ecd7l, 0xe75e94bdl, 0x3d649e2al, 0x6c413a2bl, 0x706f0d7cl, 0xdfb0127bl,
  0x4e366b55l, 0x2c825650l, 0x24205720l, 0xb5c998f7l, 0x3e95462cl, 0x756e5c72l,
  0x3259488fl, 0x11e8771al, 0xa7c0a617l, 0x577663e5l, 0x089b6401l, 0x8eab1941l,
  0xae55ef8cl, 0x3aac5460l, 0xd4e6262fl, 0x5d979a47l, 0xb19823b0l, 0x7f8d6a0cl,
  0xffa08683l, 0x0170cd0fl, 0x858cd5d8l, 0x53961c90l, 0xc4c61556l, 0x41f2f226l,
  0xcfcd062dl, 0xf24c03b8l, 0xea81df5bl, 0x7be2fa52l, 0xb361f98bl, 0xc2901316l,
  0x55ba4bbcl, 0x93b234a9l, 0x0fbc6603l, 0x80a96822l, 0x6d60491fl, 0x22bd00f8l,
  0xbcad5aadl, 0x52f3f13bl, 0x42fd2b28l, 0xb41dd01cl, 0xc52c93bfl, 0xfc663094l,
  0x8f58d100l, 0x43fecc08l, 0xc6331e5dl, 0xe6480f66l, 0xca847204l, 0x4bdf1da0l,
  0x30cc2efbl, 0x13e02deal, 0xfb49ac45l, 0xf9d4434fl, 0xf47c5b9cl, 0x148879c2l,
  0x039fc234l, 0xa3db9bfcl, 0xd1a1dc5cl, 0x763d7cd4l, 0xed6d2f93l, 0xab13af6el,
  0x1e8e054al, 0xd68f4f9al, 0xc30484b3l, 0xd7d50afal, 0x6930855fl, 0xcc07db95l,
  0xce746db1l, 0x744e967dl, 0xf16cf575l, 0x8643e8b5l, 0xf0eae38el, 0xe52de1d1l,
  0x6587dae0l, 0x0c4b8121l, 0x1c7ac567l, 0xac0db20al, 0x36c3a812l, 0x5b1a4514l,
  0xa9a3f868l, 0xb9263baal, 0xcb3ce9d2l, 0xe44fb1a4l, 0x9221bc82l, 0xb29390fel,
  0x6ab41863l, 0x974a3e2el, 0x89f531c5l, 0x255ca13el, 0x8b65d348l, 0xec248f78l,
  0xd8fc16f0l, 0x50ecdeeel, 0x09010792l, 0x3c7d1fb2l, 0xeba5426bl, 0x847b417al,
  0x468b40d9l, 0x8dc4e680l, 0x7cc1f391l, 0x2f1eb086l, 0x6e5baa6al, 0xe0b395dal,
  0xe31b2cf6l, 0xd9690b0dl, 0x729ec464l, 0x38403ddel, 0x610b80a2l, 0x5cf433abl,
  0xb0785fc4l, 0xd512e4c6l, 0xbbb7d699l, 0x5a86591bl, 0x10cf5376l, 0x12bf9f4bl,
  0x980fbaa1l, 0x992a4e70l, 0x20fa7ae7l, 0xf7996ebbl, 0xc918a2bel, 0x82de74f2l,
  0xad54209bl, 0xf66b4d74l, 0x1fc5b771l, 0x169d9229l, 0x887761dfl, 0x00b667d5l,
  0xdb425e59l, 0xb72f2844l, 0x9b0ac1f5l, 0x9c737e3al, 0x2b85476cl, 0x6722add6l,
  0x44a63297l, 0x0d688cedl, 0xabc59484l, 0x4107778al, 0x8ad94c6fl, 0xfe83df90l,
  0x0f64053fl, 0xd1292e9dl, 0xc5744356l, 0x8dd1abb4l, 0x4c4e7667l, 0xfb4a7fc1l,
  0x74f402cbl, 0x70f06afdl, 0xa82286f2l, 0x918dd076l, 0x7a97c5cel, 0x48f7bde3l,
  0x6a04d11dl, 0xac243ef7l, 0x33ac10cal, 0x2f7a341el, 0x5f75157al, 0xf4773381l,
  0x591c870el, 0x78df8cc8l, 0x22f3adb0l, 0x251a5993l, 0x09fbef66l, 0x796942a8l,
  0x97541d2el, 0x2373daa9l, 0x1bd2f142l, 0xb57e8eb2l, 0xe1a5bfdbl, 0x7d0efa92l,
  0xb3442c94l, 0xd2cb6447l, 0x386ac97el, 0x66d61805l, 0xbdada15el, 0x11bc1aa7l,
  0x14e9f6eal, 0xe533a0c0l, 0xf935ee0al, 0x8fee8a04l, 0x810d6d85l, 0x7c68b6d6l,
  0x4edc9aa2l, 0x956e897dl, 0xed87581al, 0x264be9d7l, 0xff4ddb29l, 0x823857c2l,
  0xe005a9a0l, 0xf1cc2450l, 0x6f9951e1l, 0xaade2310l, 0xe70c75f5l, 0x83e1a31fl,
  0x4f7dde8el, 0xf723b563l, 0x368e0928l, 0x86362b71l, 0x21e8982dl, 0xdfb3f92bl,
  0x44676352l, 0x99efba31l, 0x2eab4e1cl, 0xfc6ca5e7l, 0x0ebe5d4el, 0xa0717d0cl,
  0xb64f8199l, 0x946b31a1l, 0x5656cbc6l, 0xcffec3efl, 0x622766c9l, 0xfa211e35l,
  0x52f98b89l, 0x6d01674bl, 0x4978a802l, 0xf651f701l, 0x15b0d43dl, 0xd6ff4683l,
  0x3463855fl, 0x672ba29cl, 0xbc128312l, 0x4626a70dl, 0xc8927a5al, 0xb8481cf9l,
  0x1c962262l, 0xa21196bal, 0xbaba5ee9l, 0x5bb162d0l, 0x69943bd1l, 0x0c47e35cl,
  0x8cc9619al, 0xe284d948l, 0x271bf264l, 0xc27fb398l, 0x4bc70897l, 0x60cf202cl,
  0x7f42d6aal, 0xa5a13506l, 0x5d3e8860l, 0xcea63d3cl, 0x63bf0a8fl, 0xf02e9efal,
  0xb17b0674l, 0xb072b1d3l, 0x06e5723bl, 0x3737e436l, 0x24aa49c7l, 0x0ded0d18l,
  0xdb256b14l, 0x58b27877l, 0xecb49f54l, 0x6c40256al, 0x6ea92ffbl, 0x3906aa4cl,
  0xc9866fd5l, 0x4549323el, 0xa7b85fabl, 0x1918cc27l, 0x7308d7b5l, 0x1e16c7adl,
  0x71850b37l, 0x3095fd78l, 0xa63b70e6l, 0xd880e2ael, 0x3e282769l, 0xa39ba6bcl,
  0x98700fa3l, 0xf34c53e8l, 0x288af426l, 0xb99d930fl, 0xf5b99df1l, 0xe9d0c8cfl,
  0x5ac8405dl, 0x50e7217bl, 0x511fbbbel, 0x2ca2e639l, 0xc020301bl, 0x356dbc00l,
  0x8e43ddb9l, 0x4d327b4al, 0xf20ff3edl, 0x1dbb29bdl, 0x43d44779l, 0xa1b68f70l,
  0x6114455bl, 0xe63d280bl, 0x6bf6ff65l, 0x10fc39e5l, 0x3dae126el, 0xc1d7cf11l,
  0xcb60b795l, 0x1789d5b3l, 0x9bca36b7l, 0x08306075l, 0x84615608l, 0x8b3a0186l,
  0xe88fbecdl, 0x7ba47c4dl, 0x2de44dacl, 0x653fe58dl, 0xcca0b968l, 0xd7fa0e72l,
  0x93901780l, 0x1f2c26ccl, 0xae595b6bl, 0xa9ecea9bl, 0xe3dbf8c4l, 0x319cc130l,
  0x12981196l, 0x01a3a4del, 0x32c454b6l, 0x755bd817l, 0x3cd871e4l, 0xa48bb8dal,
  0x02fdec09l, 0xfd2dc2e2l, 0x9e578088l, 0x9a9f916dl, 0x4065fe6cl, 0x1853999el,
  0xc7793f23l, 0xdc1016bbl, 0x969355ffl, 0x7ef292f6l, 0xcdce4adcl, 0x05e24416l,
  0x85c16c46l, 0xd441d37fl, 0x57bd6855l, 0x8746f54fl, 0x9ca773dfl, 0x770bae22l,
  0x54828413l, 0xb75e4b19l, 0x04c35c03l, 0xbf7cca07l, 0x2955c4ddl, 0x721db041l,
  0xb2394f33l, 0x03f51387l, 0x89b73c9fl, 0x0b1737f3l, 0x07e69024l, 0x9231d245l,
  0x76193861l, 0x88159c15l, 0xdeb552d9l, 0xd9767e40l, 0x20c6c0c3l, 0x4281977cl,
  0xf8afe1e0l, 0xd32a0751l, 0x3fc27432l, 0xddf1dcc5l, 0x68581f34l, 0x3bcd5025l,
  0x0091b2eel, 0x4aeb6944l, 0x1602e743l, 0xea09eb58l, 0xef0a2a8bl, 0x641e03a5l,
  0xeb50e021l, 0x5c8ccef8l, 0x802ff0b8l, 0xd5e3edfel, 0xc4dd1b49l, 0x5334cd2al,
  0x13f82d2fl, 0x47450c20l, 0x55dafbd2l, 0xbec0c6f4l, 0xb45d7959l, 0x3ad36e8cl,
  0x0aa8ac57l, 0x1a3c8d73l, 0xe45aafb1l, 0x9f664838l, 0xc6880053l, 0xd0039bbfl,
  0xee5f19ebl, 0xca0041d8l, 0xbbea3aafl, 0xda628291l, 0x9d5c95d4l, 0xadd504a6l,
  0xc39ab482l, 0x5e9e14a4l, 0x2be065f0l, 0x2a13fc3al, 0x9052e8ecl, 0xaf6f5afcl,
  0x519aa8b5l, 0xbb303da9l, 0xe00e2b10l, 0xdfa6c1dbl, 0x2e6b952el, 0xee10dc23l,
  0x37936d09l, 0x1fc42e92l, 0x39b25a9fl, 0x13ff89f4l, 0xc8f53feal, 0x18500bc7l,
  0x95a0379dl, 0x98f751c2l, 0x2289c42fl, 0xa21e4098l, 0x6f391f41l, 0xf27e7e58l,
  0x0d0df887l, 0x4b79d540l, 0x8e8409aal, 0x71fe46f8l, 0x688a9b29l, 0x3f08b548l,
  0x84abe03al, 0x5e91b6c1l, 0xfde4c2ael, 0x251d0e72l, 0x92d4fee5l, 0xf9371967l,
  0x9175108fl, 0xe6e81835l, 0x8c8cb8eel, 0xb55a67b3l, 0xcef138ccl, 0x8b256268l,
  0x00d815f5l, 0xe8810812l, 0x77826189l, 0xea73267dl, 0x19b90f8dl, 0x45c33bb4l,
  0x82477056l, 0xe1770075l, 0x09467aa6l, 0xa7c6f54al, 0x79768742l, 0x61b86bcal,
  0xd6644a44l, 0xe33f0171l, 0xc229fbcdl, 0x41b08febl, 0xd1903e30l, 0x65ec9080l,
  0x563d6fbdl, 0xf56da488l, 0xebf64cd8l, 0x4934426bl, 0x7c8592fcl, 0x6aca8cf2l,
  0x1cea111bl, 0x3a57ee7al, 0xace11c0dl, 0x9942d85el, 0xc4613407l, 0xfa8e643bl,
  0x327fc701l, 0x4ca9be82l, 0x3352526dl, 0x2c047f63l, 0xf3a8f7ddl, 0x1a4a98a8l,
  0x762ed4d1l, 0x27c75008l, 0xbdf497c0l, 0x7a7b84dfl, 0x315c28abl, 0x801f93e3l,
  0xf19b0ca1l, 0x8f14e46al, 0xe48ba333l, 0x9605e625l, 0xf03ecb60l, 0x60385f2dl,
  0x902845bal, 0x7f96d66fl, 0x24bff05cl, 0x2820730bl, 0x947133cbl, 0xd444828al,
  0xb343f6f1l, 0x0bef4705l, 0x8da574f9l, 0x01e25d6cl, 0x1732793el, 0x4f0f7b27l,
  0x364b7117l, 0xb2d1da77l, 0xa6c5f1e9l, 0x574ca5b1l, 0x386a3076l, 0xad6894d6l,
  0x1156d7fal, 0xa48d1d9al, 0x4794c0afl, 0x150c0aa0l, 0x26d348acl, 0x29fdeabel,
  0xa5dede53l, 0x81671e8el, 0x594ee3bfl, 0xa96c56e6l, 0x3426a726l, 0xc5976579l,
  0xbc22e5e4l, 0xc1006319l, 0xdaafdd2al, 0xa1a1aa83l, 0x3badd0e7l, 0xc3b14981l,
  0xd770b155l, 0xccd7c693l, 0x42e944c5l, 0x03e0064fl, 0xca95b4efl, 0x3dee81c3l,
  0xfbbcd98cl, 0x1e07e15bl, 0x667ce949l, 0xe7d6773fl, 0x21b6124bl, 0x6b2a6ef7l,
  0xd3278a9cl, 0x9a988304l, 0x75d2ae9bl, 0xfe49e2ffl, 0x9bc24f46l, 0x74cc2cf6l,
  0xa3139f36l, 0x6c9ef35al, 0x9fc1dffel, 0x9e5facdcl, 0xaadc8bbbl, 0x5abdbc5fl,
  0x44b3b390l, 0xf754efa7l, 0x5fe3bdb7l, 0x4e59c886l, 0x06a4c984l, 0xa0338878l,
  0xcd513cd7l, 0x63ebd27el, 0x8aba80adl, 0x50da144el, 0x5d9f4e97l, 0x025b751cl,
  0x2d580200l, 0xb6c05837l, 0x580aa15dl, 0x54022a6el, 0xb41a5415l, 0x4863fab6l,
  0xb0b79957l, 0x46d0d159l, 0xdc2b8650l, 0x20a7bb0cl, 0x4a032974l, 0xec8636a2l,
  0x8548f24cl, 0xf6a2bf16l, 0x1088f4b0l, 0x0c2f3a94l, 0x525dc396l, 0x14065785l,
  0x2b4dca52l, 0x08aeed39l, 0xabedfc99l, 0xb1dbcf18l, 0x87f85bbcl, 0xae3aff61l,
  0x433ccd70l, 0x5b23cc64l, 0x7b453213l, 0x5355c545l, 0x9318ec0al, 0x78692d31l,
  0x0a21693dl, 0xd5666814l, 0x05fb59d9l, 0xc71985b2l, 0x2abb8e0el, 0xcf6e6c91l,
  0xd9cfe7c6l, 0xefe7132cl, 0x9711ab28l, 0x3ce52732l, 0x12d516d2l, 0x7209a0d0l,
  0xd278d306l, 0x70fa4b7bl, 0x1d407dd3l, 0xdb0beba4l, 0xbfd97621l, 0xa8be21e1l,
  0x1b6f1b66l, 0x30650ddal, 0xba7ddbb9l, 0x7df953fbl, 0x9d1c3902l, 0xedf0e8d5l,
  0xb8741ae0l, 0x0f240565l, 0x62cd438bl, 0xc616a924l, 0xaf7a96a3l, 0x35365538l,
  0xe583af4dl, 0x73415eb8l, 0x23176a47l, 0xfc9ccee8l, 0x7efc9de2l, 0x695e03cfl,
  0xf8ce66d4l, 0x88b4781dl, 0x67dd9c03l, 0x3e8f9e73l, 0xc0c95c51l, 0xbe314d22l,
  0x55aa0795l, 0xcb1bb011l, 0xe980fdc8l, 0x9c62b7cel, 0xde2d239el, 0x042cadf3l,
  0xffdf04del, 0x5ce6a60fl, 0xd8c831edl, 0xb7b5b9ecl, 0xb9cbf962l, 0xe253b254l,
  0x0735ba1fl, 0x16ac917fl, 0xdd607c2bl, 0x64a335c4l, 0x40159a7cl, 0x869222f0l,
  0x6ef21769l, 0x839d20a5l, 0xd03b24c9l, 0xf412601el, 0x6d72a243l, 0x0e018dfdl,
  0x89f3721al, 0xc94f4134l, 0x2f992f20l, 0x4d87253cl };


/**
 * Initialize the hashing state.
 */
void snefru_init(struct snefru_ctx *ctx)
{
	ctx->hashlen = 0;
	ctx->index = 0;
	mutils_bzero(ctx->hash, SNEFRU256_DIGEST_SIZE);
}


/**
 * The core snefru transform.
 */
static void snefru(mutils_word32 *block, mutils_word32 len)
{
	__const mutils_word32 st[4] = {16, 8, 16, 24};
	mutils_word32 isave[SNEFRU256_DIGEST_LEN];
	mutils_word32 *sbox, x;
	mutils_word32 i, j;
	mutils_word32 limit;

	mutils_memcpy(isave, block, SNEFRU256_DIGEST_SIZE);
  
	limit = SBOX + 512 * R;

	for (sbox = (mutils_word32 *) SBOX; sbox < limit; sbox += 512)
	{
		for (j = 0; j < 4; j++)
		{
			for (i = 0; i < SNEFRU_BLOCK_LEN; i++)
			{
				x = sbox[(i << 7 & 0x100) + (block[i] & 0xff)];
				block[(i + SNEFRU_BLOCK_LEN - 1) % SNEFRU_BLOCK_LEN] ^= x;
				block[(i + 1) % SNEFRU_BLOCK_LEN] ^= x;
			}
			for (i = 0; i < SNEFRU_BLOCK_LEN; i++)
			{
				block[i] = block[i] >> st[j] | block[i] << (32 - st[j]);
			}
		}
	}
	for (i = 0; i < len; i++)
	{
		block[i] = block[SNEFRU_BLOCK_LEN - 1 - i] ^ isave[i];
	}
}


static void processBuffer(struct snefru_ctx *ctx, int len)
{
	mutils_word32 i;
	mutils_word8 *buf = ctx->buffer;

	for(i = len; i < SNEFRU_BLOCK_LEN; i++)
	{
		ctx->hash[i] = 
		  (mutils_word32)buf[0] << 24 | (mutils_word32)buf[1] << 16 |
		  (mutils_word32)buf[2] << 8 | (mutils_word32)buf[3];
		buf += 4;
	}
	snefru(ctx->hash, len);
}

/**
 * Update the hashing state.
 */
static void snefru_update(struct snefru_ctx *ctx, __const mutils_word8 *data, mutils_word32 length,
			  mutils_word32 data_size, mutils_word32 digest_len) 
{
	if (ctx->index)
	{	/* Try to fill partial block */
		mutils_word32 left = data_size - ctx->index;
		if (length < left)
		{
			mutils_memcpy(ctx->buffer + ctx->index, data, length);
			ctx->index += length;
			return;	/* Finished */
		}
		else
		{
			mutils_memcpy(ctx->buffer + ctx->index, data, left);
			processBuffer(ctx, digest_len);
			ctx->hashlen += 8 * data_size;
			data += left;
			length -= left;
		}
	}
	while (length >= data_size)
	{
		mutils_memcpy(ctx->buffer, data, data_size);
		processBuffer(ctx, digest_len);
		ctx->hashlen += 8 * data_size;
		data += data_size;
		length -= data_size;
	}
	mutils_memcpy(ctx->buffer, data, length);
	ctx->index = length;
}

void snefru128_update(struct snefru_ctx *ctx, __const mutils_word8 *data, mutils_word32 length)
{
	snefru_update(ctx, data, length, SNEFRU128_DATA_SIZE, SNEFRU128_DIGEST_LEN);
}

void snefru256_update(struct snefru_ctx *ctx, __const mutils_word8 *data, mutils_word32 length)
{
	snefru_update(ctx, data, length, SNEFRU256_DATA_SIZE, SNEFRU256_DIGEST_LEN);
}



/**
 * Finalize the hash value calculation by appending the (padded) length field.
 */
void snefru128_final(struct snefru_ctx *ctx) 
{
	if (ctx->index)
	{    /* pad the last data block if partially filled */
		mutils_bzero(ctx->buffer + ctx->index, SNEFRU128_DATA_SIZE - ctx->index);
		processBuffer(ctx, SNEFRU128_DIGEST_LEN);
		ctx->hashlen += 8 * ctx->index;
	}

	mutils_bzero(ctx->hash + SNEFRU128_DIGEST_LEN,
		     SNEFRU_BLOCK_SIZE - SNEFRU128_DIGEST_SIZE - sizeof(ctx->hashlen));
	ctx->hash[SNEFRU_BLOCK_LEN - 2] = ctx->hashlen >> 32;
	ctx->hash[SNEFRU_BLOCK_LEN - 1] = ctx->hashlen & 0xffffffff;

	snefru(ctx->hash, SNEFRU128_DIGEST_LEN);
}

void snefru256_final(struct snefru_ctx *ctx)
{
	if (ctx->index)
	{    /* pad the last data block if partially filled */
		mutils_bzero(ctx->buffer + ctx->index, SNEFRU256_DATA_SIZE - ctx->index);
		processBuffer(ctx, SNEFRU256_DIGEST_LEN);
		ctx->hashlen += 8 * ctx->index;
	}

	mutils_bzero(ctx->hash + SNEFRU256_DIGEST_LEN, 
		     SNEFRU_BLOCK_SIZE - SNEFRU256_DIGEST_SIZE - sizeof(ctx->hashlen));
	ctx->hash[SNEFRU_BLOCK_LEN - 2] = ctx->hashlen >> 32;
	ctx->hash[SNEFRU_BLOCK_LEN - 1] = ctx->hashlen & 0xffffffff;
  
	snefru(ctx->hash, SNEFRU256_DIGEST_LEN);
}


static void snefru_digest(__const struct snefru_ctx *ctx, mutils_word8 *digest, mutils_word32 len)
{
	mutils_word32 i;

	for (i = 0; i < len; i++)
	{
		*digest++ = ctx->hash[i] >> 24;
		*digest++ = (ctx->hash[i] >> 16) & 0xff;
		*digest++ = (ctx->hash[i] >> 8) & 0xff;
		*digest++ = ctx->hash[i] & 0xff;
	}
}

void snefru128_digest(__const struct snefru_ctx *ctx, mutils_word8 *digest)
{
	snefru_digest(ctx, digest, SNEFRU128_DIGEST_LEN);
}

void snefru256_digest(__const struct snefru_ctx *ctx, mutils_word8 *digest)
{
	snefru_digest(ctx, digest, SNEFRU256_DIGEST_LEN);
}

#endif /* ENABLE_SNEFRU */
