# $Id$
# Copyright 2005 Six Apart Ltd
# Licensed under the same terms as Perl itself

package MT::Plugin::TransferEntry;

use strict;
use base qw( MT::Plugin );

use vars qw($VERSION);
$VERSION = '1.0';

sub BEGIN {
    MT->add_plugin(MT::Plugin::TransferEntry->new({
        name => 'Transfer',
        description => 'Adds an option to List Entries that allows moving entries from one blog to another.',
        version => $VERSION,
        author_name => 'Six Apart, Ltd.',
        author_link => 'http://www.sixapart.com/',
    }));
}

sub init_app {
    my $plugin = shift;
    my($app) = @_;
    return unless $app->isa('MT::App::CMS');
    $app->add_itemset_action({
        type  => 'entry',
        key   => 'transfer_entries',
        label => 'Transfer Entries',
        code  => sub { start_transfer($plugin, @_) },
    });
    $app->add_methods(
        entry_transfer => sub { transfer($plugin, @_) },
    );
}

sub start_transfer {
    my $plugin = shift;
    my($app) = @_;
    my @ids = $app->param('id');
    my @blogs;
    require MT::Blog;
    if ($app->user->is_superuser) {
        @blogs = map { { id => $_->id, name => $_->name } } MT::Blog->load;
    } else {
        for my $perms (MT::Permission->load({ author_id => $app->user->id })) {
            next unless $perms->can_post;
            my $blog = MT::Blog->load($perms->blog_id);
            push @blogs, { id => $blog->id, name => $blog->name };
        }
    }
    $app->build_page($plugin->load_tmpl('select-blog.tmpl'), {
        ids         => [ map { { id => $_ } } @ids ],
        blogs       => \@blogs,
        return_to   => $app->return_args,
    });
}

sub transfer {
    my $plugin = shift;
    my($app) = @_;
    $app->validate_magic or return;
    my @ids = $app->param('id');
    my $dest_blog_id = $app->param('blog_id');
    my $return = $app->param('return_to');
    my $user = $app->user;
    require MT::Entry;
    my %perms;
    for my $id (@ids) {
        my $entry = MT::Entry->load($id)
            or return $app->error("Can't load entry $id");
        my $perms = $perms{$entry->blog_id} ||=
                    MT::Permission->load({ author_id => $user->id, 
                                           blog_id   => $entry->blog_id });
        $app->log($user->name . " can't move entry $id"), next
            unless $perms && $perms->can_edit_entry($entry, $user);
        $plugin->_transfer_entry($entry, $dest_blog_id);
    }
    $app->return_args($return);
    $app->add_return_arg(saved => 1);
    $app->call_return;
}

sub _transfer_entry {
    my $plugin = shift;
    my($entry, $dest_blog_id) = @_;

    require MT::Comment;
    require MT::Trackback;
    require MT::TBPing;

    my @comments = MT::Comment->load({ entry_id => $entry->id });
    for my $comment (@comments) {
        $comment->blog_id($dest_blog_id);
        $comment->save;
    }

    if (my $tb = MT::Trackback->load({ entry_id => $entry->id })) {
        $tb->blog_id($dest_blog_id);
        $tb->save;

        my @pings = MT::TBPing->load({ tb_id => $tb->id });
        for my $ping (@pings) {
            $ping->blog_id($dest_blog_id);
            $ping->save;
        }
    }

    $entry->blog_id($dest_blog_id);
    $entry->save;
}

1;
