// RMISSLServerSocketFactory.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: RMISSLServerSocketFactory.java,v 1.4 1999/02/20 20:02:20 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.rmi;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.rmi.*;
import java.rmi.server.*;
import java.security.cert.*;
import de.tu_darmstadt.sp.ssl.*;
import javax.net.ssl.*;

/**
 * Class RMISSLServerSocketFactory provides server sockets for RMI calls.
 * It actually produces <code>SSLRMIServerSocket</code> objects for secure
 * connections. <code>SSLRMIServerSocket</code> extends <code>SSLeayServerSocket</code>.
 * The sockets are configured to require peer (client) authentication.
 * We produce <code>RMIServerSocket</code> because we need some
 * sort of accounting of all accepted connections. The <code>RMIServerSocket</code>
 * provides  this additional functionality.
 * <p>
 * The instantiation of the factory will fail if the necessary security information
 * (e.g. ca-file,certificate,private key) is not provided.
 * <p>
 * The methods <code>getClient</code>, <code>getClientCertificateChain</code> and
 * <code>getClientCertificate</code> should be called from inside the remote server
 * object to find out the identity of the caller.
 *
 * @version	$Revision: 1.4 $
 * @author	Andrei Popovici
 */
public
class RMISSLServerSocketFactory extends SSLeayServerSocketFactory
  implements RMIServerSocketFactory, Serializable

{


  /**
   * Create a new RMISSLServerSocketFactory which produces <code>RMIServerSocket</code>
   * objects. We produce <code>RMIServerSocket</code> because we need some
   * sort of accounting of all accepted connections. The <code>RMIServerSocket</code>
   * provide  this additional functionality.
   *
   */
  public RMISSLServerSocketFactory()
  {
    super();
    try
      {
	Class sscls =
	  Class.forName("de.tu_darmstadt.sp.rmi.RMISSLServerSocket");
	setServerSocketClass(sscls);
      }
    catch (ClassNotFoundException e)
      {
	throw new Error("RMISSLServerSocketFactory.init()");
      }


  }

  /** We need to re-specify the policy for our server sockets to require peer
    * authentication.
    * <p>
    * <code>RMIServerSocket</code>s are basicly <code>SSLeayServerSocket</code>s,
    * which by default do not authentify the peer (in our case, the
    * client, the caller).
    */
  protected void setSSLPolicy (SSLeayServerSocket s)
    {
      s.setNeedClientAuth(true);
    }

  /** Return the socket used by the currnt RMI call. It is the socket used for
    * the connection to the client.
    */
  public static SSLSocket getClient()
    {
      return RMISSLSocket.getCurrentServerSideSocket();
    }

  /** Return the certificate chain used by the peer(client, caller) in
    * the current RMI call. The first certificate is the identity of the peer.
    * <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getClient</code>.
    */
  public static X509Certificate[] getClientCertificateChain()
    throws SSLPeerUnverifiedException
    {
      SSLSocket  sock = getClient();
      SSLSession sess = sock.getSession();
      return sess.getPeerCertificateChain();
    }

  /** Return the certificate representing the identity of the peer
    * (client,caller) in the current RMI call.
    *  <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getClient</code>.
    */
  public static X509Certificate getClientCertificate()
    throws SSLPeerUnverifiedException
    {
      X509Certificate[] peerCerts = getClientCertificateChain();
      return peerCerts[0];
    }

}


//======================================================================
//
// $Log: RMISSLServerSocketFactory.java,v $
// Revision 1.4  1999/02/20 20:02:20  apopovic
// 'connections' and accounting removed; 'getClient' redefined using RMISSLSocket.
//
// Revision 1.3  1999/02/13 15:31:16  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.2  1999/01/27 22:17:29  apopovic
// RMISSLServerSocket.java
//
// Revision 1.1  1999/01/25 08:18:48  apopovic
// Initial Revision
//
