// RMISSLSocketFactory.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: RMISSLSocketFactory.java,v 1.5 1999/02/19 14:27:01 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.rmi;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.rmi.*;
import java.rmi.server.*;
import java.security.cert.*;
import de.tu_darmstadt.sp.ssl.*;
import javax.net.ssl.*;

/**
 * Class RMISSLSocketFactory provides <code>RMISSLSockets</code> for
   RMI calls. The sockets are configured to require peer authentication (default).
 *
 * <p>
 * Every aplication can query the identity of the last call's peer using
 * the <code>getServer</code>, <code>getServerCertificateChain</code> and
 * <code>getServerCertificate</code>.
 * <p>
 * The posibility of querying the last call's peer identity is based on the
 * <em>assumption</em> that the thread performing a rmi call does all the
 * steps of the remote call. This is not specified in the RMI specification,
 * but it is very likely to be true, since RMI is syncronous. Thus,
 * using the <code>getCurrentClientSideSocket</code> method of the
 * <code>RMISSLSocket</code> class, we can obtain the socket used for the
 * last RMI call, which contains all session data, including peer identity
 * and so on.. Life would be easier if RMI would provide the means
 * to reach a Socket used for a remote call..
 *
 * @version	$Revision: 1.5 $
 * @author	Andrei Popovici
 */
public
class RMISSLSocketFactory extends SSLeaySocketFactory
  implements RMIClientSocketFactory, Serializable {


    public RMISSLSocketFactory()
    {
      super();

      setSocketPrototype(new RMISSLSocket(context));
    }



  /** Return the Socket used by the last RMI call in the  current thread. It is the
    * socket used for the connection to the server.
    */
  public static SSLSocket getServer()
    {
      return RMISSLSocket.getCurrentClientSideSocket();
    }

  /** Return the certificate chain used by the peer in the last RMI call of the
    * current thread. The first certificate is the identity of the peer.
    * <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getServer</code>.
    */
  public static X509Certificate[] getServerCertificateChain()
    throws SSLPeerUnverifiedException
    {
      SSLSocket  sock = getServer();
      SSLSession sess = sock.getSession();
      return sess.getPeerCertificateChain();
    }

  /** Return the certificate representing the identity of the peer in the last RMI
    * call of the current thread.
    *  <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>fetRemoteCallSocket</code>.
    */
  public static X509Certificate getServerCertificate()
    throws SSLPeerUnverifiedException
    {
      X509Certificate[] peerCerts = getServerCertificateChain();
      return peerCerts[0];
    }
}


//======================================================================
//
// $Log: RMISSLSocketFactory.java,v $
// Revision 1.5  1999/02/19 14:27:01  apopovic
// Factory creates now 'RMISSLSocket' objects; it uses the RMISSLSocket reccords for the 'getServer' method
//
// Revision 1.4  1999/02/13 15:31:16  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.3  1999/02/12 08:48:40  apopovic
// Is now serializable
//
// Revision 1.2  1999/01/27 22:17:29  apopovic
// RMISSLServerSocket.java
//
// Revision 1.1  1999/01/25 08:18:48  apopovic
// Initial Revision
//
