// RMISecureSocketFactory.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: RMISecureSocketFactory.java,v 1.3 1999/02/13 15:31:16 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.rmi;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.rmi.*;
import java.rmi.server.*;
import java.security.cert.*;
import java.net.*;
import javax.net.ssl.*;

/**
 * Class RMISecureSocketFactory creates secures sockets for RMI connections. It
 * uses SSLeaySockets. Additionaly it adds new functionality which enables the
 * RMI client (caller) or server (calee) to find out each other's identities,
 * security parameters of the connection, etc.
 *
 * @version	$Revision: 1.3 $
 * @author	Andrei Popovici
 */
public
class RMISecureSocketFactory extends RMISocketFactory {

    /** the factory for client sockets */
    protected RMISSLSocketFactory  clientFactory;

    /** the factory for sever sockets */
    protected RMISSLServerSocketFactory serverFactory;

    /** Create a <code>RMISecureSocketFactory</code> which produces
      * secure sockets obtained from <code>clientFactory</code> and
      * <code>serverFactory</code>. The the default initialisation for the
      * two factories is <code>RMISSLSocketFactory</code>, respectively
      * <code>RMISSLSeverSocketFactory</code>.
      * <p>
      * If you want to use other socket types than those produced by
      * the default factories, you have to subclass this class and
      * respecify <code>clientFactory</code> and <code>severFactory</code>
      * in the subclasse's constructors.
      *
      */
  public RMISecureSocketFactory()
  {
    clientFactory = new RMISSLSocketFactory();

    // We want the instantiation to fail if the identity data is not present,
    // so we temporarily change it
    String idReq = System.getProperty("de.tu_darmstadt.sp.ssl.clientIdentityRequired");
    System.setProperty("de.tu_darmstadt.sp.ssl.clientIdentityRequired", "true");
    serverFactory = new RMISSLServerSocketFactory();
    if (idReq != null)
      System.setProperty("de.tu_darmstadt.sp.ssl.clientIdentityRequired",idReq);
    else
      System.setProperty("de.tu_darmstadt.sp.ssl.clientIdentityRequired","false");
  }

    /** Return a Server Socket listening on the specified port. This factory
      * returns <code>RMISSLServerSocket</code> objects. This may change in
      * subclasses.
      */
    public ServerSocket createServerSocket (int port) throws IOException
    {
      return  serverFactory.createServerSocket(port);
    }
    /** Return a socket connected to the specified host on the given port.
      * This factory returns <code>SSLeaySocket</code> objects. This may
      * change in subclasses.
      */
    public Socket createSocket(String host, int port)
      throws IOException
    {
      return clientFactory.createSocket(host,port);
    }


    /** Return the Socket used by the last RMI call in the  current thread. It is the
    * socket used for the connection to the server.
    */
  public static SSLSocket getServer()
    {
      return RMISSLSocketFactory.getServer();
    }

  /** Return the certificate chain used by the peer in the last RMI call of the
    * current thread. The first certificate is the identity of the peer.
    * <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getServer</code>.
    */
  public static X509Certificate[] getServerCertificateChain() throws SSLPeerUnverifiedException
    {
      return RMISSLSocketFactory.getServerCertificateChain();
    }

  /** Return the certificate representing the identity of the peer in the last RMI
    * call of the current thread.
    *  <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>fetRemoteCallSocket</code>.
    */
  public static X509Certificate getServerCertificate() throws SSLPeerUnverifiedException
    {
      return RMISSLSocketFactory.getServerCertificate();
    }
  /** Return the Socket used by the currnt RMI call. It is the socket used for
    * the connection to the client.
    */
  public static SSLSocket getClient()
    {
      return RMISSLServerSocketFactory.getClient();
    }

  /** Return the certificate chain used by the peer(client, caller) in
    * the current RMI call. The first certificate is the identity of the peer.
    * <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getClient</code>.
    */
  public static X509Certificate[] getClientCertificateChain()
    throws SSLPeerUnverifiedException
    {
      return RMISSLServerSocketFactory.getClientCertificateChain();
    }

  /** Return the certificate representing the identity of the peer
    * (client,caller) in the current RMI call.
    *  <p>
    * This method is just "syntactic sugar" since the same information can be obtained
    * from the socket returned by <code>getClient</code>.
    */
  public static X509Certificate getClientCertificate()
    throws SSLPeerUnverifiedException
    {
      return RMISSLServerSocketFactory.getClientCertificate();
    }
}


//======================================================================
//
// $Log: RMISecureSocketFactory.java,v $
// Revision 1.3  1999/02/13 15:31:16  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.2  1999/01/27 22:17:30  apopovic
// RMISSLServerSocket.java
//
// Revision 1.1  1999/01/25 08:18:48  apopovic
// Initial Revision
//
