// RegistryImpl.java -- ITISSL Program
// ITISSL - a Java 2 implementation for Sun's reference SSL API  using SSLeay
// Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// he Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
// Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

// $Id: RegistryImpl.java,v 1.3 1999/02/20 19:58:26 apopovic Exp $
// =====================================================================
//
// (history at end)
//

package de.tu_darmstadt.sp.rmi;

// used packages
import java.lang.*;
import java.util.*;
import java.io.*;
import java.rmi.*;
import java.rmi.server.*;
import java.rmi.registry.*;
import java.net.*;

/**
 * Class RegistryImpl implements the <code>Registry</code> interface using an implementation
 * object obtained from <code>LocateRegistry</code>.
 * <p>
 * It is intended for use with secure sockets. The 'traditional' rmiregistry in jdk
 * has the advantage of using one well defined port and always the same type of sockets.
 * Additionally, the <code>Naming</code> class facilitates the access to the standard registry.
 * In Java 2, a comunity of RMI objects may use their own custom sockets and
 * therefore need their private 'custom' rmiregistry.
 * <p>
 * Simply replacing sun's rmiregistry doesn't work, since the <code>Naming</code>
 * has hardcoded informations to what kind of registry it is
 * speaking.. On the other hand, creating a new 'MyNaming' class
 * wouln't solve the problem, since it will work until somebody needs yet another
 * type of rmiregistry. The 'rmiregistry' provided with JDK 1.2 can communicate using
 * custom sockets. HOwever, if the factories need additional classes, the normal
 * rmiregistry does not work well..
 * <p>
 * This registry simply can register itself on a specified (or anonymous) port
 * on the local registry under a given name like any other remote server. Objects interested
 * in getting a reference to such a secure registry, should use the name to lookup the
 * registry object.  Therefore, a local rmiregistry has to run on the local host
 * before starting this registry implementation.
 * <p>
 * <em>At the present moment, the <code>LocateRegistry.createRegistry</code> doens't seem to
 * work. This is the main motivation in using this registry..</em>
 *
 *
 * @version	$Revision: 1.3 $
 * @author	Andrei Popovici
 */
public
class RegistryImpl extends UnicastRemoteObject implements Registry,Remote {

  private Registry impl;

  /** Create a remote object being a registry, using normal sockets.
    *
    * @param  port the port to listen to
    *
    * @exception MalformedURLException if the name is not a correct URL-like
    * rmi string.
    */
  public RegistryImpl(int port) throws RemoteException, MalformedURLException
  {
    super(port);
    impl = LocateRegistry.createRegistry(0);
  }


  /** Create a remote object being a registry, using custom sockets.
    *
    * @param  port the port to listen to
    * @param csf client socket factory
    * @param ssf server socket factory
    * @exception MalformedURLException <code>name</code> is not a correct
    * URL-like RMI string.
    */
  public RegistryImpl(int port,RMIClientSocketFactory csf,
         RMIServerSocketFactory ssf) throws RemoteException,MalformedURLException
    {
      super(port,csf,ssf);
      impl = LocateRegistry.createRegistry(0);
    }
  /**
    *  Returns a reference, a stub, for the remote object associated with the specified name.
    */
public Remote lookup(String name)
              throws RemoteException,
                     NotBoundException,
                     AccessException
    {
      return impl.lookup(name);
    }

  /**
   * Binds the specified name to a remote object.
   */
  public void bind(String name,
                 Remote obj)
          throws RemoteException,
                 AlreadyBoundException,
                 AccessException
    {
      impl.bind(name,obj);
    }

  /**
    * Destroys the binding for the specified name that is associated with a remote object.
    */
  public void unbind(String name)
            throws RemoteException,
                   NotBoundException,
                   AccessException
    {
       impl.unbind(name);
    }

  /**
    * Rebinds the specified name to a new remote object. Any existing binding for the name is replaced.
    */
  public void rebind(String name,
                   Remote obj)
            throws RemoteException,
                   AccessException
    {
      impl.rebind(name,obj);
    }
  /** Returns an array of the names bound in the registry. The names are URL-formatted strings. The
    * array contains a snapshot of the names present in the registry at the time of the call.
    */
  public String[] list()
              throws RemoteException,
                     AccessException
    {
      return impl.list();
    }
}


//======================================================================
//
// $Log: RegistryImpl.java,v $
// Revision 1.3  1999/02/20 19:58:26  apopovic
// Self rebinding eliminated: now a real implementation of Registry
//
// Revision 1.2  1999/02/13 15:31:17  apopovic
// pakage renaming iti -> de.tu_darmstadt.sp
//
// Revision 1.1  1999/02/11 22:34:30  apopovic
// Initial Revision
//
