/* SSLeayHandle.c -- ITISSL Program */
/* ITISSL - a Java 2 implementation for  Sun's reference SSL API  using SSLeay
 * Copyright (C) 1999 Andrei Popovici (apopovic@iti.informatik.tu-darmstadt.de)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/* $Id: SSLeayHandle.c,v 1.5 1999/02/20 19:50:54 apopovic Exp $
 *----------------------------------------------------------------------
 * Written by Andrei Popovici.
 */

/* Native method implementation of iti.ssl.SSleayHandle.java
 *
 *
 * (history at end)
 */

/** Implementation hints:
  *
  *  The errors are those  last pushed by SSL to its error stack.
  *
  *  A `SSLeayHandle.java' should call
  *   - initSSLeayHandle in the handshake
  *   - close in finalize
  *
  * Note: there are two dirty things in this class: the package protected 'impl' field
  * of the 'java.net.Socket' is used. I hope sun will make the socket implementation
  * protected, so as to be visible in subclasses; In this case, the whole could be
  * made very nice...
  * The  second one is the use of the 'fd' private field of the 'java.io.FileDescriptor'
  * class. THIS IS HACK, and and don't think its going to be done soon.
  */
#include "itissl.h"
#include "SSLeayHandle.h"


/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    setup
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_setup
   (JNIEnv *env, jobject this)

{
 SSL *con = NULL;
 SSL_CTX* context;

 jclass   clsSSLeayHandle;       /** the class iti.ssl.SSLeayHandle */
 jfieldID idSSLeayHandle_connection;
 /** the id of the 'connection' field of the iti.ssl.SSLeayHandle class */

 TRACE("SSLeayHandle_initSSLeayHandle: Entering\n");
 clsSSLeayHandle          = (*env) -> GetObjectClass(env,this);
 idSSLeayHandle_connection= (*env) -> GetFieldID(env,clsSSLeayHandle,CONNECTION_FIELD,"I");

 /** create the SSL structure*/
 context = ITISSL_get_Handle_context(env,this);
 TRACE1("  Using context structure; Value: %d\n",(jint)context);
 if (context == NULL)
     {
	 ITISSLThrowException(env,"javax/net/ssl/SSLException","Context not initialized");
/** 	return; */
     }
 con = SSL_new(context);
 TRACE1("  Initializing Handle (connection %d)\n",(int)con);
 if (con == NULL)
     {
	 ITISSLThrowException(env,"javax/net/ssl/SSLException","Cannot create SSL handle");
	 return;
     }

 /** set the corresponding fields in the handle */
 (*env) -> SetIntField(env,this,idSSLeayHandle_connection,(jint)con);
 TRACE("SSLeayHandle_initSSLeayHandle: Leaving\n");
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    doEnableTransport
 * Signature: (Ljava/net/Socket;I)V
  */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_doEnableTransport
  (JNIEnv *env, jobject this, jobject sock, jint javaMinorVersion)
{
    jclass   clsSocket;             /* corresponds to the java.net.Socket class */
    jfieldID idSocket_impl;         /* the id of the 'impl' field in the Socket class */
    jobject  fieldSocket_impl;      /* the value of the 'impl' field in the Socket class */
    jclass   clsSocketImpl;         /* corresponds to the java.net.SocketImpl class */
    jfieldID idSocketImpl_fd;       /* the id of the 'fd' field of the java.net.SocketImpl class */
    jobject  fieldSocketImpl_fd;    /* the value of the 'fd' field of the java.net.SocketImpl class */
    jclass   clsFileDescriptor;     /* the class java.io.FileDescriptor */
    jfieldID idFileDescriptor_fd;   /* the id of the 'fd' field in the java.io.FileDescriptor class */
    jint     fieldFileDescriptor_fd;/* the value of 'fd' field in the java.io.FileDescriptor class */

    jint plattform_fd;
    SSL *con = NULL;

    TRACE("SSLeayHandle_bindSSLeayHandle: Entering\n");
    /** get the SocketImpl */
    clsSocket                = (*env) -> GetObjectClass(env,sock);
    idSocket_impl            = (*env) -> GetFieldID(env,clsSocket,"impl","Ljava/net/SocketImpl;");
    fieldSocket_impl         = (*env) -> GetObjectField(env,sock,idSocket_impl);

    /** get the Filedescriptor OBJECT */
    clsSocketImpl            = (*env) -> GetObjectClass(env,fieldSocket_impl);
    idSocketImpl_fd          = (*env) -> GetFieldID(env,clsSocketImpl,"fd","Ljava/io/FileDescriptor;");
    fieldSocketImpl_fd       = (*env) -> GetObjectField(env,fieldSocket_impl,idSocketImpl_fd);

    /** get the Plattform file descriptor */
    clsFileDescriptor        = (*env) -> GetObjectClass(env,fieldSocketImpl_fd);
    idFileDescriptor_fd      = (*env) -> GetFieldID(env,clsFileDescriptor,"fd","I");
    fieldFileDescriptor_fd   = (*env) -> GetIntField(env,fieldSocketImpl_fd,idFileDescriptor_fd);
    if (javaMinorVersion == 2)
	plattform_fd = fieldFileDescriptor_fd;
    else
	plattform_fd = fieldFileDescriptor_fd - 1;

    TRACE1("  Got plattform file descriptor; Value: %d\n",plattform_fd);

    con = ITISSL_get_Handle_connection(env,this);
    if (con == NULL)
     {
	 ITISSLThrowException(env,"javax/net/ssl/SSLException","Cannot create SSL handle");
	 return;
     }
    /** associate the file descriptor with the program */
    SSL_set_fd(con,(int)plattform_fd);
    TRACE("SSLeayHandle_bindSSLeayHandle: Leaving\n");
  }




/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    clientHandshake
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_clientHandshake
  (JNIEnv *env, jobject this)
{
    /* variables */
    char* errMsg;
    int   connect_result;
    int   verify_result;
    SSL*  con;
    TRACE("SSLeayHandle_clientHandshake: Entering\n");

    /* get connection and connect */
    con = ITISSL_get_Handle_connection(env,this);
    TRACE1("  Attempting client handshake (connection %d)\n",(int)con);
    connect_result = SSL_connect(con);
    verify_result  = SSL_get_verify_result(con);
    errMsg = X509_verify_cert_error_string(verify_result);

    /* exception if failed connect */
    if (connect_result == -1)
	ITISSLThrowDetailException(env,"javax/net/ssl/SSLException",
				   "Connect failed",errMsg);

    TRACE("SSLeayHandle_clientHandshake: Leaving\n");
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    serverHandshake
 * Signature: ()V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_serverHandshake
  (JNIEnv *env, jobject this)
{
    /* variables */
    int connectSuccess;
    SSL* con;
    TRACE("SSLeayHandle_serverHandshake: Entering\n");

    /* get connection and accept */
    con = ITISSL_get_Handle_connection(env,this);
    TRACE1("  Attempting server handshake (connection %d)\n",(int)con);
    connectSuccess = SSL_accept(con);
    if (  connectSuccess <= 0 )
      ITISSLThrowException(env,"javax/net/ssl/SSLException","Accept failed");

    TRACE("SSLeayHandle_serverHandshake: Leaving\n");
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    writeBytes
 * Signature: ([BII)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_writeBytes
  (JNIEnv *env, jobject this, jbyteArray buf, jint off, jint len)
{
    int result;
    jbyte* buffer;
    SSL* con = ITISSL_get_Handle_connection(env,this);
    if ( con == NULL)
	{
	    ITISSLThrowException(env,"javax/net/ssl/SSLException","unconnected SSLeayHandle");
	    return;
	}


    /* pin down buffer */
    buffer = (*env)->GetByteArrayElements( env, buf, NULL );

    /* write out data from buffer */
    TRACE1("Attempting to write (connection %d) \n",(jint)con);
    result = SSL_write(con, (char*)buffer + off, len );
    TRACE2("%d bytes wrote (connection %d)\n",result,(int)con);
    /* handle errors */
    if (result <= 0 )
      ITISSLThrowException(env,"java/io/IOException","write failed");
    else if (result != len)
      ITISSLThrowException(env,"java/io/IOException","not all bytes written");

    /* free resources */
    (*env)->ReleaseByteArrayElements( env, buf, buffer, 0 );
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    readBytes
 * Signature: ([BII)I
 */
JNIEXPORT jint JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_readBytes
  (JNIEnv *env, jobject this, jbyteArray buf, jint off, jint len)
{
        int result;
    jbyte* buffer;
    SSL* con = ITISSL_get_Handle_connection(env,this);
    if ( con == NULL)
	{

	  ITISSLThrowException(env,"javax/net/ssl/SSLException","unconnected SSLeayHandle");
	  return -1;
	}

    /* pin down buffer */
    buffer = (*env)->GetByteArrayElements( env, buf, NULL );

    /* write out data from buffer */
    TRACE3("Attempting to read %d bytes from offset %d (connection %d)\n",len,off,(int)con);
    result = SSL_read(con, (char*)buffer + off, len );
    TRACE2("%d bytes read (connection %d)\n",result,(int)con);

    /* handle errors */
    if (result <= 0 )
      ITISSLThrowException(env,"java/io/IOException","read failed");

    /* free resources */
    (*env)->ReleaseByteArrayElements( env, buf, buffer, 0 );
    TRACE1("Read completed (connection %d)\n", (int)con);
    if (result <= 0 )
      return -1;
    else
      return result;
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    close
 * Signature: ()I
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_close
  (JNIEnv * env, jobject this)
{
    /* variables */
    jclass   clsSSLeayHandle;
    jfieldID idSSLeayHandle_connection;
    SSL*     con;
    TRACE("SSLeayHandle_close: Entering\n");

    /* get connection */
    clsSSLeayHandle = (*env) -> GetObjectClass(env,this);
    idSSLeayHandle_connection = (*env) -> GetFieldID(env,clsSSLeayHandle,
						     CONNECTION_FIELD,"I");
    con  = ITISSL_get_Handle_connection(env,this);


    /** set the corresponding field in the handle to NULL*/

    (*env) -> SetIntField(env,this,idSSLeayHandle_connection,(jint)NULL);


    /** Shutdown con */
    TRACE1("  Shutdown (connection %d)\n",(int)con);
    if ( con != NULL )
      SSL_shutdown(con);

    /** free the connection */
    if ( con != NULL )
      SSL_free(con);
    TRACE("  Free connection\n");
    /** a cute SSLeayHandle should now close the socket;
      * Anyway, writeBytes doesn't work any more at this point, so at least
      * we cannot send unencrypted information..
      */
    TRACE("SSLeayHandle_close: Leaving");
}



/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    setVerificationMode
 * Signature: (I)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_setVerificationPolicy
  (JNIEnv *env, jobject this, jboolean isClient, jboolean verifyPeer)
{
    /* variables */
    SSL* con ;
    int mode = SSL_VERIFY_NONE;
    TRACE("SSLeayHandle_setVerificationPolicy: Entering\n");


    /* set SSL options */
    if (verifyPeer)
      {
	if (isClient)
	  {
	    mode = SSL_VERIFY_PEER;
	  }
	else
	  {
	    mode = SSL_VERIFY_PEER |
	      SSL_VERIFY_FAIL_IF_NO_PEER_CERT |
	      SSL_VERIFY_CLIENT_ONCE;
	  }
      }

    con = ITISSL_get_Handle_connection(env,this);
    TRACE3("  connection=%d, isClient =%d veriftyPeer=%d \n",(jint)con,
	   isClient,verifyPeer);
    SSL_set_verify(con,mode,con->verify_callback);

    TRACE("SSLeayHandle_setVerificationPolicy: Leaving\n");
}
/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    getEnabledCipherSuites
 * Signature: ()[Ljava/lang/String;
 */
JNIEXPORT jobjectArray JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_getEnabledCipherSuites
(JNIEnv* env, jobject this)
{
    /* variables */
  STACK* ciphers;
  jarray result;
  SSL*   con;
  TRACE("SSLeayHandle_getEnabledCipherSuites: Entering\n");

   con = ITISSL_get_Handle_connection(env,this);
   ciphers = SSL_get_ciphers(con);
   result = ITISSL_ciphers_to_jstringarr(env,ciphers);

   TRACE("SSLeayHandle_getEnabledCipherSuites: Leaving\n");
   return result;

}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    doSetEnabledCipherSuites
 * Signature: (Ljava/lang/String;)V
 */
JNIEXPORT void JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_doSetEnabledCipherSuites
(JNIEnv* env, jobject this, jstring ciphers)
{
    /* variables */
 char *cipher_list;
 SSL*  con;
 TRACE("SSLeayHandle_doSetEnabledCipherSuites: Entering\n");

 con=ITISSL_get_Handle_connection(env,this);
 cipher_list  = (char*)(*env)->GetStringUTFChars(env, ciphers, 0);
 SSL_set_cipher_list(con,cipher_list);
 (*env) -> ReleaseStringUTFChars(env, ciphers, cipher_list);

 TRACE("SSLeayHandle_doSetEnabledCipherSuites: Leaving\n");
}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    doGetSession
 * Signature: ()I
 */
JNIEXPORT jint JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_doGetSession
 (JNIEnv* env, jobject this)
{
    /* variables */
  SSL* con;
  SSL_SESSION* session;
  TRACE("SSLeayHandle_doGetSession: Entering\n");

  con      = ITISSL_get_Handle_connection(env,this);
  session = SSL_get_session(con);

  TRACE("SSLeayHandle_doGetSession: Leaving\n");
  return (jint)session;

}

/*
 * Class:    de_tu_0005fdarmstadt_sp_ssl_SSLeayHandle
 * Method:    getPeerCommonName
 * Signature: ()Ljava/lang/String;
 */
JNIEXPORT jstring JNICALL Java_de_tu_1darmstadt_sp_ssl_SSLeayHandle_getPeerDN
  (JNIEnv *env, jobject this)
{
    /* variables */
    X509*    peerCert;
    char*    subjectName;
    jstring  jdistName;
    SSL*     con;

    TRACE("SSLeayHandle_getPeerDN: Entering\n");
    con = ITISSL_get_Handle_connection(env,this);
    TRACE1("Attempt to get Peer DN (connection %d)\n",(int)con);
    peerCert    = SSL_get_peer_certificate(con);
    if ( peerCert == NULL)
      {
	ITISSLThrowException(env,"javax/net/ssl/SSLException","Cannot get peer certificate");
	return NULL;
      }

    subjectName = X509_NAME_oneline(X509_get_subject_name(peerCert),NULL,0);

    jdistName =(*env)->NewStringUTF(env,subjectName);
    TRACE2("Peer DN is %s (connection %d)\n",subjectName,(int)con);
    TRACE("SSLeayHandle_getPeerDN: Leaving\n");
    return jdistName;
}


/*======================================================================
 *
 * $Log: SSLeayHandle.c,v $
 * Revision 1.5  1999/02/20 19:50:54  apopovic
 * Renamings from iti to de_tu_darmstasdt..
 *
 * Revision 1.4  1999/02/05 10:26:23  apopovic
 * doEnableTransport sets now plattform_fd according to javaMinorVersion
 *
 * Revision 1.3  1999/01/27 18:42:43  apopovic
 * Documentation spell-check
 *
 * Revision 1.2  1999/01/22 10:14:32  apopovic
 * Documentation and Debug messages improved.
 * Inconsistecies in Exception throwing removed.
 *
 * Revision 1.1  1999/01/08 10:41:12  apopovic
 * Initial Revision
 *
 *
 */
